/**
 * Copyright (c) 2013-present, Facebook, Inc.
 *
 * This source code is licensed under the MIT license found in the
 * LICENSE file in the root directory of this source tree.
 *
 * 
 * @format
 */

'use strict';

var _extends3 = _interopRequireDefault(require('babel-runtime/helpers/extends'));

var _classCallCheck3 = _interopRequireDefault(require('babel-runtime/helpers/classCallCheck'));

var _possibleConstructorReturn3 = _interopRequireDefault(require('babel-runtime/helpers/possibleConstructorReturn'));

var _inherits3 = _interopRequireDefault(require('babel-runtime/helpers/inherits'));

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { 'default': obj }; }

var _require = require('relay-runtime'),
    deepFreeze = _require.deepFreeze;

var NETWORK_ONLY = 'NETWORK_ONLY';
var STORE_THEN_NETWORK = 'STORE_THEN_NETWORK';
var DataFromEnum = {
  NETWORK_ONLY: NETWORK_ONLY,
  STORE_THEN_NETWORK: STORE_THEN_NETWORK
};

/**
 * @public
 *
 * Orchestrates fetching and rendering data for a single view or view hierarchy:
 * - Fetches the query/variables using the given network implementation.
 * - Normalizes the response(s) to that query, publishing them to the given
 *   store.
 * - Renders the pending/fail/success states with the provided render function.
 * - Subscribes for updates to the root data and re-renders with any changes.
 */
var ReactRelayQueryRenderer = function (_React$Component) {
  (0, _inherits3['default'])(ReactRelayQueryRenderer, _React$Component);

  // TODO T25783053 Update this component to use the new React context API,
  // Once we have confirmed that it's okay to raise min React version to 16.3.
  function ReactRelayQueryRenderer(props, context) {
    (0, _classCallCheck3['default'])(this, ReactRelayQueryRenderer);

    // Callbacks are attached to the current instance and shared with static
    // lifecyles by bundling with state. This is okay to do because the
    // callbacks don't change in reaction to props. However we should not
    // "leak" them before mounting (since we would be unable to clean up). For
    // that reason, we define them as null initially and fill them in after
    // mounting to avoid leaking memory.
    var _this = (0, _possibleConstructorReturn3['default'])(this, _React$Component.call(this, props, context));

    _this._relayContext = {
      // $FlowFixMe TODO t16225453 QueryRenderer works with old+new environment.
      environment: _this.props.environment,
      variables: _this.props.variables
    };
    var retryCallbacks = {
      handleDataChange: null,
      handleRetryAfterError: null
    };

    var queryFetcher = new (require('./ReactRelayQueryFetcher'))();

    _this.state = (0, _extends3['default'])({
      prevPropsEnvironment: props.environment,
      prevPropsVariables: props.variables,
      prevQuery: props.query,
      queryFetcher: queryFetcher,
      retryCallbacks: retryCallbacks
    }, fetchQueryAndComputeStateFromProps(props, queryFetcher, retryCallbacks));
    return _this;
  }

  ReactRelayQueryRenderer.getDerivedStateFromProps = function getDerivedStateFromProps(nextProps, prevState) {
    if (prevState.prevQuery !== nextProps.query || prevState.prevPropsEnvironment !== nextProps.environment || !require('fbjs/lib/areEqual')(prevState.prevPropsVariables, nextProps.variables)) {
      return (0, _extends3['default'])({
        prevQuery: nextProps.query,
        prevPropsEnvironment: nextProps.environment,
        prevPropsVariables: nextProps.variables
      }, fetchQueryAndComputeStateFromProps(nextProps, prevState.queryFetcher, prevState.retryCallbacks));
    }

    return null;
  };

  ReactRelayQueryRenderer.prototype.componentDidMount = function componentDidMount() {
    var _this2 = this;

    var _state = this.state,
        retryCallbacks = _state.retryCallbacks,
        queryFetcher = _state.queryFetcher;


    retryCallbacks.handleDataChange = function (params) {
      var error = params.error == null ? null : params.error;
      var snapshot = params.snapshot == null ? null : params.snapshot;

      _this2.setState(function (prevState) {
        // Don't update state if nothing has changed.
        if (snapshot === prevState.snapshot && error === prevState.error) {
          return null;
        }
        return {
          renderProps: getRenderProps(error, snapshot, queryFetcher, retryCallbacks),
          snapshot: snapshot
        };
      });
    };

    retryCallbacks.handleRetryAfterError = function (error) {
      return _this2.setState({ renderProps: getLoadingRenderProps() });
    };

    // Re-initialize the ReactRelayQueryFetcher with callbacks.
    // If data has changed since constructions, this will re-render.
    if (this.props.query) {
      queryFetcher.setOnDataChange(retryCallbacks.handleDataChange);
    }
  };

  ReactRelayQueryRenderer.prototype.componentWillUnmount = function componentWillUnmount() {
    this.state.queryFetcher.dispose();
  };

  ReactRelayQueryRenderer.prototype.shouldComponentUpdate = function shouldComponentUpdate(nextProps, nextState) {
    return nextProps.render !== this.props.render || nextState.renderProps !== this.state.renderProps;
  };

  ReactRelayQueryRenderer.prototype.getChildContext = function getChildContext() {
    return {
      relay: this._relayContext
    };
  };

  ReactRelayQueryRenderer.prototype.render = function render() {
    var _state2 = this.state,
        relayContextEnvironment = _state2.relayContextEnvironment,
        relayContextVariables = _state2.relayContextVariables,
        renderProps = _state2.renderProps;

    // HACK Mutate the context.relay object before updating children,
    // To account for any changes made by static gDSFP.
    // Updating this value in gDSFP would be less safe, since props changes
    // could be interrupted and we might re-render based on a setState call.
    // Child containers rely on context.relay being mutated (also for gDSFP).
    // $FlowFixMe TODO t16225453 QueryRenderer works with old+new environment.

    this._relayContext.environment = relayContextEnvironment;
    this._relayContext.variables = relayContextVariables;

    // Note that the root fragment results in `renderProps.props` is already
    // frozen by the store; this call is to freeze the renderProps object and
    // error property if set.
    if (process.env.NODE_ENV !== 'production') {
      deepFreeze(renderProps);
    }
    return this.props.render(renderProps);
  };

  return ReactRelayQueryRenderer;
}(require('react').Component);

ReactRelayQueryRenderer.childContextTypes = {
  relay: require('./RelayPropTypes').Relay
};


function getLoadingRenderProps() {
  return {
    error: null,
    props: null, // `props: null` indicates that the data is being fetched (i.e. loading)
    retry: null
  };
}

function getEmptyRenderProps() {
  return {
    error: null,
    props: {}, // `props: {}` indicates no data available
    retry: null
  };
}

function getRenderProps(error, snapshot, queryFetcher, retryCallbacks) {
  return {
    error: error ? error : null,
    props: snapshot ? snapshot.data : null,
    retry: function retry() {
      var syncSnapshot = queryFetcher.retry();
      if (syncSnapshot && typeof retryCallbacks.handleDataChange === 'function') {
        retryCallbacks.handleDataChange({ snapshot: syncSnapshot });
      } else if (error && typeof retryCallbacks.handleRetryAfterError === 'function') {
        // If retrying after an error and no synchronous result available,
        // reset the render props
        retryCallbacks.handleRetryAfterError(error);
      }
    }
  };
}

function fetchQueryAndComputeStateFromProps(props, queryFetcher, retryCallbacks) {
  var environment = props.environment,
      query = props.query,
      variables = props.variables;

  if (query) {
    // $FlowFixMe TODO t16225453 QueryRenderer works with old+new environment.
    var genericEnvironment = environment;

    var _genericEnvironment$u = genericEnvironment.unstable_internal,
        createOperationSelector = _genericEnvironment$u.createOperationSelector,
        getRequest = _genericEnvironment$u.getRequest;

    var request = getRequest(query);
    var operation = createOperationSelector(request, variables);

    try {
      var storeSnapshot = props.dataFrom === STORE_THEN_NETWORK ? queryFetcher.lookupInStore(genericEnvironment, operation) : null;
      var querySnapshot = queryFetcher.fetch({
        cacheConfig: props.cacheConfig,
        dataFrom: props.dataFrom,
        environment: genericEnvironment,
        onDataChange: retryCallbacks.handleDataChange,
        operation: operation
      });
      // Use network data first, since it may be fresher
      var _snapshot = querySnapshot || storeSnapshot;
      if (!_snapshot) {
        return {
          error: null,
          relayContextEnvironment: environment,
          relayContextVariables: operation.variables,
          renderProps: getLoadingRenderProps(),
          snapshot: null
        };
      }

      return {
        error: null,
        relayContextEnvironment: environment,
        relayContextVariables: operation.variables,
        renderProps: getRenderProps(null, _snapshot, queryFetcher, retryCallbacks),
        snapshot: _snapshot
      };
    } catch (error) {
      return {
        error: error,
        relayContextEnvironment: environment,
        relayContextVariables: operation.variables,
        renderProps: getRenderProps(error, null, queryFetcher, retryCallbacks),
        snapshot: null
      };
    }
  } else {
    queryFetcher.dispose();

    return {
      error: null,
      relayContextEnvironment: environment,
      relayContextVariables: variables,
      renderProps: getEmptyRenderProps()
    };
  }
}

module.exports = ReactRelayQueryRenderer;