'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.Operation = exports.RendererProcess = exports.Worker = undefined;

var _extends = Object.assign || function (target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i]; for (var key in source) { if (Object.prototype.hasOwnProperty.call(source, key)) { target[key] = source[key]; } } } return target; };

var _path = require('path');

var _path2 = _interopRequireDefault(_path);

var _querystring = require('querystring');

var _querystring2 = _interopRequireDefault(_querystring);

var _electron = require('electron');

var _eventKit = require('event-kit');

var _helpers = require('./helpers');

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

const { BrowserWindow } = _electron.remote;
class WorkerManager {

  static getInstance() {
    if (!this.instance) {
      this.instance = new WorkerManager();
    }
    return this.instance;
  }

  static reset(force) {
    if (this.instance) {
      this.instance.destroy(force);
    }
    this.instance = null;
  }

  constructor() {
    (0, _helpers.autobind)(this, 'onDestroyed', 'onCrashed', 'onSick');

    this.workers = new Set();
    this.activeWorker = null;
    this.createNewWorker();
  }

  isReady() {
    return this.activeWorker.isReady();
  }

  request(data) {
    if (this.destroyed) {
      throw new Error('Worker is destroyed');
    }
    let operation;
    const requestPromise = new Promise((resolve, reject) => {
      operation = new Operation(data, resolve, reject);
      return this.activeWorker.executeOperation(operation);
    });
    operation.setPromise(requestPromise);
    return {
      cancel: () => this.activeWorker.cancelOperation(operation),
      promise: requestPromise
    };
  }

  createNewWorker({ operationCountLimit } = { operationCountLimit: 10 }) {
    if (this.destroyed) {
      return;
    }
    this.activeWorker = new Worker({
      operationCountLimit,
      onDestroyed: this.onDestroyed,
      onCrashed: this.onCrashed,
      onSick: this.onSick
    });
    this.workers.add(this.activeWorker);
  }

  onDestroyed(destroyedWorker) {
    this.workers.delete(destroyedWorker);
  }

  onCrashed(crashedWorker) {
    if (crashedWorker === this.getActiveWorker()) {
      this.createNewWorker({ operationCountLimit: crashedWorker.getOperationCountLimit() });
    }
    crashedWorker.getRemainingOperations().forEach(operation => this.activeWorker.executeOperation(operation));
  }

  onSick(sickWorker) {
    if (!atom.inSpecMode()) {
      // eslint-disable-next-line no-console
      console.warn(`Sick worker detected.
        operationCountLimit: ${sickWorker.getOperationCountLimit()},
        completed operation count: ${sickWorker.getCompletedOperationCount()}`);
    }
    const operationCountLimit = this.calculateNewOperationCountLimit(sickWorker);
    return this.createNewWorker({ operationCountLimit });
  }

  calculateNewOperationCountLimit(lastWorker) {
    let operationCountLimit = 10;
    if (lastWorker.getOperationCountLimit() >= lastWorker.getCompletedOperationCount()) {
      operationCountLimit = Math.min(lastWorker.getOperationCountLimit() * 2, 100);
    }
    return operationCountLimit;
  }

  getActiveWorker() {
    return this.activeWorker;
  }

  getReadyPromise() {
    return this.activeWorker.getReadyPromise();
  }

  destroy(force) {
    this.destroyed = true;
    this.workers.forEach(worker => worker.destroy(force));
  }
}

exports.default = WorkerManager;
WorkerManager.instance = null;
class Worker {

  constructor({ operationCountLimit, onSick, onCrashed, onDestroyed }) {
    (0, _helpers.autobind)(this, 'handleDataReceived', 'onOperationComplete', 'handleCancelled', 'handleExecStarted', 'handleSpawnError', 'handleStdinError', 'handleSick', 'handleCrashed');

    this.operationCountLimit = operationCountLimit;
    this.onSick = onSick;
    this.onCrashed = onCrashed;
    this.onDestroyed = onDestroyed;

    this.operationsById = new Map();
    this.completedOperationCount = 0;
    this.sick = false;

    this.rendererProcess = new RendererProcess({
      loadUrl: this.getLoadUrl(operationCountLimit),
      onData: this.handleDataReceived,
      onCancelled: this.handleCancelled,
      onExecStarted: this.handleExecStarted,
      onSpawnError: this.handleSpawnError,
      onStdinError: this.handleStdinError,
      onSick: this.handleSick,
      onCrashed: this.handleCrashed,
      onDestroyed: this.destroy
    });
  }

  isReady() {
    return this.rendererProcess.isReady();
  }

  getLoadUrl(operationCountLimit) {
    const htmlPath = _path2.default.join((0, _helpers.getPackageRoot)(), 'lib', 'renderer.html');
    const rendererJsPath = _path2.default.join((0, _helpers.getPackageRoot)(), 'lib', 'worker.js');
    const qs = _querystring2.default.stringify({
      js: rendererJsPath,
      managerWebContentsId: this.getWebContentsId(),
      operationCountLimit,
      channelName: Worker.channelName
    });
    return `file://${htmlPath}?${qs}`;
  }

  getWebContentsId() {
    return _electron.remote.getCurrentWebContents().id;
  }

  executeOperation(operation) {
    this.operationsById.set(operation.id, operation);
    operation.onComplete(this.onOperationComplete);
    return this.rendererProcess.executeOperation(operation);
  }

  cancelOperation(operation) {
    return this.rendererProcess.cancelOperation(operation);
  }

  handleDataReceived({ id, results }) {
    const operation = this.operationsById.get(id);
    operation.complete(results, data => {
      const { timing } = data;
      const totalInternalTime = timing.execTime + timing.spawnTime;
      const ipcTime = operation.getExecutionTime() - totalInternalTime;
      data.timing.ipcTime = ipcTime;
      return data;
    });
  }

  onOperationComplete(operation) {
    this.completedOperationCount++;
    this.operationsById.delete(operation.id);

    if (this.sick && this.operationsById.size === 0) {
      this.destroy();
    }
  }

  handleCancelled({ id }) {
    const operation = this.operationsById.get(id);
    if (operation) {
      // handleDataReceived() can be received before handleCancelled()
      operation.wasCancelled();
    }
  }

  handleExecStarted({ id }) {
    const operation = this.operationsById.get(id);
    operation.setInProgress();
  }

  handleSpawnError({ id, err }) {
    const operation = this.operationsById.get(id);
    operation.error(err);
  }

  handleStdinError({ id, stdin, err }) {
    const operation = this.operationsById.get(id);
    operation.error(err);
  }

  handleSick() {
    this.sick = true;
    this.onSick(this);
  }

  handleCrashed() {
    this.onCrashed(this);
    this.destroy();
  }

  getOperationCountLimit() {
    return this.operationCountLimit;
  }

  getCompletedOperationCount() {
    return this.completedOperationCount;
  }

  getRemainingOperations() {
    return Array.from(this.operationsById.values());
  }

  getPid() {
    return this.rendererProcess.getPid();
  }

  getReadyPromise() {
    return this.rendererProcess.getReadyPromise();
  }

  async destroy(force) {
    this.onDestroyed(this);
    if (this.operationsById.size > 0 && !force) {
      const remainingOperationPromises = this.getRemainingOperations().map(operation => operation.getPromise().catch(() => null));
      await Promise.all(remainingOperationPromises);
    }
    this.rendererProcess.destroy();
  }
}

exports.Worker = Worker; /*
                         Sends operations to renderer processes
                         */

Worker.channelName = 'github:renderer-ipc';
class RendererProcess {
  constructor({ loadUrl,
    onDestroyed, onCrashed, onSick, onData, onCancelled, onSpawnError, onStdinError, onExecStarted }) {
    (0, _helpers.autobind)(this, 'handleDestroy');
    this.onDestroyed = onDestroyed;
    this.onCrashed = onCrashed;
    this.onSick = onSick;
    this.onData = onData;
    this.onCancelled = onCancelled;
    this.onSpawnError = onSpawnError;
    this.onStdinError = onStdinError;
    this.onExecStarted = onExecStarted;

    this.win = new BrowserWindow({ show: !!process.env.ATOM_GITHUB_SHOW_RENDERER_WINDOW });
    this.webContents = this.win.webContents;
    // this.webContents.openDevTools();

    this.emitter = new _eventKit.Emitter();
    this.subscriptions = new _eventKit.CompositeDisposable();
    this.registerListeners();

    this.win.loadURL(loadUrl);
    this.win.webContents.on('crashed', this.handleDestroy);
    this.win.webContents.on('destroyed', this.handleDestroy);
    this.subscriptions.add(new _eventKit.Disposable(() => {
      if (!this.win.isDestroyed()) {
        this.win.webContents.removeListener('crashed', this.handleDestroy);
        this.win.webContents.removeListener('destroyed', this.handleDestroy);
        this.win.destroy();
      }
    }), this.emitter);

    this.ready = false;
    this.readyPromise = new Promise(resolve => {
      this.resolveReady = resolve;
    });
  }

  isReady() {
    return this.ready;
  }

  handleDestroy(...args) {
    this.destroy();
    this.onCrashed(...args);
  }

  registerListeners() {
    const handleMessages = (event, { sourceWebContentsId, type, data }) => {
      if (sourceWebContentsId === this.win.webContents.id) {
        this.emitter.emit(type, data);
      }
    };

    _electron.ipcRenderer.on(Worker.channelName, handleMessages);
    this.emitter.on('renderer-ready', ({ pid }) => {
      this.pid = pid;
      this.ready = true;
      this.resolveReady();
    });
    this.emitter.on('git-data', this.onData);
    this.emitter.on('git-cancelled', this.onCancelled);
    this.emitter.on('git-spawn-error', this.onSpawnError);
    this.emitter.on('git-stdin-error', this.onStdinError);
    this.emitter.on('slow-spawns', this.onSick);

    // not currently used to avoid clogging up ipc channel
    // keeping it around as it's potentially useful for avoiding duplicate write operations upon renderer crashing
    this.emitter.on('exec-started', this.onExecStarted);

    this.subscriptions.add(new _eventKit.Disposable(() => _electron.ipcRenderer.removeListener(Worker.channelName, handleMessages)));
  }

  executeOperation(operation) {
    return operation.execute(payload => {
      if (this.destroyed) {
        return null;
      }
      return this.webContents.send(Worker.channelName, {
        type: 'git-exec',
        data: payload
      });
    });
  }

  cancelOperation(operation) {
    return operation.cancel(payload => {
      if (this.destroyed) {
        return null;
      }
      return this.webContents.send(Worker.channelName, {
        type: 'git-cancel',
        data: payload
      });
    });
  }

  getPid() {
    return this.pid;
  }

  getReadyPromise() {
    return this.readyPromise;
  }

  destroy() {
    this.destroyed = true;
    this.subscriptions.dispose();
  }
}

exports.RendererProcess = RendererProcess;
class Operation {

  constructor(data, resolve, reject) {
    this.id = Operation.id++;
    this.data = data;
    this.resolve = resolve;
    this.reject = reject;
    this.promise = null;
    this.cancellationResolve = () => {};
    this.startTime = null;
    this.endTime = null;
    this.status = Operation.status.PENDING;
    this.results = null;
    this.emitter = new _eventKit.Emitter();
  }

  onComplete(cb) {
    return this.emitter.on('complete', cb);
  }

  setPromise(promise) {
    this.promise = promise;
  }

  getPromise() {
    return this.promise;
  }

  setInProgress() {
    // after exec has been called but before results a received
    this.status = Operation.status.INPROGRESS;
  }

  getExecutionTime() {
    if (!this.startTime || !this.endTime) {
      return NaN;
    } else {
      return this.endTime - this.startTime;
    }
  }

  complete(results, mutate = data => data) {
    this.endTime = performance.now();
    this.results = results;
    this.resolve(mutate(results));
    this.cancellationResolve();
    this.status = Operation.status.COMPLETE;
    this.emitter.emit('complete', this);
    this.emitter.dispose();
  }

  wasCancelled() {
    this.status = Operation.status.CANCELLED;
    this.cancellationResolve();
  }

  error(results) {
    this.endTime = performance.now();
    const err = new Error(results.message, results.fileName, results.lineNumber);
    err.stack = results.stack;
    this.reject(err);
  }

  execute(execFn) {
    this.startTime = performance.now();
    return execFn(_extends({}, this.data, { id: this.id }));
  }

  cancel(execFn) {
    return new Promise(resolve => {
      this.status = Operation.status.CANCELLING;
      this.cancellationResolve = resolve;
      execFn({ id: this.id });
    });
  }
}
exports.Operation = Operation;
Operation.status = {
  PENDING: Symbol('pending'),
  INPROGRESS: Symbol('in-progress'),
  COMPLETE: Symbol('complete'),
  CANCELLING: Symbol('cancelling'),
  CANCELLED: Symbol('canceled')
};
Operation.id = 0;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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