'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _helpers = require('./helpers');

const refreshMapPerUniqueId = new WeakMap();

class PeriodicRefresher {
  static getRefreshMap(uniqueId) {
    let refreshMap = refreshMapPerUniqueId.get(uniqueId);
    if (!refreshMap) {
      refreshMap = new Map();
      refreshMapPerUniqueId.set(uniqueId, refreshMap);
    }

    return refreshMap;
  }

  constructor(uniqueId, options) {
    (0, _helpers.autobind)(this, 'refreshNow');

    this.options = options;
    this._refreshesPerId = PeriodicRefresher.getRefreshMap(uniqueId);
  }

  start() {
    if (!this._timer) {
      this._scheduleNext();
    }
  }

  _clearTimer() {
    if (this._timer) {
      clearTimeout(this._timer);
      delete this._timer;
    }
  }

  _scheduleNext() {
    this._timer = setTimeout(this.refreshNow, this.options.interval());
  }

  refreshNow(force = false) {
    const currentId = this.options.getCurrentId();
    const lastRefreshForId = this._refreshesPerId.get(currentId) || 0;
    const delta = performance.now() - lastRefreshForId;
    if (force || delta > this.options.minimumIntervalPerId) {
      this._refreshesPerId.set(currentId, performance.now());
      this.options.refresh();
    }
    // Schedule another refresh if one is already scheduled
    if (this._timer) {
      this._clearTimer();
      this._scheduleNext();
    }
  }

  destroy() {
    this._clearTimer();
  }
}
exports.default = PeriodicRefresher;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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