'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _commit = require('../commit');

var _branchSet = require('../branch-set');

var _branchSet2 = _interopRequireDefault(_branchSet);

var _remoteSet = require('../remote-set');

var _remoteSet2 = _interopRequireDefault(_remoteSet);

var _operationStates = require('../operation-states');

var _multiFilePatch = require('../patch/multi-file-patch');

var _multiFilePatch2 = _interopRequireDefault(_multiFilePatch);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/**
 * Map of registered subclasses to allow states to transition to one another without circular dependencies.
 * Subclasses of State should call `State.register` to add themselves here.
 */
const stateConstructors = new Map();

/**
 * Base class for Repository states. Implements default "null" behavior.
 */
class State {
  constructor(repository) {
    this.repository = repository;
  }

  static register(Subclass) {
    stateConstructors.set(Subclass.name, Subclass);
  }

  // This state has just been entered. Perform any asynchronous initialization that needs to occur.
  start() {
    return Promise.resolve();
  }

  // State probe predicates ////////////////////////////////////////////////////////////////////////////////////////////
  // Allow external callers to identify which state a Repository is in if necessary.

  isLoadingGuess() {
    return false;
  }

  isAbsentGuess() {
    return false;
  }

  isAbsent() {
    return false;
  }

  isLoading() {
    return false;
  }

  isEmpty() {
    return false;
  }

  isPresent() {
    return false;
  }

  isTooLarge() {
    return false;
  }

  isDestroyed() {
    return false;
  }

  // Behavior probe predicates /////////////////////////////////////////////////////////////////////////////////////////
  // Determine specific rendering behavior based on the current state.

  isUndetermined() {
    return false;
  }

  showGitTabInit() {
    return false;
  }

  showGitTabInitInProgress() {
    return false;
  }

  showGitTabLoading() {
    return false;
  }

  showStatusBarTiles() {
    return false;
  }

  hasDirectory() {
    return true;
  }

  // Lifecycle actions /////////////////////////////////////////////////////////////////////////////////////////////////
  // These generally default to rejecting a Promise with an error.

  init() {
    return unsupportedOperationPromise(this, 'init');
  }

  clone(remoteUrl) {
    return unsupportedOperationPromise(this, 'clone');
  }

  destroy() {
    return this.transitionTo('Destroyed');
  }

  refresh() {
    // No-op
  }

  observeFilesystemChange(events) {
    this.repository.refresh();
  }

  updateCommitMessageAfterFileSystemChange() {
    // this is only used in unit tests, we don't need no stinkin coverage
    /* istanbul ignore next */
    this.repository.refresh();
  }

  // Git operations ////////////////////////////////////////////////////////////////////////////////////////////////////
  // These default to rejecting a Promise with an error stating that the operation is not supported in the current
  // state.

  // Staging and unstaging

  stageFiles(paths) {
    return unsupportedOperationPromise(this, 'stageFiles');
  }

  unstageFiles(paths) {
    return unsupportedOperationPromise(this, 'unstageFiles');
  }

  stageFilesFromParentCommit(paths) {
    return unsupportedOperationPromise(this, 'stageFilesFromParentCommit');
  }

  applyPatchToIndex(patch) {
    return unsupportedOperationPromise(this, 'applyPatchToIndex');
  }

  applyPatchToWorkdir(patch) {
    return unsupportedOperationPromise(this, 'applyPatchToWorkdir');
  }

  // Committing

  commit(message, options) {
    return unsupportedOperationPromise(this, 'commit');
  }

  // Merging

  merge(branchName) {
    return unsupportedOperationPromise(this, 'merge');
  }

  abortMerge() {
    return unsupportedOperationPromise(this, 'abortMerge');
  }

  checkoutSide(side, paths) {
    return unsupportedOperationPromise(this, 'checkoutSide');
  }

  mergeFile(oursPath, commonBasePath, theirsPath, resultPath) {
    return unsupportedOperationPromise(this, 'mergeFile');
  }

  writeMergeConflictToIndex(filePath, commonBaseSha, oursSha, theirsSha) {
    return unsupportedOperationPromise(this, 'writeMergeConflictToIndex');
  }

  // Checkout

  checkout(revision, options = {}) {
    return unsupportedOperationPromise(this, 'checkout');
  }

  checkoutPathsAtRevision(paths, revision = 'HEAD') {
    return unsupportedOperationPromise(this, 'checkoutPathsAtRevision');
  }

  // Reset

  undoLastCommit() {
    return unsupportedOperationPromise(this, 'undoLastCommit');
  }

  // Remote interactions

  fetch(branchName) {
    return unsupportedOperationPromise(this, 'fetch');
  }

  pull(branchName) {
    return unsupportedOperationPromise(this, 'pull');
  }

  push(branchName) {
    return unsupportedOperationPromise(this, 'push');
  }

  // Configuration

  setConfig(option, value, { replaceAll } = {}) {
    return unsupportedOperationPromise(this, 'setConfig');
  }

  unsetConfig(option) {
    return unsupportedOperationPromise(this, 'unsetConfig');
  }

  // Direct blob interactions

  createBlob({ filePath, stdin } = {}) {
    return unsupportedOperationPromise(this, 'createBlob');
  }

  expandBlobToFile(absFilePath, sha) {
    return unsupportedOperationPromise(this, 'expandBlobToFile');
  }

  // Discard history

  createDiscardHistoryBlob() {
    return unsupportedOperationPromise(this, 'createDiscardHistoryBlob');
  }

  updateDiscardHistory() {
    return unsupportedOperationPromise(this, 'updateDiscardHistory');
  }

  storeBeforeAndAfterBlobs(filePaths, isSafe, destructiveAction, partialDiscardFilePath = null) {
    return unsupportedOperationPromise(this, 'storeBeforeAndAfterBlobs');
  }

  restoreLastDiscardInTempFiles(isSafe, partialDiscardFilePath = null) {
    return unsupportedOperationPromise(this, 'restoreLastDiscardInTempFiles');
  }

  popDiscardHistory(partialDiscardFilePath = null) {
    return unsupportedOperationPromise(this, 'popDiscardHistory');
  }

  clearDiscardHistory(partialDiscardFilePath = null) {
    return unsupportedOperationPromise(this, 'clearDiscardHistory');
  }

  discardWorkDirChangesForPaths(paths) {
    return unsupportedOperationPromise(this, 'discardWorkDirChangesForPaths');
  }

  // Accessors /////////////////////////////////////////////////////////////////////////////////////////////////////////
  // When possible, these default to "empty" results when invoked in states that don't have information available, or
  // fail in a way that's consistent with the requested information not being found.

  // Index queries

  getStatusBundle() {
    return Promise.resolve({
      stagedFiles: {},
      unstagedFiles: {},
      mergeConflictFiles: {},
      branch: {
        oid: null,
        head: null,
        upstream: null,
        aheadBehind: { ahead: null, behind: null }
      }
    });
  }

  getStatusesForChangedFiles() {
    return Promise.resolve({
      stagedFiles: [],
      unstagedFiles: [],
      mergeConflictFiles: []
    });
  }

  getFilePatchForPath(filePath, options = {}) {
    return Promise.resolve(new _multiFilePatch2.default({}));
  }

  getStagedChangesPatch() {
    return Promise.resolve(new _multiFilePatch2.default({}));
  }

  readFileFromIndex(filePath) {
    return Promise.reject(new Error(`fatal: Path ${filePath} does not exist (neither on disk nor in the index).`));
  }

  // Commit access

  getLastCommit() {
    return Promise.resolve(_commit.nullCommit);
  }

  getCommit() {
    return Promise.resolve(_commit.nullCommit);
  }

  getRecentCommits() {
    return Promise.resolve([]);
  }

  isCommitPushed(sha) {
    return false;
  }

  // Author information

  getAuthors() {
    return Promise.resolve([]);
  }

  // Branches

  getBranches() {
    return Promise.resolve(new _branchSet2.default());
  }

  getHeadDescription() {
    return Promise.resolve('(no repository)');
  }

  // Merging and rebasing status

  isMerging() {
    return Promise.resolve(false);
  }

  isRebasing() {
    return Promise.resolve(false);
  }

  // Remotes

  getRemotes() {
    return Promise.resolve(new _remoteSet2.default([]));
  }

  addRemote() {
    return unsupportedOperationPromise(this, 'addRemote');
  }

  getAheadCount(branchName) {
    return Promise.resolve(null);
  }

  getBehindCount(branchName) {
    return Promise.resolve(null);
  }

  getConfig(option, { local } = {}) {
    return Promise.resolve(null);
  }

  // Direct blob access

  getBlobContents(sha) {
    return Promise.reject(new Error(`fatal: Not a valid object name ${sha}`));
  }

  // Discard history

  hasDiscardHistory(partialDiscardFilePath = null) {
    return false;
  }

  getDiscardHistory(partialDiscardFilePath = null) {
    return [];
  }

  getLastHistorySnapshots(partialDiscardFilePath = null) {
    return null;
  }

  // Atom repo state

  getOperationStates() {
    return _operationStates.nullOperationStates;
  }

  setCommitMessage(message) {
    return unsupportedOperationPromise(this, 'setCommitMessage');
  }

  getCommitMessage() {
    return '';
  }

  fetchCommitMessageTemplate() {
    return unsupportedOperationPromise(this, 'fetchCommitMessageTemplate');
  }

  // Cache

  getCache() {
    return null;
  }

  // Internal //////////////////////////////////////////////////////////////////////////////////////////////////////////
  // Non-delegated methods that provide subclasses with convenient access to Repository properties.

  git() {
    return this.repository.git;
  }

  workdir() {
    return this.repository.getWorkingDirectoryPath();
  }

  // Call methods on the active Repository state, even if the state has transitioned beneath you.
  // Use this to perform operations within `start()` methods to guard against interrupted state transitions.
  current() {
    return this.repository.state;
  }

  // pipeline
  executePipelineAction(...args) {
    return this.repository.executePipelineAction(...args);
  }

  // Return a Promise that will resolve once the state transitions from Loading.
  getLoadPromise() {
    return this.repository.getLoadPromise();
  }

  getRemoteForBranch(branchName) {
    return this.repository.getRemoteForBranch(branchName);
  }

  saveDiscardHistory() {
    return this.repository.saveDiscardHistory();
  }

  // Initiate a transition to another state.
  transitionTo(stateName, ...payload) {
    const StateConstructor = stateConstructors.get(stateName);
    if (StateConstructor === undefined) {
      throw new Error(`Attempt to transition to unrecognized state ${stateName}`);
    }
    return this.repository.transition(this, StateConstructor, ...payload);
  }

  // Event broadcast

  didDestroy() {
    return this.repository.emitter.emit('did-destroy');
  }

  didUpdate() {
    return this.repository.emitter.emit('did-update');
  }

  // Direct git access
  // Non-delegated git operations for internal use within states.

  directResolveDotGitDir() {
    return Promise.resolve(null);
  }

  directGetConfig(key, options = {}) {
    return Promise.resolve(null);
  }

  directGetBlobContents() {
    return Promise.reject(new Error('Not a valid object name'));
  }

  directInit() {
    return Promise.resolve();
  }

  directClone(remoteUrl, options) {
    return Promise.resolve();
  }

  // Deferred operations
  // Direct raw git operations to the current state, even if the state has been changed. Use these methods within
  // start() methods.

  resolveDotGitDir() {
    return this.current().directResolveDotGitDir();
  }

  doInit(workdir) {
    return this.current().directInit();
  }

  doClone(remoteUrl, options) {
    return this.current().directClone(remoteUrl, options);
  }

  // Parse a DiscardHistory payload from the SHA recorded in config.
  async loadHistoryPayload() {
    const historySha = await this.current().directGetConfig('atomGithub.historySha');
    if (!historySha) {
      return {};
    }

    let blob;
    try {
      blob = await this.current().directGetBlobContents(historySha);
    } catch (e) {
      if (/Not a valid object name/.test(e.stdErr)) {
        return {};
      }

      throw e;
    }

    try {
      return JSON.parse(blob);
    } catch (e) {
      return {};
    }
  }

  // Debugging assistance.

  toString() {
    return this.constructor.name;
  }
}

exports.default = State;
function unsupportedOperationPromise(self, opName) {
  return Promise.reject(new Error(`${opName} is not available in ${self} state`));
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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