'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _extends = Object.assign || function (target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i]; for (var key in source) { if (Object.prototype.hasOwnProperty.call(source, key)) { target[key] = source[key]; } } } return target; };

var _path = require('path');

var _path2 = _interopRequireDefault(_path);

var _eventKit = require('event-kit');

var _fsExtra = require('fs-extra');

var _fsExtra2 = _interopRequireDefault(_fsExtra);

var _state = require('./state');

var _state2 = _interopRequireDefault(_state);

var _gitShellOutStrategy = require('../../git-shell-out-strategy');

var _workspaceChangeObserver = require('../workspace-change-observer');

var _patch = require('../patch');

var _discardHistory = require('../discard-history');

var _discardHistory2 = _interopRequireDefault(_discardHistory);

var _branch = require('../branch');

var _branch2 = _interopRequireDefault(_branch);

var _author = require('../author');

var _author2 = _interopRequireDefault(_author);

var _branchSet = require('../branch-set');

var _branchSet2 = _interopRequireDefault(_branchSet);

var _remote = require('../remote');

var _remote2 = _interopRequireDefault(_remote);

var _remoteSet = require('../remote-set');

var _remoteSet2 = _interopRequireDefault(_remoteSet);

var _commit = require('../commit');

var _commit2 = _interopRequireDefault(_commit);

var _operationStates = require('../operation-states');

var _operationStates2 = _interopRequireDefault(_operationStates);

var _reporterProxy = require('../../reporter-proxy');

var _helpers = require('../../helpers');

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/**
 * State used when the working directory contains a valid git repository and can be interacted with. Performs
 * actual git operations, caching the results, and broadcasts `onDidUpdate` events when write actions are
 * performed.
 */
class Present extends _state2.default {
  constructor(repository, history) {
    super(repository);

    this.cache = new Cache();

    this.discardHistory = new _discardHistory2.default(this.createBlob.bind(this), this.expandBlobToFile.bind(this), this.mergeFile.bind(this), this.workdir(), { maxHistoryLength: 60 });

    this.operationStates = new _operationStates2.default({ didUpdate: this.didUpdate.bind(this) });

    this.commitMessage = '';
    this.commitMessageTemplate = null;
    this.fetchInitialMessage();

    /* istanbul ignore else */
    if (history) {
      this.discardHistory.updateHistory(history);
    }
  }

  setCommitMessage(message, { suppressUpdate } = { suppressUpdate: false }) {
    this.commitMessage = message;
    if (!suppressUpdate) {
      this.didUpdate();
    }
  }

  setCommitMessageTemplate(template) {
    this.commitMessageTemplate = template;
  }

  async fetchInitialMessage() {
    const mergeMessage = await this.repository.getMergeMessage();
    const template = await this.fetchCommitMessageTemplate();
    if (template) {
      this.commitMessageTemplate = template;
    }
    if (mergeMessage) {
      this.setCommitMessage(mergeMessage);
    } else if (template) {
      this.setCommitMessage(template);
    }
  }

  getCommitMessage() {
    return this.commitMessage;
  }

  fetchCommitMessageTemplate() {
    return this.git().fetchCommitMessageTemplate();
  }

  getOperationStates() {
    return this.operationStates;
  }

  isPresent() {
    return true;
  }

  destroy() {
    this.cache.destroy();
    super.destroy();
  }

  showStatusBarTiles() {
    return true;
  }

  acceptInvalidation(spec) {
    this.cache.invalidate(spec());
    this.didUpdate();
  }

  invalidateCacheAfterFilesystemChange(events) {
    const paths = events.map(e => e.special || e.path);
    const keys = new Set();
    for (let i = 0; i < paths.length; i++) {
      const fullPath = paths[i];

      if (fullPath === _workspaceChangeObserver.FOCUS) {
        keys.add(Keys.statusBundle);
        for (const k of Keys.filePatch.eachWithOpts({ staged: false })) {
          keys.add(k);
        }
        continue;
      }

      const includes = (...segments) => fullPath.includes(_path2.default.join(...segments));

      if ((0, _helpers.filePathEndsWith)(fullPath, '.git', 'index')) {
        keys.add(Keys.stagedChanges);
        keys.add(Keys.filePatch.all);
        keys.add(Keys.index.all);
        keys.add(Keys.statusBundle);
        continue;
      }

      if ((0, _helpers.filePathEndsWith)(fullPath, '.git', 'HEAD')) {
        keys.add(Keys.branches);
        keys.add(Keys.lastCommit);
        keys.add(Keys.recentCommits);
        keys.add(Keys.statusBundle);
        keys.add(Keys.headDescription);
        keys.add(Keys.authors);
        continue;
      }

      if (includes('.git', 'refs', 'heads')) {
        keys.add(Keys.branches);
        keys.add(Keys.lastCommit);
        keys.add(Keys.recentCommits);
        keys.add(Keys.headDescription);
        keys.add(Keys.authors);
        continue;
      }

      if (includes('.git', 'refs', 'remotes')) {
        keys.add(Keys.remotes);
        keys.add(Keys.statusBundle);
        keys.add(Keys.headDescription);
        continue;
      }

      if ((0, _helpers.filePathEndsWith)(fullPath, '.git', 'config')) {
        keys.add(Keys.remotes);
        keys.add(Keys.config.all);
        keys.add(Keys.statusBundle);
        continue;
      }

      // File change within the working directory
      const relativePath = _path2.default.relative(this.workdir(), fullPath);
      keys.add(Keys.filePatch.oneWith(relativePath, { staged: false }));
      keys.add(Keys.statusBundle);
    }

    /* istanbul ignore else */
    if (keys.size > 0) {
      this.cache.invalidate(Array.from(keys));
      this.didUpdate();
    }
  }

  isCommitMessageClean() {
    if (this.commitMessage.trim() === '') {
      return true;
    } else if (this.commitMessageTemplate) {
      return this.commitMessage === this.commitMessageTemplate;
    }
    return false;
  }

  async updateCommitMessageAfterFileSystemChange(events) {
    for (let i = 0; i < events.length; i++) {
      const event = events[i];

      if (!event.path) {
        continue;
      }

      if ((0, _helpers.filePathEndsWith)(event.path, '.git', 'MERGE_HEAD')) {
        if (event.action === 'created') {
          if (this.isCommitMessageClean()) {
            this.setCommitMessage((await this.repository.getMergeMessage()));
          }
        } else if (event.action === 'deleted') {
          this.setCommitMessage(this.commitMessageTemplate || '');
        }
      }

      if ((0, _helpers.filePathEndsWith)(event.path, '.git', 'config')) {
        // this won't catch changes made to the template file itself...
        const template = await this.fetchCommitMessageTemplate();
        if (template === null) {
          this.setCommitMessage('');
        } else if (this.commitMessageTemplate !== template) {
          this.setCommitMessage(template);
        }
        this.setCommitMessageTemplate(template);
      }
    }
  }

  observeFilesystemChange(events) {
    this.invalidateCacheAfterFilesystemChange(events);
    this.updateCommitMessageAfterFileSystemChange(events);
  }

  refresh() {
    this.cache.clear();
    this.didUpdate();
  }

  init() {
    return super.init().catch(e => {
      e.stdErr = 'This directory already contains a git repository';
      return Promise.reject(e);
    });
  }

  clone() {
    return super.clone().catch(e => {
      e.stdErr = 'This directory already contains a git repository';
      return Promise.reject(e);
    });
  }

  // Git operations ////////////////////////////////////////////////////////////////////////////////////////////////////

  // Staging and unstaging

  stageFiles(paths) {
    return this.invalidate(() => Keys.cacheOperationKeys(paths), () => this.git().stageFiles(paths));
  }

  unstageFiles(paths) {
    return this.invalidate(() => Keys.cacheOperationKeys(paths), () => this.git().unstageFiles(paths));
  }

  stageFilesFromParentCommit(paths) {
    return this.invalidate(() => Keys.cacheOperationKeys(paths), () => this.git().unstageFiles(paths, 'HEAD~'));
  }

  stageFileModeChange(filePath, fileMode) {
    return this.invalidate(() => Keys.cacheOperationKeys([filePath]), () => this.git().stageFileModeChange(filePath, fileMode));
  }

  stageFileSymlinkChange(filePath) {
    return this.invalidate(() => Keys.cacheOperationKeys([filePath]), () => this.git().stageFileSymlinkChange(filePath));
  }

  applyPatchToIndex(multiFilePatch) {
    return this.invalidate(() => Keys.cacheOperationKeys(Array.from(multiFilePatch.getPathSet())), () => {
      const patchStr = multiFilePatch.toString();
      return this.git().applyPatch(patchStr, { index: true });
    });
  }

  applyPatchToWorkdir(multiFilePatch) {
    return this.invalidate(() => Keys.workdirOperationKeys(Array.from(multiFilePatch.getPathSet())), () => {
      const patchStr = multiFilePatch.toString();
      return this.git().applyPatch(patchStr);
    });
  }

  // Committing

  commit(message, options) {
    return this.invalidate(() => [...Keys.headOperationKeys(), ...Keys.filePatch.eachWithOpts({ staged: true }), Keys.headDescription, Keys.branches, Keys.stagedChanges],
    // eslint-disable-next-line no-shadow
    () => this.executePipelineAction('COMMIT', async (message, options = {}) => {
      const coAuthors = options.coAuthors;
      const opts = !coAuthors ? options : _extends({}, options, {
        coAuthors: coAuthors.map(author => {
          return { email: author.getEmail(), name: author.getFullName() };
        })
      });

      await this.git().commit(message, opts);

      // Collect commit metadata metrics
      // note: in GitShellOutStrategy we have counters for all git commands, including `commit`, but here we have
      //       access to additional metadata (unstaged file count) so it makes sense to collect commit events here
      const { unstagedFiles, mergeConflictFiles } = await this.getStatusesForChangedFiles();
      const unstagedCount = Object.keys(_extends({}, unstagedFiles, mergeConflictFiles)).length;
      (0, _reporterProxy.addEvent)('commit', {
        package: 'github',
        partial: unstagedCount > 0,
        amend: !!options.amend,
        coAuthorCount: coAuthors ? coAuthors.length : 0
      });
    }, message, options));
  }

  // Merging

  merge(branchName) {
    return this.invalidate(() => [...Keys.headOperationKeys(), Keys.index.all, Keys.headDescription], () => this.git().merge(branchName));
  }

  abortMerge() {
    return this.invalidate(() => [Keys.statusBundle, Keys.stagedChanges, Keys.filePatch.all, Keys.index.all], async () => {
      await this.git().abortMerge();
      this.setCommitMessage(this.commitMessageTemplate || '');
    });
  }

  checkoutSide(side, paths) {
    return this.git().checkoutSide(side, paths);
  }

  mergeFile(oursPath, commonBasePath, theirsPath, resultPath) {
    return this.git().mergeFile(oursPath, commonBasePath, theirsPath, resultPath);
  }

  writeMergeConflictToIndex(filePath, commonBaseSha, oursSha, theirsSha) {
    return this.invalidate(() => [Keys.statusBundle, Keys.stagedChanges, ...Keys.filePatch.eachWithFileOpts([filePath], [{ staged: false }, { staged: true }]), Keys.index.oneWith(filePath)], () => this.git().writeMergeConflictToIndex(filePath, commonBaseSha, oursSha, theirsSha));
  }

  // Checkout

  checkout(revision, options = {}) {
    return this.invalidate(() => [Keys.stagedChanges, Keys.lastCommit, Keys.recentCommits, Keys.authors, Keys.statusBundle, Keys.index.all, ...Keys.filePatch.eachWithOpts({ staged: true }), Keys.headDescription, Keys.branches],
    // eslint-disable-next-line no-shadow
    () => this.executePipelineAction('CHECKOUT', (revision, options) => {
      return this.git().checkout(revision, options);
    }, revision, options));
  }

  checkoutPathsAtRevision(paths, revision = 'HEAD') {
    return this.invalidate(() => [Keys.statusBundle, Keys.stagedChanges, ...paths.map(fileName => Keys.index.oneWith(fileName)), ...Keys.filePatch.eachWithFileOpts(paths, [{ staged: true }])], () => this.git().checkoutFiles(paths, revision));
  }

  // Reset

  undoLastCommit() {
    return this.invalidate(() => [Keys.stagedChanges, Keys.lastCommit, Keys.recentCommits, Keys.authors, Keys.statusBundle, Keys.index.all, ...Keys.filePatch.eachWithOpts({ staged: true }), Keys.headDescription], async () => {
      try {
        await this.git().reset('soft', 'HEAD~');
        (0, _reporterProxy.addEvent)('undo-last-commit', { package: 'github' });
      } catch (e) {
        if (/unknown revision/.test(e.stdErr)) {
          // Initial commit
          await this.git().deleteRef('HEAD');
        } else {
          throw e;
        }
      }
    });
  }

  // Remote interactions

  fetch(branchName, options = {}) {
    return this.invalidate(() => [Keys.statusBundle, Keys.headDescription],
    // eslint-disable-next-line no-shadow
    () => this.executePipelineAction('FETCH', async branchName => {
      let finalRemoteName = options.remoteName;
      if (!finalRemoteName) {
        const remote = await this.getRemoteForBranch(branchName);
        if (!remote.isPresent()) {
          return null;
        }
        finalRemoteName = remote.getName();
      }
      return this.git().fetch(finalRemoteName, branchName);
    }, branchName));
  }

  pull(branchName, options = {}) {
    return this.invalidate(() => [...Keys.headOperationKeys(), Keys.index.all, Keys.headDescription, Keys.branches],
    // eslint-disable-next-line no-shadow
    () => this.executePipelineAction('PULL', async branchName => {
      let finalRemoteName = options.remoteName;
      if (!finalRemoteName) {
        const remote = await this.getRemoteForBranch(branchName);
        if (!remote.isPresent()) {
          return null;
        }
        finalRemoteName = remote.getName();
      }
      return this.git().pull(finalRemoteName, branchName, options);
    }, branchName));
  }

  push(branchName, options = {}) {
    return this.invalidate(() => {
      const keys = [Keys.statusBundle, Keys.headDescription];

      if (options.setUpstream) {
        keys.push(Keys.branches);
        keys.push(...Keys.config.eachWithSetting(`branch.${branchName}.remote`));
      }

      return keys;
    },
    // eslint-disable-next-line no-shadow
    () => this.executePipelineAction('PUSH', async (branchName, options) => {
      const remote = options.remote || (await this.getRemoteForBranch(branchName));
      return this.git().push(remote.getNameOr('origin'), branchName, options);
    }, branchName, options));
  }

  // Configuration

  setConfig(setting, value, options) {
    return this.invalidate(() => Keys.config.eachWithSetting(setting), () => this.git().setConfig(setting, value, options));
  }

  unsetConfig(setting) {
    return this.invalidate(() => Keys.config.eachWithSetting(setting), () => this.git().unsetConfig(setting));
  }

  // Direct blob interactions

  createBlob(options) {
    return this.git().createBlob(options);
  }

  expandBlobToFile(absFilePath, sha) {
    return this.git().expandBlobToFile(absFilePath, sha);
  }

  // Discard history

  createDiscardHistoryBlob() {
    return this.discardHistory.createHistoryBlob();
  }

  async updateDiscardHistory() {
    const history = await this.loadHistoryPayload();
    this.discardHistory.updateHistory(history);
  }

  async storeBeforeAndAfterBlobs(filePaths, isSafe, destructiveAction, partialDiscardFilePath = null) {
    const snapshots = await this.discardHistory.storeBeforeAndAfterBlobs(filePaths, isSafe, destructiveAction, partialDiscardFilePath);
    /* istanbul ignore else */
    if (snapshots) {
      await this.saveDiscardHistory();
    }
    return snapshots;
  }

  restoreLastDiscardInTempFiles(isSafe, partialDiscardFilePath = null) {
    return this.discardHistory.restoreLastDiscardInTempFiles(isSafe, partialDiscardFilePath);
  }

  async popDiscardHistory(partialDiscardFilePath = null) {
    const removed = await this.discardHistory.popHistory(partialDiscardFilePath);
    if (removed) {
      await this.saveDiscardHistory();
    }
  }

  clearDiscardHistory(partialDiscardFilePath = null) {
    this.discardHistory.clearHistory(partialDiscardFilePath);
    return this.saveDiscardHistory();
  }

  discardWorkDirChangesForPaths(paths) {
    return this.invalidate(() => [Keys.statusBundle, ...paths.map(filePath => Keys.filePatch.oneWith(filePath, { staged: false }))], async () => {
      const untrackedFiles = await this.git().getUntrackedFiles();
      const [filesToRemove, filesToCheckout] = partition(paths, f => untrackedFiles.includes(f));
      await this.git().checkoutFiles(filesToCheckout);
      await Promise.all(filesToRemove.map(filePath => {
        const absPath = _path2.default.join(this.workdir(), filePath);
        return _fsExtra2.default.remove(absPath);
      }));
    });
  }

  // Accessors /////////////////////////////////////////////////////////////////////////////////////////////////////////

  // Index queries

  getStatusBundle() {
    return this.cache.getOrSet(Keys.statusBundle, async () => {
      try {
        const bundle = await this.git().getStatusBundle();
        const results = await this.formatChangedFiles(bundle);
        results.branch = bundle.branch;
        if (!results.branch.aheadBehind) {
          results.branch.aheadBehind = { ahead: null, behind: null };
        }
        return results;
      } catch (err) {
        if (err instanceof _gitShellOutStrategy.LargeRepoError) {
          this.transitionTo('TooLarge');
          return {
            branch: {},
            stagedFiles: {},
            unstagedFiles: {},
            mergeConflictFiles: {}
          };
        } else {
          throw err;
        }
      }
    });
  }

  async formatChangedFiles({ changedEntries, untrackedEntries, renamedEntries, unmergedEntries }) {
    const statusMap = {
      A: 'added',
      M: 'modified',
      D: 'deleted',
      U: 'modified',
      T: 'typechange'
    };

    const stagedFiles = {};
    const unstagedFiles = {};
    const mergeConflictFiles = {};

    changedEntries.forEach(entry => {
      if (entry.stagedStatus) {
        stagedFiles[entry.filePath] = statusMap[entry.stagedStatus];
      }
      if (entry.unstagedStatus) {
        unstagedFiles[entry.filePath] = statusMap[entry.unstagedStatus];
      }
    });

    untrackedEntries.forEach(entry => {
      unstagedFiles[entry.filePath] = statusMap.A;
    });

    renamedEntries.forEach(entry => {
      if (entry.stagedStatus === 'R') {
        stagedFiles[entry.filePath] = statusMap.A;
        stagedFiles[entry.origFilePath] = statusMap.D;
      }
      if (entry.unstagedStatus === 'R') {
        unstagedFiles[entry.filePath] = statusMap.A;
        unstagedFiles[entry.origFilePath] = statusMap.D;
      }
      if (entry.stagedStatus === 'C') {
        stagedFiles[entry.filePath] = statusMap.A;
      }
      if (entry.unstagedStatus === 'C') {
        unstagedFiles[entry.filePath] = statusMap.A;
      }
    });

    let statusToHead;

    for (let i = 0; i < unmergedEntries.length; i++) {
      const { stagedStatus, unstagedStatus, filePath } = unmergedEntries[i];
      if (stagedStatus === 'U' || unstagedStatus === 'U' || stagedStatus === 'A' && unstagedStatus === 'A') {
        // Skipping this check here because we only run a single `await`
        // and we only run it in the main, synchronous body of the for loop.
        // eslint-disable-next-line no-await-in-loop
        if (!statusToHead) {
          statusToHead = await this.git().diffFileStatus({ target: 'HEAD' });
        }
        mergeConflictFiles[filePath] = {
          ours: statusMap[stagedStatus],
          theirs: statusMap[unstagedStatus],
          file: statusToHead[filePath] || 'equivalent'
        };
      }
    }

    return { stagedFiles, unstagedFiles, mergeConflictFiles };
  }

  async getStatusesForChangedFiles() {
    const { stagedFiles, unstagedFiles, mergeConflictFiles } = await this.getStatusBundle();
    return { stagedFiles, unstagedFiles, mergeConflictFiles };
  }

  getFilePatchForPath(filePath, { staged } = { staged: false }) {
    return this.cache.getOrSet(Keys.filePatch.oneWith(filePath, { staged }), async () => {
      const diffs = await this.git().getDiffsForFilePath(filePath, { staged });
      return (0, _patch.buildFilePatch)(diffs);
    });
  }

  getStagedChangesPatch() {
    return this.cache.getOrSet(Keys.stagedChanges, () => {
      return this.git().getStagedChangesPatch().then(_patch.buildMultiFilePatch);
    });
  }

  readFileFromIndex(filePath) {
    return this.cache.getOrSet(Keys.index.oneWith(filePath), () => {
      return this.git().readFileFromIndex(filePath);
    });
  }

  // Commit access

  getLastCommit() {
    return this.cache.getOrSet(Keys.lastCommit, async () => {
      const headCommit = await this.git().getHeadCommit();
      return headCommit.unbornRef ? _commit2.default.createUnborn() : new _commit2.default(headCommit);
    });
  }

  getCommit(sha) {
    return this.cache.getOrSet(Keys.blob.oneWith(sha), async () => {
      const [rawCommit] = await this.git().getCommits({ max: 1, ref: sha, includePatch: true });
      const commit = new _commit2.default(rawCommit);
      return commit;
    });
  }

  getRecentCommits(options) {
    return this.cache.getOrSet(Keys.recentCommits, async () => {
      const commits = await this.git().getCommits(_extends({ ref: 'HEAD' }, options));
      return commits.map(commit => new _commit2.default(commit));
    });
  }

  async isCommitPushed(sha) {
    const remoteBranchesWithCommit = await this.git().getBranchesWithCommit(sha, { showLocal: false, showRemote: true });
    const currentRemote = (await this.repository.getCurrentBranch()).getUpstream();
    return remoteBranchesWithCommit.includes(currentRemote.getFullRef());
  }

  // Author information

  getAuthors(options) {
    // For now we'll do the naive thing and invalidate anytime HEAD moves. This ensures that we get new authors
    // introduced by newly created commits or pulled commits.
    // This means that we are constantly re-fetching data. If performance becomes a concern we can optimize
    return this.cache.getOrSet(Keys.authors, async () => {
      const authorMap = await this.git().getAuthors(options);
      return Object.keys(authorMap).map(email => new _author2.default(email, authorMap[email]));
    });
  }

  // Branches

  getBranches() {
    return this.cache.getOrSet(Keys.branches, async () => {
      const payloads = await this.git().getBranches();
      const branches = new _branchSet2.default();
      for (const payload of payloads) {
        let upstream = _branch.nullBranch;
        if (payload.upstream) {
          upstream = payload.upstream.remoteName ? _branch2.default.createRemoteTracking(payload.upstream.trackingRef, payload.upstream.remoteName, payload.upstream.remoteRef) : new _branch2.default(payload.upstream.trackingRef);
        }

        let push = upstream;
        if (payload.push) {
          push = payload.push.remoteName ? _branch2.default.createRemoteTracking(payload.push.trackingRef, payload.push.remoteName, payload.push.remoteRef) : new _branch2.default(payload.push.trackingRef);
        }

        branches.add(new _branch2.default(payload.name, upstream, push, payload.head, { sha: payload.sha }));
      }
      return branches;
    });
  }

  getHeadDescription() {
    return this.cache.getOrSet(Keys.headDescription, () => {
      return this.git().describeHead();
    });
  }

  // Merging and rebasing status

  isMerging() {
    return this.git().isMerging(this.repository.getGitDirectoryPath());
  }

  isRebasing() {
    return this.git().isRebasing(this.repository.getGitDirectoryPath());
  }

  // Remotes

  getRemotes() {
    return this.cache.getOrSet(Keys.remotes, async () => {
      const remotesInfo = await this.git().getRemotes();
      return new _remoteSet2.default(remotesInfo.map(({ name, url }) => new _remote2.default(name, url)));
    });
  }

  addRemote(name, url) {
    return this.invalidate(() => [...Keys.config.eachWithSetting(`remote.${name}.url`), ...Keys.config.eachWithSetting(`remote.${name}.fetch`), Keys.remotes],
    // eslint-disable-next-line no-shadow
    () => this.executePipelineAction('ADDREMOTE', async (name, url) => {
      await this.git().addRemote(name, url);
      return new _remote2.default(name, url);
    }, name, url));
  }

  async getAheadCount(branchName) {
    const bundle = await this.getStatusBundle();
    return bundle.branch.aheadBehind.ahead;
  }

  async getBehindCount(branchName) {
    const bundle = await this.getStatusBundle();
    return bundle.branch.aheadBehind.behind;
  }

  getConfig(option, { local } = { local: false }) {
    return this.cache.getOrSet(Keys.config.oneWith(option, { local }), () => {
      return this.git().getConfig(option, { local });
    });
  }

  // Direct blob access

  getBlobContents(sha) {
    return this.cache.getOrSet(Keys.blob(sha), () => {
      return this.git().getBlobContents(sha);
    });
  }

  // Discard history

  hasDiscardHistory(partialDiscardFilePath = null) {
    return this.discardHistory.hasHistory(partialDiscardFilePath);
  }

  getDiscardHistory(partialDiscardFilePath = null) {
    return this.discardHistory.getHistory(partialDiscardFilePath);
  }

  getLastHistorySnapshots(partialDiscardFilePath = null) {
    return this.discardHistory.getLastSnapshots(partialDiscardFilePath);
  }

  // Cache

  getCache() {
    return this.cache;
  }

  invalidate(spec, body) {
    return body().then(result => {
      this.acceptInvalidation(spec);
      return result;
    }, err => {
      this.acceptInvalidation(spec);
      return Promise.reject(err);
    });
  }
}

exports.default = Present;
_state2.default.register(Present);

function partition(array, predicate) {
  const matches = [];
  const nonmatches = [];
  array.forEach(item => {
    if (predicate(item)) {
      matches.push(item);
    } else {
      nonmatches.push(item);
    }
  });
  return [matches, nonmatches];
}

class Cache {
  constructor() {
    this.storage = new Map();
    this.byGroup = new Map();

    this.emitter = new _eventKit.Emitter();
  }

  getOrSet(key, operation) {
    const primary = key.getPrimary();
    const existing = this.storage.get(primary);
    if (existing !== undefined) {
      existing.hits++;
      return existing.promise;
    }

    const created = operation();

    this.storage.set(primary, {
      createdAt: performance.now(),
      hits: 0,
      promise: created
    });

    const groups = key.getGroups();
    for (let i = 0; i < groups.length; i++) {
      const group = groups[i];
      let groupSet = this.byGroup.get(group);
      if (groupSet === undefined) {
        groupSet = new Set();
        this.byGroup.set(group, groupSet);
      }
      groupSet.add(key);
    }

    this.didUpdate();

    return created;
  }

  invalidate(keys) {
    for (let i = 0; i < keys.length; i++) {
      keys[i].removeFromCache(this);
    }

    if (keys.length > 0) {
      this.didUpdate();
    }
  }

  keysInGroup(group) {
    return this.byGroup.get(group) || [];
  }

  removePrimary(primary) {
    this.storage.delete(primary);
    this.didUpdate();
  }

  removeFromGroup(group, key) {
    const groupSet = this.byGroup.get(group);
    groupSet && groupSet.delete(key);
    this.didUpdate();
  }

  [Symbol.iterator]() {
    return this.storage[Symbol.iterator]();
  }

  clear() {
    this.storage.clear();
    this.byGroup.clear();
    this.didUpdate();
  }

  didUpdate() {
    this.emitter.emit('did-update');
  }

  onDidUpdate(callback) {
    return this.emitter.on('did-update', callback);
  }

  destroy() {
    this.emitter.dispose();
  }
}

class CacheKey {
  constructor(primary, groups = []) {
    this.primary = primary;
    this.groups = groups;
  }

  getPrimary() {
    return this.primary;
  }

  getGroups() {
    return this.groups;
  }

  removeFromCache(cache, withoutGroup = null) {
    cache.removePrimary(this.getPrimary());

    const groups = this.getGroups();
    for (let i = 0; i < groups.length; i++) {
      const group = groups[i];
      if (group === withoutGroup) {
        continue;
      }

      cache.removeFromGroup(group, this);
    }
  }

  toString() {
    return `CacheKey(${this.primary})`;
  }
}

class GroupKey {
  constructor(group) {
    this.group = group;
  }

  removeFromCache(cache) {
    for (const matchingKey of cache.keysInGroup(this.group)) {
      matchingKey.removeFromCache(cache, this.group);
    }
  }

  toString() {
    return `GroupKey(${this.group})`;
  }
}

const Keys = {
  statusBundle: new CacheKey('status-bundle'),

  stagedChanges: new CacheKey('staged-changes'),

  filePatch: {
    _optKey: ({ staged }) => {
      return staged ? 's' : 'u';
    },

    oneWith: (fileName, options) => {
      // <-- Keys.filePatch
      const optKey = Keys.filePatch._optKey(options);
      return new CacheKey(`file-patch:${optKey}:${fileName}`, ['file-patch', `file-patch:${optKey}`]);
    },

    eachWithFileOpts: (fileNames, opts) => {
      const keys = [];
      for (let i = 0; i < fileNames.length; i++) {
        for (let j = 0; j < opts.length; j++) {
          keys.push(Keys.filePatch.oneWith(fileNames[i], opts[j]));
        }
      }
      return keys;
    },

    eachWithOpts: (...opts) => opts.map(opt => new GroupKey(`file-patch:${Keys.filePatch._optKey(opt)}`)),

    all: new GroupKey('file-patch')
  },

  index: {
    oneWith: fileName => new CacheKey(`index:${fileName}`, ['index']),

    all: new GroupKey('index')
  },

  lastCommit: new CacheKey('last-commit'),

  recentCommits: new CacheKey('recent-commits'),

  authors: new CacheKey('authors'),

  branches: new CacheKey('branches'),

  headDescription: new CacheKey('head-description'),

  remotes: new CacheKey('remotes'),

  config: {
    _optKey: options => options.local ? 'l' : '',

    oneWith: (setting, options) => {
      const optKey = Keys.config._optKey(options);
      return new CacheKey(`config:${optKey}:${setting}`, ['config', `config:${optKey}`]);
    },

    eachWithSetting: setting => [Keys.config.oneWith(setting, { local: true }), Keys.config.oneWith(setting, { local: false })],

    all: new GroupKey('config')
  },

  blob: {
    oneWith: sha => new CacheKey(`blob:${sha}`, ['blob'])
  },

  // Common collections of keys and patterns for use with invalidate().

  workdirOperationKeys: fileNames => [Keys.statusBundle, ...Keys.filePatch.eachWithFileOpts(fileNames, [{ staged: false }])],

  cacheOperationKeys: fileNames => [...Keys.workdirOperationKeys(fileNames), ...Keys.filePatch.eachWithFileOpts(fileNames, [{ staged: true }]), ...fileNames.map(Keys.index.oneWith), Keys.stagedChanges],

  headOperationKeys: () => [...Keys.filePatch.eachWithOpts({ staged: true }), Keys.stagedChanges, Keys.lastCommit, Keys.recentCommits, Keys.authors, Keys.statusBundle]
};
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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