'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _path = require('path');

var _path2 = _interopRequireDefault(_path);

var _electron = require('electron');

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

let headless = null;

class EventLogger {
  constructor(kind) {
    this.kind = kind;
    this.directory = '<unknown>';
    this.shortDirectory = '<unknown>';

    if (headless === null) {
      headless = !_electron.remote.getCurrentWindow().isVisible();
    }
  }

  showStarted(directory, implementation) {
    this.directory = directory;
    this.shortDirectory = directory.split(_path2.default.sep).slice(-2).join(_path2.default.sep);

    if (!this.isEnabled()) {
      return;
    }

    this.shortLog(`${implementation} watcher started`);
  }

  showEvents(events) {
    if (!this.isEnabled()) {
      return;
    }

    const uniqueRelativeNames = new Set(events.map(event => {
      return _path2.default.relative(this.directory, event.path);
    }));

    const fileNames = Array.from(uniqueRelativeNames).slice(0, 3);
    const elipses = uniqueRelativeNames.size > 3 ? '...' : '';
    const summary = `${this.getShortName()}: ${fileNames.join(', ')}${elipses}`;

    /* eslint-disable no-console */
    if (headless) {
      const eventText = events.map(event => {
        if (event.action === 'renamed') {
          return `  ${event.oldPath} => ${event.path} (${event.action})\n`;
        } else {
          return `  ${event.path} (${event.action})\n`;
        }
      }).join('\n');
      console.log(summary + '\n' + eventText);
    } else {
      console.groupCollapsed(summary);
      console.table(events, ['action', 'path', 'oldPath']);
      console.groupEnd();
    }
    /* eslint-enable no-console */
  }

  showFocusEvent() {
    if (!this.isEnabled()) {
      return;
    }

    this.shortLog('focus triggered');
  }

  showWorkdirOrHeadEvents() {
    if (!this.isEnabled()) {
      return;
    }

    this.shortLog('working directory or HEAD change');
  }

  showStopped() {
    if (!this.isEnabled()) {
      return;
    }

    this.shortLog('stopped');
  }

  isEnabled() {
    return process.env.ATOM_GITHUB_FS_EVENT_LOG || atom.config.get('github.filesystemEventDiagnostics');
  }

  getShortName() {
    return `${this.kind} @ ${this.shortDirectory}`;
  }

  shortLog(line) {
    if (headless) {
      // eslint-disable-next-line no-console
      console.log(`${this.getShortName()}: ${line}`);
      return;
    }

    // eslint-disable-next-line no-console
    console.log('%c%s%c: %s', 'font-weight: bold; color: blue;', this.getShortName(), 'font-weight: normal; color: black;', line);
  }
}
exports.default = EventLogger;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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