'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _markerTools = require('./marker-tools');

class Side {
  constructor(editor, marker, blockMarker, source, position, banner, originalText) {
    this.editor = editor;
    this.marker = marker;
    this.blockMarker = blockMarker;
    this.source = source;
    this.position = position;
    this.banner = banner;
    this.originalText = originalText;
  }

  getMarker() {
    return this.marker;
  }

  getBannerMarker() {
    return this.banner.getMarker();
  }

  getSource() {
    return this.source;
  }

  getBlockMarker() {
    return this.blockMarker;
  }

  getBlockPosition() {
    return this.position.when({
      top: () => 'before',
      middle: () => 'before',
      bottom: () => 'after'
    });
  }

  getLineCSSClass() {
    if (this.isModified() || this.isBannerModified()) {
      return 'github-ConflictModified';
    } else {
      return this.source.getCSSClass();
    }
  }

  getBannerCSSClass() {
    if (this.isModified() || this.isBannerModified()) {
      return 'github-ConflictModifiedBanner';
    } else {
      return this.source.getBannerCSSClass();
    }
  }

  getBlockCSSClasses() {
    const cxs = ['github-ConflictBlock'];
    cxs.push(this.source.getBlockCSSClass());
    cxs.push(this.position.getBlockCSSClass());
    if (this.isModified() || this.isBannerModified()) {
      cxs.push('github-ConflictModifiedBlock');
    }
    return cxs.join(' ');
  }

  getPosition() {
    return this.position;
  }

  getRange() {
    const bannerRange = this.banner.getRange();
    const bodyRange = this.marker.getBufferRange();
    return bannerRange.union(bodyRange);
  }

  includesPoint(point) {
    return this.getRange().containsPoint(point);
  }

  getText() {
    return this.editor.getTextInBufferRange(this.getMarker().getBufferRange());
  }

  isBannerModified() {
    return this.banner.isModified();
  }

  isModified() {
    return this.getText() !== this.originalText;
  }

  isEmpty() {
    return this.marker.getBufferRange().isEmpty();
  }

  revertBanner() {
    this.banner.revert();
  }

  revert() {
    const range = this.getMarker().getBufferRange();
    this.editor.setTextInBufferRange(range, this.originalText);
  }

  deleteBanner() {
    this.banner.delete();
  }

  delete() {
    (0, _markerTools.deleteMarkerIn)(this.getMarker(), this.editor);
  }

  appendText(text) {
    const insertionPoint = this.getMarker().getBufferRange().end;
    return this.editor.setTextInBufferRange([insertionPoint, insertionPoint], text);
  }
}
exports.default = Side;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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