'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _eventKit = require('event-kit');

let key = 0;
class StubItem {
  // StubItems should only be created by `create` and never constructed directly.
  static create(name, props, uri = '') {
    const stub = new StubItem(name, props, uri);
    const override = {
      _getStub: () => stub,
      getElement: () => stub.getElement(),
      destroy: stub.destroy.bind(stub)
    };
    const proxy = new Proxy(override, {
      get(target, propName) {
        const item = stub.getRealItem();
        if (Reflect.has(target, propName)) {
          return target[propName];
        } else if (item && Reflect.has(item, propName)) {
          let val = item[propName];
          if (typeof val === 'function') {
            val = val.bind(item);
          }
          return val;
        } else {
          let val = stub[propName];
          if (typeof val === 'function') {
            val = val.bind(stub);
          }
          return val;
        }
      }
    });
    return proxy;
  }

  constructor(name, props = {}, uri) {
    this.emitter = new _eventKit.Emitter();
    this.subscriptions = new _eventKit.CompositeDisposable();

    this.name = name;
    this.props = props;
    this.uri = uri;
    this.key = ++key;
    this.element = document.createElement('div');
    this.element.classList.add(`github-StubItem-${name}`);
    this.realItem = null;
    this.realItemPromise = new Promise(res => {
      this.resolveRealItemPromise = res;
    });
  }

  setRealItem(item) {
    this.realItem = item;

    if (this.realItem.getRealItemPromise) {
      this.realItem.getRealItemPromise().then(this.resolveRealItemPromise);
    } else {
      this.resolveRealItemPromise(this.realItem);
    }

    this.emitter.emit('did-change-title');
    this.emitter.emit('did-change-icon');

    if (item.onDidChangeTitle) {
      this.subscriptions.add(item.onDidChangeTitle((...args) => this.emitter.emit('did-change-title', ...args)));
    }

    if (item.onDidChangeIcon) {
      this.subscriptions.add(item.onDidChangeIcon((...args) => this.emitter.emit('did-change-icon', ...args)));
    }

    if (item.onDidDestroy) {
      this.subscriptions.add(item.onDidDestroy((...args) => {
        this.realItem = null;
        this.emitter.emit('did-destroy', ...args);
      }));
    }
  }

  getRealItemPromise() {
    return this.realItemPromise;
  }

  getRealItem() {
    return this.realItem;
  }

  getURI() {
    return this.uri;
  }

  getTitle() {
    return this.props.title || null;
  }

  getIconName() {
    return this.props.iconName || null;
  }

  onDidChangeTitle(cb) {
    return this.emitter.on('did-change-title', cb);
  }

  onDidChangeIcon(cb) {
    return this.emitter.on('did-change-icon', cb);
  }

  getElement() {
    return this.element;
  }

  onDidDestroy(cb) {
    return this.emitter.on('did-destroy', cb);
  }

  destroy() {
    this.resolveRealItemPromise(null);
    this.subscriptions.dispose();
    this.emitter.dispose();
    if (this.realItem) {
      this.realItem.destroy && this.realItem.destroy();
    }
  }
}
exports.default = StubItem;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbInN0dWItaXRlbS5qcyJdLCJuYW1lcyI6WyJrZXkiLCJTdHViSXRlbSIsImNyZWF0ZSIsIm5hbWUiLCJwcm9wcyIsInVyaSIsInN0dWIiLCJvdmVycmlkZSIsIl9nZXRTdHViIiwiZ2V0RWxlbWVudCIsImRlc3Ryb3kiLCJiaW5kIiwicHJveHkiLCJQcm94eSIsImdldCIsInRhcmdldCIsInByb3BOYW1lIiwiaXRlbSIsImdldFJlYWxJdGVtIiwiUmVmbGVjdCIsImhhcyIsInZhbCIsImNvbnN0cnVjdG9yIiwiZW1pdHRlciIsIkVtaXR0ZXIiLCJzdWJzY3JpcHRpb25zIiwiQ29tcG9zaXRlRGlzcG9zYWJsZSIsImVsZW1lbnQiLCJkb2N1bWVudCIsImNyZWF0ZUVsZW1lbnQiLCJjbGFzc0xpc3QiLCJhZGQiLCJyZWFsSXRlbSIsInJlYWxJdGVtUHJvbWlzZSIsIlByb21pc2UiLCJyZXMiLCJyZXNvbHZlUmVhbEl0ZW1Qcm9taXNlIiwic2V0UmVhbEl0ZW0iLCJnZXRSZWFsSXRlbVByb21pc2UiLCJ0aGVuIiwiZW1pdCIsIm9uRGlkQ2hhbmdlVGl0bGUiLCJhcmdzIiwib25EaWRDaGFuZ2VJY29uIiwib25EaWREZXN0cm95IiwiZ2V0VVJJIiwiZ2V0VGl0bGUiLCJ0aXRsZSIsImdldEljb25OYW1lIiwiaWNvbk5hbWUiLCJjYiIsIm9uIiwiZGlzcG9zZSJdLCJtYXBwaW5ncyI6Ijs7Ozs7O0FBQUE7O0FBRUEsSUFBSUEsTUFBTSxDQUFWO0FBQ2UsTUFBTUMsUUFBTixDQUFlO0FBQzVCO0FBQ0EsU0FBT0MsTUFBUCxDQUFjQyxJQUFkLEVBQW9CQyxLQUFwQixFQUEyQkMsTUFBTSxFQUFqQyxFQUFxQztBQUNuQyxVQUFNQyxPQUFPLElBQUlMLFFBQUosQ0FBYUUsSUFBYixFQUFtQkMsS0FBbkIsRUFBMEJDLEdBQTFCLENBQWI7QUFDQSxVQUFNRSxXQUFXO0FBQ2ZDLGdCQUFVLE1BQU1GLElBREQ7QUFFZkcsa0JBQVksTUFBTUgsS0FBS0csVUFBTCxFQUZIO0FBR2ZDLGVBQVNKLEtBQUtJLE9BQUwsQ0FBYUMsSUFBYixDQUFrQkwsSUFBbEI7QUFITSxLQUFqQjtBQUtBLFVBQU1NLFFBQVEsSUFBSUMsS0FBSixDQUFVTixRQUFWLEVBQW9CO0FBQ2hDTyxVQUFJQyxNQUFKLEVBQVlDLFFBQVosRUFBc0I7QUFDcEIsY0FBTUMsT0FBT1gsS0FBS1ksV0FBTCxFQUFiO0FBQ0EsWUFBSUMsUUFBUUMsR0FBUixDQUFZTCxNQUFaLEVBQW9CQyxRQUFwQixDQUFKLEVBQW1DO0FBQ2pDLGlCQUFPRCxPQUFPQyxRQUFQLENBQVA7QUFDRCxTQUZELE1BRU8sSUFBSUMsUUFBUUUsUUFBUUMsR0FBUixDQUFZSCxJQUFaLEVBQWtCRCxRQUFsQixDQUFaLEVBQXlDO0FBQzlDLGNBQUlLLE1BQU1KLEtBQUtELFFBQUwsQ0FBVjtBQUNBLGNBQUksT0FBT0ssR0FBUCxLQUFlLFVBQW5CLEVBQStCO0FBQzdCQSxrQkFBTUEsSUFBSVYsSUFBSixDQUFTTSxJQUFULENBQU47QUFDRDtBQUNELGlCQUFPSSxHQUFQO0FBQ0QsU0FOTSxNQU1BO0FBQ0wsY0FBSUEsTUFBTWYsS0FBS1UsUUFBTCxDQUFWO0FBQ0EsY0FBSSxPQUFPSyxHQUFQLEtBQWUsVUFBbkIsRUFBK0I7QUFDN0JBLGtCQUFNQSxJQUFJVixJQUFKLENBQVNMLElBQVQsQ0FBTjtBQUNEO0FBQ0QsaUJBQU9lLEdBQVA7QUFDRDtBQUNGO0FBbEIrQixLQUFwQixDQUFkO0FBb0JBLFdBQU9ULEtBQVA7QUFDRDs7QUFFRFUsY0FBWW5CLElBQVosRUFBa0JDLFFBQVEsRUFBMUIsRUFBOEJDLEdBQTlCLEVBQW1DO0FBQ2pDLFNBQUtrQixPQUFMLEdBQWUsSUFBSUMsaUJBQUosRUFBZjtBQUNBLFNBQUtDLGFBQUwsR0FBcUIsSUFBSUMsNkJBQUosRUFBckI7O0FBRUEsU0FBS3ZCLElBQUwsR0FBWUEsSUFBWjtBQUNBLFNBQUtDLEtBQUwsR0FBYUEsS0FBYjtBQUNBLFNBQUtDLEdBQUwsR0FBV0EsR0FBWDtBQUNBLFNBQUtMLEdBQUwsR0FBVyxFQUFFQSxHQUFiO0FBQ0EsU0FBSzJCLE9BQUwsR0FBZUMsU0FBU0MsYUFBVCxDQUF1QixLQUF2QixDQUFmO0FBQ0EsU0FBS0YsT0FBTCxDQUFhRyxTQUFiLENBQXVCQyxHQUF2QixDQUE0QixtQkFBa0I1QixJQUFLLEVBQW5EO0FBQ0EsU0FBSzZCLFFBQUwsR0FBZ0IsSUFBaEI7QUFDQSxTQUFLQyxlQUFMLEdBQXVCLElBQUlDLE9BQUosQ0FBWUMsT0FBTztBQUN4QyxXQUFLQyxzQkFBTCxHQUE4QkQsR0FBOUI7QUFDRCxLQUZzQixDQUF2QjtBQUdEOztBQUVERSxjQUFZcEIsSUFBWixFQUFrQjtBQUNoQixTQUFLZSxRQUFMLEdBQWdCZixJQUFoQjs7QUFFQSxRQUFJLEtBQUtlLFFBQUwsQ0FBY00sa0JBQWxCLEVBQXNDO0FBQ3BDLFdBQUtOLFFBQUwsQ0FBY00sa0JBQWQsR0FBbUNDLElBQW5DLENBQXdDLEtBQUtILHNCQUE3QztBQUNELEtBRkQsTUFFTztBQUNMLFdBQUtBLHNCQUFMLENBQTRCLEtBQUtKLFFBQWpDO0FBQ0Q7O0FBRUQsU0FBS1QsT0FBTCxDQUFhaUIsSUFBYixDQUFrQixrQkFBbEI7QUFDQSxTQUFLakIsT0FBTCxDQUFhaUIsSUFBYixDQUFrQixpQkFBbEI7O0FBRUEsUUFBSXZCLEtBQUt3QixnQkFBVCxFQUEyQjtBQUN6QixXQUFLaEIsYUFBTCxDQUFtQk0sR0FBbkIsQ0FBdUJkLEtBQUt3QixnQkFBTCxDQUFzQixDQUFDLEdBQUdDLElBQUosS0FBYSxLQUFLbkIsT0FBTCxDQUFhaUIsSUFBYixDQUFrQixrQkFBbEIsRUFBc0MsR0FBR0UsSUFBekMsQ0FBbkMsQ0FBdkI7QUFDRDs7QUFFRCxRQUFJekIsS0FBSzBCLGVBQVQsRUFBMEI7QUFDeEIsV0FBS2xCLGFBQUwsQ0FBbUJNLEdBQW5CLENBQXVCZCxLQUFLMEIsZUFBTCxDQUFxQixDQUFDLEdBQUdELElBQUosS0FBYSxLQUFLbkIsT0FBTCxDQUFhaUIsSUFBYixDQUFrQixpQkFBbEIsRUFBcUMsR0FBR0UsSUFBeEMsQ0FBbEMsQ0FBdkI7QUFDRDs7QUFFRCxRQUFJekIsS0FBSzJCLFlBQVQsRUFBdUI7QUFDckIsV0FBS25CLGFBQUwsQ0FBbUJNLEdBQW5CLENBQXVCZCxLQUFLMkIsWUFBTCxDQUFrQixDQUFDLEdBQUdGLElBQUosS0FBYTtBQUNwRCxhQUFLVixRQUFMLEdBQWdCLElBQWhCO0FBQ0EsYUFBS1QsT0FBTCxDQUFhaUIsSUFBYixDQUFrQixhQUFsQixFQUFpQyxHQUFHRSxJQUFwQztBQUNELE9BSHNCLENBQXZCO0FBSUQ7QUFDRjs7QUFFREosdUJBQXFCO0FBQ25CLFdBQU8sS0FBS0wsZUFBWjtBQUNEOztBQUVEZixnQkFBYztBQUNaLFdBQU8sS0FBS2MsUUFBWjtBQUNEOztBQUVEYSxXQUFTO0FBQ1AsV0FBTyxLQUFLeEMsR0FBWjtBQUNEOztBQUVEeUMsYUFBVztBQUNULFdBQU8sS0FBSzFDLEtBQUwsQ0FBVzJDLEtBQVgsSUFBb0IsSUFBM0I7QUFDRDs7QUFFREMsZ0JBQWM7QUFDWixXQUFPLEtBQUs1QyxLQUFMLENBQVc2QyxRQUFYLElBQXVCLElBQTlCO0FBQ0Q7O0FBRURSLG1CQUFpQlMsRUFBakIsRUFBcUI7QUFDbkIsV0FBTyxLQUFLM0IsT0FBTCxDQUFhNEIsRUFBYixDQUFnQixrQkFBaEIsRUFBb0NELEVBQXBDLENBQVA7QUFDRDs7QUFFRFAsa0JBQWdCTyxFQUFoQixFQUFvQjtBQUNsQixXQUFPLEtBQUszQixPQUFMLENBQWE0QixFQUFiLENBQWdCLGlCQUFoQixFQUFtQ0QsRUFBbkMsQ0FBUDtBQUNEOztBQUVEekMsZUFBYTtBQUNYLFdBQU8sS0FBS2tCLE9BQVo7QUFDRDs7QUFFRGlCLGVBQWFNLEVBQWIsRUFBaUI7QUFDZixXQUFPLEtBQUszQixPQUFMLENBQWE0QixFQUFiLENBQWdCLGFBQWhCLEVBQStCRCxFQUEvQixDQUFQO0FBQ0Q7O0FBRUR4QyxZQUFVO0FBQ1IsU0FBSzBCLHNCQUFMLENBQTRCLElBQTVCO0FBQ0EsU0FBS1gsYUFBTCxDQUFtQjJCLE9BQW5CO0FBQ0EsU0FBSzdCLE9BQUwsQ0FBYTZCLE9BQWI7QUFDQSxRQUFJLEtBQUtwQixRQUFULEVBQW1CO0FBQ2pCLFdBQUtBLFFBQUwsQ0FBY3RCLE9BQWQsSUFBeUIsS0FBS3NCLFFBQUwsQ0FBY3RCLE9BQWQsRUFBekI7QUFDRDtBQUNGO0FBdkgyQjtrQkFBVFQsUSIsImZpbGUiOiJzdHViLWl0ZW0uanMiLCJzb3VyY2VSb290IjoiL2J1aWxkL2F0b20vc3JjL2F0b20tMS4zNS4xL291dC9hcHAvbm9kZV9tb2R1bGVzL2dpdGh1Yi9saWIvaXRlbXMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQge0VtaXR0ZXIsIENvbXBvc2l0ZURpc3Bvc2FibGV9IGZyb20gJ2V2ZW50LWtpdCc7XG5cbmxldCBrZXkgPSAwO1xuZXhwb3J0IGRlZmF1bHQgY2xhc3MgU3R1Ykl0ZW0ge1xuICAvLyBTdHViSXRlbXMgc2hvdWxkIG9ubHkgYmUgY3JlYXRlZCBieSBgY3JlYXRlYCBhbmQgbmV2ZXIgY29uc3RydWN0ZWQgZGlyZWN0bHkuXG4gIHN0YXRpYyBjcmVhdGUobmFtZSwgcHJvcHMsIHVyaSA9ICcnKSB7XG4gICAgY29uc3Qgc3R1YiA9IG5ldyBTdHViSXRlbShuYW1lLCBwcm9wcywgdXJpKTtcbiAgICBjb25zdCBvdmVycmlkZSA9IHtcbiAgICAgIF9nZXRTdHViOiAoKSA9PiBzdHViLFxuICAgICAgZ2V0RWxlbWVudDogKCkgPT4gc3R1Yi5nZXRFbGVtZW50KCksXG4gICAgICBkZXN0cm95OiBzdHViLmRlc3Ryb3kuYmluZChzdHViKSxcbiAgICB9O1xuICAgIGNvbnN0IHByb3h5ID0gbmV3IFByb3h5KG92ZXJyaWRlLCB7XG4gICAgICBnZXQodGFyZ2V0LCBwcm9wTmFtZSkge1xuICAgICAgICBjb25zdCBpdGVtID0gc3R1Yi5nZXRSZWFsSXRlbSgpO1xuICAgICAgICBpZiAoUmVmbGVjdC5oYXModGFyZ2V0LCBwcm9wTmFtZSkpIHtcbiAgICAgICAgICByZXR1cm4gdGFyZ2V0W3Byb3BOYW1lXTtcbiAgICAgICAgfSBlbHNlIGlmIChpdGVtICYmIFJlZmxlY3QuaGFzKGl0ZW0sIHByb3BOYW1lKSkge1xuICAgICAgICAgIGxldCB2YWwgPSBpdGVtW3Byb3BOYW1lXTtcbiAgICAgICAgICBpZiAodHlwZW9mIHZhbCA9PT0gJ2Z1bmN0aW9uJykge1xuICAgICAgICAgICAgdmFsID0gdmFsLmJpbmQoaXRlbSk7XG4gICAgICAgICAgfVxuICAgICAgICAgIHJldHVybiB2YWw7XG4gICAgICAgIH0gZWxzZSB7XG4gICAgICAgICAgbGV0IHZhbCA9IHN0dWJbcHJvcE5hbWVdO1xuICAgICAgICAgIGlmICh0eXBlb2YgdmFsID09PSAnZnVuY3Rpb24nKSB7XG4gICAgICAgICAgICB2YWwgPSB2YWwuYmluZChzdHViKTtcbiAgICAgICAgICB9XG4gICAgICAgICAgcmV0dXJuIHZhbDtcbiAgICAgICAgfVxuICAgICAgfSxcbiAgICB9KTtcbiAgICByZXR1cm4gcHJveHk7XG4gIH1cblxuICBjb25zdHJ1Y3RvcihuYW1lLCBwcm9wcyA9IHt9LCB1cmkpIHtcbiAgICB0aGlzLmVtaXR0ZXIgPSBuZXcgRW1pdHRlcigpO1xuICAgIHRoaXMuc3Vic2NyaXB0aW9ucyA9IG5ldyBDb21wb3NpdGVEaXNwb3NhYmxlKCk7XG5cbiAgICB0aGlzLm5hbWUgPSBuYW1lO1xuICAgIHRoaXMucHJvcHMgPSBwcm9wcztcbiAgICB0aGlzLnVyaSA9IHVyaTtcbiAgICB0aGlzLmtleSA9ICsra2V5O1xuICAgIHRoaXMuZWxlbWVudCA9IGRvY3VtZW50LmNyZWF0ZUVsZW1lbnQoJ2RpdicpO1xuICAgIHRoaXMuZWxlbWVudC5jbGFzc0xpc3QuYWRkKGBnaXRodWItU3R1Ykl0ZW0tJHtuYW1lfWApO1xuICAgIHRoaXMucmVhbEl0ZW0gPSBudWxsO1xuICAgIHRoaXMucmVhbEl0ZW1Qcm9taXNlID0gbmV3IFByb21pc2UocmVzID0+IHtcbiAgICAgIHRoaXMucmVzb2x2ZVJlYWxJdGVtUHJvbWlzZSA9IHJlcztcbiAgICB9KTtcbiAgfVxuXG4gIHNldFJlYWxJdGVtKGl0ZW0pIHtcbiAgICB0aGlzLnJlYWxJdGVtID0gaXRlbTtcblxuICAgIGlmICh0aGlzLnJlYWxJdGVtLmdldFJlYWxJdGVtUHJvbWlzZSkge1xuICAgICAgdGhpcy5yZWFsSXRlbS5nZXRSZWFsSXRlbVByb21pc2UoKS50aGVuKHRoaXMucmVzb2x2ZVJlYWxJdGVtUHJvbWlzZSk7XG4gICAgfSBlbHNlIHtcbiAgICAgIHRoaXMucmVzb2x2ZVJlYWxJdGVtUHJvbWlzZSh0aGlzLnJlYWxJdGVtKTtcbiAgICB9XG5cbiAgICB0aGlzLmVtaXR0ZXIuZW1pdCgnZGlkLWNoYW5nZS10aXRsZScpO1xuICAgIHRoaXMuZW1pdHRlci5lbWl0KCdkaWQtY2hhbmdlLWljb24nKTtcblxuICAgIGlmIChpdGVtLm9uRGlkQ2hhbmdlVGl0bGUpIHtcbiAgICAgIHRoaXMuc3Vic2NyaXB0aW9ucy5hZGQoaXRlbS5vbkRpZENoYW5nZVRpdGxlKCguLi5hcmdzKSA9PiB0aGlzLmVtaXR0ZXIuZW1pdCgnZGlkLWNoYW5nZS10aXRsZScsIC4uLmFyZ3MpKSk7XG4gICAgfVxuXG4gICAgaWYgKGl0ZW0ub25EaWRDaGFuZ2VJY29uKSB7XG4gICAgICB0aGlzLnN1YnNjcmlwdGlvbnMuYWRkKGl0ZW0ub25EaWRDaGFuZ2VJY29uKCguLi5hcmdzKSA9PiB0aGlzLmVtaXR0ZXIuZW1pdCgnZGlkLWNoYW5nZS1pY29uJywgLi4uYXJncykpKTtcbiAgICB9XG5cbiAgICBpZiAoaXRlbS5vbkRpZERlc3Ryb3kpIHtcbiAgICAgIHRoaXMuc3Vic2NyaXB0aW9ucy5hZGQoaXRlbS5vbkRpZERlc3Ryb3koKC4uLmFyZ3MpID0+IHtcbiAgICAgICAgdGhpcy5yZWFsSXRlbSA9IG51bGw7XG4gICAgICAgIHRoaXMuZW1pdHRlci5lbWl0KCdkaWQtZGVzdHJveScsIC4uLmFyZ3MpO1xuICAgICAgfSkpO1xuICAgIH1cbiAgfVxuXG4gIGdldFJlYWxJdGVtUHJvbWlzZSgpIHtcbiAgICByZXR1cm4gdGhpcy5yZWFsSXRlbVByb21pc2U7XG4gIH1cblxuICBnZXRSZWFsSXRlbSgpIHtcbiAgICByZXR1cm4gdGhpcy5yZWFsSXRlbTtcbiAgfVxuXG4gIGdldFVSSSgpIHtcbiAgICByZXR1cm4gdGhpcy51cmk7XG4gIH1cblxuICBnZXRUaXRsZSgpIHtcbiAgICByZXR1cm4gdGhpcy5wcm9wcy50aXRsZSB8fCBudWxsO1xuICB9XG5cbiAgZ2V0SWNvbk5hbWUoKSB7XG4gICAgcmV0dXJuIHRoaXMucHJvcHMuaWNvbk5hbWUgfHwgbnVsbDtcbiAgfVxuXG4gIG9uRGlkQ2hhbmdlVGl0bGUoY2IpIHtcbiAgICByZXR1cm4gdGhpcy5lbWl0dGVyLm9uKCdkaWQtY2hhbmdlLXRpdGxlJywgY2IpO1xuICB9XG5cbiAgb25EaWRDaGFuZ2VJY29uKGNiKSB7XG4gICAgcmV0dXJuIHRoaXMuZW1pdHRlci5vbignZGlkLWNoYW5nZS1pY29uJywgY2IpO1xuICB9XG5cbiAgZ2V0RWxlbWVudCgpIHtcbiAgICByZXR1cm4gdGhpcy5lbGVtZW50O1xuICB9XG5cbiAgb25EaWREZXN0cm95KGNiKSB7XG4gICAgcmV0dXJuIHRoaXMuZW1pdHRlci5vbignZGlkLWRlc3Ryb3knLCBjYik7XG4gIH1cblxuICBkZXN0cm95KCkge1xuICAgIHRoaXMucmVzb2x2ZVJlYWxJdGVtUHJvbWlzZShudWxsKTtcbiAgICB0aGlzLnN1YnNjcmlwdGlvbnMuZGlzcG9zZSgpO1xuICAgIHRoaXMuZW1pdHRlci5kaXNwb3NlKCk7XG4gICAgaWYgKHRoaXMucmVhbEl0ZW0pIHtcbiAgICAgIHRoaXMucmVhbEl0ZW0uZGVzdHJveSAmJiB0aGlzLnJlYWxJdGVtLmRlc3Ryb3koKTtcbiAgICB9XG4gIH1cbn1cbiJdfQ==