'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _react = require('react');

var _react2 = _interopRequireDefault(_react);

var _propTypes = require('prop-types');

var _propTypes2 = _interopRequireDefault(_propTypes);

var _eventKit = require('event-kit');

var _helpers = require('../helpers');

var _propTypes3 = require('../prop-types');

var _reporterProxy = require('../reporter-proxy');

var _repository = require('../models/repository');

var _repository2 = _interopRequireDefault(_repository);

var _endpoint = require('../models/endpoint');

var _issueishDetailContainer = require('../containers/issueish-detail-container');

var _issueishDetailContainer2 = _interopRequireDefault(_issueishDetailContainer);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

class IssueishDetailItem extends _react.Component {

  static buildURI(host, owner, repo, number, workdir = null) {
    const encodedWorkdir = workdir ? encodeURIComponent(workdir) : '';

    return 'atom-github://issueish/' + encodeURIComponent(host) + '/' + encodeURIComponent(owner) + '/' + encodeURIComponent(repo) + '/' + encodeURIComponent(number) + '?workdir=' + encodedWorkdir;
  }

  constructor(props) {
    super(props);

    this.destroy = () => {
      /* istanbul ignore else */
      if (!this.isDestroyed) {
        this.emitter.emit('did-destroy');
        this.isDestroyed = true;
      }
    };

    (0, _helpers.autobind)(this, 'switchToIssueish', 'handleTitleChanged');

    this.emitter = new _eventKit.Emitter();
    this.title = `${this.props.owner}/${this.props.repo}#${this.props.issueishNumber}`;
    this.hasTerminatedPendingState = false;

    const repository = this.props.workingDirectory === '' ? _repository2.default.absent() : this.props.workdirContextPool.add(this.props.workingDirectory).getRepository();

    this.state = {
      host: this.props.host,
      owner: this.props.owner,
      repo: this.props.repo,
      issueishNumber: this.props.issueishNumber,
      repository
    };

    if (repository.isAbsent()) {
      this.switchToIssueish(this.props.owner, this.props.repo, this.props.issueishNumber);
    }
  }

  render() {
    return _react2.default.createElement(_issueishDetailContainer2.default, {
      endpoint: (0, _endpoint.getEndpoint)(this.state.host),
      owner: this.state.owner,
      repo: this.state.repo,
      issueishNumber: this.state.issueishNumber,

      repository: this.state.repository,
      workspace: this.props.workspace,
      loginModel: this.props.loginModel,

      onTitleChange: this.handleTitleChanged,
      switchToIssueish: this.switchToIssueish,
      commands: this.props.commands,
      keymaps: this.props.keymaps,
      tooltips: this.props.tooltips,
      config: this.props.config,

      destroy: this.destroy,
      itemType: this.constructor
    });
  }

  async switchToIssueish(owner, repo, issueishNumber) {
    const pool = this.props.workdirContextPool;
    const prev = {
      owner: this.state.owner,
      repo: this.state.repo,
      issueishNumber: this.state.issueishNumber
    };

    const matchingRepositories = (await Promise.all(pool.withResidentContexts((workdir, context) => {
      const repository = context.getRepository();
      return repository.hasGitHubRemote(this.state.host, owner, repo).then(hasRemote => hasRemote ? repository : null);
    }))).filter(Boolean);
    const nextRepository = matchingRepositories.length === 1 ? matchingRepositories[0] : _repository2.default.absent();

    await new Promise(resolve => {
      this.setState((prevState, props) => {
        if (pool === props.workdirContextPool && prevState.owner === prev.owner && prevState.repo === prev.repo && prevState.issueishNumber === prev.issueishNumber) {
          (0, _reporterProxy.addEvent)('open-issueish-in-pane', { package: 'github', from: 'issueish-link', target: 'current-tab' });
          return {
            owner,
            repo,
            issueishNumber,
            repository: nextRepository
          };
        }

        return {};
      }, resolve);
    });
  }

  handleTitleChanged(title) {
    if (this.title !== title) {
      this.title = title;
      this.emitter.emit('did-change-title', title);
    }
  }

  onDidChangeTitle(cb) {
    return this.emitter.on('did-change-title', cb);
  }

  terminatePendingState() {
    if (!this.hasTerminatedPendingState) {
      this.emitter.emit('did-terminate-pending-state');
      this.hasTerminatedPendingState = true;
    }
  }

  onDidTerminatePendingState(callback) {
    return this.emitter.on('did-terminate-pending-state', callback);
  }

  onDidDestroy(callback) {
    return this.emitter.on('did-destroy', callback);
  }

  serialize() {
    return {
      uri: this.getURI(),
      deserializer: 'IssueishDetailItem'
    };
  }

  getTitle() {
    return this.title;
  }
}
exports.default = IssueishDetailItem;
IssueishDetailItem.propTypes = {
  // Issueish selection criteria
  // Parsed from item URI
  host: _propTypes2.default.string.isRequired,
  owner: _propTypes2.default.string.isRequired,
  repo: _propTypes2.default.string.isRequired,
  issueishNumber: _propTypes2.default.number.isRequired,
  workingDirectory: _propTypes2.default.string.isRequired,

  // Package models
  workdirContextPool: _propTypes3.WorkdirContextPoolPropType.isRequired,
  loginModel: _propTypes3.GithubLoginModelPropType.isRequired,

  // Atom environment
  workspace: _propTypes2.default.object.isRequired,
  commands: _propTypes2.default.object.isRequired,
  keymaps: _propTypes2.default.object.isRequired,
  tooltips: _propTypes2.default.object.isRequired,
  config: _propTypes2.default.object.isRequired
};
IssueishDetailItem.uriPattern = 'atom-github://issueish/{host}/{owner}/{repo}/{issueishNumber}?workdir={workingDirectory}';
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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