'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _extends = Object.assign || function (target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i]; for (var key in source) { if (Object.prototype.hasOwnProperty.call(source, key)) { target[key] = source[key]; } } } return target; };

var _fsExtra = require('fs-extra');

var _fsExtra2 = _interopRequireDefault(_fsExtra);

var _path = require('path');

var _path2 = _interopRequireDefault(_path);

var _electron = require('electron');

var _react = require('react');

var _react2 = _interopRequireDefault(_react);

var _propTypes = require('prop-types');

var _propTypes2 = _interopRequireDefault(_propTypes);

var _eventKit = require('event-kit');

var _statusBar = require('../atom/status-bar');

var _statusBar2 = _interopRequireDefault(_statusBar);

var _panel = require('../atom/panel');

var _panel2 = _interopRequireDefault(_panel);

var _paneItem = require('../atom/pane-item');

var _paneItem2 = _interopRequireDefault(_paneItem);

var _cloneDialog = require('../views/clone-dialog');

var _cloneDialog2 = _interopRequireDefault(_cloneDialog);

var _openIssueishDialog = require('../views/open-issueish-dialog');

var _openIssueishDialog2 = _interopRequireDefault(_openIssueishDialog);

var _openCommitDialog = require('../views/open-commit-dialog');

var _openCommitDialog2 = _interopRequireDefault(_openCommitDialog);

var _initDialog = require('../views/init-dialog');

var _initDialog2 = _interopRequireDefault(_initDialog);

var _credentialDialog = require('../views/credential-dialog');

var _credentialDialog2 = _interopRequireDefault(_credentialDialog);

var _commands = require('../atom/commands');

var _commands2 = _interopRequireDefault(_commands);

var _gitTimingsView = require('../views/git-timings-view');

var _gitTimingsView2 = _interopRequireDefault(_gitTimingsView);

var _changedFileItem = require('../items/changed-file-item');

var _changedFileItem2 = _interopRequireDefault(_changedFileItem);

var _issueishDetailItem = require('../items/issueish-detail-item');

var _issueishDetailItem2 = _interopRequireDefault(_issueishDetailItem);

var _commitDetailItem = require('../items/commit-detail-item');

var _commitDetailItem2 = _interopRequireDefault(_commitDetailItem);

var _commitPreviewItem = require('../items/commit-preview-item');

var _commitPreviewItem2 = _interopRequireDefault(_commitPreviewItem);

var _gitTabItem = require('../items/git-tab-item');

var _gitTabItem2 = _interopRequireDefault(_gitTabItem);

var _githubTabItem = require('../items/github-tab-item');

var _githubTabItem2 = _interopRequireDefault(_githubTabItem);

var _statusBarTileController = require('./status-bar-tile-controller');

var _statusBarTileController2 = _interopRequireDefault(_statusBarTileController);

var _repositoryConflictController = require('./repository-conflict-controller');

var _repositoryConflictController2 = _interopRequireDefault(_repositoryConflictController);

var _gitCacheView = require('../views/git-cache-view');

var _gitCacheView2 = _interopRequireDefault(_gitCacheView);

var _conflict = require('../models/conflicts/conflict');

var _conflict2 = _interopRequireDefault(_conflict);

var _switchboard = require('../switchboard');

var _switchboard2 = _interopRequireDefault(_switchboard);

var _propTypes3 = require('../prop-types');

var _helpers = require('../helpers');

var _gitShellOutStrategy = require('../git-shell-out-strategy');

var _reporterProxy = require('../reporter-proxy');

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

class RootController extends _react2.default.Component {

  constructor(props, context) {
    super(props, context);

    this.toggleCommitPreviewItem = () => {
      const workdir = this.props.repository.getWorkingDirectoryPath();
      return this.props.workspace.toggle(_commitPreviewItem2.default.buildURI(workdir));
    };

    this.showOpenCommitDialog = () => {
      this.setState({ openCommitDialogActive: true });
    };

    this.isValidCommit = async ref => {
      try {
        await this.props.repository.getCommit(ref);
        return true;
      } catch (error) {
        if (error instanceof _gitShellOutStrategy.GitError && error.code === 128) {
          return false;
        } else {
          throw error;
        }
      }
    };

    this.acceptOpenCommit = ({ ref }) => {
      const workdir = this.props.repository.getWorkingDirectoryPath();
      const uri = _commitDetailItem2.default.buildURI(workdir, ref);
      this.setState({ openCommitDialogActive: false });
      this.props.workspace.open(uri).then(() => {
        (0, _reporterProxy.addEvent)('open-commit-in-pane', { package: 'github', from: _openCommitDialog2.default.name });
      });
    };

    this.cancelOpenCommit = () => {
      this.setState({ openCommitDialogActive: false });
    };

    this.surfaceFromFileAtPath = (filePath, stagingStatus) => {
      const gitTab = this.gitTabTracker.getComponent();
      return gitTab && gitTab.focusAndSelectStagingItem(filePath, stagingStatus);
    };

    this.surfaceToCommitPreviewButton = () => {
      const gitTab = this.gitTabTracker.getComponent();
      return gitTab && gitTab.focusAndSelectCommitPreviewButton();
    };

    this.surfaceToRecentCommit = () => {
      const gitTab = this.gitTabTracker.getComponent();
      return gitTab && gitTab.focusAndSelectRecentCommit();
    };

    (0, _helpers.autobind)(this, 'installReactDevTools', 'clearGithubToken', 'initializeRepo', 'showOpenIssueishDialog', 'showWaterfallDiagnostics', 'showCacheDiagnostics', 'acceptClone', 'cancelClone', 'acceptInit', 'cancelInit', 'acceptOpenIssueish', 'cancelOpenIssueish', 'destroyFilePatchPaneItems', 'destroyEmptyFilePatchPaneItems', 'openCloneDialog', 'quietlySelectItem', 'viewUnstagedChangesForCurrentFile', 'viewStagedChangesForCurrentFile', 'openFiles', 'getUnsavedFiles', 'ensureNoUnsavedFiles', 'discardWorkDirChangesForPaths', 'discardLines', 'undoLastDiscard', 'refreshResolutionProgress');

    this.state = {
      cloneDialogActive: false,
      cloneDialogInProgress: false,
      initDialogActive: false,
      initDialogPath: null,
      initDialogResolve: null,
      credentialDialogQuery: null
    };

    this.gitTabTracker = new TabTracker('git', {
      uri: _gitTabItem2.default.buildURI(),
      getWorkspace: () => this.props.workspace
    });

    this.githubTabTracker = new TabTracker('github', {
      uri: _githubTabItem2.default.buildURI(),
      getWorkspace: () => this.props.workspace
    });

    this.subscription = new _eventKit.CompositeDisposable(this.props.repository.onPullError(this.gitTabTracker.ensureVisible));

    this.props.commandRegistry.onDidDispatch(event => {
      if (event.type && event.type.startsWith('github:') && event.detail && event.detail[0] && event.detail[0].contextCommand) {
        (0, _reporterProxy.addEvent)('context-menu-action', {
          package: 'github',
          command: event.type
        });
      }
    });
  }

  componentDidMount() {
    this.openTabs();
  }

  render() {
    return _react2.default.createElement(
      _react.Fragment,
      null,
      this.renderCommands(),
      this.renderStatusBarTile(),
      this.renderPaneItems(),
      this.renderDialogs(),
      this.renderConflictResolver()
    );
  }

  renderCommands() {
    const devMode = global.atom && global.atom.inDevMode();

    return _react2.default.createElement(
      _commands2.default,
      { registry: this.props.commandRegistry, target: 'atom-workspace' },
      devMode && _react2.default.createElement(_commands.Command, { command: 'github:install-react-dev-tools', callback: this.installReactDevTools }),
      _react2.default.createElement(_commands.Command, { command: 'github:toggle-commit-preview', callback: this.toggleCommitPreviewItem }),
      _react2.default.createElement(_commands.Command, { command: 'github:logout', callback: this.clearGithubToken }),
      _react2.default.createElement(_commands.Command, { command: 'github:show-waterfall-diagnostics', callback: this.showWaterfallDiagnostics }),
      _react2.default.createElement(_commands.Command, { command: 'github:show-cache-diagnostics', callback: this.showCacheDiagnostics }),
      _react2.default.createElement(_commands.Command, { command: 'github:open-issue-or-pull-request', callback: this.showOpenIssueishDialog }),
      _react2.default.createElement(_commands.Command, { command: 'github:toggle-git-tab', callback: this.gitTabTracker.toggle }),
      _react2.default.createElement(_commands.Command, { command: 'github:toggle-git-tab-focus', callback: this.gitTabTracker.toggleFocus }),
      _react2.default.createElement(_commands.Command, { command: 'github:toggle-github-tab', callback: this.githubTabTracker.toggle }),
      _react2.default.createElement(_commands.Command, { command: 'github:toggle-github-tab-focus', callback: this.githubTabTracker.toggleFocus }),
      _react2.default.createElement(_commands.Command, { command: 'github:clone', callback: this.openCloneDialog }),
      _react2.default.createElement(_commands.Command, { command: 'github:open-commit', callback: this.showOpenCommitDialog }),
      _react2.default.createElement(_commands.Command, {
        command: 'github:view-unstaged-changes-for-current-file',
        callback: this.viewUnstagedChangesForCurrentFile
      }),
      _react2.default.createElement(_commands.Command, {
        command: 'github:view-staged-changes-for-current-file',
        callback: this.viewStagedChangesForCurrentFile
      }),
      _react2.default.createElement(_commands.Command, {
        command: 'github:close-all-diff-views',
        callback: this.destroyFilePatchPaneItems
      }),
      _react2.default.createElement(_commands.Command, {
        command: 'github:close-empty-diff-views',
        callback: this.destroyEmptyFilePatchPaneItems
      })
    );
  }

  renderStatusBarTile() {
    return _react2.default.createElement(
      _statusBar2.default,
      {
        statusBar: this.props.statusBar,
        onConsumeStatusBar: sb => this.onConsumeStatusBar(sb),
        className: 'github-StatusBarTileController' },
      _react2.default.createElement(_statusBarTileController2.default, {
        pipelineManager: this.props.pipelineManager,
        workspace: this.props.workspace,
        repository: this.props.repository,
        commandRegistry: this.props.commandRegistry,
        notificationManager: this.props.notificationManager,
        tooltips: this.props.tooltips,
        confirm: this.props.confirm,
        toggleGitTab: this.gitTabTracker.toggle,
        toggleGithubTab: this.githubTabTracker.toggle
      })
    );
  }

  renderDialogs() {
    return _react2.default.createElement(
      _react.Fragment,
      null,
      this.renderInitDialog(),
      this.renderCloneDialog(),
      this.renderCredentialDialog(),
      this.renderOpenIssueishDialog(),
      this.renderOpenCommitDialog()
    );
  }

  renderInitDialog() {
    if (!this.state.initDialogActive) {
      return null;
    }

    return _react2.default.createElement(
      _panel2.default,
      { workspace: this.props.workspace, location: 'modal' },
      _react2.default.createElement(_initDialog2.default, {
        config: this.props.config,
        commandRegistry: this.props.commandRegistry,
        initPath: this.state.initDialogPath,
        didAccept: this.acceptInit,
        didCancel: this.cancelInit
      })
    );
  }

  renderCloneDialog() {
    if (!this.state.cloneDialogActive) {
      return null;
    }

    return _react2.default.createElement(
      _panel2.default,
      { workspace: this.props.workspace, location: 'modal' },
      _react2.default.createElement(_cloneDialog2.default, {
        config: this.props.config,
        commandRegistry: this.props.commandRegistry,
        didAccept: this.acceptClone,
        didCancel: this.cancelClone,
        inProgress: this.state.cloneDialogInProgress
      })
    );
  }

  renderOpenIssueishDialog() {
    if (!this.state.openIssueishDialogActive) {
      return null;
    }

    return _react2.default.createElement(
      _panel2.default,
      { workspace: this.props.workspace, location: 'modal' },
      _react2.default.createElement(_openIssueishDialog2.default, {
        commandRegistry: this.props.commandRegistry,
        didAccept: this.acceptOpenIssueish,
        didCancel: this.cancelOpenIssueish
      })
    );
  }

  renderOpenCommitDialog() {
    if (!this.state.openCommitDialogActive) {
      return null;
    }

    return _react2.default.createElement(
      _panel2.default,
      { workspace: this.props.workspace, location: 'modal' },
      _react2.default.createElement(_openCommitDialog2.default, {
        commandRegistry: this.props.commandRegistry,
        didAccept: this.acceptOpenCommit,
        didCancel: this.cancelOpenCommit,
        isValidEntry: this.isValidCommit
      })
    );
  }

  renderCredentialDialog() {
    if (this.state.credentialDialogQuery === null) {
      return null;
    }

    return _react2.default.createElement(
      _panel2.default,
      { workspace: this.props.workspace, location: 'modal' },
      _react2.default.createElement(_credentialDialog2.default, _extends({ commandRegistry: this.props.commandRegistry }, this.state.credentialDialogQuery))
    );
  }

  renderConflictResolver() {
    if (!this.props.repository) {
      return null;
    }

    return _react2.default.createElement(_repositoryConflictController2.default, {
      workspace: this.props.workspace,
      config: this.props.config,
      repository: this.props.repository,
      resolutionProgress: this.props.resolutionProgress,
      refreshResolutionProgress: this.refreshResolutionProgress,
      commandRegistry: this.props.commandRegistry
    });
  }

  renderPaneItems() {
    return _react2.default.createElement(
      _react.Fragment,
      null,
      _react2.default.createElement(
        _paneItem2.default,
        {
          workspace: this.props.workspace,
          uriPattern: _gitTabItem2.default.uriPattern,
          className: 'github-Git-root' },
        ({ itemHolder }) => _react2.default.createElement(_gitTabItem2.default, {
          ref: itemHolder.setter,
          workspace: this.props.workspace,
          commandRegistry: this.props.commandRegistry,
          notificationManager: this.props.notificationManager,
          tooltips: this.props.tooltips,
          grammars: this.props.grammars,
          project: this.props.project,
          confirm: this.props.confirm,
          config: this.props.config,
          repository: this.props.repository,
          loginModel: this.props.loginModel,
          initializeRepo: this.initializeRepo,
          resolutionProgress: this.props.resolutionProgress,
          ensureGitTab: this.gitTabTracker.ensureVisible,
          openFiles: this.openFiles,
          discardWorkDirChangesForPaths: this.discardWorkDirChangesForPaths,
          undoLastDiscard: this.undoLastDiscard,
          refreshResolutionProgress: this.refreshResolutionProgress
        })
      ),
      _react2.default.createElement(
        _paneItem2.default,
        {
          workspace: this.props.workspace,
          uriPattern: _githubTabItem2.default.uriPattern,
          className: 'github-GitHub-root' },
        ({ itemHolder }) => _react2.default.createElement(_githubTabItem2.default, {
          ref: itemHolder.setter,
          repository: this.props.repository,
          loginModel: this.props.loginModel,
          workspace: this.props.workspace
        })
      ),
      _react2.default.createElement(
        _paneItem2.default,
        {
          workspace: this.props.workspace,
          uriPattern: _changedFileItem2.default.uriPattern },
        ({ itemHolder, params }) => _react2.default.createElement(_changedFileItem2.default, {
          ref: itemHolder.setter,

          workdirContextPool: this.props.workdirContextPool,
          relPath: _path2.default.join(...params.relPath),
          workingDirectory: params.workingDirectory,
          stagingStatus: params.stagingStatus,

          tooltips: this.props.tooltips,
          commands: this.props.commandRegistry,
          keymaps: this.props.keymaps,
          workspace: this.props.workspace,
          config: this.props.config,

          discardLines: this.discardLines,
          undoLastDiscard: this.undoLastDiscard,
          surfaceFileAtPath: this.surfaceFromFileAtPath
        })
      ),
      _react2.default.createElement(
        _paneItem2.default,
        {
          workspace: this.props.workspace,
          uriPattern: _commitPreviewItem2.default.uriPattern,
          className: 'github-CommitPreview-root' },
        ({ itemHolder, params }) => _react2.default.createElement(_commitPreviewItem2.default, {
          ref: itemHolder.setter,

          workdirContextPool: this.props.workdirContextPool,
          workingDirectory: params.workingDirectory,
          workspace: this.props.workspace,
          commands: this.props.commandRegistry,
          keymaps: this.props.keymaps,
          tooltips: this.props.tooltips,
          config: this.props.config,

          discardLines: this.discardLines,
          undoLastDiscard: this.undoLastDiscard,
          surfaceToCommitPreviewButton: this.surfaceToCommitPreviewButton
        })
      ),
      _react2.default.createElement(
        _paneItem2.default,
        {
          workspace: this.props.workspace,
          uriPattern: _commitDetailItem2.default.uriPattern,
          className: 'github-CommitDetail-root' },
        ({ itemHolder, params }) => _react2.default.createElement(_commitDetailItem2.default, {
          ref: itemHolder.setter,

          workdirContextPool: this.props.workdirContextPool,
          workingDirectory: params.workingDirectory,
          workspace: this.props.workspace,
          commands: this.props.commandRegistry,
          keymaps: this.props.keymaps,
          tooltips: this.props.tooltips,
          config: this.props.config,

          sha: params.sha,
          surfaceCommit: this.surfaceToRecentCommit
        })
      ),
      _react2.default.createElement(
        _paneItem2.default,
        { workspace: this.props.workspace, uriPattern: _issueishDetailItem2.default.uriPattern },
        ({ itemHolder, params }) => _react2.default.createElement(_issueishDetailItem2.default, {
          ref: itemHolder.setter,

          host: params.host,
          owner: params.owner,
          repo: params.repo,
          issueishNumber: parseInt(params.issueishNumber, 10),

          workingDirectory: params.workingDirectory,
          workdirContextPool: this.props.workdirContextPool,
          loginModel: this.props.loginModel,

          workspace: this.props.workspace,
          commands: this.props.commandRegistry,
          keymaps: this.props.keymaps,
          tooltips: this.props.tooltips,
          config: this.props.config
        })
      ),
      _react2.default.createElement(
        _paneItem2.default,
        { workspace: this.props.workspace, uriPattern: _gitTimingsView2.default.uriPattern },
        ({ itemHolder }) => _react2.default.createElement(_gitTimingsView2.default, { ref: itemHolder.setter })
      ),
      _react2.default.createElement(
        _paneItem2.default,
        { workspace: this.props.workspace, uriPattern: _gitCacheView2.default.uriPattern },
        ({ itemHolder }) => _react2.default.createElement(_gitCacheView2.default, { ref: itemHolder.setter, repository: this.props.repository })
      )
    );
  }

  async openTabs() {
    if (this.props.startOpen) {
      await Promise.all([this.gitTabTracker.ensureRendered(false), this.githubTabTracker.ensureRendered(false)]);
    }

    if (this.props.startRevealed) {
      const docks = new Set([_gitTabItem2.default.buildURI(), _githubTabItem2.default.buildURI()].map(uri => this.props.workspace.paneContainerForURI(uri)).filter(container => container && typeof container.show === 'function'));

      for (const dock of docks) {
        dock.show();
      }
    }
  }

  async installReactDevTools() {
    // Prevent electron-link from attempting to descend into electron-devtools-installer, which is not available
    // when we're bundled in Atom.
    const devToolsName = 'electron-devtools-installer';
    const devTools = require(devToolsName);

    const crossUnzipName = 'cross-unzip';
    const unzip = require(crossUnzipName);

    const reactId = devTools.REACT_DEVELOPER_TOOLS.id;

    const url = 'https://clients2.google.com/service/update2/crx?' + `response=redirect&x=id%3D${reactId}%26uc&prodversion=32`;
    const extensionFolder = _path2.default.resolve(_electron.remote.app.getPath('userData'), `extensions/${reactId}`);
    const extensionFile = `${extensionFolder}.crx`;
    await _fsExtra2.default.ensureDir(_path2.default.dirname(extensionFile));
    const response = await fetch(url, { method: 'GET' });
    const body = Buffer.from((await response.arrayBuffer()));
    await _fsExtra2.default.writeFile(extensionFile, body);

    await new Promise((resolve, reject) => {
      unzip(extensionFile, extensionFolder, async err => {
        if (err && !(await _fsExtra2.default.exists(_path2.default.join(extensionFolder, 'manifest.json')))) {
          reject(err);
        }

        resolve();
      });
    });

    await _fsExtra2.default.ensureDir(extensionFolder, 0o755);
    await devTools.default(devTools.REACT_DEVELOPER_TOOLS);

    this.props.notificationManager.addSuccess('🌈 Reload your window to start using the React dev tools!');
  }

  componentWillUnmount() {
    this.subscription.dispose();
  }

  componentDidUpdate() {
    this.subscription.dispose();
    this.subscription = new _eventKit.CompositeDisposable(this.props.repository.onPullError(() => this.gitTabTracker.ensureVisible()));
  }

  onConsumeStatusBar(statusBar) {
    if (statusBar.disableGitInfoTile) {
      statusBar.disableGitInfoTile();
    }
  }

  clearGithubToken() {
    return this.props.loginModel.removeToken('https://api.github.com');
  }

  initializeRepo(initDialogPath) {
    if (this.state.initDialogActive) {
      return null;
    }

    if (!initDialogPath) {
      initDialogPath = this.props.repository.getWorkingDirectoryPath();
    }

    return new Promise(resolve => {
      this.setState({ initDialogActive: true, initDialogPath, initDialogResolve: resolve });
    });
  }

  showOpenIssueishDialog() {
    this.setState({ openIssueishDialogActive: true });
  }

  showWaterfallDiagnostics() {
    this.props.workspace.open(_gitTimingsView2.default.buildURI());
  }

  showCacheDiagnostics() {
    this.props.workspace.open(_gitCacheView2.default.buildURI());
  }

  async acceptClone(remoteUrl, projectPath) {
    this.setState({ cloneDialogInProgress: true });
    try {
      await this.props.cloneRepositoryForProjectPath(remoteUrl, projectPath);
      (0, _reporterProxy.addEvent)('clone-repo', { package: 'github' });
    } catch (e) {
      this.props.notificationManager.addError(`Unable to clone ${remoteUrl}`, { detail: e.stdErr, dismissable: true });
    } finally {
      this.setState({ cloneDialogInProgress: false, cloneDialogActive: false });
    }
  }

  cancelClone() {
    this.setState({ cloneDialogActive: false });
  }

  async acceptInit(projectPath) {
    try {
      await this.props.createRepositoryForProjectPath(projectPath);
      if (this.state.initDialogResolve) {
        this.state.initDialogResolve(projectPath);
      }
    } catch (e) {
      this.props.notificationManager.addError(`Unable to initialize git repository in ${projectPath}`, { detail: e.stdErr, dismissable: true });
    } finally {
      this.setState({ initDialogActive: false, initDialogPath: null, initDialogResolve: null });
    }
  }

  cancelInit() {
    if (this.state.initDialogResolve) {
      this.state.initDialogResolve(false);
    }
    this.setState({ initDialogActive: false, initDialogPath: null, initDialogResolve: null });
  }

  acceptOpenIssueish({ repoOwner, repoName, issueishNumber }) {
    const uri = _issueishDetailItem2.default.buildURI('github.com', repoOwner, repoName, issueishNumber);
    this.setState({ openIssueishDialogActive: false });
    this.props.workspace.open(uri).then(() => {
      (0, _reporterProxy.addEvent)('open-issueish-in-pane', { package: 'github', from: 'dialog' });
    });
  }

  cancelOpenIssueish() {
    this.setState({ openIssueishDialogActive: false });
  }

  destroyFilePatchPaneItems() {
    (0, _helpers.destroyFilePatchPaneItems)({ onlyStaged: false }, this.props.workspace);
  }

  destroyEmptyFilePatchPaneItems() {
    (0, _helpers.destroyEmptyFilePatchPaneItems)(this.props.workspace);
  }

  openCloneDialog() {
    this.setState({ cloneDialogActive: true });
  }

  quietlySelectItem(filePath, stagingStatus) {
    const gitTab = this.gitTabTracker.getComponent();
    return gitTab && gitTab.quietlySelectItem(filePath, stagingStatus);
  }

  async viewChangesForCurrentFile(stagingStatus) {
    const editor = this.props.workspace.getActiveTextEditor();
    if (!editor.getPath()) {
      return;
    }

    const absFilePath = await _fsExtra2.default.realpath(editor.getPath());
    const repoPath = this.props.repository.getWorkingDirectoryPath();
    if (repoPath === null) {
      const [projectPath] = this.props.project.relativizePath(editor.getPath());
      const notification = this.props.notificationManager.addInfo("Hmm, there's nothing to compare this file to", {
        description: 'You can create a Git repository to track changes to the files in your project.',
        dismissable: true,
        buttons: [{
          className: 'btn btn-primary',
          text: 'Create a repository now',
          onDidClick: async () => {
            notification.dismiss();
            const createdPath = await this.initializeRepo(projectPath);
            // If the user confirmed repository creation for this project path,
            // retry the operation that got them here in the first place
            if (createdPath === projectPath) {
              this.viewChangesForCurrentFile(stagingStatus);
            }
          }
        }]
      });
      return;
    }
    if (absFilePath.startsWith(repoPath)) {
      const filePath = absFilePath.slice(repoPath.length + 1);
      this.quietlySelectItem(filePath, stagingStatus);
      const splitDirection = this.props.config.get('github.viewChangesForCurrentFileDiffPaneSplitDirection');
      const pane = this.props.workspace.getActivePane();
      if (splitDirection === 'right') {
        pane.splitRight();
      } else if (splitDirection === 'down') {
        pane.splitDown();
      }
      const lineNum = editor.getCursorBufferPosition().row + 1;
      const item = await this.props.workspace.open(_changedFileItem2.default.buildURI(filePath, repoPath, stagingStatus), { pending: true, activatePane: true, activateItem: true });
      await item.getRealItemPromise();
      await item.getFilePatchLoadedPromise();
      item.goToDiffLine(lineNum);
      item.focus();
    } else {
      throw new Error(`${absFilePath} does not belong to repo ${repoPath}`);
    }
  }

  viewUnstagedChangesForCurrentFile() {
    return this.viewChangesForCurrentFile('unstaged');
  }

  viewStagedChangesForCurrentFile() {
    return this.viewChangesForCurrentFile('staged');
  }

  openFiles(filePaths, repository = this.props.repository) {
    return Promise.all(filePaths.map(filePath => {
      const absolutePath = _path2.default.join(repository.getWorkingDirectoryPath(), filePath);
      return this.props.workspace.open(absolutePath, { pending: filePaths.length === 1 });
    }));
  }

  getUnsavedFiles(filePaths, workdirPath) {
    const isModifiedByPath = new Map();
    this.props.workspace.getTextEditors().forEach(editor => {
      isModifiedByPath.set(editor.getPath(), editor.isModified());
    });
    return filePaths.filter(filePath => {
      const absFilePath = _path2.default.join(workdirPath, filePath);
      return isModifiedByPath.get(absFilePath);
    });
  }

  ensureNoUnsavedFiles(filePaths, message, workdirPath = this.props.repository.getWorkingDirectoryPath()) {
    const unsavedFiles = this.getUnsavedFiles(filePaths, workdirPath).map(filePath => `\`${filePath}\``).join('<br>');
    if (unsavedFiles.length) {
      this.props.notificationManager.addError(message, {
        description: `You have unsaved changes in:<br>${unsavedFiles}.`,
        dismissable: true
      });
      return false;
    } else {
      return true;
    }
  }

  async discardWorkDirChangesForPaths(filePaths) {
    const destructiveAction = () => {
      return this.props.repository.discardWorkDirChangesForPaths(filePaths);
    };
    return await this.props.repository.storeBeforeAndAfterBlobs(filePaths, () => this.ensureNoUnsavedFiles(filePaths, 'Cannot discard changes in selected files.'), destructiveAction);
  }

  async discardLines(multiFilePatch, lines, repository = this.props.repository) {
    // (kuychaco) For now we only support discarding rows for MultiFilePatches that contain a single file patch
    // The only way to access this method from the UI is to be in a ChangedFileItem, which only has a single file patch
    if (multiFilePatch.getFilePatches().length !== 1) {
      return Promise.resolve(null);
    }

    const filePath = multiFilePatch.getFilePatches()[0].getPath();
    const destructiveAction = async () => {
      const discardFilePatch = multiFilePatch.getUnstagePatchForLines(lines);
      await repository.applyPatchToWorkdir(discardFilePatch);
    };
    return await repository.storeBeforeAndAfterBlobs([filePath], () => this.ensureNoUnsavedFiles([filePath], 'Cannot discard lines.', repository.getWorkingDirectoryPath()), destructiveAction, filePath);
  }

  getFilePathsForLastDiscard(partialDiscardFilePath = null) {
    let lastSnapshots = this.props.repository.getLastHistorySnapshots(partialDiscardFilePath);
    if (partialDiscardFilePath) {
      lastSnapshots = lastSnapshots ? [lastSnapshots] : [];
    }
    return lastSnapshots.map(snapshot => snapshot.filePath);
  }

  async undoLastDiscard(partialDiscardFilePath = null, repository = this.props.repository) {
    const filePaths = this.getFilePathsForLastDiscard(partialDiscardFilePath);
    try {
      const results = await repository.restoreLastDiscardInTempFiles(() => this.ensureNoUnsavedFiles(filePaths, 'Cannot undo last discard.'), partialDiscardFilePath);
      if (results.length === 0) {
        return;
      }
      await this.proceedOrPromptBasedOnResults(results, partialDiscardFilePath);
    } catch (e) {
      if (e instanceof _gitShellOutStrategy.GitError && e.stdErr.match(/fatal: Not a valid object name/)) {
        this.cleanUpHistoryForFilePaths(filePaths, partialDiscardFilePath);
      } else {
        // eslint-disable-next-line no-console
        console.error(e);
      }
    }
  }

  async proceedOrPromptBasedOnResults(results, partialDiscardFilePath = null) {
    const conflicts = results.filter(({ conflict }) => conflict);
    if (conflicts.length === 0) {
      await this.proceedWithLastDiscardUndo(results, partialDiscardFilePath);
    } else {
      await this.promptAboutConflicts(results, conflicts, partialDiscardFilePath);
    }
  }

  async promptAboutConflicts(results, conflicts, partialDiscardFilePath = null) {
    const conflictedFiles = conflicts.map(({ filePath }) => `\t${filePath}`).join('\n');
    const choice = this.props.confirm({
      message: 'Undoing will result in conflicts...',
      detailedMessage: `for the following files:\n${conflictedFiles}\n` + 'Would you like to apply the changes with merge conflict markers, ' + 'or open the text with merge conflict markers in a new file?',
      buttons: ['Merge with conflict markers', 'Open in new file', 'Cancel']
    });
    if (choice === 0) {
      await this.proceedWithLastDiscardUndo(results, partialDiscardFilePath);
    } else if (choice === 1) {
      await this.openConflictsInNewEditors(conflicts.map(({ resultPath }) => resultPath));
    }
  }

  cleanUpHistoryForFilePaths(filePaths, partialDiscardFilePath = null) {
    this.props.repository.clearDiscardHistory(partialDiscardFilePath);
    const filePathsStr = filePaths.map(filePath => `\`${filePath}\``).join('<br>');
    this.props.notificationManager.addError('Discard history has expired.', {
      description: `Cannot undo discard for<br>${filePathsStr}<br>Stale discard history has been deleted.`,
      dismissable: true
    });
  }

  async proceedWithLastDiscardUndo(results, partialDiscardFilePath = null) {
    const promises = results.map(async result => {
      const { filePath, resultPath, deleted, conflict, theirsSha, commonBaseSha, currentSha } = result;
      const absFilePath = _path2.default.join(this.props.repository.getWorkingDirectoryPath(), filePath);
      if (deleted && resultPath === null) {
        await _fsExtra2.default.remove(absFilePath);
      } else {
        await _fsExtra2.default.copy(resultPath, absFilePath);
      }
      if (conflict) {
        await this.props.repository.writeMergeConflictToIndex(filePath, commonBaseSha, currentSha, theirsSha);
      }
    });
    await Promise.all(promises);
    await this.props.repository.popDiscardHistory(partialDiscardFilePath);
  }

  async openConflictsInNewEditors(resultPaths) {
    const editorPromises = resultPaths.map(resultPath => {
      return this.props.workspace.open(resultPath);
    });
    return await Promise.all(editorPromises);
  }

  /*
   * Asynchronously count the conflict markers present in a file specified by full path.
   */
  refreshResolutionProgress(fullPath) {
    const readStream = _fsExtra2.default.createReadStream(fullPath, { encoding: 'utf8' });
    return new Promise(resolve => {
      _conflict2.default.countFromStream(readStream).then(count => {
        this.props.resolutionProgress.reportMarkerCount(fullPath, count);
      });
    });
  }

  /*
   * Display the credential entry dialog. Return a Promise that will resolve with the provided credentials on accept
   * or reject on cancel.
   */
  promptForCredentials(query) {
    return new Promise((resolve, reject) => {
      this.setState({
        credentialDialogQuery: _extends({}, query, {
          onSubmit: response => this.setState({ credentialDialogQuery: null }, () => resolve(response)),
          onCancel: () => this.setState({ credentialDialogQuery: null }, reject)
        })
      });
    });
  }
}

exports.default = RootController;
RootController.propTypes = {
  workspace: _propTypes2.default.object.isRequired,
  commandRegistry: _propTypes2.default.object.isRequired,
  deserializers: _propTypes2.default.object.isRequired,
  notificationManager: _propTypes2.default.object.isRequired,
  tooltips: _propTypes2.default.object.isRequired,
  keymaps: _propTypes2.default.object.isRequired,
  grammars: _propTypes2.default.object.isRequired,
  config: _propTypes2.default.object.isRequired,
  project: _propTypes2.default.object.isRequired,
  loginModel: _propTypes2.default.object.isRequired,
  confirm: _propTypes2.default.func.isRequired,
  createRepositoryForProjectPath: _propTypes2.default.func,
  cloneRepositoryForProjectPath: _propTypes2.default.func,
  repository: _propTypes2.default.object.isRequired,
  resolutionProgress: _propTypes2.default.object.isRequired,
  statusBar: _propTypes2.default.object,
  switchboard: _propTypes2.default.instanceOf(_switchboard2.default),
  startOpen: _propTypes2.default.bool,
  startRevealed: _propTypes2.default.bool,
  pipelineManager: _propTypes2.default.object,
  workdirContextPool: _propTypes3.WorkdirContextPoolPropType.isRequired
};
RootController.defaultProps = {
  switchboard: new _switchboard2.default(),
  startOpen: false,
  startRevealed: false
};
class TabTracker {
  constructor(name, { getWorkspace, uri }) {
    (0, _helpers.autobind)(this, 'toggle', 'toggleFocus', 'ensureVisible');
    this.name = name;

    this.getWorkspace = getWorkspace;
    this.uri = uri;
  }

  async toggle() {
    const focusToRestore = document.activeElement;
    let shouldRestoreFocus = false;

    // Rendered => the dock item is being rendered, whether or not the dock is visible or the item
    //   is visible within its dock.
    // Visible => the item is active and the dock item is active within its dock.
    const wasRendered = this.isRendered();
    const wasVisible = this.isVisible();

    if (!wasRendered || !wasVisible) {
      // Not rendered, or rendered but not an active item in a visible dock.
      await this.reveal();
      shouldRestoreFocus = true;
    } else {
      // Rendered and an active item within a visible dock.
      await this.hide();
      shouldRestoreFocus = false;
    }

    if (shouldRestoreFocus) {
      process.nextTick(() => focusToRestore.focus());
    }
  }

  async toggleFocus() {
    const hadFocus = this.hasFocus();
    await this.ensureVisible();

    if (hadFocus) {
      let workspace = this.getWorkspace();
      if (workspace.getCenter) {
        workspace = workspace.getCenter();
      }
      workspace.getActivePane().activate();
    } else {
      this.focus();
    }
  }

  async ensureVisible() {
    if (!this.isVisible()) {
      await this.reveal();
      return true;
    }
    return false;
  }

  ensureRendered() {
    return this.getWorkspace().open(this.uri, { searchAllPanes: true, activateItem: false, activatePane: false });
  }

  reveal() {
    (0, _reporterProxy.incrementCounter)(`${this.name}-tab-open`);
    return this.getWorkspace().open(this.uri, { searchAllPanes: true, activateItem: true, activatePane: true });
  }

  hide() {
    (0, _reporterProxy.incrementCounter)(`${this.name}-tab-close`);
    return this.getWorkspace().hide(this.uri);
  }

  focus() {
    this.getComponent().restoreFocus();
  }

  getItem() {
    const pane = this.getWorkspace().paneForURI(this.uri);
    if (!pane) {
      return null;
    }

    const paneItem = pane.itemForURI(this.uri);
    if (!paneItem) {
      return null;
    }

    return paneItem;
  }

  getComponent() {
    const paneItem = this.getItem();
    if (!paneItem) {
      return null;
    }
    if (typeof paneItem.getRealItem !== 'function') {
      return null;
    }

    return paneItem.getRealItem();
  }

  getDOMElement() {
    const paneItem = this.getItem();
    if (!paneItem) {
      return null;
    }
    if (typeof paneItem.getElement !== 'function') {
      return null;
    }

    return paneItem.getElement();
  }

  isRendered() {
    return !!this.getWorkspace().paneForURI(this.uri);
  }

  isVisible() {
    const workspace = this.getWorkspace();
    return workspace.getPaneContainers().filter(container => container === workspace.getCenter() || container.isVisible()).some(container => container.getPanes().some(pane => {
      const item = pane.getActiveItem();
      return item && item.getURI && item.getURI() === this.uri;
    }));
  }

  hasFocus() {
    const root = this.getDOMElement();
    return root && root.contains(document.activeElement);
  }
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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