'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.COMMIT_GRAMMAR_SCOPE = undefined;

var _extends = Object.assign || function (target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i]; for (var key in source) { if (Object.prototype.hasOwnProperty.call(source, key)) { target[key] = source[key]; } } } return target; };

var _path = require('path');

var _path2 = _interopRequireDefault(_path);

var _atom = require('atom');

var _react = require('react');

var _react2 = _interopRequireDefault(_react);

var _propTypes = require('prop-types');

var _propTypes2 = _interopRequireDefault(_propTypes);

var _eventKit = require('event-kit');

var _fsExtra = require('fs-extra');

var _fsExtra2 = _interopRequireDefault(_fsExtra);

var _commitView = require('../views/commit-view');

var _commitView2 = _interopRequireDefault(_commitView);

var _refHolder = require('../models/ref-holder');

var _refHolder2 = _interopRequireDefault(_refHolder);

var _commitPreviewItem = require('../items/commit-preview-item');

var _commitPreviewItem2 = _interopRequireDefault(_commitPreviewItem);

var _propTypes3 = require('../prop-types');

var _watchWorkspaceItem = require('../watch-workspace-item');

var _helpers = require('../helpers');

var _reporterProxy = require('../reporter-proxy');

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

const COMMIT_GRAMMAR_SCOPE = exports.COMMIT_GRAMMAR_SCOPE = 'text.git-commit';

class CommitController extends _react2.default.Component {

  constructor(props, context) {
    super(props, context);

    this.activateCommitPreview = () => {
      const uri = _commitPreviewItem2.default.buildURI(this.props.repository.getWorkingDirectoryPath());
      return this.props.workspace.open(uri, { searchAllPanes: true, pending: true, activate: true });
    };

    (0, _helpers.autobind)(this, 'commit', 'handleMessageChange', 'toggleExpandedCommitMessageEditor', 'grammarAdded', 'toggleCommitPreview');

    this.subscriptions = new _eventKit.CompositeDisposable();
    this.refCommitView = new _refHolder2.default();

    this.commitMessageBuffer = new _atom.TextBuffer({ text: this.props.repository.getCommitMessage() });
    this.subscriptions.add(this.commitMessageBuffer.onDidChange(this.handleMessageChange));

    this.previewWatcher = (0, _watchWorkspaceItem.watchWorkspaceItem)(this.props.workspace, _commitPreviewItem2.default.buildURI(this.props.repository.getWorkingDirectoryPath()), this, 'commitPreviewActive');
    this.subscriptions.add(this.previewWatcher);
  }

  componentDidMount() {
    this.subscriptions.add(this.props.workspace.onDidAddTextEditor(({ textEditor }) => {
      if (this.props.repository.isPresent() && textEditor.getPath() === this.getCommitMessagePath()) {
        const grammar = this.props.grammars.grammarForScopeName(COMMIT_GRAMMAR_SCOPE);
        if (grammar) {
          textEditor.setGrammar(grammar);
        }
      }
    }), this.props.workspace.onDidDestroyPaneItem(async ({ item }) => {
      if (this.props.repository.isPresent() && item.getPath && item.getPath() === this.getCommitMessagePath() && this.getCommitMessageEditors().length === 0) {
        // we closed the last editor pointing to the commit message file
        try {
          this.commitMessageBuffer.setText((await _fsExtra2.default.readFile(this.getCommitMessagePath(), { encoding: 'utf8' })));
        } catch (e) {
          if (e.code !== 'ENOENT') {
            throw e;
          }
        }
      }
    }));
  }

  render() {
    const operationStates = this.props.repository.getOperationStates();

    return _react2.default.createElement(_commitView2.default, {
      ref: this.refCommitView.setter,
      workspace: this.props.workspace,
      tooltips: this.props.tooltips,
      config: this.props.config,
      stagedChangesExist: this.props.stagedChangesExist,
      mergeConflictsExist: this.props.mergeConflictsExist,
      prepareToCommit: this.props.prepareToCommit,
      commit: this.commit,
      abortMerge: this.props.abortMerge,
      commandRegistry: this.props.commandRegistry,
      maximumCharacterLimit: 72,
      messageBuffer: this.commitMessageBuffer,
      isMerging: this.props.isMerging,
      isCommitting: operationStates.isCommitInProgress(),
      lastCommit: this.props.lastCommit,
      currentBranch: this.props.currentBranch,
      toggleExpandedCommitMessageEditor: this.toggleExpandedCommitMessageEditor,
      deactivateCommitBox: this.isCommitMessageEditorExpanded(),
      userStore: this.props.userStore,
      selectedCoAuthors: this.props.selectedCoAuthors,
      updateSelectedCoAuthors: this.props.updateSelectedCoAuthors,
      toggleCommitPreview: this.toggleCommitPreview,
      activateCommitPreview: this.activateCommitPreview,
      commitPreviewActive: this.state.commitPreviewActive
    });
  }

  componentDidUpdate(prevProps) {
    this.commitMessageBuffer.setTextViaDiff(this.getCommitMessage());

    if (prevProps.repository !== this.props.repository) {
      this.previewWatcher.setPattern(_commitPreviewItem2.default.buildURI(this.props.repository.getWorkingDirectoryPath()));
    }
  }

  componentWillUnmount() {
    this.subscriptions.dispose();
  }

  commit(message, coAuthors = [], amend = false) {
    let msg, verbatim;
    if (this.isCommitMessageEditorExpanded()) {
      msg = this.getCommitMessageEditors()[0].getText();
      verbatim = false;
    } else {
      const wrapMessage = this.props.config.get('github.automaticCommitMessageWrapping');
      msg = wrapMessage ? wrapCommitMessage(message) : message;
      verbatim = true;
    }

    return this.props.commit(msg.trim(), { amend, coAuthors, verbatim });
  }

  setCommitMessage(message, options) {
    if (!this.props.repository.isPresent()) {
      return;
    }
    this.props.repository.setCommitMessage(message, options);
  }

  getCommitMessage() {
    return this.props.repository.getCommitMessage();
  }

  getCommitMessagePath() {
    return _path2.default.join(this.props.repository.getGitDirectoryPath(), 'ATOM_COMMIT_EDITMSG');
  }

  handleMessageChange() {
    if (!this.props.repository.isPresent()) {
      return;
    }
    this.setCommitMessage(this.commitMessageBuffer.getText(), { suppressUpdate: true });
  }

  getCommitMessageEditors() {
    if (!this.props.repository.isPresent()) {
      return [];
    }
    return this.props.workspace.getTextEditors().filter(editor => editor.getPath() === this.getCommitMessagePath());
  }

  async toggleExpandedCommitMessageEditor(messageFromBox) {
    if (this.isCommitMessageEditorExpanded()) {
      if (this.commitMessageEditorIsInForeground()) {
        await this.closeAllOpenCommitMessageEditors();
        this.forceUpdate();
      } else {
        this.activateCommitMessageEditor();
      }
    } else {
      await this.openCommitMessageEditor(messageFromBox);
      this.forceUpdate();
    }
  }

  isCommitMessageEditorExpanded() {
    return this.getCommitMessageEditors().length > 0;
  }

  commitMessageEditorIsInForeground() {
    const commitMessageEditorsInForeground = this.props.workspace.getPanes().map(pane => pane.getActiveItem()).filter(item => item && item.getPath && item.getPath() === this.getCommitMessagePath());
    return commitMessageEditorsInForeground.length > 0;
  }

  activateCommitMessageEditor() {
    const panes = this.props.workspace.getPanes();
    let editor;
    const paneWithEditor = panes.find(pane => {
      editor = pane.getItems().find(item => item.getPath && item.getPath() === this.getCommitMessagePath());
      return !!editor;
    });
    paneWithEditor.activate();
    paneWithEditor.activateItem(editor);
  }

  closeAllOpenCommitMessageEditors() {
    return Promise.all(this.props.workspace.getPanes().map(pane => {
      return Promise.all(pane.getItems().map(async item => {
        if (item && item.getPath && item.getPath() === this.getCommitMessagePath()) {
          const destroyed = await pane.destroyItem(item);
          if (!destroyed) {
            pane.activateItem(item);
          }
        }
      }));
    }));
  }

  async openCommitMessageEditor(messageFromBox) {
    await _fsExtra2.default.writeFile(this.getCommitMessagePath(), messageFromBox, 'utf8');
    const commitEditor = await this.props.workspace.open(this.getCommitMessagePath());
    (0, _reporterProxy.addEvent)('open-commit-message-editor', { package: 'github' });

    const grammar = this.props.grammars.grammarForScopeName(COMMIT_GRAMMAR_SCOPE);
    if (grammar) {
      commitEditor.setGrammar(grammar);
    } else {
      this.grammarSubscription = this.props.grammars.onDidAddGrammar(this.grammarAdded);
      this.subscriptions.add(this.grammarSubscription);
    }
  }

  grammarAdded(grammar) {
    if (grammar.scopeName !== COMMIT_GRAMMAR_SCOPE) {
      return;
    }

    this.getCommitMessageEditors().forEach(editor => editor.setGrammar(grammar));
    this.grammarSubscription.dispose();
  }

  getFocus(element) {
    return this.refCommitView.map(view => view.getFocus(element)).getOr(null);
  }

  setFocus(focus) {
    return this.refCommitView.map(view => view.setFocus(focus)).getOr(false);
  }

  advanceFocusFrom(...args) {
    return this.refCommitView.map(view => view.advanceFocusFrom(...args)).getOr(false);
  }

  retreatFocusFrom(...args) {
    return this.refCommitView.map(view => view.retreatFocusFrom(...args)).getOr(false);
  }

  toggleCommitPreview() {
    (0, _reporterProxy.addEvent)('toggle-commit-preview', { package: 'github' });
    const uri = _commitPreviewItem2.default.buildURI(this.props.repository.getWorkingDirectoryPath());
    if (this.props.workspace.hide(uri)) {
      return Promise.resolve();
    } else {
      return this.props.workspace.open(uri, { searchAllPanes: true, pending: true });
    }
  }

}

exports.default = CommitController;
CommitController.focus = _extends({}, _commitView2.default.focus);
CommitController.propTypes = {
  workspace: _propTypes2.default.object.isRequired,
  grammars: _propTypes2.default.object.isRequired,
  commandRegistry: _propTypes2.default.object.isRequired,
  config: _propTypes2.default.object.isRequired,
  tooltips: _propTypes2.default.object.isRequired,

  repository: _propTypes2.default.object.isRequired,
  isMerging: _propTypes2.default.bool.isRequired,
  mergeConflictsExist: _propTypes2.default.bool.isRequired,
  stagedChangesExist: _propTypes2.default.bool.isRequired,
  lastCommit: _propTypes2.default.object.isRequired,
  currentBranch: _propTypes2.default.object.isRequired,
  userStore: _propTypes3.UserStorePropType.isRequired,
  selectedCoAuthors: _propTypes2.default.arrayOf(_propTypes3.AuthorPropType),
  updateSelectedCoAuthors: _propTypes2.default.func,
  prepareToCommit: _propTypes2.default.func.isRequired,
  commit: _propTypes2.default.func.isRequired,
  abortMerge: _propTypes2.default.func.isRequired
};
function wrapCommitMessage(message) {
  // hard wrap message (except for first line) at 72 characters
  let results = [];
  message.split('\n').forEach((line, index) => {
    if (line.length <= 72 || index === 0) {
      results.push(line);
    } else {
      const matches = line.match(/.{1,72}(\s|$)|\S+?(\s|$)/g).map(match => {
        return match.endsWith('\n') ? match.substr(0, match.length - 1) : match;
      });
      results = results.concat(matches);
    }
  });

  return results.join('\n');
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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