/**
 * Copyright (c) 2013-present, Facebook, Inc.
 *
 * This source code is licensed under the MIT license found in the
 * LICENSE file in the root directory of this source tree.
 *
 * 
 * @format
 */

'use strict';

var _classCallCheck3 = _interopRequireDefault(require('babel-runtime/helpers/classCallCheck'));

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { 'default': obj }; }

var ReactRelayQueryFetcher = function () {
  function ReactRelayQueryFetcher() {
    (0, _classCallCheck3['default'])(this, ReactRelayQueryFetcher);
    this._selectionReferences = [];
    this._callOnDataChangeWhenSet = false;
  } // results of the root fragment;
  // fetch error


  ReactRelayQueryFetcher.prototype.lookupInStore = function lookupInStore(environment, operation) {
    if (environment.check(operation.root)) {
      this._retainCachedOperation(environment, operation);
      return environment.lookup(operation.fragment);
    }
    return null;
  };

  ReactRelayQueryFetcher.prototype.execute = function execute(_ref) {
    var _this = this;

    var environment = _ref.environment,
        operation = _ref.operation,
        cacheConfig = _ref.cacheConfig,
        _ref$preservePrevious = _ref.preservePreviousReferences,
        preservePreviousReferences = _ref$preservePrevious === undefined ? false : _ref$preservePrevious;
    var createOperationSelector = environment.unstable_internal.createOperationSelector;

    var nextReferences = [];

    return environment.execute({ operation: operation, cacheConfig: cacheConfig }).map(function (payload) {
      var operationForPayload = createOperationSelector(operation.node, payload.variables, payload.operation);
      nextReferences.push(environment.retain(operationForPayload.root));
      return payload;
    })['do']({
      error: function error() {
        // We may have partially fulfilled the request, so let the next request
        // or the unmount dispose of the references.
        _this._selectionReferences = _this._selectionReferences.concat(nextReferences);
      },
      complete: function complete() {
        if (!preservePreviousReferences) {
          _this._disposeSelectionReferences();
        }
        _this._selectionReferences = _this._selectionReferences.concat(nextReferences);
      },
      unsubscribe: function unsubscribe() {
        // Let the next request or the unmount code dispose of the references.
        // We may have partially fulfilled the request.
        _this._selectionReferences = _this._selectionReferences.concat(nextReferences);
      }
    });
  };

  ReactRelayQueryFetcher.prototype.setOnDataChange = function setOnDataChange(onDataChange) {
    require('fbjs/lib/invariant')(this._fetchOptions, 'ReactRelayQueryFetcher: `setOnDataChange` should have been called after having called `fetch`');

    // Mutate the most recent fetchOptions in place,
    // So that in-progress requests can access the updated callback.
    this._fetchOptions.onDataChange = onDataChange;

    if (this._callOnDataChangeWhenSet && typeof onDataChange === 'function') {
      this._callOnDataChangeWhenSet = false;
      if (this._error != null) {
        onDataChange({ error: this._error });
      } else if (this._snapshot != null) {
        onDataChange({ snapshot: this._snapshot });
      }
    }
  };

  /**
   * `fetch` fetches the data for the given operation.
   * If a result is immediately available synchronously, it will be synchronously
   * returned by this function.
   *
   * Otherwise, the fetched result will be communicated via the `onDataChange` callback.
   * `onDataChange` will be called with the first result (**if it wasn't returned synchronously**),
   * and then subsequently whenever the data changes.
   */


  ReactRelayQueryFetcher.prototype.fetch = function fetch(fetchOptions) {
    var _this2 = this;

    var cacheConfig = fetchOptions.cacheConfig,
        environment = fetchOptions.environment,
        operation = fetchOptions.operation;

    var fetchHasReturned = false;
    var error = void 0;

    this._disposeRequest();
    this._fetchOptions = fetchOptions;

    var request = this.execute({
      environment: environment,
      operation: operation,
      cacheConfig: cacheConfig
    })['finally'](function () {
      _this2._pendingRequest = null;
    }).subscribe({
      next: function next() {
        var onDataChange = _this2._fetchOptions ? _this2._fetchOptions.onDataChange : null;

        // If we received a response when we didn't have a change callback,
        // Make a note that to notify the callback when it's later added.
        _this2._callOnDataChangeWhenSet = typeof onDataChange !== 'function';
        _this2._error = null;

        // Only notify of the first result if `next` is being called **asynchronously**
        // (i.e. after `fetch` has returned).
        _this2._onQueryDataAvailable({ notifyFirstResult: fetchHasReturned });
      },
      error: function (_error) {
        function error(_x) {
          return _error.apply(this, arguments);
        }

        error.toString = function () {
          return _error.toString();
        };

        return error;
      }(function (err) {
        var onDataChange = _this2._fetchOptions ? _this2._fetchOptions.onDataChange : null;

        // If we received a response when we didn't have a change callback,
        // Make a note that to notify the callback when it's later added.
        _this2._callOnDataChangeWhenSet = typeof onDataChange !== 'function';
        _this2._error = err;
        _this2._snapshot = null;

        // Only notify of error if `error` is being called **asynchronously**
        // (i.e. after `fetch` has returned).
        if (fetchHasReturned) {
          if (typeof onDataChange === 'function') {
            onDataChange({ error: err });
          }
        } else {
          error = err;
        }
      })
    });

    this._pendingRequest = {
      dispose: function dispose() {
        request.unsubscribe();
      }
    };

    fetchHasReturned = true;

    if (error) {
      throw error;
    }

    return this._snapshot;
  };

  ReactRelayQueryFetcher.prototype.retry = function retry() {
    require('fbjs/lib/invariant')(this._fetchOptions, 'ReactRelayQueryFetcher: `retry` should be called after having called `fetch`');
    return this.fetch(this._fetchOptions);
  };

  ReactRelayQueryFetcher.prototype.dispose = function dispose() {
    this._disposeRequest();
    this._disposeSelectionReferences();
  };

  ReactRelayQueryFetcher.prototype._disposeRequest = function _disposeRequest() {
    this._error = null;
    this._snapshot = null;

    // order is important, dispose of pendingFetch before selectionReferences
    if (this._pendingRequest) {
      this._pendingRequest.dispose();
    }
    if (this._rootSubscription) {
      this._rootSubscription.dispose();
      this._rootSubscription = null;
    }
  };

  ReactRelayQueryFetcher.prototype._retainCachedOperation = function _retainCachedOperation(environment, operation) {
    this._disposeCacheSelectionReference();
    this._cacheSelectionReference = environment.retain(operation.root);
  };

  ReactRelayQueryFetcher.prototype._disposeCacheSelectionReference = function _disposeCacheSelectionReference() {
    this._cacheSelectionReference && this._cacheSelectionReference.dispose();
    this._cacheSelectionReference = null;
  };

  ReactRelayQueryFetcher.prototype._disposeSelectionReferences = function _disposeSelectionReferences() {
    this._disposeCacheSelectionReference();
    this._selectionReferences.forEach(function (r) {
      return r.dispose();
    });
    this._selectionReferences = [];
  };

  ReactRelayQueryFetcher.prototype._onQueryDataAvailable = function _onQueryDataAvailable(_ref2) {
    var _this3 = this;

    var notifyFirstResult = _ref2.notifyFirstResult;

    require('fbjs/lib/invariant')(this._fetchOptions, 'ReactRelayQueryFetcher: `_onQueryDataAvailable` should have been called after having called `fetch`');
    var _fetchOptions = this._fetchOptions,
        environment = _fetchOptions.environment,
        onDataChange = _fetchOptions.onDataChange,
        operation = _fetchOptions.operation;

    // `_onQueryDataAvailable` can be called synchronously the first time and can be called
    // multiple times by network layers that support data subscriptions.
    // Wait until the first payload to call `onDataChange` and subscribe for data updates.

    if (this._snapshot) {
      return;
    }

    this._snapshot = environment.lookup(operation.fragment);

    // Subscribe to changes in the data of the root fragment
    this._rootSubscription = environment.subscribe(this._snapshot, function (snapshot) {
      // Read from this._fetchOptions in case onDataChange() was lazily added.
      if (_this3._fetchOptions != null) {
        var maybeNewOnDataChange = _this3._fetchOptions.onDataChange;
        if (typeof maybeNewOnDataChange === 'function') {
          maybeNewOnDataChange({ snapshot: snapshot });
        }
      }
    });

    if (this._snapshot && notifyFirstResult && typeof onDataChange === 'function') {
      onDataChange({ snapshot: this._snapshot });
    }
  };

  return ReactRelayQueryFetcher;
}();

module.exports = ReactRelayQueryFetcher;