'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _extends = Object.assign || function (target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i]; for (var key in source) { if (Object.prototype.hasOwnProperty.call(source, key)) { target[key] = source[key]; } } } return target; };

var _react = require('react');

var _react2 = _interopRequireDefault(_react);

var _propTypes = require('prop-types');

var _propTypes2 = _interopRequireDefault(_propTypes);

var _classnames = require('classnames');

var _classnames2 = _interopRequireDefault(_classnames);

var _atom = require('atom');

var _eventKit = require('event-kit');

var _helpers = require('../helpers');

var _propTypes3 = require('../prop-types');

var _atomTextEditor = require('../atom/atom-text-editor');

var _atomTextEditor2 = _interopRequireDefault(_atomTextEditor);

var _marker = require('../atom/marker');

var _marker2 = _interopRequireDefault(_marker);

var _markerLayer = require('../atom/marker-layer');

var _markerLayer2 = _interopRequireDefault(_markerLayer);

var _decoration = require('../atom/decoration');

var _decoration2 = _interopRequireDefault(_decoration);

var _gutter = require('../atom/gutter');

var _gutter2 = _interopRequireDefault(_gutter);

var _commands = require('../atom/commands');

var _commands2 = _interopRequireDefault(_commands);

var _filePatchHeaderView = require('./file-patch-header-view');

var _filePatchHeaderView2 = _interopRequireDefault(_filePatchHeaderView);

var _filePatchMetaView = require('./file-patch-meta-view');

var _filePatchMetaView2 = _interopRequireDefault(_filePatchMetaView);

var _hunkHeaderView = require('./hunk-header-view');

var _hunkHeaderView2 = _interopRequireDefault(_hunkHeaderView);

var _refHolder = require('../models/ref-holder');

var _refHolder2 = _interopRequireDefault(_refHolder);

var _changedFileItem = require('../items/changed-file-item');

var _changedFileItem2 = _interopRequireDefault(_changedFileItem);

var _commitDetailItem = require('../items/commit-detail-item');

var _commitDetailItem2 = _interopRequireDefault(_commitDetailItem);

var _issueishDetailItem = require('../items/issueish-detail-item');

var _issueishDetailItem2 = _interopRequireDefault(_issueishDetailItem);

var _file = require('../models/patch/file');

var _file2 = _interopRequireDefault(_file);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

const executableText = {
  [_file2.default.modes.NORMAL]: 'non executable',
  [_file2.default.modes.EXECUTABLE]: 'executable'
};

const NBSP_CHARACTER = '\u00a0';

const BLANK_LABEL = () => NBSP_CHARACTER;

class MultiFilePatchView extends _react2.default.Component {

  constructor(props) {
    super(props);

    this.renderFilePatchDecorations = filePatch => {
      return _react2.default.createElement(
        _react.Fragment,
        { key: filePatch.getPath() },
        _react2.default.createElement(
          _marker2.default,
          { invalidate: 'never', bufferRange: filePatch.getStartRange() },
          _react2.default.createElement(
            _decoration2.default,
            { type: 'block', className: 'github-FilePatchView-controlBlock' },
            _react2.default.createElement(_filePatchHeaderView2.default, {
              itemType: this.props.itemType,
              relPath: filePatch.getPath(),
              newPath: filePatch.getStatus() === 'renamed' ? filePatch.getNewPath() : null,
              stagingStatus: this.props.stagingStatus,
              isPartiallyStaged: this.props.isPartiallyStaged,
              hasHunks: filePatch.getHunks().length > 0,
              hasUndoHistory: this.props.hasUndoHistory,
              hasMultipleFileSelections: this.props.hasMultipleFileSelections,

              tooltips: this.props.tooltips,

              undoLastDiscard: () => this.undoLastDiscardFromButton(filePatch),
              diveIntoMirrorPatch: () => this.props.diveIntoMirrorPatch(filePatch),
              openFile: () => this.didOpenFile({ selectedFilePatch: filePatch }),
              toggleFile: () => this.props.toggleFile(filePatch)
            }),
            this.renderSymlinkChangeMeta(filePatch),
            this.renderExecutableModeChangeMeta(filePatch)
          )
        ),
        this.renderHunkHeaders(filePatch)
      );
    };

    this.undoLastDiscardFromCoreUndo = () => {
      if (this.props.hasUndoHistory) {
        const selectedFilePatches = Array.from(this.getSelectedFilePatches());
        if (this.props.itemType === _changedFileItem2.default) {
          this.props.undoLastDiscard(selectedFilePatches[0], { eventSource: { command: 'core:undo' } });
        }
      }
    };

    this.undoLastDiscardFromButton = filePatch => {
      this.props.undoLastDiscard(filePatch, { eventSource: 'button' });
    };

    this.discardSelectionFromCommand = () => {
      return this.props.discardRows(this.props.selectedRows, this.props.selectionMode, { eventSource: { command: 'github:discard-selected-lines' } });
    };

    this.didToggleModeChange = () => {
      return Promise.all(Array.from(this.getSelectedFilePatches()).filter(fp => fp.didChangeExecutableMode()).map(this.props.toggleModeChange));
    };

    this.didToggleSymlinkChange = () => {
      return Promise.all(Array.from(this.getSelectedFilePatches()).filter(fp => fp.hasTypechange()).map(this.props.toggleSymlinkChange));
    };

    (0, _helpers.autobind)(this, 'didMouseDownOnHeader', 'didMouseDownOnLineNumber', 'didMouseMoveOnLineNumber', 'didMouseUp', 'didConfirm', 'didToggleSelectionMode', 'selectNextHunk', 'selectPreviousHunk', 'didOpenFile', 'didAddSelection', 'didChangeSelectionRange', 'didDestroySelection', 'oldLineNumberLabel', 'newLineNumberLabel');

    this.mouseSelectionInProgress = false;
    this.lastMouseMoveLine = null;
    this.nextSelectionMode = null;
    this.suppressChanges = false;
    this.refRoot = new _refHolder2.default();
    this.refEditor = new _refHolder2.default();
    this.refEditorElement = new _refHolder2.default();

    this.subs = new _eventKit.CompositeDisposable();

    this.subs.add(this.refEditor.observe(editor => {
      this.refEditorElement.setter(editor.getElement());
    }), this.refEditorElement.observe(element => {
      this.props.refInitialFocus && this.props.refInitialFocus.setter(element);
    }));
  }

  componentDidMount() {
    window.addEventListener('mouseup', this.didMouseUp);
    this.refEditor.map(editor => {
      // this.props.multiFilePatch is guaranteed to contain at least one FilePatch if <AtomTextEditor> is rendered.
      const [firstPatch] = this.props.multiFilePatch.getFilePatches();
      const [firstHunk] = firstPatch.getHunks();
      if (!firstHunk) {
        return null;
      }

      this.nextSelectionMode = 'hunk';
      editor.setSelectedBufferRange(firstHunk.getRange());
      return null;
    });

    this.subs.add(this.props.config.onDidChange('github.showDiffIconGutter', ({ newValue }) => this.forceUpdate()));
  }

  getSnapshotBeforeUpdate(prevProps) {
    let newSelectionRange = null;
    if (this.props.multiFilePatch !== prevProps.multiFilePatch) {
      // Heuristically adjust the editor selection based on the old file patch, the old row selection state, and
      // the incoming patch.
      newSelectionRange = this.props.multiFilePatch.getNextSelectionRange(prevProps.multiFilePatch, prevProps.selectedRows);

      this.suppressChanges = true;
      this.props.multiFilePatch.adoptBufferFrom(prevProps.multiFilePatch);
      this.suppressChanges = false;
    }
    return newSelectionRange;
  }

  componentDidUpdate(prevProps, prevState, newSelectionRange) {
    if (prevProps.refInitialFocus !== this.props.refInitialFocus) {
      prevProps.refInitialFocus && prevProps.refInitialFocus.setter(null);
      this.props.refInitialFocus && this.refEditorElement.map(this.props.refInitialFocus.setter);
    }

    if (newSelectionRange) {
      this.refEditor.map(editor => {
        if (this.props.selectionMode === 'line') {
          this.nextSelectionMode = 'line';
          editor.setSelectedBufferRange(newSelectionRange);
        } else {
          const nextHunks = new Set(_atom.Range.fromObject(newSelectionRange).getRows().map(row => this.props.multiFilePatch.getHunkAt(row)).filter(Boolean));
          const nextRanges = nextHunks.size > 0 ? Array.from(nextHunks, hunk => hunk.getRange()) : [[[0, 0], [0, 0]]];

          this.nextSelectionMode = 'hunk';
          editor.setSelectedBufferRanges(nextRanges);
        }

        return null;
      });
    } else {
      this.nextSelectionMode = null;
    }
  }

  componentWillUnmount() {
    window.removeEventListener('mouseup', this.didMouseUp);
    this.subs.dispose();
  }

  render() {
    const rootClass = (0, _classnames2.default)('github-FilePatchView', { [`github-FilePatchView--${this.props.stagingStatus}`]: this.props.stagingStatus }, { 'github-FilePatchView--blank': !this.props.multiFilePatch.anyPresent() }, { 'github-FilePatchView--hunkMode': this.props.selectionMode === 'hunk' });

    return _react2.default.createElement(
      'div',
      { className: rootClass, ref: this.refRoot.setter },
      this.renderCommands(),
      _react2.default.createElement(
        'main',
        { className: 'github-FilePatchView-container' },
        this.props.multiFilePatch.anyPresent() ? this.renderNonEmptyPatch() : this.renderEmptyPatch()
      )
    );
  }

  renderCommands() {
    if (this.props.itemType === _commitDetailItem2.default || this.props.itemType === _issueishDetailItem2.default) {
      return _react2.default.createElement(
        _commands2.default,
        { registry: this.props.commands, target: this.refRoot },
        _react2.default.createElement(_commands.Command, { command: 'github:select-next-hunk', callback: this.selectNextHunk }),
        _react2.default.createElement(_commands.Command, { command: 'github:select-previous-hunk', callback: this.selectPreviousHunk }),
        _react2.default.createElement(_commands.Command, { command: 'github:toggle-patch-selection-mode', callback: this.didToggleSelectionMode })
      );
    }

    let stageModeCommand = null;
    let stageSymlinkCommand = null;

    if (this.props.multiFilePatch.didAnyChangeExecutableMode()) {
      const command = this.props.stagingStatus === 'unstaged' ? 'github:stage-file-mode-change' : 'github:unstage-file-mode-change';
      stageModeCommand = _react2.default.createElement(_commands.Command, { command: command, callback: this.didToggleModeChange });
    }

    if (this.props.multiFilePatch.anyHaveTypechange()) {
      const command = this.props.stagingStatus === 'unstaged' ? 'github:stage-symlink-change' : 'github:unstage-symlink-change';
      stageSymlinkCommand = _react2.default.createElement(_commands.Command, { command: command, callback: this.didToggleSymlinkChange });
    }

    return _react2.default.createElement(
      _commands2.default,
      { registry: this.props.commands, target: this.refRoot },
      _react2.default.createElement(_commands.Command, { command: 'github:select-next-hunk', callback: this.selectNextHunk }),
      _react2.default.createElement(_commands.Command, { command: 'github:select-previous-hunk', callback: this.selectPreviousHunk }),
      _react2.default.createElement(_commands.Command, { command: 'core:confirm', callback: this.didConfirm }),
      _react2.default.createElement(_commands.Command, { command: 'core:undo', callback: this.undoLastDiscardFromCoreUndo }),
      _react2.default.createElement(_commands.Command, { command: 'github:discard-selected-lines', callback: this.discardSelectionFromCommand }),
      _react2.default.createElement(_commands.Command, { command: 'github:jump-to-file', callback: this.didOpenFile }),
      _react2.default.createElement(_commands.Command, { command: 'github:surface', callback: this.props.surface }),
      _react2.default.createElement(_commands.Command, { command: 'github:toggle-patch-selection-mode', callback: this.didToggleSelectionMode }),
      stageModeCommand,
      stageSymlinkCommand
    );
  }

  renderEmptyPatch() {
    return _react2.default.createElement(
      'p',
      { className: 'github-FilePatchView-message icon icon-info' },
      'No changes to display'
    );
  }

  renderNonEmptyPatch() {
    return _react2.default.createElement(
      _atomTextEditor2.default,
      {
        workspace: this.props.workspace,

        buffer: this.props.multiFilePatch.getBuffer(),
        lineNumberGutterVisible: false,
        autoWidth: false,
        autoHeight: false,
        readOnly: true,
        softWrapped: true,

        didAddSelection: this.didAddSelection,
        didChangeSelectionRange: this.didChangeSelectionRange,
        didDestroySelection: this.didDestroySelection,
        refModel: this.refEditor },
      _react2.default.createElement(_gutter2.default, {
        name: 'old-line-numbers',
        priority: 1,
        className: 'old',
        type: 'line-number',
        labelFn: this.oldLineNumberLabel,
        onMouseDown: this.didMouseDownOnLineNumber,
        onMouseMove: this.didMouseMoveOnLineNumber
      }),
      _react2.default.createElement(_gutter2.default, {
        name: 'new-line-numbers',
        priority: 2,
        className: 'new',
        type: 'line-number',
        labelFn: this.newLineNumberLabel,
        onMouseDown: this.didMouseDownOnLineNumber,
        onMouseMove: this.didMouseMoveOnLineNumber
      }),
      this.props.config.get('github.showDiffIconGutter') && _react2.default.createElement(_gutter2.default, {
        name: 'diff-icons',
        priority: 3,
        type: 'line-number',
        className: 'icons',
        labelFn: BLANK_LABEL,
        onMouseDown: this.didMouseDownOnLineNumber,
        onMouseMove: this.didMouseMoveOnLineNumber
      }),
      this.props.multiFilePatch.getFilePatches().map(this.renderFilePatchDecorations),
      this.renderLineDecorations(Array.from(this.props.selectedRows, row => _atom.Range.fromObject([[row, 0], [row, Infinity]])), 'github-FilePatchView-line--selected', { gutter: true, icon: true, line: true }),
      this.renderDecorationsOnLayer(this.props.multiFilePatch.getAdditionLayer(), 'github-FilePatchView-line--added', { icon: true, line: true }),
      this.renderDecorationsOnLayer(this.props.multiFilePatch.getDeletionLayer(), 'github-FilePatchView-line--deleted', { icon: true, line: true }),
      this.renderDecorationsOnLayer(this.props.multiFilePatch.getNoNewlineLayer(), 'github-FilePatchView-line--nonewline', { icon: true, line: true })
    );
  }

  renderExecutableModeChangeMeta(filePatch) {
    if (!filePatch.didChangeExecutableMode()) {
      return null;
    }

    const oldMode = filePatch.getOldMode();
    const newMode = filePatch.getNewMode();

    const attrs = this.props.stagingStatus === 'unstaged' ? {
      actionIcon: 'icon-move-down',
      actionText: 'Stage Mode Change'
    } : {
      actionIcon: 'icon-move-up',
      actionText: 'Unstage Mode Change'
    };

    return _react2.default.createElement(
      _filePatchMetaView2.default,
      {
        title: 'Mode change',
        actionIcon: attrs.actionIcon,
        actionText: attrs.actionText,
        itemType: this.props.itemType,
        action: () => this.props.toggleModeChange(filePatch) },
      _react2.default.createElement(
        _react.Fragment,
        null,
        'File changed mode',
        _react2.default.createElement(
          'span',
          { className: 'github-FilePatchView-metaDiff github-FilePatchView-metaDiff--removed' },
          'from ',
          executableText[oldMode],
          ' ',
          _react2.default.createElement(
            'code',
            null,
            oldMode
          )
        ),
        _react2.default.createElement(
          'span',
          { className: 'github-FilePatchView-metaDiff github-FilePatchView-metaDiff--added' },
          'to ',
          executableText[newMode],
          ' ',
          _react2.default.createElement(
            'code',
            null,
            newMode
          )
        )
      )
    );
  }

  renderSymlinkChangeMeta(filePatch) {
    if (!filePatch.hasSymlink()) {
      return null;
    }

    let detail = _react2.default.createElement('div', null);
    let title = '';
    const oldSymlink = filePatch.getOldSymlink();
    const newSymlink = filePatch.getNewSymlink();
    if (oldSymlink && newSymlink) {
      detail = _react2.default.createElement(
        _react.Fragment,
        null,
        'Symlink changed',
        _react2.default.createElement(
          'span',
          { className: (0, _classnames2.default)('github-FilePatchView-metaDiff', 'github-FilePatchView-metaDiff--fullWidth', 'github-FilePatchView-metaDiff--removed') },
          'from ',
          _react2.default.createElement(
            'code',
            null,
            oldSymlink
          )
        ),
        _react2.default.createElement(
          'span',
          { className: (0, _classnames2.default)('github-FilePatchView-metaDiff', 'github-FilePatchView-metaDiff--fullWidth', 'github-FilePatchView-metaDiff--added') },
          'to ',
          _react2.default.createElement(
            'code',
            null,
            newSymlink
          )
        ),
        '.'
      );
      title = 'Symlink changed';
    } else if (oldSymlink && !newSymlink) {
      detail = _react2.default.createElement(
        _react.Fragment,
        null,
        'Symlink',
        _react2.default.createElement(
          'span',
          { className: 'github-FilePatchView-metaDiff github-FilePatchView-metaDiff--removed' },
          'to ',
          _react2.default.createElement(
            'code',
            null,
            oldSymlink
          )
        ),
        'deleted.'
      );
      title = 'Symlink deleted';
    } else {
      detail = _react2.default.createElement(
        _react.Fragment,
        null,
        'Symlink',
        _react2.default.createElement(
          'span',
          { className: 'github-FilePatchView-metaDiff github-FilePatchView-metaDiff--added' },
          'to ',
          _react2.default.createElement(
            'code',
            null,
            newSymlink
          )
        ),
        'created.'
      );
      title = 'Symlink created';
    }

    const attrs = this.props.stagingStatus === 'unstaged' ? {
      actionIcon: 'icon-move-down',
      actionText: 'Stage Symlink Change'
    } : {
      actionIcon: 'icon-move-up',
      actionText: 'Unstage Symlink Change'
    };

    return _react2.default.createElement(
      _filePatchMetaView2.default,
      {
        title: title,
        actionIcon: attrs.actionIcon,
        actionText: attrs.actionText,
        itemType: this.props.itemType,
        action: () => this.props.toggleSymlinkChange(filePatch) },
      _react2.default.createElement(
        _react.Fragment,
        null,
        detail
      )
    );
  }

  renderHunkHeaders(filePatch) {
    const toggleVerb = this.props.stagingStatus === 'unstaged' ? 'Stage' : 'Unstage';
    const selectedHunks = new Set(Array.from(this.props.selectedRows, row => this.props.multiFilePatch.getHunkAt(row)));

    return _react2.default.createElement(
      _react.Fragment,
      null,
      _react2.default.createElement(
        _markerLayer2.default,
        null,
        filePatch.getHunks().map((hunk, index) => {
          const containsSelection = this.props.selectionMode === 'line' && selectedHunks.has(hunk);
          const isSelected = this.props.selectionMode === 'hunk' && selectedHunks.has(hunk);

          let buttonSuffix = '';
          if (containsSelection) {
            buttonSuffix += 'Selected Line';
            if (this.props.selectedRows.size > 1) {
              buttonSuffix += 's';
            }
          } else {
            buttonSuffix += 'Hunk';
            if (selectedHunks.size > 1) {
              buttonSuffix += 's';
            }
          }

          const toggleSelectionLabel = `${toggleVerb} ${buttonSuffix}`;
          const discardSelectionLabel = `Discard ${buttonSuffix}`;

          const startPoint = hunk.getRange().start;
          const startRange = new _atom.Range(startPoint, startPoint);

          return _react2.default.createElement(
            _marker2.default,
            { key: `hunkHeader-${index}`, bufferRange: startRange, invalidate: 'never' },
            _react2.default.createElement(
              _decoration2.default,
              { type: 'block', className: 'github-FilePatchView-controlBlock' },
              _react2.default.createElement(_hunkHeaderView2.default, {
                refTarget: this.refEditorElement,
                hunk: hunk,
                isSelected: isSelected,
                stagingStatus: this.props.stagingStatus,
                selectionMode: 'line',
                toggleSelectionLabel: toggleSelectionLabel,
                discardSelectionLabel: discardSelectionLabel,

                tooltips: this.props.tooltips,
                keymaps: this.props.keymaps,

                toggleSelection: () => this.toggleHunkSelection(hunk, containsSelection),
                discardSelection: () => this.discardHunkSelection(hunk, containsSelection),
                mouseDown: this.didMouseDownOnHeader,
                itemType: this.props.itemType
              })
            )
          );
        })
      )
    );
  }

  renderLineDecorations(ranges, lineClass, { line, gutter, icon, refHolder }) {
    if (ranges.length === 0) {
      return null;
    }

    const holder = refHolder || new _refHolder2.default();
    return _react2.default.createElement(
      _markerLayer2.default,
      { handleLayer: holder.setter },
      ranges.map((range, index) => {
        return _react2.default.createElement(_marker2.default, {
          key: `line-${lineClass}-${index}`,
          bufferRange: range,
          invalidate: 'never'
        });
      }),
      this.renderDecorations(lineClass, { line, gutter, icon })
    );
  }

  renderDecorationsOnLayer(layer, lineClass, { line, gutter, icon }) {
    if (layer.getMarkerCount() === 0) {
      return null;
    }

    return _react2.default.createElement(
      _markerLayer2.default,
      { external: layer },
      this.renderDecorations(lineClass, { line, gutter, icon })
    );
  }

  renderDecorations(lineClass, { line, gutter, icon }) {
    return _react2.default.createElement(
      _react.Fragment,
      null,
      line && _react2.default.createElement(_decoration2.default, {
        type: 'line',
        className: lineClass,
        omitEmptyLastRow: false
      }),
      gutter && _react2.default.createElement(
        _react.Fragment,
        null,
        _react2.default.createElement(_decoration2.default, {
          type: 'line-number',
          gutterName: 'old-line-numbers',
          className: lineClass,
          omitEmptyLastRow: false
        }),
        _react2.default.createElement(_decoration2.default, {
          type: 'line-number',
          gutterName: 'new-line-numbers',
          className: lineClass,
          omitEmptyLastRow: false
        })
      ),
      icon && _react2.default.createElement(_decoration2.default, {
        type: 'line-number',
        gutterName: 'diff-icons',
        className: lineClass,
        omitEmptyLastRow: false
      })
    );
  }

  toggleHunkSelection(hunk, containsSelection) {
    if (containsSelection) {
      return this.props.toggleRows(this.props.selectedRows, this.props.selectionMode, { eventSource: 'button' });
    } else {
      const changeRows = new Set(hunk.getChanges().reduce((rows, change) => {
        rows.push(...change.getBufferRows());
        return rows;
      }, []));
      return this.props.toggleRows(changeRows, 'hunk', { eventSource: 'button' });
    }
  }

  discardHunkSelection(hunk, containsSelection) {
    if (containsSelection) {
      return this.props.discardRows(this.props.selectedRows, this.props.selectionMode, { eventSource: 'button' });
    } else {
      const changeRows = new Set(hunk.getChanges().reduce((rows, change) => {
        rows.push(...change.getBufferRows());
        return rows;
      }, []));
      return this.props.discardRows(changeRows, 'hunk', { eventSource: 'button' });
    }
  }

  didMouseDownOnHeader(event, hunk) {
    this.nextSelectionMode = 'hunk';
    this.handleSelectionEvent(event, hunk.getRange());
  }

  didMouseDownOnLineNumber(event) {
    const line = event.bufferRow;
    if (line === undefined || isNaN(line)) {
      return;
    }

    this.nextSelectionMode = 'line';
    if (this.handleSelectionEvent(event.domEvent, [[line, 0], [line, Infinity]])) {
      this.mouseSelectionInProgress = true;
    }
  }

  didMouseMoveOnLineNumber(event) {
    if (!this.mouseSelectionInProgress) {
      return;
    }

    const line = event.bufferRow;
    if (this.lastMouseMoveLine === line || line === undefined || isNaN(line)) {
      return;
    }
    this.lastMouseMoveLine = line;

    this.nextSelectionMode = 'line';
    this.handleSelectionEvent(event.domEvent, [[line, 0], [line, Infinity]], { add: true });
  }

  didMouseUp() {
    this.mouseSelectionInProgress = false;
  }

  handleSelectionEvent(event, rangeLike, opts) {
    if (event.button !== 0) {
      return false;
    }

    const isWindows = process.platform === 'win32';
    if (event.ctrlKey && !isWindows) {
      // Allow the context menu to open.
      return false;
    }

    const options = _extends({
      add: false
    }, opts);

    // Normalize the target selection range
    const converted = _atom.Range.fromObject(rangeLike);
    const range = this.refEditor.map(editor => editor.clipBufferRange(converted)).getOr(converted);

    if (event.metaKey || /* istanbul ignore next */event.ctrlKey && isWindows) {
      this.refEditor.map(editor => {
        let intersects = false;
        let without = null;

        for (const selection of editor.getSelections()) {
          if (selection.intersectsBufferRange(range)) {
            // Remove range from this selection by truncating it to the "near edge" of the range and creating a
            // new selection from the "far edge" to the previous end. Omit either side if it is empty.
            intersects = true;
            const selectionRange = selection.getBufferRange();

            const newRanges = [];

            if (!range.start.isEqual(selectionRange.start)) {
              // Include the bit from the selection's previous start to the range's start.
              let nudged = range.start;
              if (range.start.column === 0) {
                const lastColumn = editor.getBuffer().lineLengthForRow(range.start.row - 1);
                nudged = [range.start.row - 1, lastColumn];
              }

              newRanges.push([selectionRange.start, nudged]);
            }

            if (!range.end.isEqual(selectionRange.end)) {
              // Include the bit from the range's end to the selection's end.
              let nudged = range.end;
              const lastColumn = editor.getBuffer().lineLengthForRow(range.end.row);
              if (range.end.column === lastColumn) {
                nudged = [range.end.row + 1, 0];
              }

              newRanges.push([nudged, selectionRange.end]);
            }

            if (newRanges.length > 0) {
              selection.setBufferRange(newRanges[0]);
              for (const newRange of newRanges.slice(1)) {
                editor.addSelectionForBufferRange(newRange, { reversed: selection.isReversed() });
              }
            } else {
              without = selection;
            }
          }
        }

        if (without !== null) {
          const replacementRanges = editor.getSelections().filter(each => each !== without).map(each => each.getBufferRange());
          if (replacementRanges.length > 0) {
            editor.setSelectedBufferRanges(replacementRanges);
          }
        }

        if (!intersects) {
          // Add this range as a new, distinct selection.
          editor.addSelectionForBufferRange(range);
        }

        return null;
      });
    } else if (options.add || event.shiftKey) {
      // Extend the existing selection to encompass this range.
      this.refEditor.map(editor => {
        const lastSelection = editor.getLastSelection();
        const lastSelectionRange = lastSelection.getBufferRange();

        // You are now entering the wall of ternery operators. This is your last exit before the tollbooth
        const isBefore = range.start.isLessThan(lastSelectionRange.start);
        const farEdge = isBefore ? range.start : range.end;
        const newRange = isBefore ? [farEdge, lastSelectionRange.end] : [lastSelectionRange.start, farEdge];

        lastSelection.setBufferRange(newRange, { reversed: isBefore });
        return null;
      });
    } else {
      this.refEditor.map(editor => editor.setSelectedBufferRange(range));
    }

    return true;
  }

  didConfirm() {
    return this.props.toggleRows(this.props.selectedRows, this.props.selectionMode);
  }

  didToggleSelectionMode() {
    const selectedHunks = this.getSelectedHunks();
    this.withSelectionMode({
      line: () => {
        const hunkRanges = selectedHunks.map(hunk => hunk.getRange());
        this.nextSelectionMode = 'hunk';
        this.refEditor.map(editor => editor.setSelectedBufferRanges(hunkRanges));
      },
      hunk: () => {
        let firstChangeRow = Infinity;
        for (const hunk of selectedHunks) {
          const [firstChange] = hunk.getChanges();
          if (firstChange && (!firstChangeRow || firstChange.getStartBufferRow() < firstChangeRow)) {
            firstChangeRow = firstChange.getStartBufferRow();
          }
        }

        this.nextSelectionMode = 'line';
        this.refEditor.map(editor => {
          editor.setSelectedBufferRanges([[[firstChangeRow, 0], [firstChangeRow, Infinity]]]);
          return null;
        });
      }
    });
  }

  selectNextHunk() {
    this.refEditor.map(editor => {
      const nextHunks = new Set(this.withSelectedHunks(hunk => this.getHunkAfter(hunk) || hunk));
      const nextRanges = Array.from(nextHunks, hunk => hunk.getRange());
      this.nextSelectionMode = 'hunk';
      editor.setSelectedBufferRanges(nextRanges);
      return null;
    });
  }

  selectPreviousHunk() {
    this.refEditor.map(editor => {
      const nextHunks = new Set(this.withSelectedHunks(hunk => this.getHunkBefore(hunk) || hunk));
      const nextRanges = Array.from(nextHunks, hunk => hunk.getRange());
      this.nextSelectionMode = 'hunk';
      editor.setSelectedBufferRanges(nextRanges);
      return null;
    });
  }

  didOpenFile({ selectedFilePatch } = {}) {
    const cursorsByFilePatch = new Map();

    this.refEditor.map(editor => {
      const placedRows = new Set();

      for (const cursor of editor.getCursors()) {
        const cursorRow = cursor.getBufferPosition().row;
        const hunk = this.props.multiFilePatch.getHunkAt(cursorRow);
        const filePatch = this.props.multiFilePatch.getFilePatchAt(cursorRow);
        /* istanbul ignore next */
        if (!hunk) {
          continue;
        }

        let newRow = hunk.getNewRowAt(cursorRow);
        let newColumn = cursor.getBufferPosition().column;
        if (newRow === null) {
          let nearestRow = hunk.getNewStartRow();
          for (const region of hunk.getRegions()) {
            if (!region.includesBufferRow(cursorRow)) {
              region.when({
                unchanged: () => {
                  nearestRow += region.bufferRowCount();
                },
                addition: () => {
                  nearestRow += region.bufferRowCount();
                }
              });
            } else {
              break;
            }
          }

          if (!placedRows.has(nearestRow)) {
            newRow = nearestRow;
            newColumn = 0;
            placedRows.add(nearestRow);
          }
        }

        if (newRow !== null) {
          // Why is this needed? I _think_ everything is in terms of buffer position
          // so there shouldn't be an off-by-one issue
          newRow -= 1;
          const cursors = cursorsByFilePatch.get(filePatch);
          if (!cursors) {
            cursorsByFilePatch.set(filePatch, [[newRow, newColumn]]);
          } else {
            cursors.push([newRow, newColumn]);
          }
        }
      }

      return null;
    });

    const filePatchesWithCursors = new Set(cursorsByFilePatch.keys());
    if (selectedFilePatch && !filePatchesWithCursors.has(selectedFilePatch)) {
      const [firstHunk] = selectedFilePatch.getHunks();
      const cursorRow = firstHunk ? firstHunk.getNewStartRow() - 1 : 0;
      return this.props.openFile(selectedFilePatch, [[cursorRow, 0]], true);
    } else {
      const pending = cursorsByFilePatch.size === 1;
      return Promise.all(Array.from(cursorsByFilePatch, value => {
        const [filePatch, cursors] = value;
        return this.props.openFile(filePatch, cursors, pending);
      }));
    }
  }

  getSelectedRows() {
    return this.refEditor.map(editor => {
      return new Set(editor.getSelections().map(selection => selection.getBufferRange()).reduce((acc, range) => {
        for (const row of range.getRows()) {
          if (this.isChangeRow(row)) {
            acc.push(row);
          }
        }
        return acc;
      }, []));
    }).getOr(new Set());
  }

  didAddSelection() {
    this.didChangeSelectedRows();
  }

  didChangeSelectionRange(event) {
    if (!event || event.oldBufferRange.start.row !== event.newBufferRange.start.row || event.oldBufferRange.end.row !== event.newBufferRange.end.row) {
      this.didChangeSelectedRows();
    }
  }

  didDestroySelection() {
    this.didChangeSelectedRows();
  }

  didChangeSelectedRows() {
    if (this.suppressChanges) {
      return;
    }

    const nextCursorRows = this.refEditor.map(editor => {
      return editor.getCursorBufferPositions().map(position => position.row);
    }).getOr([]);
    const hasMultipleFileSelections = this.props.multiFilePatch.spansMultipleFiles(nextCursorRows);

    this.props.selectedRowsChanged(this.getSelectedRows(), this.nextSelectionMode || 'line', hasMultipleFileSelections);
  }

  oldLineNumberLabel({ bufferRow, softWrapped }) {
    const hunk = this.props.multiFilePatch.getHunkAt(bufferRow);
    if (hunk === undefined) {
      return this.pad('');
    }

    const oldRow = hunk.getOldRowAt(bufferRow);
    if (softWrapped) {
      return this.pad(oldRow === null ? '' : '•');
    }

    return this.pad(oldRow);
  }

  newLineNumberLabel({ bufferRow, softWrapped }) {
    const hunk = this.props.multiFilePatch.getHunkAt(bufferRow);
    if (hunk === undefined) {
      return this.pad('');
    }

    const newRow = hunk.getNewRowAt(bufferRow);
    if (softWrapped) {
      return this.pad(newRow === null ? '' : '•');
    }
    return this.pad(newRow);
  }

  /*
   * Return a Set of the Hunks that include at least one editor selection. The selection need not contain an actual
   * change row.
   */
  getSelectedHunks() {
    return this.withSelectedHunks(each => each);
  }

  withSelectedHunks(callback) {
    return this.refEditor.map(editor => {
      const seen = new Set();
      return editor.getSelectedBufferRanges().reduce((acc, range) => {
        for (const row of range.getRows()) {
          const hunk = this.props.multiFilePatch.getHunkAt(row);
          if (!hunk || seen.has(hunk)) {
            continue;
          }

          seen.add(hunk);
          acc.push(callback(hunk));
        }
        return acc;
      }, []);
    }).getOr([]);
  }

  /*
   * Return a Set of FilePatches that include at least one editor selection. The selection need not contain an actual
   * change row.
   */
  getSelectedFilePatches() {
    return this.refEditor.map(editor => {
      const patches = new Set();
      for (const range of editor.getSelectedBufferRanges()) {
        for (const row of range.getRows()) {
          const patch = this.props.multiFilePatch.getFilePatchAt(row);
          patches.add(patch);
        }
      }
      return patches;
    }).getOr(new Set());
  }

  getHunkBefore(hunk) {
    const prevRow = hunk.getRange().start.row - 1;
    return this.props.multiFilePatch.getHunkAt(prevRow);
  }

  getHunkAfter(hunk) {
    const nextRow = hunk.getRange().end.row + 1;
    return this.props.multiFilePatch.getHunkAt(nextRow);
  }

  isChangeRow(bufferRow) {
    const changeLayers = [this.props.multiFilePatch.getAdditionLayer(), this.props.multiFilePatch.getDeletionLayer()];
    return changeLayers.some(layer => layer.findMarkers({ intersectsRow: bufferRow }).length > 0);
  }

  withSelectionMode(callbacks) {
    const callback = callbacks[this.props.selectionMode];
    /* istanbul ignore if */
    if (!callback) {
      throw new Error(`Unknown selection mode: ${this.props.selectionMode}`);
    }
    return callback();
  }

  pad(num) {
    const maxDigits = this.props.multiFilePatch.getMaxLineNumberWidth();
    if (num === null) {
      return NBSP_CHARACTER.repeat(maxDigits);
    } else {
      return NBSP_CHARACTER.repeat(maxDigits - num.toString().length) + num.toString();
    }
  }

}
exports.default = MultiFilePatchView;
MultiFilePatchView.propTypes = {
  stagingStatus: _propTypes2.default.oneOf(['staged', 'unstaged']),
  isPartiallyStaged: _propTypes2.default.bool,
  multiFilePatch: _propTypes3.MultiFilePatchPropType.isRequired,
  selectionMode: _propTypes2.default.oneOf(['hunk', 'line']).isRequired,
  selectedRows: _propTypes2.default.object.isRequired,
  hasMultipleFileSelections: _propTypes2.default.bool.isRequired,
  repository: _propTypes2.default.object.isRequired,
  hasUndoHistory: _propTypes2.default.bool,

  workspace: _propTypes2.default.object.isRequired,
  commands: _propTypes2.default.object.isRequired,
  keymaps: _propTypes2.default.object.isRequired,
  tooltips: _propTypes2.default.object.isRequired,
  config: _propTypes2.default.object.isRequired,

  selectedRowsChanged: _propTypes2.default.func,

  diveIntoMirrorPatch: _propTypes2.default.func,
  surface: _propTypes2.default.func,
  openFile: _propTypes2.default.func,
  toggleFile: _propTypes2.default.func,
  toggleRows: _propTypes2.default.func,
  toggleModeChange: _propTypes2.default.func,
  toggleSymlinkChange: _propTypes2.default.func,
  undoLastDiscard: _propTypes2.default.func,
  discardRows: _propTypes2.default.func,
  refInitialFocus: _propTypes3.RefHolderPropType,
  itemType: _propTypes3.ItemTypePropType.isRequired
};
//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIm11bHRpLWZpbGUtcGF0Y2gtdmlldy5qcyJdLCJuYW1lcyI6WyJleGVjdXRhYmxlVGV4dCIsIkZpbGUiLCJtb2RlcyIsIk5PUk1BTCIsIkVYRUNVVEFCTEUiLCJOQlNQX0NIQVJBQ1RFUiIsIkJMQU5LX0xBQkVMIiwiTXVsdGlGaWxlUGF0Y2hWaWV3IiwiUmVhY3QiLCJDb21wb25lbnQiLCJjb25zdHJ1Y3RvciIsInByb3BzIiwicmVuZGVyRmlsZVBhdGNoRGVjb3JhdGlvbnMiLCJmaWxlUGF0Y2giLCJnZXRQYXRoIiwiZ2V0U3RhcnRSYW5nZSIsIml0ZW1UeXBlIiwiZ2V0U3RhdHVzIiwiZ2V0TmV3UGF0aCIsInN0YWdpbmdTdGF0dXMiLCJpc1BhcnRpYWxseVN0YWdlZCIsImdldEh1bmtzIiwibGVuZ3RoIiwiaGFzVW5kb0hpc3RvcnkiLCJoYXNNdWx0aXBsZUZpbGVTZWxlY3Rpb25zIiwidG9vbHRpcHMiLCJ1bmRvTGFzdERpc2NhcmRGcm9tQnV0dG9uIiwiZGl2ZUludG9NaXJyb3JQYXRjaCIsImRpZE9wZW5GaWxlIiwic2VsZWN0ZWRGaWxlUGF0Y2giLCJ0b2dnbGVGaWxlIiwicmVuZGVyU3ltbGlua0NoYW5nZU1ldGEiLCJyZW5kZXJFeGVjdXRhYmxlTW9kZUNoYW5nZU1ldGEiLCJyZW5kZXJIdW5rSGVhZGVycyIsInVuZG9MYXN0RGlzY2FyZEZyb21Db3JlVW5kbyIsInNlbGVjdGVkRmlsZVBhdGNoZXMiLCJBcnJheSIsImZyb20iLCJnZXRTZWxlY3RlZEZpbGVQYXRjaGVzIiwiQ2hhbmdlZEZpbGVJdGVtIiwidW5kb0xhc3REaXNjYXJkIiwiZXZlbnRTb3VyY2UiLCJjb21tYW5kIiwiZGlzY2FyZFNlbGVjdGlvbkZyb21Db21tYW5kIiwiZGlzY2FyZFJvd3MiLCJzZWxlY3RlZFJvd3MiLCJzZWxlY3Rpb25Nb2RlIiwiZGlkVG9nZ2xlTW9kZUNoYW5nZSIsIlByb21pc2UiLCJhbGwiLCJmaWx0ZXIiLCJmcCIsImRpZENoYW5nZUV4ZWN1dGFibGVNb2RlIiwibWFwIiwidG9nZ2xlTW9kZUNoYW5nZSIsImRpZFRvZ2dsZVN5bWxpbmtDaGFuZ2UiLCJoYXNUeXBlY2hhbmdlIiwidG9nZ2xlU3ltbGlua0NoYW5nZSIsIm1vdXNlU2VsZWN0aW9uSW5Qcm9ncmVzcyIsImxhc3RNb3VzZU1vdmVMaW5lIiwibmV4dFNlbGVjdGlvbk1vZGUiLCJzdXBwcmVzc0NoYW5nZXMiLCJyZWZSb290IiwiUmVmSG9sZGVyIiwicmVmRWRpdG9yIiwicmVmRWRpdG9yRWxlbWVudCIsInN1YnMiLCJDb21wb3NpdGVEaXNwb3NhYmxlIiwiYWRkIiwib2JzZXJ2ZSIsImVkaXRvciIsInNldHRlciIsImdldEVsZW1lbnQiLCJlbGVtZW50IiwicmVmSW5pdGlhbEZvY3VzIiwiY29tcG9uZW50RGlkTW91bnQiLCJ3aW5kb3ciLCJhZGRFdmVudExpc3RlbmVyIiwiZGlkTW91c2VVcCIsImZpcnN0UGF0Y2giLCJtdWx0aUZpbGVQYXRjaCIsImdldEZpbGVQYXRjaGVzIiwiZmlyc3RIdW5rIiwic2V0U2VsZWN0ZWRCdWZmZXJSYW5nZSIsImdldFJhbmdlIiwiY29uZmlnIiwib25EaWRDaGFuZ2UiLCJuZXdWYWx1ZSIsImZvcmNlVXBkYXRlIiwiZ2V0U25hcHNob3RCZWZvcmVVcGRhdGUiLCJwcmV2UHJvcHMiLCJuZXdTZWxlY3Rpb25SYW5nZSIsImdldE5leHRTZWxlY3Rpb25SYW5nZSIsImFkb3B0QnVmZmVyRnJvbSIsImNvbXBvbmVudERpZFVwZGF0ZSIsInByZXZTdGF0ZSIsIm5leHRIdW5rcyIsIlNldCIsIlJhbmdlIiwiZnJvbU9iamVjdCIsImdldFJvd3MiLCJyb3ciLCJnZXRIdW5rQXQiLCJCb29sZWFuIiwibmV4dFJhbmdlcyIsInNpemUiLCJodW5rIiwic2V0U2VsZWN0ZWRCdWZmZXJSYW5nZXMiLCJjb21wb25lbnRXaWxsVW5tb3VudCIsInJlbW92ZUV2ZW50TGlzdGVuZXIiLCJkaXNwb3NlIiwicmVuZGVyIiwicm9vdENsYXNzIiwiYW55UHJlc2VudCIsInJlbmRlckNvbW1hbmRzIiwicmVuZGVyTm9uRW1wdHlQYXRjaCIsInJlbmRlckVtcHR5UGF0Y2giLCJDb21taXREZXRhaWxJdGVtIiwiSXNzdWVpc2hEZXRhaWxJdGVtIiwiY29tbWFuZHMiLCJzZWxlY3ROZXh0SHVuayIsInNlbGVjdFByZXZpb3VzSHVuayIsImRpZFRvZ2dsZVNlbGVjdGlvbk1vZGUiLCJzdGFnZU1vZGVDb21tYW5kIiwic3RhZ2VTeW1saW5rQ29tbWFuZCIsImRpZEFueUNoYW5nZUV4ZWN1dGFibGVNb2RlIiwiYW55SGF2ZVR5cGVjaGFuZ2UiLCJkaWRDb25maXJtIiwic3VyZmFjZSIsIndvcmtzcGFjZSIsImdldEJ1ZmZlciIsImRpZEFkZFNlbGVjdGlvbiIsImRpZENoYW5nZVNlbGVjdGlvblJhbmdlIiwiZGlkRGVzdHJveVNlbGVjdGlvbiIsIm9sZExpbmVOdW1iZXJMYWJlbCIsImRpZE1vdXNlRG93bk9uTGluZU51bWJlciIsImRpZE1vdXNlTW92ZU9uTGluZU51bWJlciIsIm5ld0xpbmVOdW1iZXJMYWJlbCIsImdldCIsInJlbmRlckxpbmVEZWNvcmF0aW9ucyIsIkluZmluaXR5IiwiZ3V0dGVyIiwiaWNvbiIsImxpbmUiLCJyZW5kZXJEZWNvcmF0aW9uc09uTGF5ZXIiLCJnZXRBZGRpdGlvbkxheWVyIiwiZ2V0RGVsZXRpb25MYXllciIsImdldE5vTmV3bGluZUxheWVyIiwib2xkTW9kZSIsImdldE9sZE1vZGUiLCJuZXdNb2RlIiwiZ2V0TmV3TW9kZSIsImF0dHJzIiwiYWN0aW9uSWNvbiIsImFjdGlvblRleHQiLCJoYXNTeW1saW5rIiwiZGV0YWlsIiwidGl0bGUiLCJvbGRTeW1saW5rIiwiZ2V0T2xkU3ltbGluayIsIm5ld1N5bWxpbmsiLCJnZXROZXdTeW1saW5rIiwidG9nZ2xlVmVyYiIsInNlbGVjdGVkSHVua3MiLCJpbmRleCIsImNvbnRhaW5zU2VsZWN0aW9uIiwiaGFzIiwiaXNTZWxlY3RlZCIsImJ1dHRvblN1ZmZpeCIsInRvZ2dsZVNlbGVjdGlvbkxhYmVsIiwiZGlzY2FyZFNlbGVjdGlvbkxhYmVsIiwic3RhcnRQb2ludCIsInN0YXJ0Iiwic3RhcnRSYW5nZSIsImtleW1hcHMiLCJ0b2dnbGVIdW5rU2VsZWN0aW9uIiwiZGlzY2FyZEh1bmtTZWxlY3Rpb24iLCJkaWRNb3VzZURvd25PbkhlYWRlciIsInJhbmdlcyIsImxpbmVDbGFzcyIsInJlZkhvbGRlciIsImhvbGRlciIsInJhbmdlIiwicmVuZGVyRGVjb3JhdGlvbnMiLCJsYXllciIsImdldE1hcmtlckNvdW50IiwidG9nZ2xlUm93cyIsImNoYW5nZVJvd3MiLCJnZXRDaGFuZ2VzIiwicmVkdWNlIiwicm93cyIsImNoYW5nZSIsInB1c2giLCJnZXRCdWZmZXJSb3dzIiwiZXZlbnQiLCJoYW5kbGVTZWxlY3Rpb25FdmVudCIsImJ1ZmZlclJvdyIsInVuZGVmaW5lZCIsImlzTmFOIiwiZG9tRXZlbnQiLCJyYW5nZUxpa2UiLCJvcHRzIiwiYnV0dG9uIiwiaXNXaW5kb3dzIiwicHJvY2VzcyIsInBsYXRmb3JtIiwiY3RybEtleSIsIm9wdGlvbnMiLCJjb252ZXJ0ZWQiLCJjbGlwQnVmZmVyUmFuZ2UiLCJnZXRPciIsIm1ldGFLZXkiLCJpbnRlcnNlY3RzIiwid2l0aG91dCIsInNlbGVjdGlvbiIsImdldFNlbGVjdGlvbnMiLCJpbnRlcnNlY3RzQnVmZmVyUmFuZ2UiLCJzZWxlY3Rpb25SYW5nZSIsImdldEJ1ZmZlclJhbmdlIiwibmV3UmFuZ2VzIiwiaXNFcXVhbCIsIm51ZGdlZCIsImNvbHVtbiIsImxhc3RDb2x1bW4iLCJsaW5lTGVuZ3RoRm9yUm93IiwiZW5kIiwic2V0QnVmZmVyUmFuZ2UiLCJuZXdSYW5nZSIsInNsaWNlIiwiYWRkU2VsZWN0aW9uRm9yQnVmZmVyUmFuZ2UiLCJyZXZlcnNlZCIsImlzUmV2ZXJzZWQiLCJyZXBsYWNlbWVudFJhbmdlcyIsImVhY2giLCJzaGlmdEtleSIsImxhc3RTZWxlY3Rpb24iLCJnZXRMYXN0U2VsZWN0aW9uIiwibGFzdFNlbGVjdGlvblJhbmdlIiwiaXNCZWZvcmUiLCJpc0xlc3NUaGFuIiwiZmFyRWRnZSIsImdldFNlbGVjdGVkSHVua3MiLCJ3aXRoU2VsZWN0aW9uTW9kZSIsImh1bmtSYW5nZXMiLCJmaXJzdENoYW5nZVJvdyIsImZpcnN0Q2hhbmdlIiwiZ2V0U3RhcnRCdWZmZXJSb3ciLCJ3aXRoU2VsZWN0ZWRIdW5rcyIsImdldEh1bmtBZnRlciIsImdldEh1bmtCZWZvcmUiLCJjdXJzb3JzQnlGaWxlUGF0Y2giLCJNYXAiLCJwbGFjZWRSb3dzIiwiY3Vyc29yIiwiZ2V0Q3Vyc29ycyIsImN1cnNvclJvdyIsImdldEJ1ZmZlclBvc2l0aW9uIiwiZ2V0RmlsZVBhdGNoQXQiLCJuZXdSb3ciLCJnZXROZXdSb3dBdCIsIm5ld0NvbHVtbiIsIm5lYXJlc3RSb3ciLCJnZXROZXdTdGFydFJvdyIsInJlZ2lvbiIsImdldFJlZ2lvbnMiLCJpbmNsdWRlc0J1ZmZlclJvdyIsIndoZW4iLCJ1bmNoYW5nZWQiLCJidWZmZXJSb3dDb3VudCIsImFkZGl0aW9uIiwiY3Vyc29ycyIsInNldCIsImZpbGVQYXRjaGVzV2l0aEN1cnNvcnMiLCJrZXlzIiwib3BlbkZpbGUiLCJwZW5kaW5nIiwidmFsdWUiLCJnZXRTZWxlY3RlZFJvd3MiLCJhY2MiLCJpc0NoYW5nZVJvdyIsImRpZENoYW5nZVNlbGVjdGVkUm93cyIsIm9sZEJ1ZmZlclJhbmdlIiwibmV3QnVmZmVyUmFuZ2UiLCJuZXh0Q3Vyc29yUm93cyIsImdldEN1cnNvckJ1ZmZlclBvc2l0aW9ucyIsInBvc2l0aW9uIiwic3BhbnNNdWx0aXBsZUZpbGVzIiwic2VsZWN0ZWRSb3dzQ2hhbmdlZCIsInNvZnRXcmFwcGVkIiwicGFkIiwib2xkUm93IiwiZ2V0T2xkUm93QXQiLCJjYWxsYmFjayIsInNlZW4iLCJnZXRTZWxlY3RlZEJ1ZmZlclJhbmdlcyIsInBhdGNoZXMiLCJwYXRjaCIsInByZXZSb3ciLCJuZXh0Um93IiwiY2hhbmdlTGF5ZXJzIiwic29tZSIsImZpbmRNYXJrZXJzIiwiaW50ZXJzZWN0c1JvdyIsImNhbGxiYWNrcyIsIkVycm9yIiwibnVtIiwibWF4RGlnaXRzIiwiZ2V0TWF4TGluZU51bWJlcldpZHRoIiwicmVwZWF0IiwidG9TdHJpbmciLCJwcm9wVHlwZXMiLCJQcm9wVHlwZXMiLCJvbmVPZiIsImJvb2wiLCJNdWx0aUZpbGVQYXRjaFByb3BUeXBlIiwiaXNSZXF1aXJlZCIsIm9iamVjdCIsInJlcG9zaXRvcnkiLCJmdW5jIiwiUmVmSG9sZGVyUHJvcFR5cGUiLCJJdGVtVHlwZVByb3BUeXBlIl0sIm1hcHBpbmdzIjoiOzs7Ozs7OztBQUFBOzs7O0FBQ0E7Ozs7QUFDQTs7OztBQUNBOztBQUNBOztBQUVBOztBQUNBOztBQUNBOzs7O0FBQ0E7Ozs7QUFDQTs7OztBQUNBOzs7O0FBQ0E7Ozs7QUFDQTs7OztBQUNBOzs7O0FBQ0E7Ozs7QUFDQTs7OztBQUNBOzs7O0FBQ0E7Ozs7QUFDQTs7OztBQUNBOzs7O0FBQ0E7Ozs7OztBQUVBLE1BQU1BLGlCQUFpQjtBQUNyQixHQUFDQyxlQUFLQyxLQUFMLENBQVdDLE1BQVosR0FBcUIsZ0JBREE7QUFFckIsR0FBQ0YsZUFBS0MsS0FBTCxDQUFXRSxVQUFaLEdBQXlCO0FBRkosQ0FBdkI7O0FBS0EsTUFBTUMsaUJBQWlCLFFBQXZCOztBQUVBLE1BQU1DLGNBQWMsTUFBTUQsY0FBMUI7O0FBRWUsTUFBTUUsa0JBQU4sU0FBaUNDLGdCQUFNQyxTQUF2QyxDQUFpRDs7QUFnQzlEQyxjQUFZQyxLQUFaLEVBQW1CO0FBQ2pCLFVBQU1BLEtBQU47O0FBRGlCLFNBc1BuQkMsMEJBdFBtQixHQXNQVUMsYUFBYTtBQUN4QyxhQUNFO0FBQUMsdUJBQUQ7QUFBQSxVQUFVLEtBQUtBLFVBQVVDLE9BQVYsRUFBZjtBQUNFO0FBQUMsMEJBQUQ7QUFBQSxZQUFRLFlBQVcsT0FBbkIsRUFBMkIsYUFBYUQsVUFBVUUsYUFBVixFQUF4QztBQUNFO0FBQUMsZ0NBQUQ7QUFBQSxjQUFZLE1BQUssT0FBakIsRUFBeUIsV0FBVSxtQ0FBbkM7QUFDRSwwQ0FBQyw2QkFBRDtBQUNFLHdCQUFVLEtBQUtKLEtBQUwsQ0FBV0ssUUFEdkI7QUFFRSx1QkFBU0gsVUFBVUMsT0FBVixFQUZYO0FBR0UsdUJBQVNELFVBQVVJLFNBQVYsT0FBMEIsU0FBMUIsR0FBc0NKLFVBQVVLLFVBQVYsRUFBdEMsR0FBK0QsSUFIMUU7QUFJRSw2QkFBZSxLQUFLUCxLQUFMLENBQVdRLGFBSjVCO0FBS0UsaUNBQW1CLEtBQUtSLEtBQUwsQ0FBV1MsaUJBTGhDO0FBTUUsd0JBQVVQLFVBQVVRLFFBQVYsR0FBcUJDLE1BQXJCLEdBQThCLENBTjFDO0FBT0UsOEJBQWdCLEtBQUtYLEtBQUwsQ0FBV1ksY0FQN0I7QUFRRSx5Q0FBMkIsS0FBS1osS0FBTCxDQUFXYSx5QkFSeEM7O0FBVUUsd0JBQVUsS0FBS2IsS0FBTCxDQUFXYyxRQVZ2Qjs7QUFZRSwrQkFBaUIsTUFBTSxLQUFLQyx5QkFBTCxDQUErQmIsU0FBL0IsQ0FaekI7QUFhRSxtQ0FBcUIsTUFBTSxLQUFLRixLQUFMLENBQVdnQixtQkFBWCxDQUErQmQsU0FBL0IsQ0FiN0I7QUFjRSx3QkFBVSxNQUFNLEtBQUtlLFdBQUwsQ0FBaUIsRUFBQ0MsbUJBQW1CaEIsU0FBcEIsRUFBakIsQ0FkbEI7QUFlRSwwQkFBWSxNQUFNLEtBQUtGLEtBQUwsQ0FBV21CLFVBQVgsQ0FBc0JqQixTQUF0QjtBQWZwQixjQURGO0FBa0JHLGlCQUFLa0IsdUJBQUwsQ0FBNkJsQixTQUE3QixDQWxCSDtBQW1CRyxpQkFBS21CLDhCQUFMLENBQW9DbkIsU0FBcEM7QUFuQkg7QUFERixTQURGO0FBeUJHLGFBQUtvQixpQkFBTCxDQUF1QnBCLFNBQXZCO0FBekJILE9BREY7QUE2QkQsS0FwUmtCOztBQUFBLFNBK2dCbkJxQiwyQkEvZ0JtQixHQStnQlcsTUFBTTtBQUNsQyxVQUFJLEtBQUt2QixLQUFMLENBQVdZLGNBQWYsRUFBK0I7QUFDN0IsY0FBTVksc0JBQXNCQyxNQUFNQyxJQUFOLENBQVcsS0FBS0Msc0JBQUwsRUFBWCxDQUE1QjtBQUNBLFlBQUksS0FBSzNCLEtBQUwsQ0FBV0ssUUFBWCxLQUF3QnVCLHlCQUE1QixFQUE2QztBQUMzQyxlQUFLNUIsS0FBTCxDQUFXNkIsZUFBWCxDQUEyQkwsb0JBQW9CLENBQXBCLENBQTNCLEVBQW1ELEVBQUNNLGFBQWEsRUFBQ0MsU0FBUyxXQUFWLEVBQWQsRUFBbkQ7QUFDRDtBQUNGO0FBQ0YsS0F0aEJrQjs7QUFBQSxTQXdoQm5CaEIseUJBeGhCbUIsR0F3aEJTYixhQUFhO0FBQ3ZDLFdBQUtGLEtBQUwsQ0FBVzZCLGVBQVgsQ0FBMkIzQixTQUEzQixFQUFzQyxFQUFDNEIsYUFBYSxRQUFkLEVBQXRDO0FBQ0QsS0ExaEJrQjs7QUFBQSxTQTRoQm5CRSwyQkE1aEJtQixHQTRoQlcsTUFBTTtBQUNsQyxhQUFPLEtBQUtoQyxLQUFMLENBQVdpQyxXQUFYLENBQ0wsS0FBS2pDLEtBQUwsQ0FBV2tDLFlBRE4sRUFFTCxLQUFLbEMsS0FBTCxDQUFXbUMsYUFGTixFQUdMLEVBQUNMLGFBQWEsRUFBQ0MsU0FBUywrQkFBVixFQUFkLEVBSEssQ0FBUDtBQUtELEtBbGlCa0I7O0FBQUEsU0F3dkJuQkssbUJBeHZCbUIsR0F3dkJHLE1BQU07QUFDMUIsYUFBT0MsUUFBUUMsR0FBUixDQUNMYixNQUFNQyxJQUFOLENBQVcsS0FBS0Msc0JBQUwsRUFBWCxFQUNHWSxNQURILENBQ1VDLE1BQU1BLEdBQUdDLHVCQUFILEVBRGhCLEVBRUdDLEdBRkgsQ0FFTyxLQUFLMUMsS0FBTCxDQUFXMkMsZ0JBRmxCLENBREssQ0FBUDtBQUtELEtBOXZCa0I7O0FBQUEsU0Fnd0JuQkMsc0JBaHdCbUIsR0Fnd0JNLE1BQU07QUFDN0IsYUFBT1AsUUFBUUMsR0FBUixDQUNMYixNQUFNQyxJQUFOLENBQVcsS0FBS0Msc0JBQUwsRUFBWCxFQUNHWSxNQURILENBQ1VDLE1BQU1BLEdBQUdLLGFBQUgsRUFEaEIsRUFFR0gsR0FGSCxDQUVPLEtBQUsxQyxLQUFMLENBQVc4QyxtQkFGbEIsQ0FESyxDQUFQO0FBS0QsS0F0d0JrQjs7QUFFakIsMkJBQ0UsSUFERixFQUVFLHNCQUZGLEVBRTBCLDBCQUYxQixFQUVzRCwwQkFGdEQsRUFFa0YsWUFGbEYsRUFHRSxZQUhGLEVBR2dCLHdCQUhoQixFQUcwQyxnQkFIMUMsRUFHNEQsb0JBSDVELEVBSUUsYUFKRixFQUlpQixpQkFKakIsRUFJb0MseUJBSnBDLEVBSStELHFCQUovRCxFQUtFLG9CQUxGLEVBS3dCLG9CQUx4Qjs7QUFRQSxTQUFLQyx3QkFBTCxHQUFnQyxLQUFoQztBQUNBLFNBQUtDLGlCQUFMLEdBQXlCLElBQXpCO0FBQ0EsU0FBS0MsaUJBQUwsR0FBeUIsSUFBekI7QUFDQSxTQUFLQyxlQUFMLEdBQXVCLEtBQXZCO0FBQ0EsU0FBS0MsT0FBTCxHQUFlLElBQUlDLG1CQUFKLEVBQWY7QUFDQSxTQUFLQyxTQUFMLEdBQWlCLElBQUlELG1CQUFKLEVBQWpCO0FBQ0EsU0FBS0UsZ0JBQUwsR0FBd0IsSUFBSUYsbUJBQUosRUFBeEI7O0FBRUEsU0FBS0csSUFBTCxHQUFZLElBQUlDLDZCQUFKLEVBQVo7O0FBRUEsU0FBS0QsSUFBTCxDQUFVRSxHQUFWLENBQ0UsS0FBS0osU0FBTCxDQUFlSyxPQUFmLENBQXVCQyxVQUFVO0FBQy9CLFdBQUtMLGdCQUFMLENBQXNCTSxNQUF0QixDQUE2QkQsT0FBT0UsVUFBUCxFQUE3QjtBQUNELEtBRkQsQ0FERixFQUlFLEtBQUtQLGdCQUFMLENBQXNCSSxPQUF0QixDQUE4QkksV0FBVztBQUN2QyxXQUFLOUQsS0FBTCxDQUFXK0QsZUFBWCxJQUE4QixLQUFLL0QsS0FBTCxDQUFXK0QsZUFBWCxDQUEyQkgsTUFBM0IsQ0FBa0NFLE9BQWxDLENBQTlCO0FBQ0QsS0FGRCxDQUpGO0FBUUQ7O0FBRURFLHNCQUFvQjtBQUNsQkMsV0FBT0MsZ0JBQVAsQ0FBd0IsU0FBeEIsRUFBbUMsS0FBS0MsVUFBeEM7QUFDQSxTQUFLZCxTQUFMLENBQWVYLEdBQWYsQ0FBbUJpQixVQUFVO0FBQzNCO0FBQ0EsWUFBTSxDQUFDUyxVQUFELElBQWUsS0FBS3BFLEtBQUwsQ0FBV3FFLGNBQVgsQ0FBMEJDLGNBQTFCLEVBQXJCO0FBQ0EsWUFBTSxDQUFDQyxTQUFELElBQWNILFdBQVcxRCxRQUFYLEVBQXBCO0FBQ0EsVUFBSSxDQUFDNkQsU0FBTCxFQUFnQjtBQUNkLGVBQU8sSUFBUDtBQUNEOztBQUVELFdBQUt0QixpQkFBTCxHQUF5QixNQUF6QjtBQUNBVSxhQUFPYSxzQkFBUCxDQUE4QkQsVUFBVUUsUUFBVixFQUE5QjtBQUNBLGFBQU8sSUFBUDtBQUNELEtBWEQ7O0FBYUEsU0FBS2xCLElBQUwsQ0FBVUUsR0FBVixDQUNFLEtBQUt6RCxLQUFMLENBQVcwRSxNQUFYLENBQWtCQyxXQUFsQixDQUE4QiwyQkFBOUIsRUFBMkQsQ0FBQyxFQUFDQyxRQUFELEVBQUQsS0FBZ0IsS0FBS0MsV0FBTCxFQUEzRSxDQURGO0FBR0Q7O0FBRURDLDBCQUF3QkMsU0FBeEIsRUFBbUM7QUFDakMsUUFBSUMsb0JBQW9CLElBQXhCO0FBQ0EsUUFBSSxLQUFLaEYsS0FBTCxDQUFXcUUsY0FBWCxLQUE4QlUsVUFBVVYsY0FBNUMsRUFBNEQ7QUFDMUQ7QUFDQTtBQUNBVywwQkFBb0IsS0FBS2hGLEtBQUwsQ0FBV3FFLGNBQVgsQ0FBMEJZLHFCQUExQixDQUNsQkYsVUFBVVYsY0FEUSxFQUVsQlUsVUFBVTdDLFlBRlEsQ0FBcEI7O0FBS0EsV0FBS2dCLGVBQUwsR0FBdUIsSUFBdkI7QUFDQSxXQUFLbEQsS0FBTCxDQUFXcUUsY0FBWCxDQUEwQmEsZUFBMUIsQ0FBMENILFVBQVVWLGNBQXBEO0FBQ0EsV0FBS25CLGVBQUwsR0FBdUIsS0FBdkI7QUFDRDtBQUNELFdBQU84QixpQkFBUDtBQUNEOztBQUVERyxxQkFBbUJKLFNBQW5CLEVBQThCSyxTQUE5QixFQUF5Q0osaUJBQXpDLEVBQTREO0FBQzFELFFBQUlELFVBQVVoQixlQUFWLEtBQThCLEtBQUsvRCxLQUFMLENBQVcrRCxlQUE3QyxFQUE4RDtBQUM1RGdCLGdCQUFVaEIsZUFBVixJQUE2QmdCLFVBQVVoQixlQUFWLENBQTBCSCxNQUExQixDQUFpQyxJQUFqQyxDQUE3QjtBQUNBLFdBQUs1RCxLQUFMLENBQVcrRCxlQUFYLElBQThCLEtBQUtULGdCQUFMLENBQXNCWixHQUF0QixDQUEwQixLQUFLMUMsS0FBTCxDQUFXK0QsZUFBWCxDQUEyQkgsTUFBckQsQ0FBOUI7QUFDRDs7QUFFRCxRQUFJb0IsaUJBQUosRUFBdUI7QUFDckIsV0FBSzNCLFNBQUwsQ0FBZVgsR0FBZixDQUFtQmlCLFVBQVU7QUFDM0IsWUFBSSxLQUFLM0QsS0FBTCxDQUFXbUMsYUFBWCxLQUE2QixNQUFqQyxFQUF5QztBQUN2QyxlQUFLYyxpQkFBTCxHQUF5QixNQUF6QjtBQUNBVSxpQkFBT2Esc0JBQVAsQ0FBOEJRLGlCQUE5QjtBQUNELFNBSEQsTUFHTztBQUNMLGdCQUFNSyxZQUFZLElBQUlDLEdBQUosQ0FDaEJDLFlBQU1DLFVBQU4sQ0FBaUJSLGlCQUFqQixFQUFvQ1MsT0FBcEMsR0FDRy9DLEdBREgsQ0FDT2dELE9BQU8sS0FBSzFGLEtBQUwsQ0FBV3FFLGNBQVgsQ0FBMEJzQixTQUExQixDQUFvQ0QsR0FBcEMsQ0FEZCxFQUVHbkQsTUFGSCxDQUVVcUQsT0FGVixDQURnQixDQUFsQjtBQUtBLGdCQUFNQyxhQUFhUixVQUFVUyxJQUFWLEdBQWlCLENBQWpCLEdBQ2ZyRSxNQUFNQyxJQUFOLENBQVcyRCxTQUFYLEVBQXNCVSxRQUFRQSxLQUFLdEIsUUFBTCxFQUE5QixDQURlLEdBRWYsQ0FBQyxDQUFDLENBQUMsQ0FBRCxFQUFJLENBQUosQ0FBRCxFQUFTLENBQUMsQ0FBRCxFQUFJLENBQUosQ0FBVCxDQUFELENBRko7O0FBSUEsZUFBS3hCLGlCQUFMLEdBQXlCLE1BQXpCO0FBQ0FVLGlCQUFPcUMsdUJBQVAsQ0FBK0JILFVBQS9CO0FBQ0Q7O0FBRUQsZUFBTyxJQUFQO0FBQ0QsT0FuQkQ7QUFvQkQsS0FyQkQsTUFxQk87QUFDTCxXQUFLNUMsaUJBQUwsR0FBeUIsSUFBekI7QUFDRDtBQUNGOztBQUVEZ0QseUJBQXVCO0FBQ3JCaEMsV0FBT2lDLG1CQUFQLENBQTJCLFNBQTNCLEVBQXNDLEtBQUsvQixVQUEzQztBQUNBLFNBQUtaLElBQUwsQ0FBVTRDLE9BQVY7QUFDRDs7QUFFREMsV0FBUztBQUNQLFVBQU1DLFlBQVksMEJBQ2hCLHNCQURnQixFQUVoQixFQUFDLENBQUUseUJBQXdCLEtBQUtyRyxLQUFMLENBQVdRLGFBQWMsRUFBbkQsR0FBdUQsS0FBS1IsS0FBTCxDQUFXUSxhQUFuRSxFQUZnQixFQUdoQixFQUFDLCtCQUErQixDQUFDLEtBQUtSLEtBQUwsQ0FBV3FFLGNBQVgsQ0FBMEJpQyxVQUExQixFQUFqQyxFQUhnQixFQUloQixFQUFDLGtDQUFrQyxLQUFLdEcsS0FBTCxDQUFXbUMsYUFBWCxLQUE2QixNQUFoRSxFQUpnQixDQUFsQjs7QUFPQSxXQUNFO0FBQUE7QUFBQSxRQUFLLFdBQVdrRSxTQUFoQixFQUEyQixLQUFLLEtBQUtsRCxPQUFMLENBQWFTLE1BQTdDO0FBQ0csV0FBSzJDLGNBQUwsRUFESDtBQUdFO0FBQUE7QUFBQSxVQUFNLFdBQVUsZ0NBQWhCO0FBQ0csYUFBS3ZHLEtBQUwsQ0FBV3FFLGNBQVgsQ0FBMEJpQyxVQUExQixLQUF5QyxLQUFLRSxtQkFBTCxFQUF6QyxHQUFzRSxLQUFLQyxnQkFBTDtBQUR6RTtBQUhGLEtBREY7QUFTRDs7QUFFREYsbUJBQWlCO0FBQ2YsUUFBSSxLQUFLdkcsS0FBTCxDQUFXSyxRQUFYLEtBQXdCcUcsMEJBQXhCLElBQTRDLEtBQUsxRyxLQUFMLENBQVdLLFFBQVgsS0FBd0JzRyw0QkFBeEUsRUFBNEY7QUFDMUYsYUFDRTtBQUFDLDBCQUFEO0FBQUEsVUFBVSxVQUFVLEtBQUszRyxLQUFMLENBQVc0RyxRQUEvQixFQUF5QyxRQUFRLEtBQUt6RCxPQUF0RDtBQUNFLHNDQUFDLGlCQUFELElBQVMsU0FBUSx5QkFBakIsRUFBMkMsVUFBVSxLQUFLMEQsY0FBMUQsR0FERjtBQUVFLHNDQUFDLGlCQUFELElBQVMsU0FBUSw2QkFBakIsRUFBK0MsVUFBVSxLQUFLQyxrQkFBOUQsR0FGRjtBQUdFLHNDQUFDLGlCQUFELElBQVMsU0FBUSxvQ0FBakIsRUFBc0QsVUFBVSxLQUFLQyxzQkFBckU7QUFIRixPQURGO0FBT0Q7O0FBRUQsUUFBSUMsbUJBQW1CLElBQXZCO0FBQ0EsUUFBSUMsc0JBQXNCLElBQTFCOztBQUVBLFFBQUksS0FBS2pILEtBQUwsQ0FBV3FFLGNBQVgsQ0FBMEI2QywwQkFBMUIsRUFBSixFQUE0RDtBQUMxRCxZQUFNbkYsVUFBVSxLQUFLL0IsS0FBTCxDQUFXUSxhQUFYLEtBQTZCLFVBQTdCLEdBQ1osK0JBRFksR0FFWixpQ0FGSjtBQUdBd0cseUJBQW1CLDhCQUFDLGlCQUFELElBQVMsU0FBU2pGLE9BQWxCLEVBQTJCLFVBQVUsS0FBS0ssbUJBQTFDLEdBQW5CO0FBQ0Q7O0FBRUQsUUFBSSxLQUFLcEMsS0FBTCxDQUFXcUUsY0FBWCxDQUEwQjhDLGlCQUExQixFQUFKLEVBQW1EO0FBQ2pELFlBQU1wRixVQUFVLEtBQUsvQixLQUFMLENBQVdRLGFBQVgsS0FBNkIsVUFBN0IsR0FDWiw2QkFEWSxHQUVaLCtCQUZKO0FBR0F5Ryw0QkFBc0IsOEJBQUMsaUJBQUQsSUFBUyxTQUFTbEYsT0FBbEIsRUFBMkIsVUFBVSxLQUFLYSxzQkFBMUMsR0FBdEI7QUFDRDs7QUFFRCxXQUNFO0FBQUMsd0JBQUQ7QUFBQSxRQUFVLFVBQVUsS0FBSzVDLEtBQUwsQ0FBVzRHLFFBQS9CLEVBQXlDLFFBQVEsS0FBS3pELE9BQXREO0FBQ0Usb0NBQUMsaUJBQUQsSUFBUyxTQUFRLHlCQUFqQixFQUEyQyxVQUFVLEtBQUswRCxjQUExRCxHQURGO0FBRUUsb0NBQUMsaUJBQUQsSUFBUyxTQUFRLDZCQUFqQixFQUErQyxVQUFVLEtBQUtDLGtCQUE5RCxHQUZGO0FBR0Usb0NBQUMsaUJBQUQsSUFBUyxTQUFRLGNBQWpCLEVBQWdDLFVBQVUsS0FBS00sVUFBL0MsR0FIRjtBQUlFLG9DQUFDLGlCQUFELElBQVMsU0FBUSxXQUFqQixFQUE2QixVQUFVLEtBQUs3RiwyQkFBNUMsR0FKRjtBQUtFLG9DQUFDLGlCQUFELElBQVMsU0FBUSwrQkFBakIsRUFBaUQsVUFBVSxLQUFLUywyQkFBaEUsR0FMRjtBQU1FLG9DQUFDLGlCQUFELElBQVMsU0FBUSxxQkFBakIsRUFBdUMsVUFBVSxLQUFLZixXQUF0RCxHQU5GO0FBT0Usb0NBQUMsaUJBQUQsSUFBUyxTQUFRLGdCQUFqQixFQUFrQyxVQUFVLEtBQUtqQixLQUFMLENBQVdxSCxPQUF2RCxHQVBGO0FBUUUsb0NBQUMsaUJBQUQsSUFBUyxTQUFRLG9DQUFqQixFQUFzRCxVQUFVLEtBQUtOLHNCQUFyRSxHQVJGO0FBU0dDLHNCQVRIO0FBVUdDO0FBVkgsS0FERjtBQWNEOztBQUVEUixxQkFBbUI7QUFDakIsV0FBTztBQUFBO0FBQUEsUUFBRyxXQUFVLDZDQUFiO0FBQUE7QUFBQSxLQUFQO0FBQ0Q7O0FBRURELHdCQUFzQjtBQUNwQixXQUNFO0FBQUMsOEJBQUQ7QUFBQTtBQUNFLG1CQUFXLEtBQUt4RyxLQUFMLENBQVdzSCxTQUR4Qjs7QUFHRSxnQkFBUSxLQUFLdEgsS0FBTCxDQUFXcUUsY0FBWCxDQUEwQmtELFNBQTFCLEVBSFY7QUFJRSxpQ0FBeUIsS0FKM0I7QUFLRSxtQkFBVyxLQUxiO0FBTUUsb0JBQVksS0FOZDtBQU9FLGtCQUFVLElBUFo7QUFRRSxxQkFBYSxJQVJmOztBQVVFLHlCQUFpQixLQUFLQyxlQVZ4QjtBQVdFLGlDQUF5QixLQUFLQyx1QkFYaEM7QUFZRSw2QkFBcUIsS0FBS0MsbUJBWjVCO0FBYUUsa0JBQVUsS0FBS3JFLFNBYmpCO0FBZUUsb0NBQUMsZ0JBQUQ7QUFDRSxjQUFLLGtCQURQO0FBRUUsa0JBQVUsQ0FGWjtBQUdFLG1CQUFVLEtBSFo7QUFJRSxjQUFLLGFBSlA7QUFLRSxpQkFBUyxLQUFLc0Usa0JBTGhCO0FBTUUscUJBQWEsS0FBS0Msd0JBTnBCO0FBT0UscUJBQWEsS0FBS0M7QUFQcEIsUUFmRjtBQXdCRSxvQ0FBQyxnQkFBRDtBQUNFLGNBQUssa0JBRFA7QUFFRSxrQkFBVSxDQUZaO0FBR0UsbUJBQVUsS0FIWjtBQUlFLGNBQUssYUFKUDtBQUtFLGlCQUFTLEtBQUtDLGtCQUxoQjtBQU1FLHFCQUFhLEtBQUtGLHdCQU5wQjtBQU9FLHFCQUFhLEtBQUtDO0FBUHBCLFFBeEJGO0FBaUNHLFdBQUs3SCxLQUFMLENBQVcwRSxNQUFYLENBQWtCcUQsR0FBbEIsQ0FBc0IsMkJBQXRCLEtBQ0MsOEJBQUMsZ0JBQUQ7QUFDRSxjQUFLLFlBRFA7QUFFRSxrQkFBVSxDQUZaO0FBR0UsY0FBSyxhQUhQO0FBSUUsbUJBQVUsT0FKWjtBQUtFLGlCQUFTcEksV0FMWDtBQU1FLHFCQUFhLEtBQUtpSSx3QkFOcEI7QUFPRSxxQkFBYSxLQUFLQztBQVBwQixRQWxDSjtBQTZDRyxXQUFLN0gsS0FBTCxDQUFXcUUsY0FBWCxDQUEwQkMsY0FBMUIsR0FBMkM1QixHQUEzQyxDQUErQyxLQUFLekMsMEJBQXBELENBN0NIO0FBK0NHLFdBQUsrSCxxQkFBTCxDQUNDdkcsTUFBTUMsSUFBTixDQUFXLEtBQUsxQixLQUFMLENBQVdrQyxZQUF0QixFQUFvQ3dELE9BQU9ILFlBQU1DLFVBQU4sQ0FBaUIsQ0FBQyxDQUFDRSxHQUFELEVBQU0sQ0FBTixDQUFELEVBQVcsQ0FBQ0EsR0FBRCxFQUFNdUMsUUFBTixDQUFYLENBQWpCLENBQTNDLENBREQsRUFFQyxxQ0FGRCxFQUdDLEVBQUNDLFFBQVEsSUFBVCxFQUFlQyxNQUFNLElBQXJCLEVBQTJCQyxNQUFNLElBQWpDLEVBSEQsQ0EvQ0g7QUFxREcsV0FBS0Msd0JBQUwsQ0FDQyxLQUFLckksS0FBTCxDQUFXcUUsY0FBWCxDQUEwQmlFLGdCQUExQixFQURELEVBRUMsa0NBRkQsRUFHQyxFQUFDSCxNQUFNLElBQVAsRUFBYUMsTUFBTSxJQUFuQixFQUhELENBckRIO0FBMERHLFdBQUtDLHdCQUFMLENBQ0MsS0FBS3JJLEtBQUwsQ0FBV3FFLGNBQVgsQ0FBMEJrRSxnQkFBMUIsRUFERCxFQUVDLG9DQUZELEVBR0MsRUFBQ0osTUFBTSxJQUFQLEVBQWFDLE1BQU0sSUFBbkIsRUFIRCxDQTFESDtBQStERyxXQUFLQyx3QkFBTCxDQUNDLEtBQUtySSxLQUFMLENBQVdxRSxjQUFYLENBQTBCbUUsaUJBQTFCLEVBREQsRUFFQyxzQ0FGRCxFQUdDLEVBQUNMLE1BQU0sSUFBUCxFQUFhQyxNQUFNLElBQW5CLEVBSEQ7QUEvREgsS0FERjtBQXdFRDs7QUFrQ0QvRyxpQ0FBK0JuQixTQUEvQixFQUEwQztBQUN4QyxRQUFJLENBQUNBLFVBQVV1Qyx1QkFBVixFQUFMLEVBQTBDO0FBQ3hDLGFBQU8sSUFBUDtBQUNEOztBQUVELFVBQU1nRyxVQUFVdkksVUFBVXdJLFVBQVYsRUFBaEI7QUFDQSxVQUFNQyxVQUFVekksVUFBVTBJLFVBQVYsRUFBaEI7O0FBRUEsVUFBTUMsUUFBUSxLQUFLN0ksS0FBTCxDQUFXUSxhQUFYLEtBQTZCLFVBQTdCLEdBQ1Y7QUFDQXNJLGtCQUFZLGdCQURaO0FBRUFDLGtCQUFZO0FBRlosS0FEVSxHQUtWO0FBQ0FELGtCQUFZLGNBRFo7QUFFQUMsa0JBQVk7QUFGWixLQUxKOztBQVVBLFdBQ0U7QUFBQyxpQ0FBRDtBQUFBO0FBQ0UsZUFBTSxhQURSO0FBRUUsb0JBQVlGLE1BQU1DLFVBRnBCO0FBR0Usb0JBQVlELE1BQU1FLFVBSHBCO0FBSUUsa0JBQVUsS0FBSy9JLEtBQUwsQ0FBV0ssUUFKdkI7QUFLRSxnQkFBUSxNQUFNLEtBQUtMLEtBQUwsQ0FBVzJDLGdCQUFYLENBQTRCekMsU0FBNUIsQ0FMaEI7QUFNRTtBQUFDLHVCQUFEO0FBQUE7QUFBQTtBQUVFO0FBQUE7QUFBQSxZQUFNLFdBQVUsc0VBQWhCO0FBQUE7QUFDUWIseUJBQWVvSixPQUFmLENBRFI7QUFBQTtBQUNpQztBQUFBO0FBQUE7QUFBT0E7QUFBUDtBQURqQyxTQUZGO0FBS0U7QUFBQTtBQUFBLFlBQU0sV0FBVSxvRUFBaEI7QUFBQTtBQUNNcEoseUJBQWVzSixPQUFmLENBRE47QUFBQTtBQUMrQjtBQUFBO0FBQUE7QUFBT0E7QUFBUDtBQUQvQjtBQUxGO0FBTkYsS0FERjtBQWtCRDs7QUFFRHZILDBCQUF3QmxCLFNBQXhCLEVBQW1DO0FBQ2pDLFFBQUksQ0FBQ0EsVUFBVThJLFVBQVYsRUFBTCxFQUE2QjtBQUMzQixhQUFPLElBQVA7QUFDRDs7QUFFRCxRQUFJQyxTQUFTLDBDQUFiO0FBQ0EsUUFBSUMsUUFBUSxFQUFaO0FBQ0EsVUFBTUMsYUFBYWpKLFVBQVVrSixhQUFWLEVBQW5CO0FBQ0EsVUFBTUMsYUFBYW5KLFVBQVVvSixhQUFWLEVBQW5CO0FBQ0EsUUFBSUgsY0FBY0UsVUFBbEIsRUFBOEI7QUFDNUJKLGVBQ0U7QUFBQyx1QkFBRDtBQUFBO0FBQUE7QUFFRTtBQUFBO0FBQUEsWUFBTSxXQUFXLDBCQUNmLCtCQURlLEVBRWYsMENBRmUsRUFHZix3Q0FIZSxDQUFqQjtBQUFBO0FBS087QUFBQTtBQUFBO0FBQU9FO0FBQVA7QUFMUCxTQUZGO0FBU0U7QUFBQTtBQUFBLFlBQU0sV0FBVywwQkFDZiwrQkFEZSxFQUVmLDBDQUZlLEVBR2Ysc0NBSGUsQ0FBakI7QUFBQTtBQUtLO0FBQUE7QUFBQTtBQUFPRTtBQUFQO0FBTEwsU0FURjtBQUFBO0FBQUEsT0FERjtBQW1CQUgsY0FBUSxpQkFBUjtBQUNELEtBckJELE1BcUJPLElBQUlDLGNBQWMsQ0FBQ0UsVUFBbkIsRUFBK0I7QUFDcENKLGVBQ0U7QUFBQyx1QkFBRDtBQUFBO0FBQUE7QUFFRTtBQUFBO0FBQUEsWUFBTSxXQUFVLHNFQUFoQjtBQUFBO0FBQ0s7QUFBQTtBQUFBO0FBQU9FO0FBQVA7QUFETCxTQUZGO0FBQUE7QUFBQSxPQURGO0FBU0FELGNBQVEsaUJBQVI7QUFDRCxLQVhNLE1BV0E7QUFDTEQsZUFDRTtBQUFDLHVCQUFEO0FBQUE7QUFBQTtBQUVFO0FBQUE7QUFBQSxZQUFNLFdBQVUsb0VBQWhCO0FBQUE7QUFDSztBQUFBO0FBQUE7QUFBT0k7QUFBUDtBQURMLFNBRkY7QUFBQTtBQUFBLE9BREY7QUFTQUgsY0FBUSxpQkFBUjtBQUNEOztBQUVELFVBQU1MLFFBQVEsS0FBSzdJLEtBQUwsQ0FBV1EsYUFBWCxLQUE2QixVQUE3QixHQUNWO0FBQ0FzSSxrQkFBWSxnQkFEWjtBQUVBQyxrQkFBWTtBQUZaLEtBRFUsR0FLVjtBQUNBRCxrQkFBWSxjQURaO0FBRUFDLGtCQUFZO0FBRlosS0FMSjs7QUFVQSxXQUNFO0FBQUMsaUNBQUQ7QUFBQTtBQUNFLGVBQU9HLEtBRFQ7QUFFRSxvQkFBWUwsTUFBTUMsVUFGcEI7QUFHRSxvQkFBWUQsTUFBTUUsVUFIcEI7QUFJRSxrQkFBVSxLQUFLL0ksS0FBTCxDQUFXSyxRQUp2QjtBQUtFLGdCQUFRLE1BQU0sS0FBS0wsS0FBTCxDQUFXOEMsbUJBQVgsQ0FBK0I1QyxTQUEvQixDQUxoQjtBQU1FO0FBQUMsdUJBQUQ7QUFBQTtBQUNHK0k7QUFESDtBQU5GLEtBREY7QUFZRDs7QUFFRDNILG9CQUFrQnBCLFNBQWxCLEVBQTZCO0FBQzNCLFVBQU1xSixhQUFhLEtBQUt2SixLQUFMLENBQVdRLGFBQVgsS0FBNkIsVUFBN0IsR0FBMEMsT0FBMUMsR0FBb0QsU0FBdkU7QUFDQSxVQUFNZ0osZ0JBQWdCLElBQUlsRSxHQUFKLENBQ3BCN0QsTUFBTUMsSUFBTixDQUFXLEtBQUsxQixLQUFMLENBQVdrQyxZQUF0QixFQUFvQ3dELE9BQU8sS0FBSzFGLEtBQUwsQ0FBV3FFLGNBQVgsQ0FBMEJzQixTQUExQixDQUFvQ0QsR0FBcEMsQ0FBM0MsQ0FEb0IsQ0FBdEI7O0FBSUEsV0FDRTtBQUFDLHFCQUFEO0FBQUE7QUFDRTtBQUFDLDZCQUFEO0FBQUE7QUFDR3hGLGtCQUFVUSxRQUFWLEdBQXFCZ0MsR0FBckIsQ0FBeUIsQ0FBQ3FELElBQUQsRUFBTzBELEtBQVAsS0FBaUI7QUFDekMsZ0JBQU1DLG9CQUFvQixLQUFLMUosS0FBTCxDQUFXbUMsYUFBWCxLQUE2QixNQUE3QixJQUF1Q3FILGNBQWNHLEdBQWQsQ0FBa0I1RCxJQUFsQixDQUFqRTtBQUNBLGdCQUFNNkQsYUFBYyxLQUFLNUosS0FBTCxDQUFXbUMsYUFBWCxLQUE2QixNQUE5QixJQUF5Q3FILGNBQWNHLEdBQWQsQ0FBa0I1RCxJQUFsQixDQUE1RDs7QUFFQSxjQUFJOEQsZUFBZSxFQUFuQjtBQUNBLGNBQUlILGlCQUFKLEVBQXVCO0FBQ3JCRyw0QkFBZ0IsZUFBaEI7QUFDQSxnQkFBSSxLQUFLN0osS0FBTCxDQUFXa0MsWUFBWCxDQUF3QjRELElBQXhCLEdBQStCLENBQW5DLEVBQXNDO0FBQ3BDK0QsOEJBQWdCLEdBQWhCO0FBQ0Q7QUFDRixXQUxELE1BS087QUFDTEEsNEJBQWdCLE1BQWhCO0FBQ0EsZ0JBQUlMLGNBQWMxRCxJQUFkLEdBQXFCLENBQXpCLEVBQTRCO0FBQzFCK0QsOEJBQWdCLEdBQWhCO0FBQ0Q7QUFDRjs7QUFFRCxnQkFBTUMsdUJBQXdCLEdBQUVQLFVBQVcsSUFBR00sWUFBYSxFQUEzRDtBQUNBLGdCQUFNRSx3QkFBeUIsV0FBVUYsWUFBYSxFQUF0RDs7QUFFQSxnQkFBTUcsYUFBYWpFLEtBQUt0QixRQUFMLEdBQWdCd0YsS0FBbkM7QUFDQSxnQkFBTUMsYUFBYSxJQUFJM0UsV0FBSixDQUFVeUUsVUFBVixFQUFzQkEsVUFBdEIsQ0FBbkI7O0FBRUEsaUJBQ0U7QUFBQyw0QkFBRDtBQUFBLGNBQVEsS0FBTSxjQUFhUCxLQUFNLEVBQWpDLEVBQW9DLGFBQWFTLFVBQWpELEVBQTZELFlBQVcsT0FBeEU7QUFDRTtBQUFDLGtDQUFEO0FBQUEsZ0JBQVksTUFBSyxPQUFqQixFQUF5QixXQUFVLG1DQUFuQztBQUNFLDRDQUFDLHdCQUFEO0FBQ0UsMkJBQVcsS0FBSzVHLGdCQURsQjtBQUVFLHNCQUFNeUMsSUFGUjtBQUdFLDRCQUFZNkQsVUFIZDtBQUlFLCtCQUFlLEtBQUs1SixLQUFMLENBQVdRLGFBSjVCO0FBS0UsK0JBQWMsTUFMaEI7QUFNRSxzQ0FBc0JzSixvQkFOeEI7QUFPRSx1Q0FBdUJDLHFCQVB6Qjs7QUFTRSwwQkFBVSxLQUFLL0osS0FBTCxDQUFXYyxRQVR2QjtBQVVFLHlCQUFTLEtBQUtkLEtBQUwsQ0FBV21LLE9BVnRCOztBQVlFLGlDQUFpQixNQUFNLEtBQUtDLG1CQUFMLENBQXlCckUsSUFBekIsRUFBK0IyRCxpQkFBL0IsQ0FaekI7QUFhRSxrQ0FBa0IsTUFBTSxLQUFLVyxvQkFBTCxDQUEwQnRFLElBQTFCLEVBQWdDMkQsaUJBQWhDLENBYjFCO0FBY0UsMkJBQVcsS0FBS1ksb0JBZGxCO0FBZUUsMEJBQVUsS0FBS3RLLEtBQUwsQ0FBV0s7QUFmdkI7QUFERjtBQURGLFdBREY7QUF1QkQsU0E5Q0E7QUFESDtBQURGLEtBREY7QUFxREQ7O0FBRUQySCx3QkFBc0J1QyxNQUF0QixFQUE4QkMsU0FBOUIsRUFBeUMsRUFBQ3BDLElBQUQsRUFBT0YsTUFBUCxFQUFlQyxJQUFmLEVBQXFCc0MsU0FBckIsRUFBekMsRUFBMEU7QUFDeEUsUUFBSUYsT0FBTzVKLE1BQVAsS0FBa0IsQ0FBdEIsRUFBeUI7QUFDdkIsYUFBTyxJQUFQO0FBQ0Q7O0FBRUQsVUFBTStKLFNBQVNELGFBQWEsSUFBSXJILG1CQUFKLEVBQTVCO0FBQ0EsV0FDRTtBQUFDLDJCQUFEO0FBQUEsUUFBYSxhQUFhc0gsT0FBTzlHLE1BQWpDO0FBQ0cyRyxhQUFPN0gsR0FBUCxDQUFXLENBQUNpSSxLQUFELEVBQVFsQixLQUFSLEtBQWtCO0FBQzVCLGVBQ0UsOEJBQUMsZ0JBQUQ7QUFDRSxlQUFNLFFBQU9lLFNBQVUsSUFBR2YsS0FBTSxFQURsQztBQUVFLHVCQUFha0IsS0FGZjtBQUdFLHNCQUFXO0FBSGIsVUFERjtBQU9ELE9BUkEsQ0FESDtBQVVHLFdBQUtDLGlCQUFMLENBQXVCSixTQUF2QixFQUFrQyxFQUFDcEMsSUFBRCxFQUFPRixNQUFQLEVBQWVDLElBQWYsRUFBbEM7QUFWSCxLQURGO0FBY0Q7O0FBRURFLDJCQUF5QndDLEtBQXpCLEVBQWdDTCxTQUFoQyxFQUEyQyxFQUFDcEMsSUFBRCxFQUFPRixNQUFQLEVBQWVDLElBQWYsRUFBM0MsRUFBaUU7QUFDL0QsUUFBSTBDLE1BQU1DLGNBQU4sT0FBMkIsQ0FBL0IsRUFBa0M7QUFDaEMsYUFBTyxJQUFQO0FBQ0Q7O0FBRUQsV0FDRTtBQUFDLDJCQUFEO0FBQUEsUUFBYSxVQUFVRCxLQUF2QjtBQUNHLFdBQUtELGlCQUFMLENBQXVCSixTQUF2QixFQUFrQyxFQUFDcEMsSUFBRCxFQUFPRixNQUFQLEVBQWVDLElBQWYsRUFBbEM7QUFESCxLQURGO0FBS0Q7O0FBRUR5QyxvQkFBa0JKLFNBQWxCLEVBQTZCLEVBQUNwQyxJQUFELEVBQU9GLE1BQVAsRUFBZUMsSUFBZixFQUE3QixFQUFtRDtBQUNqRCxXQUNFO0FBQUMscUJBQUQ7QUFBQTtBQUNHQyxjQUNDLDhCQUFDLG9CQUFEO0FBQ0UsY0FBSyxNQURQO0FBRUUsbUJBQVdvQyxTQUZiO0FBR0UsMEJBQWtCO0FBSHBCLFFBRko7QUFRR3RDLGdCQUNDO0FBQUMsdUJBQUQ7QUFBQTtBQUNFLHNDQUFDLG9CQUFEO0FBQ0UsZ0JBQUssYUFEUDtBQUVFLHNCQUFXLGtCQUZiO0FBR0UscUJBQVdzQyxTQUhiO0FBSUUsNEJBQWtCO0FBSnBCLFVBREY7QUFPRSxzQ0FBQyxvQkFBRDtBQUNFLGdCQUFLLGFBRFA7QUFFRSxzQkFBVyxrQkFGYjtBQUdFLHFCQUFXQSxTQUhiO0FBSUUsNEJBQWtCO0FBSnBCO0FBUEYsT0FUSjtBQXdCR3JDLGNBQ0MsOEJBQUMsb0JBQUQ7QUFDRSxjQUFLLGFBRFA7QUFFRSxvQkFBVyxZQUZiO0FBR0UsbUJBQVdxQyxTQUhiO0FBSUUsMEJBQWtCO0FBSnBCO0FBekJKLEtBREY7QUFtQ0Q7O0FBdUJESixzQkFBb0JyRSxJQUFwQixFQUEwQjJELGlCQUExQixFQUE2QztBQUMzQyxRQUFJQSxpQkFBSixFQUF1QjtBQUNyQixhQUFPLEtBQUsxSixLQUFMLENBQVcrSyxVQUFYLENBQ0wsS0FBSy9LLEtBQUwsQ0FBV2tDLFlBRE4sRUFFTCxLQUFLbEMsS0FBTCxDQUFXbUMsYUFGTixFQUdMLEVBQUNMLGFBQWEsUUFBZCxFQUhLLENBQVA7QUFLRCxLQU5ELE1BTU87QUFDTCxZQUFNa0osYUFBYSxJQUFJMUYsR0FBSixDQUNqQlMsS0FBS2tGLFVBQUwsR0FDR0MsTUFESCxDQUNVLENBQUNDLElBQUQsRUFBT0MsTUFBUCxLQUFrQjtBQUN4QkQsYUFBS0UsSUFBTCxDQUFVLEdBQUdELE9BQU9FLGFBQVAsRUFBYjtBQUNBLGVBQU9ILElBQVA7QUFDRCxPQUpILEVBSUssRUFKTCxDQURpQixDQUFuQjtBQU9BLGFBQU8sS0FBS25MLEtBQUwsQ0FBVytLLFVBQVgsQ0FDTEMsVUFESyxFQUVMLE1BRkssRUFHTCxFQUFDbEosYUFBYSxRQUFkLEVBSEssQ0FBUDtBQUtEO0FBQ0Y7O0FBRUR1SSx1QkFBcUJ0RSxJQUFyQixFQUEyQjJELGlCQUEzQixFQUE4QztBQUM1QyxRQUFJQSxpQkFBSixFQUF1QjtBQUNyQixhQUFPLEtBQUsxSixLQUFMLENBQVdpQyxXQUFYLENBQ0wsS0FBS2pDLEtBQUwsQ0FBV2tDLFlBRE4sRUFFTCxLQUFLbEMsS0FBTCxDQUFXbUMsYUFGTixFQUdMLEVBQUNMLGFBQWEsUUFBZCxFQUhLLENBQVA7QUFLRCxLQU5ELE1BTU87QUFDTCxZQUFNa0osYUFBYSxJQUFJMUYsR0FBSixDQUNqQlMsS0FBS2tGLFVBQUwsR0FDR0MsTUFESCxDQUNVLENBQUNDLElBQUQsRUFBT0MsTUFBUCxLQUFrQjtBQUN4QkQsYUFBS0UsSUFBTCxDQUFVLEdBQUdELE9BQU9FLGFBQVAsRUFBYjtBQUNBLGVBQU9ILElBQVA7QUFDRCxPQUpILEVBSUssRUFKTCxDQURpQixDQUFuQjtBQU9BLGFBQU8sS0FBS25MLEtBQUwsQ0FBV2lDLFdBQVgsQ0FBdUIrSSxVQUF2QixFQUFtQyxNQUFuQyxFQUEyQyxFQUFDbEosYUFBYSxRQUFkLEVBQTNDLENBQVA7QUFDRDtBQUNGOztBQUVEd0ksdUJBQXFCaUIsS0FBckIsRUFBNEJ4RixJQUE1QixFQUFrQztBQUNoQyxTQUFLOUMsaUJBQUwsR0FBeUIsTUFBekI7QUFDQSxTQUFLdUksb0JBQUwsQ0FBMEJELEtBQTFCLEVBQWlDeEYsS0FBS3RCLFFBQUwsRUFBakM7QUFDRDs7QUFFRG1ELDJCQUF5QjJELEtBQXpCLEVBQWdDO0FBQzlCLFVBQU1uRCxPQUFPbUQsTUFBTUUsU0FBbkI7QUFDQSxRQUFJckQsU0FBU3NELFNBQVQsSUFBc0JDLE1BQU12RCxJQUFOLENBQTFCLEVBQXVDO0FBQ3JDO0FBQ0Q7O0FBRUQsU0FBS25GLGlCQUFMLEdBQXlCLE1BQXpCO0FBQ0EsUUFBSSxLQUFLdUksb0JBQUwsQ0FBMEJELE1BQU1LLFFBQWhDLEVBQTBDLENBQUMsQ0FBQ3hELElBQUQsRUFBTyxDQUFQLENBQUQsRUFBWSxDQUFDQSxJQUFELEVBQU9ILFFBQVAsQ0FBWixDQUExQyxDQUFKLEVBQThFO0FBQzVFLFdBQUtsRix3QkFBTCxHQUFnQyxJQUFoQztBQUNEO0FBQ0Y7O0FBRUQ4RSwyQkFBeUIwRCxLQUF6QixFQUFnQztBQUM5QixRQUFJLENBQUMsS0FBS3hJLHdCQUFWLEVBQW9DO0FBQ2xDO0FBQ0Q7O0FBRUQsVUFBTXFGLE9BQU9tRCxNQUFNRSxTQUFuQjtBQUNBLFFBQUksS0FBS3pJLGlCQUFMLEtBQTJCb0YsSUFBM0IsSUFBbUNBLFNBQVNzRCxTQUE1QyxJQUF5REMsTUFBTXZELElBQU4sQ0FBN0QsRUFBMEU7QUFDeEU7QUFDRDtBQUNELFNBQUtwRixpQkFBTCxHQUF5Qm9GLElBQXpCOztBQUVBLFNBQUtuRixpQkFBTCxHQUF5QixNQUF6QjtBQUNBLFNBQUt1SSxvQkFBTCxDQUEwQkQsTUFBTUssUUFBaEMsRUFBMEMsQ0FBQyxDQUFDeEQsSUFBRCxFQUFPLENBQVAsQ0FBRCxFQUFZLENBQUNBLElBQUQsRUFBT0gsUUFBUCxDQUFaLENBQTFDLEVBQXlFLEVBQUN4RSxLQUFLLElBQU4sRUFBekU7QUFDRDs7QUFFRFUsZUFBYTtBQUNYLFNBQUtwQix3QkFBTCxHQUFnQyxLQUFoQztBQUNEOztBQUVEeUksdUJBQXFCRCxLQUFyQixFQUE0Qk0sU0FBNUIsRUFBdUNDLElBQXZDLEVBQTZDO0FBQzNDLFFBQUlQLE1BQU1RLE1BQU4sS0FBaUIsQ0FBckIsRUFBd0I7QUFDdEIsYUFBTyxLQUFQO0FBQ0Q7O0FBRUQsVUFBTUMsWUFBWUMsUUFBUUMsUUFBUixLQUFxQixPQUF2QztBQUNBLFFBQUlYLE1BQU1ZLE9BQU4sSUFBaUIsQ0FBQ0gsU0FBdEIsRUFBaUM7QUFDL0I7QUFDQSxhQUFPLEtBQVA7QUFDRDs7QUFFRCxVQUFNSTtBQUNKM0ksV0FBSztBQURELE9BRURxSSxJQUZDLENBQU47O0FBS0E7QUFDQSxVQUFNTyxZQUFZOUcsWUFBTUMsVUFBTixDQUFpQnFHLFNBQWpCLENBQWxCO0FBQ0EsVUFBTWxCLFFBQVEsS0FBS3RILFNBQUwsQ0FBZVgsR0FBZixDQUFtQmlCLFVBQVVBLE9BQU8ySSxlQUFQLENBQXVCRCxTQUF2QixDQUE3QixFQUFnRUUsS0FBaEUsQ0FBc0VGLFNBQXRFLENBQWQ7O0FBRUEsUUFBSWQsTUFBTWlCLE9BQU4sSUFBaUIsMEJBQTRCakIsTUFBTVksT0FBTixJQUFpQkgsU0FBbEUsRUFBOEU7QUFDNUUsV0FBSzNJLFNBQUwsQ0FBZVgsR0FBZixDQUFtQmlCLFVBQVU7QUFDM0IsWUFBSThJLGFBQWEsS0FBakI7QUFDQSxZQUFJQyxVQUFVLElBQWQ7O0FBRUEsYUFBSyxNQUFNQyxTQUFYLElBQXdCaEosT0FBT2lKLGFBQVAsRUFBeEIsRUFBZ0Q7QUFDOUMsY0FBSUQsVUFBVUUscUJBQVYsQ0FBZ0NsQyxLQUFoQyxDQUFKLEVBQTRDO0FBQzFDO0FBQ0E7QUFDQThCLHlCQUFhLElBQWI7QUFDQSxrQkFBTUssaUJBQWlCSCxVQUFVSSxjQUFWLEVBQXZCOztBQUVBLGtCQUFNQyxZQUFZLEVBQWxCOztBQUVBLGdCQUFJLENBQUNyQyxNQUFNVixLQUFOLENBQVlnRCxPQUFaLENBQW9CSCxlQUFlN0MsS0FBbkMsQ0FBTCxFQUFnRDtBQUM5QztBQUNBLGtCQUFJaUQsU0FBU3ZDLE1BQU1WLEtBQW5CO0FBQ0Esa0JBQUlVLE1BQU1WLEtBQU4sQ0FBWWtELE1BQVosS0FBdUIsQ0FBM0IsRUFBOEI7QUFDNUIsc0JBQU1DLGFBQWF6SixPQUFPNEQsU0FBUCxHQUFtQjhGLGdCQUFuQixDQUFvQzFDLE1BQU1WLEtBQU4sQ0FBWXZFLEdBQVosR0FBa0IsQ0FBdEQsQ0FBbkI7QUFDQXdILHlCQUFTLENBQUN2QyxNQUFNVixLQUFOLENBQVl2RSxHQUFaLEdBQWtCLENBQW5CLEVBQXNCMEgsVUFBdEIsQ0FBVDtBQUNEOztBQUVESix3QkFBVTNCLElBQVYsQ0FBZSxDQUFDeUIsZUFBZTdDLEtBQWhCLEVBQXVCaUQsTUFBdkIsQ0FBZjtBQUNEOztBQUVELGdCQUFJLENBQUN2QyxNQUFNMkMsR0FBTixDQUFVTCxPQUFWLENBQWtCSCxlQUFlUSxHQUFqQyxDQUFMLEVBQTRDO0FBQzFDO0FBQ0Esa0JBQUlKLFNBQVN2QyxNQUFNMkMsR0FBbkI7QUFDQSxvQkFBTUYsYUFBYXpKLE9BQU80RCxTQUFQLEdBQW1COEYsZ0JBQW5CLENBQW9DMUMsTUFBTTJDLEdBQU4sQ0FBVTVILEdBQTlDLENBQW5CO0FBQ0Esa0JBQUlpRixNQUFNMkMsR0FBTixDQUFVSCxNQUFWLEtBQXFCQyxVQUF6QixFQUFxQztBQUNuQ0YseUJBQVMsQ0FBQ3ZDLE1BQU0yQyxHQUFOLENBQVU1SCxHQUFWLEdBQWdCLENBQWpCLEVBQW9CLENBQXBCLENBQVQ7QUFDRDs7QUFFRHNILHdCQUFVM0IsSUFBVixDQUFlLENBQUM2QixNQUFELEVBQVNKLGVBQWVRLEdBQXhCLENBQWY7QUFDRDs7QUFFRCxnQkFBSU4sVUFBVXJNLE1BQVYsR0FBbUIsQ0FBdkIsRUFBMEI7QUFDeEJnTSx3QkFBVVksY0FBVixDQUF5QlAsVUFBVSxDQUFWLENBQXpCO0FBQ0EsbUJBQUssTUFBTVEsUUFBWCxJQUF1QlIsVUFBVVMsS0FBVixDQUFnQixDQUFoQixDQUF2QixFQUEyQztBQUN6QzlKLHVCQUFPK0osMEJBQVAsQ0FBa0NGLFFBQWxDLEVBQTRDLEVBQUNHLFVBQVVoQixVQUFVaUIsVUFBVixFQUFYLEVBQTVDO0FBQ0Q7QUFDRixhQUxELE1BS087QUFDTGxCLHdCQUFVQyxTQUFWO0FBQ0Q7QUFDRjtBQUNGOztBQUVELFlBQUlELFlBQVksSUFBaEIsRUFBc0I7QUFDcEIsZ0JBQU1tQixvQkFBb0JsSyxPQUFPaUosYUFBUCxHQUN2QnJLLE1BRHVCLENBQ2hCdUwsUUFBUUEsU0FBU3BCLE9BREQsRUFFdkJoSyxHQUZ1QixDQUVuQm9MLFFBQVFBLEtBQUtmLGNBQUwsRUFGVyxDQUExQjtBQUdBLGNBQUljLGtCQUFrQmxOLE1BQWxCLEdBQTJCLENBQS9CLEVBQWtDO0FBQ2hDZ0QsbUJBQU9xQyx1QkFBUCxDQUErQjZILGlCQUEvQjtBQUNEO0FBQ0Y7O0FBRUQsWUFBSSxDQUFDcEIsVUFBTCxFQUFpQjtBQUNmO0FBQ0E5SSxpQkFBTytKLDBCQUFQLENBQWtDL0MsS0FBbEM7QUFDRDs7QUFFRCxlQUFPLElBQVA7QUFDRCxPQTdERDtBQThERCxLQS9ERCxNQStETyxJQUFJeUIsUUFBUTNJLEdBQVIsSUFBZThILE1BQU13QyxRQUF6QixFQUFtQztBQUN4QztBQUNBLFdBQUsxSyxTQUFMLENBQWVYLEdBQWYsQ0FBbUJpQixVQUFVO0FBQzNCLGNBQU1xSyxnQkFBZ0JySyxPQUFPc0ssZ0JBQVAsRUFBdEI7QUFDQSxjQUFNQyxxQkFBcUJGLGNBQWNqQixjQUFkLEVBQTNCOztBQUVBO0FBQ0EsY0FBTW9CLFdBQVd4RCxNQUFNVixLQUFOLENBQVltRSxVQUFaLENBQXVCRixtQkFBbUJqRSxLQUExQyxDQUFqQjtBQUNBLGNBQU1vRSxVQUFVRixXQUFXeEQsTUFBTVYsS0FBakIsR0FBeUJVLE1BQU0yQyxHQUEvQztBQUNBLGNBQU1FLFdBQVdXLFdBQVcsQ0FBQ0UsT0FBRCxFQUFVSCxtQkFBbUJaLEdBQTdCLENBQVgsR0FBK0MsQ0FBQ1ksbUJBQW1CakUsS0FBcEIsRUFBMkJvRSxPQUEzQixDQUFoRTs7QUFFQUwsc0JBQWNULGNBQWQsQ0FBNkJDLFFBQTdCLEVBQXVDLEVBQUNHLFVBQVVRLFFBQVgsRUFBdkM7QUFDQSxlQUFPLElBQVA7QUFDRCxPQVhEO0FBWUQsS0FkTSxNQWNBO0FBQ0wsV0FBSzlLLFNBQUwsQ0FBZVgsR0FBZixDQUFtQmlCLFVBQVVBLE9BQU9hLHNCQUFQLENBQThCbUcsS0FBOUIsQ0FBN0I7QUFDRDs7QUFFRCxXQUFPLElBQVA7QUFDRDs7QUFFRHZELGVBQWE7QUFDWCxXQUFPLEtBQUtwSCxLQUFMLENBQVcrSyxVQUFYLENBQXNCLEtBQUsvSyxLQUFMLENBQVdrQyxZQUFqQyxFQUErQyxLQUFLbEMsS0FBTCxDQUFXbUMsYUFBMUQsQ0FBUDtBQUNEOztBQUVENEUsMkJBQXlCO0FBQ3ZCLFVBQU15QyxnQkFBZ0IsS0FBSzhFLGdCQUFMLEVBQXRCO0FBQ0EsU0FBS0MsaUJBQUwsQ0FBdUI7QUFDckJuRyxZQUFNLE1BQU07QUFDVixjQUFNb0csYUFBYWhGLGNBQWM5RyxHQUFkLENBQWtCcUQsUUFBUUEsS0FBS3RCLFFBQUwsRUFBMUIsQ0FBbkI7QUFDQSxhQUFLeEIsaUJBQUwsR0FBeUIsTUFBekI7QUFDQSxhQUFLSSxTQUFMLENBQWVYLEdBQWYsQ0FBbUJpQixVQUFVQSxPQUFPcUMsdUJBQVAsQ0FBK0J3SSxVQUEvQixDQUE3QjtBQUNELE9BTG9CO0FBTXJCekksWUFBTSxNQUFNO0FBQ1YsWUFBSTBJLGlCQUFpQnhHLFFBQXJCO0FBQ0EsYUFBSyxNQUFNbEMsSUFBWCxJQUFtQnlELGFBQW5CLEVBQWtDO0FBQ2hDLGdCQUFNLENBQUNrRixXQUFELElBQWdCM0ksS0FBS2tGLFVBQUwsRUFBdEI7QUFDQSxjQUFJeUQsZ0JBQWdCLENBQUNELGNBQUQsSUFBbUJDLFlBQVlDLGlCQUFaLEtBQWtDRixjQUFyRSxDQUFKLEVBQTBGO0FBQ3hGQSw2QkFBaUJDLFlBQVlDLGlCQUFaLEVBQWpCO0FBQ0Q7QUFDRjs7QUFFRCxhQUFLMUwsaUJBQUwsR0FBeUIsTUFBekI7QUFDQSxhQUFLSSxTQUFMLENBQWVYLEdBQWYsQ0FBbUJpQixVQUFVO0FBQzNCQSxpQkFBT3FDLHVCQUFQLENBQStCLENBQUMsQ0FBQyxDQUFDeUksY0FBRCxFQUFpQixDQUFqQixDQUFELEVBQXNCLENBQUNBLGNBQUQsRUFBaUJ4RyxRQUFqQixDQUF0QixDQUFELENBQS9CO0FBQ0EsaUJBQU8sSUFBUDtBQUNELFNBSEQ7QUFJRDtBQXBCb0IsS0FBdkI7QUFzQkQ7O0FBa0JEcEIsbUJBQWlCO0FBQ2YsU0FBS3hELFNBQUwsQ0FBZVgsR0FBZixDQUFtQmlCLFVBQVU7QUFDM0IsWUFBTTBCLFlBQVksSUFBSUMsR0FBSixDQUNoQixLQUFLc0osaUJBQUwsQ0FBdUI3SSxRQUFRLEtBQUs4SSxZQUFMLENBQWtCOUksSUFBbEIsS0FBMkJBLElBQTFELENBRGdCLENBQWxCO0FBR0EsWUFBTUYsYUFBYXBFLE1BQU1DLElBQU4sQ0FBVzJELFNBQVgsRUFBc0JVLFFBQVFBLEtBQUt0QixRQUFMLEVBQTlCLENBQW5CO0FBQ0EsV0FBS3hCLGlCQUFMLEdBQXlCLE1BQXpCO0FBQ0FVLGFBQU9xQyx1QkFBUCxDQUErQkgsVUFBL0I7QUFDQSxhQUFPLElBQVA7QUFDRCxLQVJEO0FBU0Q7O0FBRURpQix1QkFBcUI7QUFDbkIsU0FBS3pELFNBQUwsQ0FBZVgsR0FBZixDQUFtQmlCLFVBQVU7QUFDM0IsWUFBTTBCLFlBQVksSUFBSUMsR0FBSixDQUNoQixLQUFLc0osaUJBQUwsQ0FBdUI3SSxRQUFRLEtBQUsrSSxhQUFMLENBQW1CL0ksSUFBbkIsS0FBNEJBLElBQTNELENBRGdCLENBQWxCO0FBR0EsWUFBTUYsYUFBYXBFLE1BQU1DLElBQU4sQ0FBVzJELFNBQVgsRUFBc0JVLFFBQVFBLEtBQUt0QixRQUFMLEVBQTlCLENBQW5CO0FBQ0EsV0FBS3hCLGlCQUFMLEdBQXlCLE1BQXpCO0FBQ0FVLGFBQU9xQyx1QkFBUCxDQUErQkgsVUFBL0I7QUFDQSxhQUFPLElBQVA7QUFDRCxLQVJEO0FBU0Q7O0FBRUQ1RSxjQUFZLEVBQUNDLGlCQUFELEtBQXNCLEVBQWxDLEVBQXNDO0FBQ3BDLFVBQU02TixxQkFBcUIsSUFBSUMsR0FBSixFQUEzQjs7QUFFQSxTQUFLM0wsU0FBTCxDQUFlWCxHQUFmLENBQW1CaUIsVUFBVTtBQUMzQixZQUFNc0wsYUFBYSxJQUFJM0osR0FBSixFQUFuQjs7QUFFQSxXQUFLLE1BQU00SixNQUFYLElBQXFCdkwsT0FBT3dMLFVBQVAsRUFBckIsRUFBMEM7QUFDeEMsY0FBTUMsWUFBWUYsT0FBT0csaUJBQVAsR0FBMkIzSixHQUE3QztBQUNBLGNBQU1LLE9BQU8sS0FBSy9GLEtBQUwsQ0FBV3FFLGNBQVgsQ0FBMEJzQixTQUExQixDQUFvQ3lKLFNBQXBDLENBQWI7QUFDQSxjQUFNbFAsWUFBWSxLQUFLRixLQUFMLENBQVdxRSxjQUFYLENBQTBCaUwsY0FBMUIsQ0FBeUNGLFNBQXpDLENBQWxCO0FBQ0E7QUFDQSxZQUFJLENBQUNySixJQUFMLEVBQVc7QUFDVDtBQUNEOztBQUVELFlBQUl3SixTQUFTeEosS0FBS3lKLFdBQUwsQ0FBaUJKLFNBQWpCLENBQWI7QUFDQSxZQUFJSyxZQUFZUCxPQUFPRyxpQkFBUCxHQUEyQmxDLE1BQTNDO0FBQ0EsWUFBSW9DLFdBQVcsSUFBZixFQUFxQjtBQUNuQixjQUFJRyxhQUFhM0osS0FBSzRKLGNBQUwsRUFBakI7QUFDQSxlQUFLLE1BQU1DLE1BQVgsSUFBcUI3SixLQUFLOEosVUFBTCxFQUFyQixFQUF3QztBQUN0QyxnQkFBSSxDQUFDRCxPQUFPRSxpQkFBUCxDQUF5QlYsU0FBekIsQ0FBTCxFQUEwQztBQUN4Q1EscUJBQU9HLElBQVAsQ0FBWTtBQUNWQywyQkFBVyxNQUFNO0FBQ2ZOLGdDQUFjRSxPQUFPSyxjQUFQLEVBQWQ7QUFDRCxpQkFIUztBQUlWQywwQkFBVSxNQUFNO0FBQ2RSLGdDQUFjRSxPQUFPSyxjQUFQLEVBQWQ7QUFDRDtBQU5TLGVBQVo7QUFRRCxhQVRELE1BU087QUFDTDtBQUNEO0FBQ0Y7O0FBRUQsY0FBSSxDQUFDaEIsV0FBV3RGLEdBQVgsQ0FBZStGLFVBQWYsQ0FBTCxFQUFpQztBQUMvQkgscUJBQVNHLFVBQVQ7QUFDQUQsd0JBQVksQ0FBWjtBQUNBUix1QkFBV3hMLEdBQVgsQ0FBZWlNLFVBQWY7QUFDRDtBQUNGOztBQUVELFlBQUlILFdBQVcsSUFBZixFQUFxQjtBQUNuQjtBQUNBO0FBQ0FBLG9CQUFVLENBQVY7QUFDQSxnQkFBTVksVUFBVXBCLG1CQUFtQmhILEdBQW5CLENBQXVCN0gsU0FBdkIsQ0FBaEI7QUFDQSxjQUFJLENBQUNpUSxPQUFMLEVBQWM7QUFDWnBCLCtCQUFtQnFCLEdBQW5CLENBQXVCbFEsU0FBdkIsRUFBa0MsQ0FBQyxDQUFDcVAsTUFBRCxFQUFTRSxTQUFULENBQUQsQ0FBbEM7QUFDRCxXQUZELE1BRU87QUFDTFUsb0JBQVE5RSxJQUFSLENBQWEsQ0FBQ2tFLE1BQUQsRUFBU0UsU0FBVCxDQUFiO0FBQ0Q7QUFDRjtBQUNGOztBQUVELGFBQU8sSUFBUDtBQUNELEtBcEREOztBQXNEQSxVQUFNWSx5QkFBeUIsSUFBSS9LLEdBQUosQ0FBUXlKLG1CQUFtQnVCLElBQW5CLEVBQVIsQ0FBL0I7QUFDQSxRQUFJcFAscUJBQXFCLENBQUNtUCx1QkFBdUIxRyxHQUF2QixDQUEyQnpJLGlCQUEzQixDQUExQixFQUF5RTtBQUN2RSxZQUFNLENBQUNxRCxTQUFELElBQWNyRCxrQkFBa0JSLFFBQWxCLEVBQXBCO0FBQ0EsWUFBTTBPLFlBQVk3SyxZQUFZQSxVQUFVb0wsY0FBVixLQUE2QixDQUF6QyxHQUE2QyxDQUEvRDtBQUNBLGFBQU8sS0FBSzNQLEtBQUwsQ0FBV3VRLFFBQVgsQ0FBb0JyUCxpQkFBcEIsRUFBdUMsQ0FBQyxDQUFDa08sU0FBRCxFQUFZLENBQVosQ0FBRCxDQUF2QyxFQUF5RCxJQUF6RCxDQUFQO0FBQ0QsS0FKRCxNQUlPO0FBQ0wsWUFBTW9CLFVBQVV6QixtQkFBbUJqSixJQUFuQixLQUE0QixDQUE1QztBQUNBLGFBQU96RCxRQUFRQyxHQUFSLENBQVliLE1BQU1DLElBQU4sQ0FBV3FOLGtCQUFYLEVBQStCMEIsU0FBUztBQUN6RCxjQUFNLENBQUN2USxTQUFELEVBQVlpUSxPQUFaLElBQXVCTSxLQUE3QjtBQUNBLGVBQU8sS0FBS3pRLEtBQUwsQ0FBV3VRLFFBQVgsQ0FBb0JyUSxTQUFwQixFQUErQmlRLE9BQS9CLEVBQXdDSyxPQUF4QyxDQUFQO0FBQ0QsT0FIa0IsQ0FBWixDQUFQO0FBSUQ7QUFFRjs7QUFFREUsb0JBQWtCO0FBQ2hCLFdBQU8sS0FBS3JOLFNBQUwsQ0FBZVgsR0FBZixDQUFtQmlCLFVBQVU7QUFDbEMsYUFBTyxJQUFJMkIsR0FBSixDQUNMM0IsT0FBT2lKLGFBQVAsR0FDR2xLLEdBREgsQ0FDT2lLLGFBQWFBLFVBQVVJLGNBQVYsRUFEcEIsRUFFRzdCLE1BRkgsQ0FFVSxDQUFDeUYsR0FBRCxFQUFNaEcsS0FBTixLQUFnQjtBQUN0QixhQUFLLE1BQU1qRixHQUFYLElBQWtCaUYsTUFBTWxGLE9BQU4sRUFBbEIsRUFBbUM7QUFDakMsY0FBSSxLQUFLbUwsV0FBTCxDQUFpQmxMLEdBQWpCLENBQUosRUFBMkI7QUFDekJpTCxnQkFBSXRGLElBQUosQ0FBUzNGLEdBQVQ7QUFDRDtBQUNGO0FBQ0QsZUFBT2lMLEdBQVA7QUFDRCxPQVRILEVBU0ssRUFUTCxDQURLLENBQVA7QUFZRCxLQWJNLEVBYUpwRSxLQWJJLENBYUUsSUFBSWpILEdBQUosRUFiRixDQUFQO0FBY0Q7O0FBRURrQyxvQkFBa0I7QUFDaEIsU0FBS3FKLHFCQUFMO0FBQ0Q7O0FBRURwSiwwQkFBd0I4RCxLQUF4QixFQUErQjtBQUM3QixRQUNFLENBQUNBLEtBQUQsSUFDQUEsTUFBTXVGLGNBQU4sQ0FBcUI3RyxLQUFyQixDQUEyQnZFLEdBQTNCLEtBQW1DNkYsTUFBTXdGLGNBQU4sQ0FBcUI5RyxLQUFyQixDQUEyQnZFLEdBRDlELElBRUE2RixNQUFNdUYsY0FBTixDQUFxQnhELEdBQXJCLENBQXlCNUgsR0FBekIsS0FBaUM2RixNQUFNd0YsY0FBTixDQUFxQnpELEdBQXJCLENBQXlCNUgsR0FINUQsRUFJRTtBQUNBLFdBQUttTCxxQkFBTDtBQUNEO0FBQ0Y7O0FBRURuSix3QkFBc0I7QUFDcEIsU0FBS21KLHFCQUFMO0FBQ0Q7O0FBRURBLDBCQUF3QjtBQUN0QixRQUFJLEtBQUszTixlQUFULEVBQTBCO0FBQ3hCO0FBQ0Q7O0FBRUQsVUFBTThOLGlCQUFpQixLQUFLM04sU0FBTCxDQUFlWCxHQUFmLENBQW1CaUIsVUFBVTtBQUNsRCxhQUFPQSxPQUFPc04sd0JBQVAsR0FBa0N2TyxHQUFsQyxDQUFzQ3dPLFlBQVlBLFNBQVN4TCxHQUEzRCxDQUFQO0FBQ0QsS0FGc0IsRUFFcEI2RyxLQUZvQixDQUVkLEVBRmMsQ0FBdkI7QUFHQSxVQUFNMUwsNEJBQTRCLEtBQUtiLEtBQUwsQ0FBV3FFLGNBQVgsQ0FBMEI4TSxrQkFBMUIsQ0FBNkNILGNBQTdDLENBQWxDOztBQUVBLFNBQUtoUixLQUFMLENBQVdvUixtQkFBWCxDQUNFLEtBQUtWLGVBQUwsRUFERixFQUVFLEtBQUt6TixpQkFBTCxJQUEwQixNQUY1QixFQUdFcEMseUJBSEY7QUFLRDs7QUFFRDhHLHFCQUFtQixFQUFDOEQsU0FBRCxFQUFZNEYsV0FBWixFQUFuQixFQUE2QztBQUMzQyxVQUFNdEwsT0FBTyxLQUFLL0YsS0FBTCxDQUFXcUUsY0FBWCxDQUEwQnNCLFNBQTFCLENBQW9DOEYsU0FBcEMsQ0FBYjtBQUNBLFFBQUkxRixTQUFTMkYsU0FBYixFQUF3QjtBQUN0QixhQUFPLEtBQUs0RixHQUFMLENBQVMsRUFBVCxDQUFQO0FBQ0Q7O0FBRUQsVUFBTUMsU0FBU3hMLEtBQUt5TCxXQUFMLENBQWlCL0YsU0FBakIsQ0FBZjtBQUNBLFFBQUk0RixXQUFKLEVBQWlCO0FBQ2YsYUFBTyxLQUFLQyxHQUFMLENBQVNDLFdBQVcsSUFBWCxHQUFrQixFQUFsQixHQUF1QixHQUFoQyxDQUFQO0FBQ0Q7O0FBRUQsV0FBTyxLQUFLRCxHQUFMLENBQVNDLE1BQVQsQ0FBUDtBQUNEOztBQUVEekoscUJBQW1CLEVBQUMyRCxTQUFELEVBQVk0RixXQUFaLEVBQW5CLEVBQTZDO0FBQzNDLFVBQU10TCxPQUFPLEtBQUsvRixLQUFMLENBQVdxRSxjQUFYLENBQTBCc0IsU0FBMUIsQ0FBb0M4RixTQUFwQyxDQUFiO0FBQ0EsUUFBSTFGLFNBQVMyRixTQUFiLEVBQXdCO0FBQ3RCLGFBQU8sS0FBSzRGLEdBQUwsQ0FBUyxFQUFULENBQVA7QUFDRDs7QUFFRCxVQUFNL0IsU0FBU3hKLEtBQUt5SixXQUFMLENBQWlCL0QsU0FBakIsQ0FBZjtBQUNBLFFBQUk0RixXQUFKLEVBQWlCO0FBQ2YsYUFBTyxLQUFLQyxHQUFMLENBQVMvQixXQUFXLElBQVgsR0FBa0IsRUFBbEIsR0FBdUIsR0FBaEMsQ0FBUDtBQUNEO0FBQ0QsV0FBTyxLQUFLK0IsR0FBTCxDQUFTL0IsTUFBVCxDQUFQO0FBQ0Q7O0FBRUQ7Ozs7QUFJQWpCLHFCQUFtQjtBQUNqQixXQUFPLEtBQUtNLGlCQUFMLENBQXVCZCxRQUFRQSxJQUEvQixDQUFQO0FBQ0Q7O0FBRURjLG9CQUFrQjZDLFFBQWxCLEVBQTRCO0FBQzFCLFdBQU8sS0FBS3BPLFNBQUwsQ0FBZVgsR0FBZixDQUFtQmlCLFVBQVU7QUFDbEMsWUFBTStOLE9BQU8sSUFBSXBNLEdBQUosRUFBYjtBQUNBLGFBQU8zQixPQUFPZ08sdUJBQVAsR0FBaUN6RyxNQUFqQyxDQUF3QyxDQUFDeUYsR0FBRCxFQUFNaEcsS0FBTixLQUFnQjtBQUM3RCxhQUFLLE1BQU1qRixHQUFYLElBQWtCaUYsTUFBTWxGLE9BQU4sRUFBbEIsRUFBbUM7QUFDakMsZ0JBQU1NLE9BQU8sS0FBSy9GLEtBQUwsQ0FBV3FFLGNBQVgsQ0FBMEJzQixTQUExQixDQUFvQ0QsR0FBcEMsQ0FBYjtBQUNBLGNBQUksQ0FBQ0ssSUFBRCxJQUFTMkwsS0FBSy9ILEdBQUwsQ0FBUzVELElBQVQsQ0FBYixFQUE2QjtBQUMzQjtBQUNEOztBQUVEMkwsZUFBS2pPLEdBQUwsQ0FBU3NDLElBQVQ7QUFDQTRLLGNBQUl0RixJQUFKLENBQVNvRyxTQUFTMUwsSUFBVCxDQUFUO0FBQ0Q7QUFDRCxlQUFPNEssR0FBUDtBQUNELE9BWE0sRUFXSixFQVhJLENBQVA7QUFZRCxLQWRNLEVBY0pwRSxLQWRJLENBY0UsRUFkRixDQUFQO0FBZUQ7O0FBRUQ7Ozs7QUFJQTVLLDJCQUF5QjtBQUN2QixXQUFPLEtBQUswQixTQUFMLENBQWVYLEdBQWYsQ0FBbUJpQixVQUFVO0FBQ2xDLFlBQU1pTyxVQUFVLElBQUl0TSxHQUFKLEVBQWhCO0FBQ0EsV0FBSyxNQUFNcUYsS0FBWCxJQUFvQmhILE9BQU9nTyx1QkFBUCxFQUFwQixFQUFzRDtBQUNwRCxhQUFLLE1BQU1qTSxHQUFYLElBQWtCaUYsTUFBTWxGLE9BQU4sRUFBbEIsRUFBbUM7QUFDakMsZ0JBQU1vTSxRQUFRLEtBQUs3UixLQUFMLENBQVdxRSxjQUFYLENBQTBCaUwsY0FBMUIsQ0FBeUM1SixHQUF6QyxDQUFkO0FBQ0FrTSxrQkFBUW5PLEdBQVIsQ0FBWW9PLEtBQVo7QUFDRDtBQUNGO0FBQ0QsYUFBT0QsT0FBUDtBQUNELEtBVE0sRUFTSnJGLEtBVEksQ0FTRSxJQUFJakgsR0FBSixFQVRGLENBQVA7QUFVRDs7QUFFRHdKLGdCQUFjL0ksSUFBZCxFQUFvQjtBQUNsQixVQUFNK0wsVUFBVS9MLEtBQUt0QixRQUFMLEdBQWdCd0YsS0FBaEIsQ0FBc0J2RSxHQUF0QixHQUE0QixDQUE1QztBQUNBLFdBQU8sS0FBSzFGLEtBQUwsQ0FBV3FFLGNBQVgsQ0FBMEJzQixTQUExQixDQUFvQ21NLE9BQXBDLENBQVA7QUFDRDs7QUFFRGpELGVBQWE5SSxJQUFiLEVBQW1CO0FBQ2pCLFVBQU1nTSxVQUFVaE0sS0FBS3RCLFFBQUwsR0FBZ0I2SSxHQUFoQixDQUFvQjVILEdBQXBCLEdBQTBCLENBQTFDO0FBQ0EsV0FBTyxLQUFLMUYsS0FBTCxDQUFXcUUsY0FBWCxDQUEwQnNCLFNBQTFCLENBQW9Db00sT0FBcEMsQ0FBUDtBQUNEOztBQUVEbkIsY0FBWW5GLFNBQVosRUFBdUI7QUFDckIsVUFBTXVHLGVBQWUsQ0FBQyxLQUFLaFMsS0FBTCxDQUFXcUUsY0FBWCxDQUEwQmlFLGdCQUExQixFQUFELEVBQStDLEtBQUt0SSxLQUFMLENBQVdxRSxjQUFYLENBQTBCa0UsZ0JBQTFCLEVBQS9DLENBQXJCO0FBQ0EsV0FBT3lKLGFBQWFDLElBQWIsQ0FBa0JwSCxTQUFTQSxNQUFNcUgsV0FBTixDQUFrQixFQUFDQyxlQUFlMUcsU0FBaEIsRUFBbEIsRUFBOEM5SyxNQUE5QyxHQUF1RCxDQUFsRixDQUFQO0FBQ0Q7O0FBRUQ0TixvQkFBa0I2RCxTQUFsQixFQUE2QjtBQUMzQixVQUFNWCxXQUFXVyxVQUFVLEtBQUtwUyxLQUFMLENBQVdtQyxhQUFyQixDQUFqQjtBQUNBO0FBQ0EsUUFBSSxDQUFDc1AsUUFBTCxFQUFlO0FBQ2IsWUFBTSxJQUFJWSxLQUFKLENBQVcsMkJBQTBCLEtBQUtyUyxLQUFMLENBQVdtQyxhQUFjLEVBQTlELENBQU47QUFDRDtBQUNELFdBQU9zUCxVQUFQO0FBQ0Q7O0FBRURILE1BQUlnQixHQUFKLEVBQVM7QUFDUCxVQUFNQyxZQUFZLEtBQUt2UyxLQUFMLENBQVdxRSxjQUFYLENBQTBCbU8scUJBQTFCLEVBQWxCO0FBQ0EsUUFBSUYsUUFBUSxJQUFaLEVBQWtCO0FBQ2hCLGFBQU81UyxlQUFlK1MsTUFBZixDQUFzQkYsU0FBdEIsQ0FBUDtBQUNELEtBRkQsTUFFTztBQUNMLGFBQU83UyxlQUFlK1MsTUFBZixDQUFzQkYsWUFBWUQsSUFBSUksUUFBSixHQUFlL1IsTUFBakQsSUFBMkQyUixJQUFJSSxRQUFKLEVBQWxFO0FBQ0Q7QUFDRjs7QUFqaUM2RDtrQkFBM0M5UyxrQjtBQUFBQSxrQixDQUNaK1MsUyxHQUFZO0FBQ2pCblMsaUJBQWVvUyxvQkFBVUMsS0FBVixDQUFnQixDQUFDLFFBQUQsRUFBVyxVQUFYLENBQWhCLENBREU7QUFFakJwUyxxQkFBbUJtUyxvQkFBVUUsSUFGWjtBQUdqQnpPLGtCQUFnQjBPLG1DQUF1QkMsVUFIdEI7QUFJakI3USxpQkFBZXlRLG9CQUFVQyxLQUFWLENBQWdCLENBQUMsTUFBRCxFQUFTLE1BQVQsQ0FBaEIsRUFBa0NHLFVBSmhDO0FBS2pCOVEsZ0JBQWMwUSxvQkFBVUssTUFBVixDQUFpQkQsVUFMZDtBQU1qQm5TLDZCQUEyQitSLG9CQUFVRSxJQUFWLENBQWVFLFVBTnpCO0FBT2pCRSxjQUFZTixvQkFBVUssTUFBVixDQUFpQkQsVUFQWjtBQVFqQnBTLGtCQUFnQmdTLG9CQUFVRSxJQVJUOztBQVVqQnhMLGFBQVdzTCxvQkFBVUssTUFBVixDQUFpQkQsVUFWWDtBQVdqQnBNLFlBQVVnTSxvQkFBVUssTUFBVixDQUFpQkQsVUFYVjtBQVlqQjdJLFdBQVN5SSxvQkFBVUssTUFBVixDQUFpQkQsVUFaVDtBQWFqQmxTLFlBQVU4UixvQkFBVUssTUFBVixDQUFpQkQsVUFiVjtBQWNqQnRPLFVBQVFrTyxvQkFBVUssTUFBVixDQUFpQkQsVUFkUjs7QUFnQmpCNUIsdUJBQXFCd0Isb0JBQVVPLElBaEJkOztBQWtCakJuUyx1QkFBcUI0UixvQkFBVU8sSUFsQmQ7QUFtQmpCOUwsV0FBU3VMLG9CQUFVTyxJQW5CRjtBQW9CakI1QyxZQUFVcUMsb0JBQVVPLElBcEJIO0FBcUJqQmhTLGNBQVl5UixvQkFBVU8sSUFyQkw7QUFzQmpCcEksY0FBWTZILG9CQUFVTyxJQXRCTDtBQXVCakJ4USxvQkFBa0JpUSxvQkFBVU8sSUF2Qlg7QUF3QmpCclEsdUJBQXFCOFAsb0JBQVVPLElBeEJkO0FBeUJqQnRSLG1CQUFpQitRLG9CQUFVTyxJQXpCVjtBQTBCakJsUixlQUFhMlEsb0JBQVVPLElBMUJOO0FBMkJqQnBQLG1CQUFpQnFQLDZCQTNCQTtBQTRCakIvUyxZQUFVZ1QsNkJBQWlCTDtBQTVCVixDIiwiZmlsZSI6Im11bHRpLWZpbGUtcGF0Y2gtdmlldy5qcyIsInNvdXJjZVJvb3QiOiIvYnVpbGQvYXRvbS9zcmMvYXRvbS0xLjM1LjEvb3V0L2FwcC9ub2RlX21vZHVsZXMvZ2l0aHViL2xpYi92aWV3cyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCBSZWFjdCwge0ZyYWdtZW50fSBmcm9tICdyZWFjdCc7XG5pbXBvcnQgUHJvcFR5cGVzIGZyb20gJ3Byb3AtdHlwZXMnO1xuaW1wb3J0IGN4IGZyb20gJ2NsYXNzbmFtZXMnO1xuaW1wb3J0IHtSYW5nZX0gZnJvbSAnYXRvbSc7XG5pbXBvcnQge0NvbXBvc2l0ZURpc3Bvc2FibGV9IGZyb20gJ2V2ZW50LWtpdCc7XG5cbmltcG9ydCB7YXV0b2JpbmR9IGZyb20gJy4uL2hlbHBlcnMnO1xuaW1wb3J0IHtSZWZIb2xkZXJQcm9wVHlwZSwgTXVsdGlGaWxlUGF0Y2hQcm9wVHlwZSwgSXRlbVR5cGVQcm9wVHlwZX0gZnJvbSAnLi4vcHJvcC10eXBlcyc7XG5pbXBvcnQgQXRvbVRleHRFZGl0b3IgZnJvbSAnLi4vYXRvbS9hdG9tLXRleHQtZWRpdG9yJztcbmltcG9ydCBNYXJrZXIgZnJvbSAnLi4vYXRvbS9tYXJrZXInO1xuaW1wb3J0IE1hcmtlckxheWVyIGZyb20gJy4uL2F0b20vbWFya2VyLWxheWVyJztcbmltcG9ydCBEZWNvcmF0aW9uIGZyb20gJy4uL2F0b20vZGVjb3JhdGlvbic7XG5pbXBvcnQgR3V0dGVyIGZyb20gJy4uL2F0b20vZ3V0dGVyJztcbmltcG9ydCBDb21tYW5kcywge0NvbW1hbmR9IGZyb20gJy4uL2F0b20vY29tbWFuZHMnO1xuaW1wb3J0IEZpbGVQYXRjaEhlYWRlclZpZXcgZnJvbSAnLi9maWxlLXBhdGNoLWhlYWRlci12aWV3JztcbmltcG9ydCBGaWxlUGF0Y2hNZXRhVmlldyBmcm9tICcuL2ZpbGUtcGF0Y2gtbWV0YS12aWV3JztcbmltcG9ydCBIdW5rSGVhZGVyVmlldyBmcm9tICcuL2h1bmstaGVhZGVyLXZpZXcnO1xuaW1wb3J0IFJlZkhvbGRlciBmcm9tICcuLi9tb2RlbHMvcmVmLWhvbGRlcic7XG5pbXBvcnQgQ2hhbmdlZEZpbGVJdGVtIGZyb20gJy4uL2l0ZW1zL2NoYW5nZWQtZmlsZS1pdGVtJztcbmltcG9ydCBDb21taXREZXRhaWxJdGVtIGZyb20gJy4uL2l0ZW1zL2NvbW1pdC1kZXRhaWwtaXRlbSc7XG5pbXBvcnQgSXNzdWVpc2hEZXRhaWxJdGVtIGZyb20gJy4uL2l0ZW1zL2lzc3VlaXNoLWRldGFpbC1pdGVtJztcbmltcG9ydCBGaWxlIGZyb20gJy4uL21vZGVscy9wYXRjaC9maWxlJztcblxuY29uc3QgZXhlY3V0YWJsZVRleHQgPSB7XG4gIFtGaWxlLm1vZGVzLk5PUk1BTF06ICdub24gZXhlY3V0YWJsZScsXG4gIFtGaWxlLm1vZGVzLkVYRUNVVEFCTEVdOiAnZXhlY3V0YWJsZScsXG59O1xuXG5jb25zdCBOQlNQX0NIQVJBQ1RFUiA9ICdcXHUwMGEwJztcblxuY29uc3QgQkxBTktfTEFCRUwgPSAoKSA9PiBOQlNQX0NIQVJBQ1RFUjtcblxuZXhwb3J0IGRlZmF1bHQgY2xhc3MgTXVsdGlGaWxlUGF0Y2hWaWV3IGV4dGVuZHMgUmVhY3QuQ29tcG9uZW50IHtcbiAgc3RhdGljIHByb3BUeXBlcyA9IHtcbiAgICBzdGFnaW5nU3RhdHVzOiBQcm9wVHlwZXMub25lT2YoWydzdGFnZWQnLCAndW5zdGFnZWQnXSksXG4gICAgaXNQYXJ0aWFsbHlTdGFnZWQ6IFByb3BUeXBlcy5ib29sLFxuICAgIG11bHRpRmlsZVBhdGNoOiBNdWx0aUZpbGVQYXRjaFByb3BUeXBlLmlzUmVxdWlyZWQsXG4gICAgc2VsZWN0aW9uTW9kZTogUHJvcFR5cGVzLm9uZU9mKFsnaHVuaycsICdsaW5lJ10pLmlzUmVxdWlyZWQsXG4gICAgc2VsZWN0ZWRSb3dzOiBQcm9wVHlwZXMub2JqZWN0LmlzUmVxdWlyZWQsXG4gICAgaGFzTXVsdGlwbGVGaWxlU2VsZWN0aW9uczogUHJvcFR5cGVzLmJvb2wuaXNSZXF1aXJlZCxcbiAgICByZXBvc2l0b3J5OiBQcm9wVHlwZXMub2JqZWN0LmlzUmVxdWlyZWQsXG4gICAgaGFzVW5kb0hpc3Rvcnk6IFByb3BUeXBlcy5ib29sLFxuXG4gICAgd29ya3NwYWNlOiBQcm9wVHlwZXMub2JqZWN0LmlzUmVxdWlyZWQsXG4gICAgY29tbWFuZHM6IFByb3BUeXBlcy5vYmplY3QuaXNSZXF1aXJlZCxcbiAgICBrZXltYXBzOiBQcm9wVHlwZXMub2JqZWN0LmlzUmVxdWlyZWQsXG4gICAgdG9vbHRpcHM6IFByb3BUeXBlcy5vYmplY3QuaXNSZXF1aXJlZCxcbiAgICBjb25maWc6IFByb3BUeXBlcy5vYmplY3QuaXNSZXF1aXJlZCxcblxuICAgIHNlbGVjdGVkUm93c0NoYW5nZWQ6IFByb3BUeXBlcy5mdW5jLFxuXG4gICAgZGl2ZUludG9NaXJyb3JQYXRjaDogUHJvcFR5cGVzLmZ1bmMsXG4gICAgc3VyZmFjZTogUHJvcFR5cGVzLmZ1bmMsXG4gICAgb3BlbkZpbGU6IFByb3BUeXBlcy5mdW5jLFxuICAgIHRvZ2dsZUZpbGU6IFByb3BUeXBlcy5mdW5jLFxuICAgIHRvZ2dsZVJvd3M6IFByb3BUeXBlcy5mdW5jLFxuICAgIHRvZ2dsZU1vZGVDaGFuZ2U6IFByb3BUeXBlcy5mdW5jLFxuICAgIHRvZ2dsZVN5bWxpbmtDaGFuZ2U6IFByb3BUeXBlcy5mdW5jLFxuICAgIHVuZG9MYXN0RGlzY2FyZDogUHJvcFR5cGVzLmZ1bmMsXG4gICAgZGlzY2FyZFJvd3M6IFByb3BUeXBlcy5mdW5jLFxuICAgIHJlZkluaXRpYWxGb2N1czogUmVmSG9sZGVyUHJvcFR5cGUsXG4gICAgaXRlbVR5cGU6IEl0ZW1UeXBlUHJvcFR5cGUuaXNSZXF1aXJlZCxcbiAgfVxuXG4gIGNvbnN0cnVjdG9yKHByb3BzKSB7XG4gICAgc3VwZXIocHJvcHMpO1xuICAgIGF1dG9iaW5kKFxuICAgICAgdGhpcyxcbiAgICAgICdkaWRNb3VzZURvd25PbkhlYWRlcicsICdkaWRNb3VzZURvd25PbkxpbmVOdW1iZXInLCAnZGlkTW91c2VNb3ZlT25MaW5lTnVtYmVyJywgJ2RpZE1vdXNlVXAnLFxuICAgICAgJ2RpZENvbmZpcm0nLCAnZGlkVG9nZ2xlU2VsZWN0aW9uTW9kZScsICdzZWxlY3ROZXh0SHVuaycsICdzZWxlY3RQcmV2aW91c0h1bmsnLFxuICAgICAgJ2RpZE9wZW5GaWxlJywgJ2RpZEFkZFNlbGVjdGlvbicsICdkaWRDaGFuZ2VTZWxlY3Rpb25SYW5nZScsICdkaWREZXN0cm95U2VsZWN0aW9uJyxcbiAgICAgICdvbGRMaW5lTnVtYmVyTGFiZWwnLCAnbmV3TGluZU51bWJlckxhYmVsJyxcbiAgICApO1xuXG4gICAgdGhpcy5tb3VzZVNlbGVjdGlvbkluUHJvZ3Jlc3MgPSBmYWxzZTtcbiAgICB0aGlzLmxhc3RNb3VzZU1vdmVMaW5lID0gbnVsbDtcbiAgICB0aGlzLm5leHRTZWxlY3Rpb25Nb2RlID0gbnVsbDtcbiAgICB0aGlzLnN1cHByZXNzQ2hhbmdlcyA9IGZhbHNlO1xuICAgIHRoaXMucmVmUm9vdCA9IG5ldyBSZWZIb2xkZXIoKTtcbiAgICB0aGlzLnJlZkVkaXRvciA9IG5ldyBSZWZIb2xkZXIoKTtcbiAgICB0aGlzLnJlZkVkaXRvckVsZW1lbnQgPSBuZXcgUmVmSG9sZGVyKCk7XG5cbiAgICB0aGlzLnN1YnMgPSBuZXcgQ29tcG9zaXRlRGlzcG9zYWJsZSgpO1xuXG4gICAgdGhpcy5zdWJzLmFkZChcbiAgICAgIHRoaXMucmVmRWRpdG9yLm9ic2VydmUoZWRpdG9yID0+IHtcbiAgICAgICAgdGhpcy5yZWZFZGl0b3JFbGVtZW50LnNldHRlcihlZGl0b3IuZ2V0RWxlbWVudCgpKTtcbiAgICAgIH0pLFxuICAgICAgdGhpcy5yZWZFZGl0b3JFbGVtZW50Lm9ic2VydmUoZWxlbWVudCA9PiB7XG4gICAgICAgIHRoaXMucHJvcHMucmVmSW5pdGlhbEZvY3VzICYmIHRoaXMucHJvcHMucmVmSW5pdGlhbEZvY3VzLnNldHRlcihlbGVtZW50KTtcbiAgICAgIH0pLFxuICAgICk7XG4gIH1cblxuICBjb21wb25lbnREaWRNb3VudCgpIHtcbiAgICB3aW5kb3cuYWRkRXZlbnRMaXN0ZW5lcignbW91c2V1cCcsIHRoaXMuZGlkTW91c2VVcCk7XG4gICAgdGhpcy5yZWZFZGl0b3IubWFwKGVkaXRvciA9PiB7XG4gICAgICAvLyB0aGlzLnByb3BzLm11bHRpRmlsZVBhdGNoIGlzIGd1YXJhbnRlZWQgdG8gY29udGFpbiBhdCBsZWFzdCBvbmUgRmlsZVBhdGNoIGlmIDxBdG9tVGV4dEVkaXRvcj4gaXMgcmVuZGVyZWQuXG4gICAgICBjb25zdCBbZmlyc3RQYXRjaF0gPSB0aGlzLnByb3BzLm11bHRpRmlsZVBhdGNoLmdldEZpbGVQYXRjaGVzKCk7XG4gICAgICBjb25zdCBbZmlyc3RIdW5rXSA9IGZpcnN0UGF0Y2guZ2V0SHVua3MoKTtcbiAgICAgIGlmICghZmlyc3RIdW5rKSB7XG4gICAgICAgIHJldHVybiBudWxsO1xuICAgICAgfVxuXG4gICAgICB0aGlzLm5leHRTZWxlY3Rpb25Nb2RlID0gJ2h1bmsnO1xuICAgICAgZWRpdG9yLnNldFNlbGVjdGVkQnVmZmVyUmFuZ2UoZmlyc3RIdW5rLmdldFJhbmdlKCkpO1xuICAgICAgcmV0dXJuIG51bGw7XG4gICAgfSk7XG5cbiAgICB0aGlzLnN1YnMuYWRkKFxuICAgICAgdGhpcy5wcm9wcy5jb25maWcub25EaWRDaGFuZ2UoJ2dpdGh1Yi5zaG93RGlmZkljb25HdXR0ZXInLCAoe25ld1ZhbHVlfSkgPT4gdGhpcy5mb3JjZVVwZGF0ZSgpKSxcbiAgICApO1xuICB9XG5cbiAgZ2V0U25hcHNob3RCZWZvcmVVcGRhdGUocHJldlByb3BzKSB7XG4gICAgbGV0IG5ld1NlbGVjdGlvblJhbmdlID0gbnVsbDtcbiAgICBpZiAodGhpcy5wcm9wcy5tdWx0aUZpbGVQYXRjaCAhPT0gcHJldlByb3BzLm11bHRpRmlsZVBhdGNoKSB7XG4gICAgICAvLyBIZXVyaXN0aWNhbGx5IGFkanVzdCB0aGUgZWRpdG9yIHNlbGVjdGlvbiBiYXNlZCBvbiB0aGUgb2xkIGZpbGUgcGF0Y2gsIHRoZSBvbGQgcm93IHNlbGVjdGlvbiBzdGF0ZSwgYW5kXG4gICAgICAvLyB0aGUgaW5jb21pbmcgcGF0Y2guXG4gICAgICBuZXdTZWxlY3Rpb25SYW5nZSA9IHRoaXMucHJvcHMubXVsdGlGaWxlUGF0Y2guZ2V0TmV4dFNlbGVjdGlvblJhbmdlKFxuICAgICAgICBwcmV2UHJvcHMubXVsdGlGaWxlUGF0Y2gsXG4gICAgICAgIHByZXZQcm9wcy5zZWxlY3RlZFJvd3MsXG4gICAgICApO1xuXG4gICAgICB0aGlzLnN1cHByZXNzQ2hhbmdlcyA9IHRydWU7XG4gICAgICB0aGlzLnByb3BzLm11bHRpRmlsZVBhdGNoLmFkb3B0QnVmZmVyRnJvbShwcmV2UHJvcHMubXVsdGlGaWxlUGF0Y2gpO1xuICAgICAgdGhpcy5zdXBwcmVzc0NoYW5nZXMgPSBmYWxzZTtcbiAgICB9XG4gICAgcmV0dXJuIG5ld1NlbGVjdGlvblJhbmdlO1xuICB9XG5cbiAgY29tcG9uZW50RGlkVXBkYXRlKHByZXZQcm9wcywgcHJldlN0YXRlLCBuZXdTZWxlY3Rpb25SYW5nZSkge1xuICAgIGlmIChwcmV2UHJvcHMucmVmSW5pdGlhbEZvY3VzICE9PSB0aGlzLnByb3BzLnJlZkluaXRpYWxGb2N1cykge1xuICAgICAgcHJldlByb3BzLnJlZkluaXRpYWxGb2N1cyAmJiBwcmV2UHJvcHMucmVmSW5pdGlhbEZvY3VzLnNldHRlcihudWxsKTtcbiAgICAgIHRoaXMucHJvcHMucmVmSW5pdGlhbEZvY3VzICYmIHRoaXMucmVmRWRpdG9yRWxlbWVudC5tYXAodGhpcy5wcm9wcy5yZWZJbml0aWFsRm9jdXMuc2V0dGVyKTtcbiAgICB9XG5cbiAgICBpZiAobmV3U2VsZWN0aW9uUmFuZ2UpIHtcbiAgICAgIHRoaXMucmVmRWRpdG9yLm1hcChlZGl0b3IgPT4ge1xuICAgICAgICBpZiAodGhpcy5wcm9wcy5zZWxlY3Rpb25Nb2RlID09PSAnbGluZScpIHtcbiAgICAgICAgICB0aGlzLm5leHRTZWxlY3Rpb25Nb2RlID0gJ2xpbmUnO1xuICAgICAgICAgIGVkaXRvci5zZXRTZWxlY3RlZEJ1ZmZlclJhbmdlKG5ld1NlbGVjdGlvblJhbmdlKTtcbiAgICAgICAgfSBlbHNlIHtcbiAgICAgICAgICBjb25zdCBuZXh0SHVua3MgPSBuZXcgU2V0KFxuICAgICAgICAgICAgUmFuZ2UuZnJvbU9iamVjdChuZXdTZWxlY3Rpb25SYW5nZSkuZ2V0Um93cygpXG4gICAgICAgICAgICAgIC5tYXAocm93ID0+IHRoaXMucHJvcHMubXVsdGlGaWxlUGF0Y2guZ2V0SHVua0F0KHJvdykpXG4gICAgICAgICAgICAgIC5maWx0ZXIoQm9vbGVhbiksXG4gICAgICAgICAgKTtcbiAgICAgICAgICBjb25zdCBuZXh0UmFuZ2VzID0gbmV4dEh1bmtzLnNpemUgPiAwXG4gICAgICAgICAgICA/IEFycmF5LmZyb20obmV4dEh1bmtzLCBodW5rID0+IGh1bmsuZ2V0UmFuZ2UoKSlcbiAgICAgICAgICAgIDogW1tbMCwgMF0sIFswLCAwXV1dO1xuXG4gICAgICAgICAgdGhpcy5uZXh0U2VsZWN0aW9uTW9kZSA9ICdodW5rJztcbiAgICAgICAgICBlZGl0b3Iuc2V0U2VsZWN0ZWRCdWZmZXJSYW5nZXMobmV4dFJhbmdlcyk7XG4gICAgICAgIH1cblxuICAgICAgICByZXR1cm4gbnVsbDtcbiAgICAgIH0pO1xuICAgIH0gZWxzZSB7XG4gICAgICB0aGlzLm5leHRTZWxlY3Rpb25Nb2RlID0gbnVsbDtcbiAgICB9XG4gIH1cblxuICBjb21wb25lbnRXaWxsVW5tb3VudCgpIHtcbiAgICB3aW5kb3cucmVtb3ZlRXZlbnRMaXN0ZW5lcignbW91c2V1cCcsIHRoaXMuZGlkTW91c2VVcCk7XG4gICAgdGhpcy5zdWJzLmRpc3Bvc2UoKTtcbiAgfVxuXG4gIHJlbmRlcigpIHtcbiAgICBjb25zdCByb290Q2xhc3MgPSBjeChcbiAgICAgICdnaXRodWItRmlsZVBhdGNoVmlldycsXG4gICAgICB7W2BnaXRodWItRmlsZVBhdGNoVmlldy0tJHt0aGlzLnByb3BzLnN0YWdpbmdTdGF0dXN9YF06IHRoaXMucHJvcHMuc3RhZ2luZ1N0YXR1c30sXG4gICAgICB7J2dpdGh1Yi1GaWxlUGF0Y2hWaWV3LS1ibGFuayc6ICF0aGlzLnByb3BzLm11bHRpRmlsZVBhdGNoLmFueVByZXNlbnQoKX0sXG4gICAgICB7J2dpdGh1Yi1GaWxlUGF0Y2hWaWV3LS1odW5rTW9kZSc6IHRoaXMucHJvcHMuc2VsZWN0aW9uTW9kZSA9PT0gJ2h1bmsnfSxcbiAgICApO1xuXG4gICAgcmV0dXJuIChcbiAgICAgIDxkaXYgY2xhc3NOYW1lPXtyb290Q2xhc3N9IHJlZj17dGhpcy5yZWZSb290LnNldHRlcn0+XG4gICAgICAgIHt0aGlzLnJlbmRlckNvbW1hbmRzKCl9XG5cbiAgICAgICAgPG1haW4gY2xhc3NOYW1lPVwiZ2l0aHViLUZpbGVQYXRjaFZpZXctY29udGFpbmVyXCI+XG4gICAgICAgICAge3RoaXMucHJvcHMubXVsdGlGaWxlUGF0Y2guYW55UHJlc2VudCgpID8gdGhpcy5yZW5kZXJOb25FbXB0eVBhdGNoKCkgOiB0aGlzLnJlbmRlckVtcHR5UGF0Y2goKX1cbiAgICAgICAgPC9tYWluPlxuICAgICAgPC9kaXY+XG4gICAgKTtcbiAgfVxuXG4gIHJlbmRlckNvbW1hbmRzKCkge1xuICAgIGlmICh0aGlzLnByb3BzLml0ZW1UeXBlID09PSBDb21taXREZXRhaWxJdGVtIHx8IHRoaXMucHJvcHMuaXRlbVR5cGUgPT09IElzc3VlaXNoRGV0YWlsSXRlbSkge1xuICAgICAgcmV0dXJuIChcbiAgICAgICAgPENvbW1hbmRzIHJlZ2lzdHJ5PXt0aGlzLnByb3BzLmNvbW1hbmRzfSB0YXJnZXQ9e3RoaXMucmVmUm9vdH0+XG4gICAgICAgICAgPENvbW1hbmQgY29tbWFuZD1cImdpdGh1YjpzZWxlY3QtbmV4dC1odW5rXCIgY2FsbGJhY2s9e3RoaXMuc2VsZWN0TmV4dEh1bmt9IC8+XG4gICAgICAgICAgPENvbW1hbmQgY29tbWFuZD1cImdpdGh1YjpzZWxlY3QtcHJldmlvdXMtaHVua1wiIGNhbGxiYWNrPXt0aGlzLnNlbGVjdFByZXZpb3VzSHVua30gLz5cbiAgICAgICAgICA8Q29tbWFuZCBjb21tYW5kPVwiZ2l0aHViOnRvZ2dsZS1wYXRjaC1zZWxlY3Rpb24tbW9kZVwiIGNhbGxiYWNrPXt0aGlzLmRpZFRvZ2dsZVNlbGVjdGlvbk1vZGV9IC8+XG4gICAgICAgIDwvQ29tbWFuZHM+XG4gICAgICApO1xuICAgIH1cblxuICAgIGxldCBzdGFnZU1vZGVDb21tYW5kID0gbnVsbDtcbiAgICBsZXQgc3RhZ2VTeW1saW5rQ29tbWFuZCA9IG51bGw7XG5cbiAgICBpZiAodGhpcy5wcm9wcy5tdWx0aUZpbGVQYXRjaC5kaWRBbnlDaGFuZ2VFeGVjdXRhYmxlTW9kZSgpKSB7XG4gICAgICBjb25zdCBjb21tYW5kID0gdGhpcy5wcm9wcy5zdGFnaW5nU3RhdHVzID09PSAndW5zdGFnZWQnXG4gICAgICAgID8gJ2dpdGh1YjpzdGFnZS1maWxlLW1vZGUtY2hhbmdlJ1xuICAgICAgICA6ICdnaXRodWI6dW5zdGFnZS1maWxlLW1vZGUtY2hhbmdlJztcbiAgICAgIHN0YWdlTW9kZUNvbW1hbmQgPSA8Q29tbWFuZCBjb21tYW5kPXtjb21tYW5kfSBjYWxsYmFjaz17dGhpcy5kaWRUb2dnbGVNb2RlQ2hhbmdlfSAvPjtcbiAgICB9XG5cbiAgICBpZiAodGhpcy5wcm9wcy5tdWx0aUZpbGVQYXRjaC5hbnlIYXZlVHlwZWNoYW5nZSgpKSB7XG4gICAgICBjb25zdCBjb21tYW5kID0gdGhpcy5wcm9wcy5zdGFnaW5nU3RhdHVzID09PSAndW5zdGFnZWQnXG4gICAgICAgID8gJ2dpdGh1YjpzdGFnZS1zeW1saW5rLWNoYW5nZSdcbiAgICAgICAgOiAnZ2l0aHViOnVuc3RhZ2Utc3ltbGluay1jaGFuZ2UnO1xuICAgICAgc3RhZ2VTeW1saW5rQ29tbWFuZCA9IDxDb21tYW5kIGNvbW1hbmQ9e2NvbW1hbmR9IGNhbGxiYWNrPXt0aGlzLmRpZFRvZ2dsZVN5bWxpbmtDaGFuZ2V9IC8+O1xuICAgIH1cblxuICAgIHJldHVybiAoXG4gICAgICA8Q29tbWFuZHMgcmVnaXN0cnk9e3RoaXMucHJvcHMuY29tbWFuZHN9IHRhcmdldD17dGhpcy5yZWZSb290fT5cbiAgICAgICAgPENvbW1hbmQgY29tbWFuZD1cImdpdGh1YjpzZWxlY3QtbmV4dC1odW5rXCIgY2FsbGJhY2s9e3RoaXMuc2VsZWN0TmV4dEh1bmt9IC8+XG4gICAgICAgIDxDb21tYW5kIGNvbW1hbmQ9XCJnaXRodWI6c2VsZWN0LXByZXZpb3VzLWh1bmtcIiBjYWxsYmFjaz17dGhpcy5zZWxlY3RQcmV2aW91c0h1bmt9IC8+XG4gICAgICAgIDxDb21tYW5kIGNvbW1hbmQ9XCJjb3JlOmNvbmZpcm1cIiBjYWxsYmFjaz17dGhpcy5kaWRDb25maXJtfSAvPlxuICAgICAgICA8Q29tbWFuZCBjb21tYW5kPVwiY29yZTp1bmRvXCIgY2FsbGJhY2s9e3RoaXMudW5kb0xhc3REaXNjYXJkRnJvbUNvcmVVbmRvfSAvPlxuICAgICAgICA8Q29tbWFuZCBjb21tYW5kPVwiZ2l0aHViOmRpc2NhcmQtc2VsZWN0ZWQtbGluZXNcIiBjYWxsYmFjaz17dGhpcy5kaXNjYXJkU2VsZWN0aW9uRnJvbUNvbW1hbmR9IC8+XG4gICAgICAgIDxDb21tYW5kIGNvbW1hbmQ9XCJnaXRodWI6anVtcC10by1maWxlXCIgY2FsbGJhY2s9e3RoaXMuZGlkT3BlbkZpbGV9IC8+XG4gICAgICAgIDxDb21tYW5kIGNvbW1hbmQ9XCJnaXRodWI6c3VyZmFjZVwiIGNhbGxiYWNrPXt0aGlzLnByb3BzLnN1cmZhY2V9IC8+XG4gICAgICAgIDxDb21tYW5kIGNvbW1hbmQ9XCJnaXRodWI6dG9nZ2xlLXBhdGNoLXNlbGVjdGlvbi1tb2RlXCIgY2FsbGJhY2s9e3RoaXMuZGlkVG9nZ2xlU2VsZWN0aW9uTW9kZX0gLz5cbiAgICAgICAge3N0YWdlTW9kZUNvbW1hbmR9XG4gICAgICAgIHtzdGFnZVN5bWxpbmtDb21tYW5kfVxuICAgICAgPC9Db21tYW5kcz5cbiAgICApO1xuICB9XG5cbiAgcmVuZGVyRW1wdHlQYXRjaCgpIHtcbiAgICByZXR1cm4gPHAgY2xhc3NOYW1lPVwiZ2l0aHViLUZpbGVQYXRjaFZpZXctbWVzc2FnZSBpY29uIGljb24taW5mb1wiPk5vIGNoYW5nZXMgdG8gZGlzcGxheTwvcD47XG4gIH1cblxuICByZW5kZXJOb25FbXB0eVBhdGNoKCkge1xuICAgIHJldHVybiAoXG4gICAgICA8QXRvbVRleHRFZGl0b3JcbiAgICAgICAgd29ya3NwYWNlPXt0aGlzLnByb3BzLndvcmtzcGFjZX1cblxuICAgICAgICBidWZmZXI9e3RoaXMucHJvcHMubXVsdGlGaWxlUGF0Y2guZ2V0QnVmZmVyKCl9XG4gICAgICAgIGxpbmVOdW1iZXJHdXR0ZXJWaXNpYmxlPXtmYWxzZX1cbiAgICAgICAgYXV0b1dpZHRoPXtmYWxzZX1cbiAgICAgICAgYXV0b0hlaWdodD17ZmFsc2V9XG4gICAgICAgIHJlYWRPbmx5PXt0cnVlfVxuICAgICAgICBzb2Z0V3JhcHBlZD17dHJ1ZX1cblxuICAgICAgICBkaWRBZGRTZWxlY3Rpb249e3RoaXMuZGlkQWRkU2VsZWN0aW9ufVxuICAgICAgICBkaWRDaGFuZ2VTZWxlY3Rpb25SYW5nZT17dGhpcy5kaWRDaGFuZ2VTZWxlY3Rpb25SYW5nZX1cbiAgICAgICAgZGlkRGVzdHJveVNlbGVjdGlvbj17dGhpcy5kaWREZXN0cm95U2VsZWN0aW9ufVxuICAgICAgICByZWZNb2RlbD17dGhpcy5yZWZFZGl0b3J9PlxuXG4gICAgICAgIDxHdXR0ZXJcbiAgICAgICAgICBuYW1lPVwib2xkLWxpbmUtbnVtYmVyc1wiXG4gICAgICAgICAgcHJpb3JpdHk9ezF9XG4gICAgICAgICAgY2xhc3NOYW1lPVwib2xkXCJcbiAgICAgICAgICB0eXBlPVwibGluZS1udW1iZXJcIlxuICAgICAgICAgIGxhYmVsRm49e3RoaXMub2xkTGluZU51bWJlckxhYmVsfVxuICAgICAgICAgIG9uTW91c2VEb3duPXt0aGlzLmRpZE1vdXNlRG93bk9uTGluZU51bWJlcn1cbiAgICAgICAgICBvbk1vdXNlTW92ZT17dGhpcy5kaWRNb3VzZU1vdmVPbkxpbmVOdW1iZXJ9XG4gICAgICAgIC8+XG4gICAgICAgIDxHdXR0ZXJcbiAgICAgICAgICBuYW1lPVwibmV3LWxpbmUtbnVtYmVyc1wiXG4gICAgICAgICAgcHJpb3JpdHk9ezJ9XG4gICAgICAgICAgY2xhc3NOYW1lPVwibmV3XCJcbiAgICAgICAgICB0eXBlPVwibGluZS1udW1iZXJcIlxuICAgICAgICAgIGxhYmVsRm49e3RoaXMubmV3TGluZU51bWJlckxhYmVsfVxuICAgICAgICAgIG9uTW91c2VEb3duPXt0aGlzLmRpZE1vdXNlRG93bk9uTGluZU51bWJlcn1cbiAgICAgICAgICBvbk1vdXNlTW92ZT17dGhpcy5kaWRNb3VzZU1vdmVPbkxpbmVOdW1iZXJ9XG4gICAgICAgIC8+XG4gICAgICAgIHt0aGlzLnByb3BzLmNvbmZpZy5nZXQoJ2dpdGh1Yi5zaG93RGlmZkljb25HdXR0ZXInKSAmJiAoXG4gICAgICAgICAgPEd1dHRlclxuICAgICAgICAgICAgbmFtZT1cImRpZmYtaWNvbnNcIlxuICAgICAgICAgICAgcHJpb3JpdHk9ezN9XG4gICAgICAgICAgICB0eXBlPVwibGluZS1udW1iZXJcIlxuICAgICAgICAgICAgY2xhc3NOYW1lPVwiaWNvbnNcIlxuICAgICAgICAgICAgbGFiZWxGbj17QkxBTktfTEFCRUx9XG4gICAgICAgICAgICBvbk1vdXNlRG93bj17dGhpcy5kaWRNb3VzZURvd25PbkxpbmVOdW1iZXJ9XG4gICAgICAgICAgICBvbk1vdXNlTW92ZT17dGhpcy5kaWRNb3VzZU1vdmVPbkxpbmVOdW1iZXJ9XG4gICAgICAgICAgLz5cbiAgICAgICAgKX1cblxuICAgICAgICB7dGhpcy5wcm9wcy5tdWx0aUZpbGVQYXRjaC5nZXRGaWxlUGF0Y2hlcygpLm1hcCh0aGlzLnJlbmRlckZpbGVQYXRjaERlY29yYXRpb25zKX1cblxuICAgICAgICB7dGhpcy5yZW5kZXJMaW5lRGVjb3JhdGlvbnMoXG4gICAgICAgICAgQXJyYXkuZnJvbSh0aGlzLnByb3BzLnNlbGVjdGVkUm93cywgcm93ID0+IFJhbmdlLmZyb21PYmplY3QoW1tyb3csIDBdLCBbcm93LCBJbmZpbml0eV1dKSksXG4gICAgICAgICAgJ2dpdGh1Yi1GaWxlUGF0Y2hWaWV3LWxpbmUtLXNlbGVjdGVkJyxcbiAgICAgICAgICB7Z3V0dGVyOiB0cnVlLCBpY29uOiB0cnVlLCBsaW5lOiB0cnVlfSxcbiAgICAgICAgKX1cblxuICAgICAgICB7dGhpcy5yZW5kZXJEZWNvcmF0aW9uc09uTGF5ZXIoXG4gICAgICAgICAgdGhpcy5wcm9wcy5tdWx0aUZpbGVQYXRjaC5nZXRBZGRpdGlvbkxheWVyKCksXG4gICAgICAgICAgJ2dpdGh1Yi1GaWxlUGF0Y2hWaWV3LWxpbmUtLWFkZGVkJyxcbiAgICAgICAgICB7aWNvbjogdHJ1ZSwgbGluZTogdHJ1ZX0sXG4gICAgICAgICl9XG4gICAgICAgIHt0aGlzLnJlbmRlckRlY29yYXRpb25zT25MYXllcihcbiAgICAgICAgICB0aGlzLnByb3BzLm11bHRpRmlsZVBhdGNoLmdldERlbGV0aW9uTGF5ZXIoKSxcbiAgICAgICAgICAnZ2l0aHViLUZpbGVQYXRjaFZpZXctbGluZS0tZGVsZXRlZCcsXG4gICAgICAgICAge2ljb246IHRydWUsIGxpbmU6IHRydWV9LFxuICAgICAgICApfVxuICAgICAgICB7dGhpcy5yZW5kZXJEZWNvcmF0aW9uc09uTGF5ZXIoXG4gICAgICAgICAgdGhpcy5wcm9wcy5tdWx0aUZpbGVQYXRjaC5nZXROb05ld2xpbmVMYXllcigpLFxuICAgICAgICAgICdnaXRodWItRmlsZVBhdGNoVmlldy1saW5lLS1ub25ld2xpbmUnLFxuICAgICAgICAgIHtpY29uOiB0cnVlLCBsaW5lOiB0cnVlfSxcbiAgICAgICAgKX1cblxuICAgICAgPC9BdG9tVGV4dEVkaXRvcj5cbiAgICApO1xuICB9XG5cbiAgcmVuZGVyRmlsZVBhdGNoRGVjb3JhdGlvbnMgPSBmaWxlUGF0Y2ggPT4ge1xuICAgIHJldHVybiAoXG4gICAgICA8RnJhZ21lbnQga2V5PXtmaWxlUGF0Y2guZ2V0UGF0aCgpfT5cbiAgICAgICAgPE1hcmtlciBpbnZhbGlkYXRlPVwibmV2ZXJcIiBidWZmZXJSYW5nZT17ZmlsZVBhdGNoLmdldFN0YXJ0UmFuZ2UoKX0+XG4gICAgICAgICAgPERlY29yYXRpb24gdHlwZT1cImJsb2NrXCIgY2xhc3NOYW1lPVwiZ2l0aHViLUZpbGVQYXRjaFZpZXctY29udHJvbEJsb2NrXCI+XG4gICAgICAgICAgICA8RmlsZVBhdGNoSGVhZGVyVmlld1xuICAgICAgICAgICAgICBpdGVtVHlwZT17dGhpcy5wcm9wcy5pdGVtVHlwZX1cbiAgICAgICAgICAgICAgcmVsUGF0aD17ZmlsZVBhdGNoLmdldFBhdGgoKX1cbiAgICAgICAgICAgICAgbmV3UGF0aD17ZmlsZVBhdGNoLmdldFN0YXR1cygpID09PSAncmVuYW1lZCcgPyBmaWxlUGF0Y2guZ2V0TmV3UGF0aCgpIDogbnVsbH1cbiAgICAgICAgICAgICAgc3RhZ2luZ1N0YXR1cz17dGhpcy5wcm9wcy5zdGFnaW5nU3RhdHVzfVxuICAgICAgICAgICAgICBpc1BhcnRpYWxseVN0YWdlZD17dGhpcy5wcm9wcy5pc1BhcnRpYWxseVN0YWdlZH1cbiAgICAgICAgICAgICAgaGFzSHVua3M9e2ZpbGVQYXRjaC5nZXRIdW5rcygpLmxlbmd0aCA+IDB9XG4gICAgICAgICAgICAgIGhhc1VuZG9IaXN0b3J5PXt0aGlzLnByb3BzLmhhc1VuZG9IaXN0b3J5fVxuICAgICAgICAgICAgICBoYXNNdWx0aXBsZUZpbGVTZWxlY3Rpb25zPXt0aGlzLnByb3BzLmhhc011bHRpcGxlRmlsZVNlbGVjdGlvbnN9XG5cbiAgICAgICAgICAgICAgdG9vbHRpcHM9e3RoaXMucHJvcHMudG9vbHRpcHN9XG5cbiAgICAgICAgICAgICAgdW5kb0xhc3REaXNjYXJkPXsoKSA9PiB0aGlzLnVuZG9MYXN0RGlzY2FyZEZyb21CdXR0b24oZmlsZVBhdGNoKX1cbiAgICAgICAgICAgICAgZGl2ZUludG9NaXJyb3JQYXRjaD17KCkgPT4gdGhpcy5wcm9wcy5kaXZlSW50b01pcnJvclBhdGNoKGZpbGVQYXRjaCl9XG4gICAgICAgICAgICAgIG9wZW5GaWxlPXsoKSA9PiB0aGlzLmRpZE9wZW5GaWxlKHtzZWxlY3RlZEZpbGVQYXRjaDogZmlsZVBhdGNofSl9XG4gICAgICAgICAgICAgIHRvZ2dsZUZpbGU9eygpID0+IHRoaXMucHJvcHMudG9nZ2xlRmlsZShmaWxlUGF0Y2gpfVxuICAgICAgICAgICAgLz5cbiAgICAgICAgICAgIHt0aGlzLnJlbmRlclN5bWxpbmtDaGFuZ2VNZXRhKGZpbGVQYXRjaCl9XG4gICAgICAgICAgICB7dGhpcy5yZW5kZXJFeGVjdXRhYmxlTW9kZUNoYW5nZU1ldGEoZmlsZVBhdGNoKX1cbiAgICAgICAgICA8L0RlY29yYXRpb24+XG4gICAgICAgIDwvTWFya2VyPlxuXG4gICAgICAgIHt0aGlzLnJlbmRlckh1bmtIZWFkZXJzKGZpbGVQYXRjaCl9XG4gICAgICA8L0ZyYWdtZW50PlxuICAgICk7XG4gIH1cblxuICByZW5kZXJFeGVjdXRhYmxlTW9kZUNoYW5nZU1ldGEoZmlsZVBhdGNoKSB7XG4gICAgaWYgKCFmaWxlUGF0Y2guZGlkQ2hhbmdlRXhlY3V0YWJsZU1vZGUoKSkge1xuICAgICAgcmV0dXJuIG51bGw7XG4gICAgfVxuXG4gICAgY29uc3Qgb2xkTW9kZSA9IGZpbGVQYXRjaC5nZXRPbGRNb2RlKCk7XG4gICAgY29uc3QgbmV3TW9kZSA9IGZpbGVQYXRjaC5nZXROZXdNb2RlKCk7XG5cbiAgICBjb25zdCBhdHRycyA9IHRoaXMucHJvcHMuc3RhZ2luZ1N0YXR1cyA9PT0gJ3Vuc3RhZ2VkJ1xuICAgICAgPyB7XG4gICAgICAgIGFjdGlvbkljb246ICdpY29uLW1vdmUtZG93bicsXG4gICAgICAgIGFjdGlvblRleHQ6ICdTdGFnZSBNb2RlIENoYW5nZScsXG4gICAgICB9XG4gICAgICA6IHtcbiAgICAgICAgYWN0aW9uSWNvbjogJ2ljb24tbW92ZS11cCcsXG4gICAgICAgIGFjdGlvblRleHQ6ICdVbnN0YWdlIE1vZGUgQ2hhbmdlJyxcbiAgICAgIH07XG5cbiAgICByZXR1cm4gKFxuICAgICAgPEZpbGVQYXRjaE1ldGFWaWV3XG4gICAgICAgIHRpdGxlPVwiTW9kZSBjaGFuZ2VcIlxuICAgICAgICBhY3Rpb25JY29uPXthdHRycy5hY3Rpb25JY29ufVxuICAgICAgICBhY3Rpb25UZXh0PXthdHRycy5hY3Rpb25UZXh0fVxuICAgICAgICBpdGVtVHlwZT17dGhpcy5wcm9wcy5pdGVtVHlwZX1cbiAgICAgICAgYWN0aW9uPXsoKSA9PiB0aGlzLnByb3BzLnRvZ2dsZU1vZGVDaGFuZ2UoZmlsZVBhdGNoKX0+XG4gICAgICAgIDxGcmFnbWVudD5cbiAgICAgICAgICBGaWxlIGNoYW5nZWQgbW9kZVxuICAgICAgICAgIDxzcGFuIGNsYXNzTmFtZT1cImdpdGh1Yi1GaWxlUGF0Y2hWaWV3LW1ldGFEaWZmIGdpdGh1Yi1GaWxlUGF0Y2hWaWV3LW1ldGFEaWZmLS1yZW1vdmVkXCI+XG4gICAgICAgICAgICBmcm9tIHtleGVjdXRhYmxlVGV4dFtvbGRNb2RlXX0gPGNvZGU+e29sZE1vZGV9PC9jb2RlPlxuICAgICAgICAgIDwvc3Bhbj5cbiAgICAgICAgICA8c3BhbiBjbGFzc05hbWU9XCJnaXRodWItRmlsZVBhdGNoVmlldy1tZXRhRGlmZiBnaXRodWItRmlsZVBhdGNoVmlldy1tZXRhRGlmZi0tYWRkZWRcIj5cbiAgICAgICAgICAgIHRvIHtleGVjdXRhYmxlVGV4dFtuZXdNb2RlXX0gPGNvZGU+e25ld01vZGV9PC9jb2RlPlxuICAgICAgICAgIDwvc3Bhbj5cbiAgICAgICAgPC9GcmFnbWVudD5cbiAgICAgIDwvRmlsZVBhdGNoTWV0YVZpZXc+XG4gICAgKTtcbiAgfVxuXG4gIHJlbmRlclN5bWxpbmtDaGFuZ2VNZXRhKGZpbGVQYXRjaCkge1xuICAgIGlmICghZmlsZVBhdGNoLmhhc1N5bWxpbmsoKSkge1xuICAgICAgcmV0dXJuIG51bGw7XG4gICAgfVxuXG4gICAgbGV0IGRldGFpbCA9IDxkaXYgLz47XG4gICAgbGV0IHRpdGxlID0gJyc7XG4gICAgY29uc3Qgb2xkU3ltbGluayA9IGZpbGVQYXRjaC5nZXRPbGRTeW1saW5rKCk7XG4gICAgY29uc3QgbmV3U3ltbGluayA9IGZpbGVQYXRjaC5nZXROZXdTeW1saW5rKCk7XG4gICAgaWYgKG9sZFN5bWxpbmsgJiYgbmV3U3ltbGluaykge1xuICAgICAgZGV0YWlsID0gKFxuICAgICAgICA8RnJhZ21lbnQ+XG4gICAgICAgICAgU3ltbGluayBjaGFuZ2VkXG4gICAgICAgICAgPHNwYW4gY2xhc3NOYW1lPXtjeChcbiAgICAgICAgICAgICdnaXRodWItRmlsZVBhdGNoVmlldy1tZXRhRGlmZicsXG4gICAgICAgICAgICAnZ2l0aHViLUZpbGVQYXRjaFZpZXctbWV0YURpZmYtLWZ1bGxXaWR0aCcsXG4gICAgICAgICAgICAnZ2l0aHViLUZpbGVQYXRjaFZpZXctbWV0YURpZmYtLXJlbW92ZWQnLFxuICAgICAgICAgICl9PlxuICAgICAgICAgICAgZnJvbSA8Y29kZT57b2xkU3ltbGlua308L2NvZGU+XG4gICAgICAgICAgPC9zcGFuPlxuICAgICAgICAgIDxzcGFuIGNsYXNzTmFtZT17Y3goXG4gICAgICAgICAgICAnZ2l0aHViLUZpbGVQYXRjaFZpZXctbWV0YURpZmYnLFxuICAgICAgICAgICAgJ2dpdGh1Yi1GaWxlUGF0Y2hWaWV3LW1ldGFEaWZmLS1mdWxsV2lkdGgnLFxuICAgICAgICAgICAgJ2dpdGh1Yi1GaWxlUGF0Y2hWaWV3LW1ldGFEaWZmLS1hZGRlZCcsXG4gICAgICAgICAgKX0+XG4gICAgICAgICAgICB0byA8Y29kZT57bmV3U3ltbGlua308L2NvZGU+XG4gICAgICAgICAgPC9zcGFuPi5cbiAgICAgICAgPC9GcmFnbWVudD5cbiAgICAgICk7XG4gICAgICB0aXRsZSA9ICdTeW1saW5rIGNoYW5nZWQnO1xuICAgIH0gZWxzZSBpZiAob2xkU3ltbGluayAmJiAhbmV3U3ltbGluaykge1xuICAgICAgZGV0YWlsID0gKFxuICAgICAgICA8RnJhZ21lbnQ+XG4gICAgICAgICAgU3ltbGlua1xuICAgICAgICAgIDxzcGFuIGNsYXNzTmFtZT1cImdpdGh1Yi1GaWxlUGF0Y2hWaWV3LW1ldGFEaWZmIGdpdGh1Yi1GaWxlUGF0Y2hWaWV3LW1ldGFEaWZmLS1yZW1vdmVkXCI+XG4gICAgICAgICAgICB0byA8Y29kZT57b2xkU3ltbGlua308L2NvZGU+XG4gICAgICAgICAgPC9zcGFuPlxuICAgICAgICAgIGRlbGV0ZWQuXG4gICAgICAgIDwvRnJhZ21lbnQ+XG4gICAgICApO1xuICAgICAgdGl0bGUgPSAnU3ltbGluayBkZWxldGVkJztcbiAgICB9IGVsc2Uge1xuICAgICAgZGV0YWlsID0gKFxuICAgICAgICA8RnJhZ21lbnQ+XG4gICAgICAgICAgU3ltbGlua1xuICAgICAgICAgIDxzcGFuIGNsYXNzTmFtZT1cImdpdGh1Yi1GaWxlUGF0Y2hWaWV3LW1ldGFEaWZmIGdpdGh1Yi1GaWxlUGF0Y2hWaWV3LW1ldGFEaWZmLS1hZGRlZFwiPlxuICAgICAgICAgICAgdG8gPGNvZGU+e25ld1N5bWxpbmt9PC9jb2RlPlxuICAgICAgICAgIDwvc3Bhbj5cbiAgICAgICAgICBjcmVhdGVkLlxuICAgICAgICA8L0ZyYWdtZW50PlxuICAgICAgKTtcbiAgICAgIHRpdGxlID0gJ1N5bWxpbmsgY3JlYXRlZCc7XG4gICAgfVxuXG4gICAgY29uc3QgYXR0cnMgPSB0aGlzLnByb3BzLnN0YWdpbmdTdGF0dXMgPT09ICd1bnN0YWdlZCdcbiAgICAgID8ge1xuICAgICAgICBhY3Rpb25JY29uOiAnaWNvbi1tb3ZlLWRvd24nLFxuICAgICAgICBhY3Rpb25UZXh0OiAnU3RhZ2UgU3ltbGluayBDaGFuZ2UnLFxuICAgICAgfVxuICAgICAgOiB7XG4gICAgICAgIGFjdGlvbkljb246ICdpY29uLW1vdmUtdXAnLFxuICAgICAgICBhY3Rpb25UZXh0OiAnVW5zdGFnZSBTeW1saW5rIENoYW5nZScsXG4gICAgICB9O1xuXG4gICAgcmV0dXJuIChcbiAgICAgIDxGaWxlUGF0Y2hNZXRhVmlld1xuICAgICAgICB0aXRsZT17dGl0bGV9XG4gICAgICAgIGFjdGlvbkljb249e2F0dHJzLmFjdGlvbkljb259XG4gICAgICAgIGFjdGlvblRleHQ9e2F0dHJzLmFjdGlvblRleHR9XG4gICAgICAgIGl0ZW1UeXBlPXt0aGlzLnByb3BzLml0ZW1UeXBlfVxuICAgICAgICBhY3Rpb249eygpID0+IHRoaXMucHJvcHMudG9nZ2xlU3ltbGlua0NoYW5nZShmaWxlUGF0Y2gpfT5cbiAgICAgICAgPEZyYWdtZW50PlxuICAgICAgICAgIHtkZXRhaWx9XG4gICAgICAgIDwvRnJhZ21lbnQ+XG4gICAgICA8L0ZpbGVQYXRjaE1ldGFWaWV3PlxuICAgICk7XG4gIH1cblxuICByZW5kZXJIdW5rSGVhZGVycyhmaWxlUGF0Y2gpIHtcbiAgICBjb25zdCB0b2dnbGVWZXJiID0gdGhpcy5wcm9wcy5zdGFnaW5nU3RhdHVzID09PSAndW5zdGFnZWQnID8gJ1N0YWdlJyA6ICdVbnN0YWdlJztcbiAgICBjb25zdCBzZWxlY3RlZEh1bmtzID0gbmV3IFNldChcbiAgICAgIEFycmF5LmZyb20odGhpcy5wcm9wcy5zZWxlY3RlZFJvd3MsIHJvdyA9PiB0aGlzLnByb3BzLm11bHRpRmlsZVBhdGNoLmdldEh1bmtBdChyb3cpKSxcbiAgICApO1xuXG4gICAgcmV0dXJuIChcbiAgICAgIDxGcmFnbWVudD5cbiAgICAgICAgPE1hcmtlckxheWVyPlxuICAgICAgICAgIHtmaWxlUGF0Y2guZ2V0SHVua3MoKS5tYXAoKGh1bmssIGluZGV4KSA9PiB7XG4gICAgICAgICAgICBjb25zdCBjb250YWluc1NlbGVjdGlvbiA9IHRoaXMucHJvcHMuc2VsZWN0aW9uTW9kZSA9PT0gJ2xpbmUnICYmIHNlbGVjdGVkSHVua3MuaGFzKGh1bmspO1xuICAgICAgICAgICAgY29uc3QgaXNTZWxlY3RlZCA9ICh0aGlzLnByb3BzLnNlbGVjdGlvbk1vZGUgPT09ICdodW5rJykgJiYgc2VsZWN0ZWRIdW5rcy5oYXMoaHVuayk7XG5cbiAgICAgICAgICAgIGxldCBidXR0b25TdWZmaXggPSAnJztcbiAgICAgICAgICAgIGlmIChjb250YWluc1NlbGVjdGlvbikge1xuICAgICAgICAgICAgICBidXR0b25TdWZmaXggKz0gJ1NlbGVjdGVkIExpbmUnO1xuICAgICAgICAgICAgICBpZiAodGhpcy5wcm9wcy5zZWxlY3RlZFJvd3Muc2l6ZSA+IDEpIHtcbiAgICAgICAgICAgICAgICBidXR0b25TdWZmaXggKz0gJ3MnO1xuICAgICAgICAgICAgICB9XG4gICAgICAgICAgICB9IGVsc2Uge1xuICAgICAgICAgICAgICBidXR0b25TdWZmaXggKz0gJ0h1bmsnO1xuICAgICAgICAgICAgICBpZiAoc2VsZWN0ZWRIdW5rcy5zaXplID4gMSkge1xuICAgICAgICAgICAgICAgIGJ1dHRvblN1ZmZpeCArPSAncyc7XG4gICAgICAgICAgICAgIH1cbiAgICAgICAgICAgIH1cblxuICAgICAgICAgICAgY29uc3QgdG9nZ2xlU2VsZWN0aW9uTGFiZWwgPSBgJHt0b2dnbGVWZXJifSAke2J1dHRvblN1ZmZpeH1gO1xuICAgICAgICAgICAgY29uc3QgZGlzY2FyZFNlbGVjdGlvbkxhYmVsID0gYERpc2NhcmQgJHtidXR0b25TdWZmaXh9YDtcblxuICAgICAgICAgICAgY29uc3Qgc3RhcnRQb2ludCA9IGh1bmsuZ2V0UmFuZ2UoKS5zdGFydDtcbiAgICAgICAgICAgIGNvbnN0IHN0YXJ0UmFuZ2UgPSBuZXcgUmFuZ2Uoc3RhcnRQb2ludCwgc3RhcnRQb2ludCk7XG5cbiAgICAgICAgICAgIHJldHVybiAoXG4gICAgICAgICAgICAgIDxNYXJrZXIga2V5PXtgaHVua0hlYWRlci0ke2luZGV4fWB9IGJ1ZmZlclJhbmdlPXtzdGFydFJhbmdlfSBpbnZhbGlkYXRlPVwibmV2ZXJcIj5cbiAgICAgICAgICAgICAgICA8RGVjb3JhdGlvbiB0eXBlPVwiYmxvY2tcIiBjbGFzc05hbWU9XCJnaXRodWItRmlsZVBhdGNoVmlldy1jb250cm9sQmxvY2tcIj5cbiAgICAgICAgICAgICAgICAgIDxIdW5rSGVhZGVyVmlld1xuICAgICAgICAgICAgICAgICAgICByZWZUYXJnZXQ9e3RoaXMucmVmRWRpdG9yRWxlbWVudH1cbiAgICAgICAgICAgICAgICAgICAgaHVuaz17aHVua31cbiAgICAgICAgICAgICAgICAgICAgaXNTZWxlY3RlZD17aXNTZWxlY3RlZH1cbiAgICAgICAgICAgICAgICAgICAgc3RhZ2luZ1N0YXR1cz17dGhpcy5wcm9wcy5zdGFnaW5nU3RhdHVzfVxuICAgICAgICAgICAgICAgICAgICBzZWxlY3Rpb25Nb2RlPVwibGluZVwiXG4gICAgICAgICAgICAgICAgICAgIHRvZ2dsZVNlbGVjdGlvbkxhYmVsPXt0b2dnbGVTZWxlY3Rpb25MYWJlbH1cbiAgICAgICAgICAgICAgICAgICAgZGlzY2FyZFNlbGVjdGlvbkxhYmVsPXtkaXNjYXJkU2VsZWN0aW9uTGFiZWx9XG5cbiAgICAgICAgICAgICAgICAgICAgdG9vbHRpcHM9e3RoaXMucHJvcHMudG9vbHRpcHN9XG4gICAgICAgICAgICAgICAgICAgIGtleW1hcHM9e3RoaXMucHJvcHMua2V5bWFwc31cblxuICAgICAgICAgICAgICAgICAgICB0b2dnbGVTZWxlY3Rpb249eygpID0+IHRoaXMudG9nZ2xlSHVua1NlbGVjdGlvbihodW5rLCBjb250YWluc1NlbGVjdGlvbil9XG4gICAgICAgICAgICAgICAgICAgIGRpc2NhcmRTZWxlY3Rpb249eygpID0+IHRoaXMuZGlzY2FyZEh1bmtTZWxlY3Rpb24oaHVuaywgY29udGFpbnNTZWxlY3Rpb24pfVxuICAgICAgICAgICAgICAgICAgICBtb3VzZURvd249e3RoaXMuZGlkTW91c2VEb3duT25IZWFkZXJ9XG4gICAgICAgICAgICAgICAgICAgIGl0ZW1UeXBlPXt0aGlzLnByb3BzLml0ZW1UeXBlfVxuICAgICAgICAgICAgICAgICAgLz5cbiAgICAgICAgICAgICAgICA8L0RlY29yYXRpb24+XG4gICAgICAgICAgICAgIDwvTWFya2VyPlxuICAgICAgICAgICAgKTtcbiAgICAgICAgICB9KX1cbiAgICAgICAgPC9NYXJrZXJMYXllcj5cbiAgICAgIDwvRnJhZ21lbnQ+XG4gICAgKTtcbiAgfVxuXG4gIHJlbmRlckxpbmVEZWNvcmF0aW9ucyhyYW5nZXMsIGxpbmVDbGFzcywge2xpbmUsIGd1dHRlciwgaWNvbiwgcmVmSG9sZGVyfSkge1xuICAgIGlmIChyYW5nZXMubGVuZ3RoID09PSAwKSB7XG4gICAgICByZXR1cm4gbnVsbDtcbiAgICB9XG5cbiAgICBjb25zdCBob2xkZXIgPSByZWZIb2xkZXIgfHwgbmV3IFJlZkhvbGRlcigpO1xuICAgIHJldHVybiAoXG4gICAgICA8TWFya2VyTGF5ZXIgaGFuZGxlTGF5ZXI9e2hvbGRlci5zZXR0ZXJ9PlxuICAgICAgICB7cmFuZ2VzLm1hcCgocmFuZ2UsIGluZGV4KSA9PiB7XG4gICAgICAgICAgcmV0dXJuIChcbiAgICAgICAgICAgIDxNYXJrZXJcbiAgICAgICAgICAgICAga2V5PXtgbGluZS0ke2xpbmVDbGFzc30tJHtpbmRleH1gfVxuICAgICAgICAgICAgICBidWZmZXJSYW5nZT17cmFuZ2V9XG4gICAgICAgICAgICAgIGludmFsaWRhdGU9XCJuZXZlclwiXG4gICAgICAgICAgICAvPlxuICAgICAgICAgICk7XG4gICAgICAgIH0pfVxuICAgICAgICB7dGhpcy5yZW5kZXJEZWNvcmF0aW9ucyhsaW5lQ2xhc3MsIHtsaW5lLCBndXR0ZXIsIGljb259KX1cbiAgICAgIDwvTWFya2VyTGF5ZXI+XG4gICAgKTtcbiAgfVxuXG4gIHJlbmRlckRlY29yYXRpb25zT25MYXllcihsYXllciwgbGluZUNsYXNzLCB7bGluZSwgZ3V0dGVyLCBpY29ufSkge1xuICAgIGlmIChsYXllci5nZXRNYXJrZXJDb3VudCgpID09PSAwKSB7XG4gICAgICByZXR1cm4gbnVsbDtcbiAgICB9XG5cbiAgICByZXR1cm4gKFxuICAgICAgPE1hcmtlckxheWVyIGV4dGVybmFsPXtsYXllcn0+XG4gICAgICAgIHt0aGlzLnJlbmRlckRlY29yYXRpb25zKGxpbmVDbGFzcywge2xpbmUsIGd1dHRlciwgaWNvbn0pfVxuICAgICAgPC9NYXJrZXJMYXllcj5cbiAgICApO1xuICB9XG5cbiAgcmVuZGVyRGVjb3JhdGlvbnMobGluZUNsYXNzLCB7bGluZSwgZ3V0dGVyLCBpY29ufSkge1xuICAgIHJldHVybiAoXG4gICAgICA8RnJhZ21lbnQ+XG4gICAgICAgIHtsaW5lICYmIChcbiAgICAgICAgICA8RGVjb3JhdGlvblxuICAgICAgICAgICAgdHlwZT1cImxpbmVcIlxuICAgICAgICAgICAgY2xhc3NOYW1lPXtsaW5lQ2xhc3N9XG4gICAgICAgICAgICBvbWl0RW1wdHlMYXN0Um93PXtmYWxzZX1cbiAgICAgICAgICAvPlxuICAgICAgICApfVxuICAgICAgICB7Z3V0dGVyICYmIChcbiAgICAgICAgICA8RnJhZ21lbnQ+XG4gICAgICAgICAgICA8RGVjb3JhdGlvblxuICAgICAgICAgICAgICB0eXBlPVwibGluZS1udW1iZXJcIlxuICAgICAgICAgICAgICBndXR0ZXJOYW1lPVwib2xkLWxpbmUtbnVtYmVyc1wiXG4gICAgICAgICAgICAgIGNsYXNzTmFtZT17bGluZUNsYXNzfVxuICAgICAgICAgICAgICBvbWl0RW1wdHlMYXN0Um93PXtmYWxzZX1cbiAgICAgICAgICAgIC8+XG4gICAgICAgICAgICA8RGVjb3JhdGlvblxuICAgICAgICAgICAgICB0eXBlPVwibGluZS1udW1iZXJcIlxuICAgICAgICAgICAgICBndXR0ZXJOYW1lPVwibmV3LWxpbmUtbnVtYmVyc1wiXG4gICAgICAgICAgICAgIGNsYXNzTmFtZT17bGluZUNsYXNzfVxuICAgICAgICAgICAgICBvbWl0RW1wdHlMYXN0Um93PXtmYWxzZX1cbiAgICAgICAgICAgIC8+XG4gICAgICAgICAgPC9GcmFnbWVudD5cbiAgICAgICAgKX1cbiAgICAgICAge2ljb24gJiYgKFxuICAgICAgICAgIDxEZWNvcmF0aW9uXG4gICAgICAgICAgICB0eXBlPVwibGluZS1udW1iZXJcIlxuICAgICAgICAgICAgZ3V0dGVyTmFtZT1cImRpZmYtaWNvbnNcIlxuICAgICAgICAgICAgY2xhc3NOYW1lPXtsaW5lQ2xhc3N9XG4gICAgICAgICAgICBvbWl0RW1wdHlMYXN0Um93PXtmYWxzZX1cbiAgICAgICAgICAvPlxuICAgICAgICApfVxuICAgICAgPC9GcmFnbWVudD5cbiAgICApO1xuICB9XG5cbiAgdW5kb0xhc3REaXNjYXJkRnJvbUNvcmVVbmRvID0gKCkgPT4ge1xuICAgIGlmICh0aGlzLnByb3BzLmhhc1VuZG9IaXN0b3J5KSB7XG4gICAgICBjb25zdCBzZWxlY3RlZEZpbGVQYXRjaGVzID0gQXJyYXkuZnJvbSh0aGlzLmdldFNlbGVjdGVkRmlsZVBhdGNoZXMoKSk7XG4gICAgICBpZiAodGhpcy5wcm9wcy5pdGVtVHlwZSA9PT0gQ2hhbmdlZEZpbGVJdGVtKSB7XG4gICAgICAgIHRoaXMucHJvcHMudW5kb0xhc3REaXNjYXJkKHNlbGVjdGVkRmlsZVBhdGNoZXNbMF0sIHtldmVudFNvdXJjZToge2NvbW1hbmQ6ICdjb3JlOnVuZG8nfX0pO1xuICAgICAgfVxuICAgIH1cbiAgfVxuXG4gIHVuZG9MYXN0RGlzY2FyZEZyb21CdXR0b24gPSBmaWxlUGF0Y2ggPT4ge1xuICAgIHRoaXMucHJvcHMudW5kb0xhc3REaXNjYXJkKGZpbGVQYXRjaCwge2V2ZW50U291cmNlOiAnYnV0dG9uJ30pO1xuICB9XG5cbiAgZGlzY2FyZFNlbGVjdGlvbkZyb21Db21tYW5kID0gKCkgPT4ge1xuICAgIHJldHVybiB0aGlzLnByb3BzLmRpc2NhcmRSb3dzKFxuICAgICAgdGhpcy5wcm9wcy5zZWxlY3RlZFJvd3MsXG4gICAgICB0aGlzLnByb3BzLnNlbGVjdGlvbk1vZGUsXG4gICAgICB7ZXZlbnRTb3VyY2U6IHtjb21tYW5kOiAnZ2l0aHViOmRpc2NhcmQtc2VsZWN0ZWQtbGluZXMnfX0sXG4gICAgKTtcbiAgfVxuXG4gIHRvZ2dsZUh1bmtTZWxlY3Rpb24oaHVuaywgY29udGFpbnNTZWxlY3Rpb24pIHtcbiAgICBpZiAoY29udGFpbnNTZWxlY3Rpb24pIHtcbiAgICAgIHJldHVybiB0aGlzLnByb3BzLnRvZ2dsZVJvd3MoXG4gICAgICAgIHRoaXMucHJvcHMuc2VsZWN0ZWRSb3dzLFxuICAgICAgICB0aGlzLnByb3BzLnNlbGVjdGlvbk1vZGUsXG4gICAgICAgIHtldmVudFNvdXJjZTogJ2J1dHRvbid9LFxuICAgICAgKTtcbiAgICB9IGVsc2Uge1xuICAgICAgY29uc3QgY2hhbmdlUm93cyA9IG5ldyBTZXQoXG4gICAgICAgIGh1bmsuZ2V0Q2hhbmdlcygpXG4gICAgICAgICAgLnJlZHVjZSgocm93cywgY2hhbmdlKSA9PiB7XG4gICAgICAgICAgICByb3dzLnB1c2goLi4uY2hhbmdlLmdldEJ1ZmZlclJvd3MoKSk7XG4gICAgICAgICAgICByZXR1cm4gcm93cztcbiAgICAgICAgICB9LCBbXSksXG4gICAgICApO1xuICAgICAgcmV0dXJuIHRoaXMucHJvcHMudG9nZ2xlUm93cyhcbiAgICAgICAgY2hhbmdlUm93cyxcbiAgICAgICAgJ2h1bmsnLFxuICAgICAgICB7ZXZlbnRTb3VyY2U6ICdidXR0b24nfSxcbiAgICAgICk7XG4gICAgfVxuICB9XG5cbiAgZGlzY2FyZEh1bmtTZWxlY3Rpb24oaHVuaywgY29udGFpbnNTZWxlY3Rpb24pIHtcbiAgICBpZiAoY29udGFpbnNTZWxlY3Rpb24pIHtcbiAgICAgIHJldHVybiB0aGlzLnByb3BzLmRpc2NhcmRSb3dzKFxuICAgICAgICB0aGlzLnByb3BzLnNlbGVjdGVkUm93cyxcbiAgICAgICAgdGhpcy5wcm9wcy5zZWxlY3Rpb25Nb2RlLFxuICAgICAgICB7ZXZlbnRTb3VyY2U6ICdidXR0b24nfSxcbiAgICAgICk7XG4gICAgfSBlbHNlIHtcbiAgICAgIGNvbnN0IGNoYW5nZVJvd3MgPSBuZXcgU2V0KFxuICAgICAgICBodW5rLmdldENoYW5nZXMoKVxuICAgICAgICAgIC5yZWR1Y2UoKHJvd3MsIGNoYW5nZSkgPT4ge1xuICAgICAgICAgICAgcm93cy5wdXNoKC4uLmNoYW5nZS5nZXRCdWZmZXJSb3dzKCkpO1xuICAgICAgICAgICAgcmV0dXJuIHJvd3M7XG4gICAgICAgICAgfSwgW10pLFxuICAgICAgKTtcbiAgICAgIHJldHVybiB0aGlzLnByb3BzLmRpc2NhcmRSb3dzKGNoYW5nZVJvd3MsICdodW5rJywge2V2ZW50U291cmNlOiAnYnV0dG9uJ30pO1xuICAgIH1cbiAgfVxuXG4gIGRpZE1vdXNlRG93bk9uSGVhZGVyKGV2ZW50LCBodW5rKSB7XG4gICAgdGhpcy5uZXh0U2VsZWN0aW9uTW9kZSA9ICdodW5rJztcbiAgICB0aGlzLmhhbmRsZVNlbGVjdGlvbkV2ZW50KGV2ZW50LCBodW5rLmdldFJhbmdlKCkpO1xuICB9XG5cbiAgZGlkTW91c2VEb3duT25MaW5lTnVtYmVyKGV2ZW50KSB7XG4gICAgY29uc3QgbGluZSA9IGV2ZW50LmJ1ZmZlclJvdztcbiAgICBpZiAobGluZSA9PT0gdW5kZWZpbmVkIHx8IGlzTmFOKGxpbmUpKSB7XG4gICAgICByZXR1cm47XG4gICAgfVxuXG4gICAgdGhpcy5uZXh0U2VsZWN0aW9uTW9kZSA9ICdsaW5lJztcbiAgICBpZiAodGhpcy5oYW5kbGVTZWxlY3Rpb25FdmVudChldmVudC5kb21FdmVudCwgW1tsaW5lLCAwXSwgW2xpbmUsIEluZmluaXR5XV0pKSB7XG4gICAgICB0aGlzLm1vdXNlU2VsZWN0aW9uSW5Qcm9ncmVzcyA9IHRydWU7XG4gICAgfVxuICB9XG5cbiAgZGlkTW91c2VNb3ZlT25MaW5lTnVtYmVyKGV2ZW50KSB7XG4gICAgaWYgKCF0aGlzLm1vdXNlU2VsZWN0aW9uSW5Qcm9ncmVzcykge1xuICAgICAgcmV0dXJuO1xuICAgIH1cblxuICAgIGNvbnN0IGxpbmUgPSBldmVudC5idWZmZXJSb3c7XG4gICAgaWYgKHRoaXMubGFzdE1vdXNlTW92ZUxpbmUgPT09IGxpbmUgfHwgbGluZSA9PT0gdW5kZWZpbmVkIHx8IGlzTmFOKGxpbmUpKSB7XG4gICAgICByZXR1cm47XG4gICAgfVxuICAgIHRoaXMubGFzdE1vdXNlTW92ZUxpbmUgPSBsaW5lO1xuXG4gICAgdGhpcy5uZXh0U2VsZWN0aW9uTW9kZSA9ICdsaW5lJztcbiAgICB0aGlzLmhhbmRsZVNlbGVjdGlvbkV2ZW50KGV2ZW50LmRvbUV2ZW50LCBbW2xpbmUsIDBdLCBbbGluZSwgSW5maW5pdHldXSwge2FkZDogdHJ1ZX0pO1xuICB9XG5cbiAgZGlkTW91c2VVcCgpIHtcbiAgICB0aGlzLm1vdXNlU2VsZWN0aW9uSW5Qcm9ncmVzcyA9IGZhbHNlO1xuICB9XG5cbiAgaGFuZGxlU2VsZWN0aW9uRXZlbnQoZXZlbnQsIHJhbmdlTGlrZSwgb3B0cykge1xuICAgIGlmIChldmVudC5idXR0b24gIT09IDApIHtcbiAgICAgIHJldHVybiBmYWxzZTtcbiAgICB9XG5cbiAgICBjb25zdCBpc1dpbmRvd3MgPSBwcm9jZXNzLnBsYXRmb3JtID09PSAnd2luMzInO1xuICAgIGlmIChldmVudC5jdHJsS2V5ICYmICFpc1dpbmRvd3MpIHtcbiAgICAgIC8vIEFsbG93IHRoZSBjb250ZXh0IG1lbnUgdG8gb3Blbi5cbiAgICAgIHJldHVybiBmYWxzZTtcbiAgICB9XG5cbiAgICBjb25zdCBvcHRpb25zID0ge1xuICAgICAgYWRkOiBmYWxzZSxcbiAgICAgIC4uLm9wdHMsXG4gICAgfTtcblxuICAgIC8vIE5vcm1hbGl6ZSB0aGUgdGFyZ2V0IHNlbGVjdGlvbiByYW5nZVxuICAgIGNvbnN0IGNvbnZlcnRlZCA9IFJhbmdlLmZyb21PYmplY3QocmFuZ2VMaWtlKTtcbiAgICBjb25zdCByYW5nZSA9IHRoaXMucmVmRWRpdG9yLm1hcChlZGl0b3IgPT4gZWRpdG9yLmNsaXBCdWZmZXJSYW5nZShjb252ZXJ0ZWQpKS5nZXRPcihjb252ZXJ0ZWQpO1xuXG4gICAgaWYgKGV2ZW50Lm1ldGFLZXkgfHwgLyogaXN0YW5idWwgaWdub3JlIG5leHQgKi8gKGV2ZW50LmN0cmxLZXkgJiYgaXNXaW5kb3dzKSkge1xuICAgICAgdGhpcy5yZWZFZGl0b3IubWFwKGVkaXRvciA9PiB7XG4gICAgICAgIGxldCBpbnRlcnNlY3RzID0gZmFsc2U7XG4gICAgICAgIGxldCB3aXRob3V0ID0gbnVsbDtcblxuICAgICAgICBmb3IgKGNvbnN0IHNlbGVjdGlvbiBvZiBlZGl0b3IuZ2V0U2VsZWN0aW9ucygpKSB7XG4gICAgICAgICAgaWYgKHNlbGVjdGlvbi5pbnRlcnNlY3RzQnVmZmVyUmFuZ2UocmFuZ2UpKSB7XG4gICAgICAgICAgICAvLyBSZW1vdmUgcmFuZ2UgZnJvbSB0aGlzIHNlbGVjdGlvbiBieSB0cnVuY2F0aW5nIGl0IHRvIHRoZSBcIm5lYXIgZWRnZVwiIG9mIHRoZSByYW5nZSBhbmQgY3JlYXRpbmcgYVxuICAgICAgICAgICAgLy8gbmV3IHNlbGVjdGlvbiBmcm9tIHRoZSBcImZhciBlZGdlXCIgdG8gdGhlIHByZXZpb3VzIGVuZC4gT21pdCBlaXRoZXIgc2lkZSBpZiBpdCBpcyBlbXB0eS5cbiAgICAgICAgICAgIGludGVyc2VjdHMgPSB0cnVlO1xuICAgICAgICAgICAgY29uc3Qgc2VsZWN0aW9uUmFuZ2UgPSBzZWxlY3Rpb24uZ2V0QnVmZmVyUmFuZ2UoKTtcblxuICAgICAgICAgICAgY29uc3QgbmV3UmFuZ2VzID0gW107XG5cbiAgICAgICAgICAgIGlmICghcmFuZ2Uuc3RhcnQuaXNFcXVhbChzZWxlY3Rpb25SYW5nZS5zdGFydCkpIHtcbiAgICAgICAgICAgICAgLy8gSW5jbHVkZSB0aGUgYml0IGZyb20gdGhlIHNlbGVjdGlvbidzIHByZXZpb3VzIHN0YXJ0IHRvIHRoZSByYW5nZSdzIHN0YXJ0LlxuICAgICAgICAgICAgICBsZXQgbnVkZ2VkID0gcmFuZ2Uuc3RhcnQ7XG4gICAgICAgICAgICAgIGlmIChyYW5nZS5zdGFydC5jb2x1bW4gPT09IDApIHtcbiAgICAgICAgICAgICAgICBjb25zdCBsYXN0Q29sdW1uID0gZWRpdG9yLmdldEJ1ZmZlcigpLmxpbmVMZW5ndGhGb3JSb3cocmFuZ2Uuc3RhcnQucm93IC0gMSk7XG4gICAgICAgICAgICAgICAgbnVkZ2VkID0gW3JhbmdlLnN0YXJ0LnJvdyAtIDEsIGxhc3RDb2x1bW5dO1xuICAgICAgICAgICAgICB9XG5cbiAgICAgICAgICAgICAgbmV3UmFuZ2VzLnB1c2goW3NlbGVjdGlvblJhbmdlLnN0YXJ0LCBudWRnZWRdKTtcbiAgICAgICAgICAgIH1cblxuICAgICAgICAgICAgaWYgKCFyYW5nZS5lbmQuaXNFcXVhbChzZWxlY3Rpb25SYW5nZS5lbmQpKSB7XG4gICAgICAgICAgICAgIC8vIEluY2x1ZGUgdGhlIGJpdCBmcm9tIHRoZSByYW5nZSdzIGVuZCB0byB0aGUgc2VsZWN0aW9uJ3MgZW5kLlxuICAgICAgICAgICAgICBsZXQgbnVkZ2VkID0gcmFuZ2UuZW5kO1xuICAgICAgICAgICAgICBjb25zdCBsYXN0Q29sdW1uID0gZWRpdG9yLmdldEJ1ZmZlcigpLmxpbmVMZW5ndGhGb3JSb3cocmFuZ2UuZW5kLnJvdyk7XG4gICAgICAgICAgICAgIGlmIChyYW5nZS5lbmQuY29sdW1uID09PSBsYXN0Q29sdW1uKSB7XG4gICAgICAgICAgICAgICAgbnVkZ2VkID0gW3JhbmdlLmVuZC5yb3cgKyAxLCAwXTtcbiAgICAgICAgICAgICAgfVxuXG4gICAgICAgICAgICAgIG5ld1Jhbmdlcy5wdXNoKFtudWRnZWQsIHNlbGVjdGlvblJhbmdlLmVuZF0pO1xuICAgICAgICAgICAgfVxuXG4gICAgICAgICAgICBpZiAobmV3UmFuZ2VzLmxlbmd0aCA+IDApIHtcbiAgICAgICAgICAgICAgc2VsZWN0aW9uLnNldEJ1ZmZlclJhbmdlKG5ld1Jhbmdlc1swXSk7XG4gICAgICAgICAgICAgIGZvciAoY29uc3QgbmV3UmFuZ2Ugb2YgbmV3UmFuZ2VzLnNsaWNlKDEpKSB7XG4gICAgICAgICAgICAgICAgZWRpdG9yLmFkZFNlbGVjdGlvbkZvckJ1ZmZlclJhbmdlKG5ld1JhbmdlLCB7cmV2ZXJzZWQ6IHNlbGVjdGlvbi5pc1JldmVyc2VkKCl9KTtcbiAgICAgICAgICAgICAgfVxuICAgICAgICAgICAgfSBlbHNlIHtcbiAgICAgICAgICAgICAgd2l0aG91dCA9IHNlbGVjdGlvbjtcbiAgICAgICAgICAgIH1cbiAgICAgICAgICB9XG4gICAgICAgIH1cblxuICAgICAgICBpZiAod2l0aG91dCAhPT0gbnVsbCkge1xuICAgICAgICAgIGNvbnN0IHJlcGxhY2VtZW50UmFuZ2VzID0gZWRpdG9yLmdldFNlbGVjdGlvbnMoKVxuICAgICAgICAgICAgLmZpbHRlcihlYWNoID0+IGVhY2ggIT09IHdpdGhvdXQpXG4gICAgICAgICAgICAubWFwKGVhY2ggPT4gZWFjaC5nZXRCdWZmZXJSYW5nZSgpKTtcbiAgICAgICAgICBpZiAocmVwbGFjZW1lbnRSYW5nZXMubGVuZ3RoID4gMCkge1xuICAgICAgICAgICAgZWRpdG9yLnNldFNlbGVjdGVkQnVmZmVyUmFuZ2VzKHJlcGxhY2VtZW50UmFuZ2VzKTtcbiAgICAgICAgICB9XG4gICAgICAgIH1cblxuICAgICAgICBpZiAoIWludGVyc2VjdHMpIHtcbiAgICAgICAgICAvLyBBZGQgdGhpcyByYW5nZSBhcyBhIG5ldywgZGlzdGluY3Qgc2VsZWN0aW9uLlxuICAgICAgICAgIGVkaXRvci5hZGRTZWxlY3Rpb25Gb3JCdWZmZXJSYW5nZShyYW5nZSk7XG4gICAgICAgIH1cblxuICAgICAgICByZXR1cm4gbnVsbDtcbiAgICAgIH0pO1xuICAgIH0gZWxzZSBpZiAob3B0aW9ucy5hZGQgfHwgZXZlbnQuc2hpZnRLZXkpIHtcbiAgICAgIC8vIEV4dGVuZCB0aGUgZXhpc3Rpbmcgc2VsZWN0aW9uIHRvIGVuY29tcGFzcyB0aGlzIHJhbmdlLlxuICAgICAgdGhpcy5yZWZFZGl0b3IubWFwKGVkaXRvciA9PiB7XG4gICAgICAgIGNvbnN0IGxhc3RTZWxlY3Rpb24gPSBlZGl0b3IuZ2V0TGFzdFNlbGVjdGlvbigpO1xuICAgICAgICBjb25zdCBsYXN0U2VsZWN0aW9uUmFuZ2UgPSBsYXN0U2VsZWN0aW9uLmdldEJ1ZmZlclJhbmdlKCk7XG5cbiAgICAgICAgLy8gWW91IGFyZSBub3cgZW50ZXJpbmcgdGhlIHdhbGwgb2YgdGVybmVyeSBvcGVyYXRvcnMuIFRoaXMgaXMgeW91ciBsYXN0IGV4aXQgYmVmb3JlIHRoZSB0b2xsYm9vdGhcbiAgICAgICAgY29uc3QgaXNCZWZvcmUgPSByYW5nZS5zdGFydC5pc0xlc3NUaGFuKGxhc3RTZWxlY3Rpb25SYW5nZS5zdGFydCk7XG4gICAgICAgIGNvbnN0IGZhckVkZ2UgPSBpc0JlZm9yZSA/IHJhbmdlLnN0YXJ0IDogcmFuZ2UuZW5kO1xuICAgICAgICBjb25zdCBuZXdSYW5nZSA9IGlzQmVmb3JlID8gW2ZhckVkZ2UsIGxhc3RTZWxlY3Rpb25SYW5nZS5lbmRdIDogW2xhc3RTZWxlY3Rpb25SYW5nZS5zdGFydCwgZmFyRWRnZV07XG5cbiAgICAgICAgbGFzdFNlbGVjdGlvbi5zZXRCdWZmZXJSYW5nZShuZXdSYW5nZSwge3JldmVyc2VkOiBpc0JlZm9yZX0pO1xuICAgICAgICByZXR1cm4gbnVsbDtcbiAgICAgIH0pO1xuICAgIH0gZWxzZSB7XG4gICAgICB0aGlzLnJlZkVkaXRvci5tYXAoZWRpdG9yID0+IGVkaXRvci5zZXRTZWxlY3RlZEJ1ZmZlclJhbmdlKHJhbmdlKSk7XG4gICAgfVxuXG4gICAgcmV0dXJuIHRydWU7XG4gIH1cblxuICBkaWRDb25maXJtKCkge1xuICAgIHJldHVybiB0aGlzLnByb3BzLnRvZ2dsZVJvd3ModGhpcy5wcm9wcy5zZWxlY3RlZFJvd3MsIHRoaXMucHJvcHMuc2VsZWN0aW9uTW9kZSk7XG4gIH1cblxuICBkaWRUb2dnbGVTZWxlY3Rpb25Nb2RlKCkge1xuICAgIGNvbnN0IHNlbGVjdGVkSHVua3MgPSB0aGlzLmdldFNlbGVjdGVkSHVua3MoKTtcbiAgICB0aGlzLndpdGhTZWxlY3Rpb25Nb2RlKHtcbiAgICAgIGxpbmU6ICgpID0+IHtcbiAgICAgICAgY29uc3QgaHVua1JhbmdlcyA9IHNlbGVjdGVkSHVua3MubWFwKGh1bmsgPT4gaHVuay5nZXRSYW5nZSgpKTtcbiAgICAgICAgdGhpcy5uZXh0U2VsZWN0aW9uTW9kZSA9ICdodW5rJztcbiAgICAgICAgdGhpcy5yZWZFZGl0b3IubWFwKGVkaXRvciA9PiBlZGl0b3Iuc2V0U2VsZWN0ZWRCdWZmZXJSYW5nZXMoaHVua1JhbmdlcykpO1xuICAgICAgfSxcbiAgICAgIGh1bms6ICgpID0+IHtcbiAgICAgICAgbGV0IGZpcnN0Q2hhbmdlUm93ID0gSW5maW5pdHk7XG4gICAgICAgIGZvciAoY29uc3QgaHVuayBvZiBzZWxlY3RlZEh1bmtzKSB7XG4gICAgICAgICAgY29uc3QgW2ZpcnN0Q2hhbmdlXSA9IGh1bmsuZ2V0Q2hhbmdlcygpO1xuICAgICAgICAgIGlmIChmaXJzdENoYW5nZSAmJiAoIWZpcnN0Q2hhbmdlUm93IHx8IGZpcnN0Q2hhbmdlLmdldFN0YXJ0QnVmZmVyUm93KCkgPCBmaXJzdENoYW5nZVJvdykpIHtcbiAgICAgICAgICAgIGZpcnN0Q2hhbmdlUm93ID0gZmlyc3RDaGFuZ2UuZ2V0U3RhcnRCdWZmZXJSb3coKTtcbiAgICAgICAgICB9XG4gICAgICAgIH1cblxuICAgICAgICB0aGlzLm5leHRTZWxlY3Rpb25Nb2RlID0gJ2xpbmUnO1xuICAgICAgICB0aGlzLnJlZkVkaXRvci5tYXAoZWRpdG9yID0+IHtcbiAgICAgICAgICBlZGl0b3Iuc2V0U2VsZWN0ZWRCdWZmZXJSYW5nZXMoW1tbZmlyc3RDaGFuZ2VSb3csIDBdLCBbZmlyc3RDaGFuZ2VSb3csIEluZmluaXR5XV1dKTtcbiAgICAgICAgICByZXR1cm4gbnVsbDtcbiAgICAgICAgfSk7XG4gICAgICB9LFxuICAgIH0pO1xuICB9XG5cbiAgZGlkVG9nZ2xlTW9kZUNoYW5nZSA9ICgpID0+IHtcbiAgICByZXR1cm4gUHJvbWlzZS5hbGwoXG4gICAgICBBcnJheS5mcm9tKHRoaXMuZ2V0U2VsZWN0ZWRGaWxlUGF0Y2hlcygpKVxuICAgICAgICAuZmlsdGVyKGZwID0+IGZwLmRpZENoYW5nZUV4ZWN1dGFibGVNb2RlKCkpXG4gICAgICAgIC5tYXAodGhpcy5wcm9wcy50b2dnbGVNb2RlQ2hhbmdlKSxcbiAgICApO1xuICB9XG5cbiAgZGlkVG9nZ2xlU3ltbGlua0NoYW5nZSA9ICgpID0+IHtcbiAgICByZXR1cm4gUHJvbWlzZS5hbGwoXG4gICAgICBBcnJheS5mcm9tKHRoaXMuZ2V0U2VsZWN0ZWRGaWxlUGF0Y2hlcygpKVxuICAgICAgICAuZmlsdGVyKGZwID0+IGZwLmhhc1R5cGVjaGFuZ2UoKSlcbiAgICAgICAgLm1hcCh0aGlzLnByb3BzLnRvZ2dsZVN5bWxpbmtDaGFuZ2UpLFxuICAgICk7XG4gIH1cblxuICBzZWxlY3ROZXh0SHVuaygpIHtcbiAgICB0aGlzLnJlZkVkaXRvci5tYXAoZWRpdG9yID0+IHtcbiAgICAgIGNvbnN0IG5leHRIdW5rcyA9IG5ldyBTZXQoXG4gICAgICAgIHRoaXMud2l0aFNlbGVjdGVkSHVua3MoaHVuayA9PiB0aGlzLmdldEh1bmtBZnRlcihodW5rKSB8fCBodW5rKSxcbiAgICAgICk7XG4gICAgICBjb25zdCBuZXh0UmFuZ2VzID0gQXJyYXkuZnJvbShuZXh0SHVua3MsIGh1bmsgPT4gaHVuay5nZXRSYW5nZSgpKTtcbiAgICAgIHRoaXMubmV4dFNlbGVjdGlvbk1vZGUgPSAnaHVuayc7XG4gICAgICBlZGl0b3Iuc2V0U2VsZWN0ZWRCdWZmZXJSYW5nZXMobmV4dFJhbmdlcyk7XG4gICAgICByZXR1cm4gbnVsbDtcbiAgICB9KTtcbiAgfVxuXG4gIHNlbGVjdFByZXZpb3VzSHVuaygpIHtcbiAgICB0aGlzLnJlZkVkaXRvci5tYXAoZWRpdG9yID0+IHtcbiAgICAgIGNvbnN0IG5leHRIdW5rcyA9IG5ldyBTZXQoXG4gICAgICAgIHRoaXMud2l0aFNlbGVjdGVkSHVua3MoaHVuayA9PiB0aGlzLmdldEh1bmtCZWZvcmUoaHVuaykgfHwgaHVuayksXG4gICAgICApO1xuICAgICAgY29uc3QgbmV4dFJhbmdlcyA9IEFycmF5LmZyb20obmV4dEh1bmtzLCBodW5rID0+IGh1bmsuZ2V0UmFuZ2UoKSk7XG4gICAgICB0aGlzLm5leHRTZWxlY3Rpb25Nb2RlID0gJ2h1bmsnO1xuICAgICAgZWRpdG9yLnNldFNlbGVjdGVkQnVmZmVyUmFuZ2VzKG5leHRSYW5nZXMpO1xuICAgICAgcmV0dXJuIG51bGw7XG4gICAgfSk7XG4gIH1cblxuICBkaWRPcGVuRmlsZSh7c2VsZWN0ZWRGaWxlUGF0Y2h9ID0ge30pIHtcbiAgICBjb25zdCBjdXJzb3JzQnlGaWxlUGF0Y2ggPSBuZXcgTWFwKCk7XG5cbiAgICB0aGlzLnJlZkVkaXRvci5tYXAoZWRpdG9yID0+IHtcbiAgICAgIGNvbnN0IHBsYWNlZFJvd3MgPSBuZXcgU2V0KCk7XG5cbiAgICAgIGZvciAoY29uc3QgY3Vyc29yIG9mIGVkaXRvci5nZXRDdXJzb3JzKCkpIHtcbiAgICAgICAgY29uc3QgY3Vyc29yUm93ID0gY3Vyc29yLmdldEJ1ZmZlclBvc2l0aW9uKCkucm93O1xuICAgICAgICBjb25zdCBodW5rID0gdGhpcy5wcm9wcy5tdWx0aUZpbGVQYXRjaC5nZXRIdW5rQXQoY3Vyc29yUm93KTtcbiAgICAgICAgY29uc3QgZmlsZVBhdGNoID0gdGhpcy5wcm9wcy5tdWx0aUZpbGVQYXRjaC5nZXRGaWxlUGF0Y2hBdChjdXJzb3JSb3cpO1xuICAgICAgICAvKiBpc3RhbmJ1bCBpZ25vcmUgbmV4dCAqL1xuICAgICAgICBpZiAoIWh1bmspIHtcbiAgICAgICAgICBjb250aW51ZTtcbiAgICAgICAgfVxuXG4gICAgICAgIGxldCBuZXdSb3cgPSBodW5rLmdldE5ld1Jvd0F0KGN1cnNvclJvdyk7XG4gICAgICAgIGxldCBuZXdDb2x1bW4gPSBjdXJzb3IuZ2V0QnVmZmVyUG9zaXRpb24oKS5jb2x1bW47XG4gICAgICAgIGlmIChuZXdSb3cgPT09IG51bGwpIHtcbiAgICAgICAgICBsZXQgbmVhcmVzdFJvdyA9IGh1bmsuZ2V0TmV3U3RhcnRSb3coKTtcbiAgICAgICAgICBmb3IgKGNvbnN0IHJlZ2lvbiBvZiBodW5rLmdldFJlZ2lvbnMoKSkge1xuICAgICAgICAgICAgaWYgKCFyZWdpb24uaW5jbHVkZXNCdWZmZXJSb3coY3Vyc29yUm93KSkge1xuICAgICAgICAgICAgICByZWdpb24ud2hlbih7XG4gICAgICAgICAgICAgICAgdW5jaGFuZ2VkOiAoKSA9PiB7XG4gICAgICAgICAgICAgICAgICBuZWFyZXN0Um93ICs9IHJlZ2lvbi5idWZmZXJSb3dDb3VudCgpO1xuICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAgYWRkaXRpb246ICgpID0+IHtcbiAgICAgICAgICAgICAgICAgIG5lYXJlc3RSb3cgKz0gcmVnaW9uLmJ1ZmZlclJvd0NvdW50KCk7XG4gICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgfSk7XG4gICAgICAgICAgICB9IGVsc2Uge1xuICAgICAgICAgICAgICBicmVhaztcbiAgICAgICAgICAgIH1cbiAgICAgICAgICB9XG5cbiAgICAgICAgICBpZiAoIXBsYWNlZFJvd3MuaGFzKG5lYXJlc3RSb3cpKSB7XG4gICAgICAgICAgICBuZXdSb3cgPSBuZWFyZXN0Um93O1xuICAgICAgICAgICAgbmV3Q29sdW1uID0gMDtcbiAgICAgICAgICAgIHBsYWNlZFJvd3MuYWRkKG5lYXJlc3RSb3cpO1xuICAgICAgICAgIH1cbiAgICAgICAgfVxuXG4gICAgICAgIGlmIChuZXdSb3cgIT09IG51bGwpIHtcbiAgICAgICAgICAvLyBXaHkgaXMgdGhpcyBuZWVkZWQ/IEkgX3RoaW5rXyBldmVyeXRoaW5nIGlzIGluIHRlcm1zIG9mIGJ1ZmZlciBwb3NpdGlvblxuICAgICAgICAgIC8vIHNvIHRoZXJlIHNob3VsZG4ndCBiZSBhbiBvZmYtYnktb25lIGlzc3VlXG4gICAgICAgICAgbmV3Um93IC09IDE7XG4gICAgICAgICAgY29uc3QgY3Vyc29ycyA9IGN1cnNvcnNCeUZpbGVQYXRjaC5nZXQoZmlsZVBhdGNoKTtcbiAgICAgICAgICBpZiAoIWN1cnNvcnMpIHtcbiAgICAgICAgICAgIGN1cnNvcnNCeUZpbGVQYXRjaC5zZXQoZmlsZVBhdGNoLCBbW25ld1JvdywgbmV3Q29sdW1uXV0pO1xuICAgICAgICAgIH0gZWxzZSB7XG4gICAgICAgICAgICBjdXJzb3JzLnB1c2goW25ld1JvdywgbmV3Q29sdW1uXSk7XG4gICAgICAgICAgfVxuICAgICAgICB9XG4gICAgICB9XG5cbiAgICAgIHJldHVybiBudWxsO1xuICAgIH0pO1xuXG4gICAgY29uc3QgZmlsZVBhdGNoZXNXaXRoQ3Vyc29ycyA9IG5ldyBTZXQoY3Vyc29yc0J5RmlsZVBhdGNoLmtleXMoKSk7XG4gICAgaWYgKHNlbGVjdGVkRmlsZVBhdGNoICYmICFmaWxlUGF0Y2hlc1dpdGhDdXJzb3JzLmhhcyhzZWxlY3RlZEZpbGVQYXRjaCkpIHtcbiAgICAgIGNvbnN0IFtmaXJzdEh1bmtdID0gc2VsZWN0ZWRGaWxlUGF0Y2guZ2V0SHVua3MoKTtcbiAgICAgIGNvbnN0IGN1cnNvclJvdyA9IGZpcnN0SHVuayA/IGZpcnN0SHVuay5nZXROZXdTdGFydFJvdygpIC0gMSA6IDA7XG4gICAgICByZXR1cm4gdGhpcy5wcm9wcy5vcGVuRmlsZShzZWxlY3RlZEZpbGVQYXRjaCwgW1tjdXJzb3JSb3csIDBdXSwgdHJ1ZSk7XG4gICAgfSBlbHNlIHtcbiAgICAgIGNvbnN0IHBlbmRpbmcgPSBjdXJzb3JzQnlGaWxlUGF0Y2guc2l6ZSA9PT0gMTtcbiAgICAgIHJldHVybiBQcm9taXNlLmFsbChBcnJheS5mcm9tKGN1cnNvcnNCeUZpbGVQYXRjaCwgdmFsdWUgPT4ge1xuICAgICAgICBjb25zdCBbZmlsZVBhdGNoLCBjdXJzb3JzXSA9IHZhbHVlO1xuICAgICAgICByZXR1cm4gdGhpcy5wcm9wcy5vcGVuRmlsZShmaWxlUGF0Y2gsIGN1cnNvcnMsIHBlbmRpbmcpO1xuICAgICAgfSkpO1xuICAgIH1cblxuICB9XG5cbiAgZ2V0U2VsZWN0ZWRSb3dzKCkge1xuICAgIHJldHVybiB0aGlzLnJlZkVkaXRvci5tYXAoZWRpdG9yID0+IHtcbiAgICAgIHJldHVybiBuZXcgU2V0KFxuICAgICAgICBlZGl0b3IuZ2V0U2VsZWN0aW9ucygpXG4gICAgICAgICAgLm1hcChzZWxlY3Rpb24gPT4gc2VsZWN0aW9uLmdldEJ1ZmZlclJhbmdlKCkpXG4gICAgICAgICAgLnJlZHVjZSgoYWNjLCByYW5nZSkgPT4ge1xuICAgICAgICAgICAgZm9yIChjb25zdCByb3cgb2YgcmFuZ2UuZ2V0Um93cygpKSB7XG4gICAgICAgICAgICAgIGlmICh0aGlzLmlzQ2hhbmdlUm93KHJvdykpIHtcbiAgICAgICAgICAgICAgICBhY2MucHVzaChyb3cpO1xuICAgICAgICAgICAgICB9XG4gICAgICAgICAgICB9XG4gICAgICAgICAgICByZXR1cm4gYWNjO1xuICAgICAgICAgIH0sIFtdKSxcbiAgICAgICk7XG4gICAgfSkuZ2V0T3IobmV3IFNldCgpKTtcbiAgfVxuXG4gIGRpZEFkZFNlbGVjdGlvbigpIHtcbiAgICB0aGlzLmRpZENoYW5nZVNlbGVjdGVkUm93cygpO1xuICB9XG5cbiAgZGlkQ2hhbmdlU2VsZWN0aW9uUmFuZ2UoZXZlbnQpIHtcbiAgICBpZiAoXG4gICAgICAhZXZlbnQgfHxcbiAgICAgIGV2ZW50Lm9sZEJ1ZmZlclJhbmdlLnN0YXJ0LnJvdyAhPT0gZXZlbnQubmV3QnVmZmVyUmFuZ2Uuc3RhcnQucm93IHx8XG4gICAgICBldmVudC5vbGRCdWZmZXJSYW5nZS5lbmQucm93ICE9PSBldmVudC5uZXdCdWZmZXJSYW5nZS5lbmQucm93XG4gICAgKSB7XG4gICAgICB0aGlzLmRpZENoYW5nZVNlbGVjdGVkUm93cygpO1xuICAgIH1cbiAgfVxuXG4gIGRpZERlc3Ryb3lTZWxlY3Rpb24oKSB7XG4gICAgdGhpcy5kaWRDaGFuZ2VTZWxlY3RlZFJvd3MoKTtcbiAgfVxuXG4gIGRpZENoYW5nZVNlbGVjdGVkUm93cygpIHtcbiAgICBpZiAodGhpcy5zdXBwcmVzc0NoYW5nZXMpIHtcbiAgICAgIHJldHVybjtcbiAgICB9XG5cbiAgICBjb25zdCBuZXh0Q3Vyc29yUm93cyA9IHRoaXMucmVmRWRpdG9yLm1hcChlZGl0b3IgPT4ge1xuICAgICAgcmV0dXJuIGVkaXRvci5nZXRDdXJzb3JCdWZmZXJQb3NpdGlvbnMoKS5tYXAocG9zaXRpb24gPT4gcG9zaXRpb24ucm93KTtcbiAgICB9KS5nZXRPcihbXSk7XG4gICAgY29uc3QgaGFzTXVsdGlwbGVGaWxlU2VsZWN0aW9ucyA9IHRoaXMucHJvcHMubXVsdGlGaWxlUGF0Y2guc3BhbnNNdWx0aXBsZUZpbGVzKG5leHRDdXJzb3JSb3dzKTtcblxuICAgIHRoaXMucHJvcHMuc2VsZWN0ZWRSb3dzQ2hhbmdlZChcbiAgICAgIHRoaXMuZ2V0U2VsZWN0ZWRSb3dzKCksXG4gICAgICB0aGlzLm5leHRTZWxlY3Rpb25Nb2RlIHx8ICdsaW5lJyxcbiAgICAgIGhhc011bHRpcGxlRmlsZVNlbGVjdGlvbnMsXG4gICAgKTtcbiAgfVxuXG4gIG9sZExpbmVOdW1iZXJMYWJlbCh7YnVmZmVyUm93LCBzb2Z0V3JhcHBlZH0pIHtcbiAgICBjb25zdCBodW5rID0gdGhpcy5wcm9wcy5tdWx0aUZpbGVQYXRjaC5nZXRIdW5rQXQoYnVmZmVyUm93KTtcbiAgICBpZiAoaHVuayA9PT0gdW5kZWZpbmVkKSB7XG4gICAgICByZXR1cm4gdGhpcy5wYWQoJycpO1xuICAgIH1cblxuICAgIGNvbnN0IG9sZFJvdyA9IGh1bmsuZ2V0T2xkUm93QXQoYnVmZmVyUm93KTtcbiAgICBpZiAoc29mdFdyYXBwZWQpIHtcbiAgICAgIHJldHVybiB0aGlzLnBhZChvbGRSb3cgPT09IG51bGwgPyAnJyA6ICfigKInKTtcbiAgICB9XG5cbiAgICByZXR1cm4gdGhpcy5wYWQob2xkUm93KTtcbiAgfVxuXG4gIG5ld0xpbmVOdW1iZXJMYWJlbCh7YnVmZmVyUm93LCBzb2Z0V3JhcHBlZH0pIHtcbiAgICBjb25zdCBodW5rID0gdGhpcy5wcm9wcy5tdWx0aUZpbGVQYXRjaC5nZXRIdW5rQXQoYnVmZmVyUm93KTtcbiAgICBpZiAoaHVuayA9PT0gdW5kZWZpbmVkKSB7XG4gICAgICByZXR1cm4gdGhpcy5wYWQoJycpO1xuICAgIH1cblxuICAgIGNvbnN0IG5ld1JvdyA9IGh1bmsuZ2V0TmV3Um93QXQoYnVmZmVyUm93KTtcbiAgICBpZiAoc29mdFdyYXBwZWQpIHtcbiAgICAgIHJldHVybiB0aGlzLnBhZChuZXdSb3cgPT09IG51bGwgPyAnJyA6ICfigKInKTtcbiAgICB9XG4gICAgcmV0dXJuIHRoaXMucGFkKG5ld1Jvdyk7XG4gIH1cblxuICAvKlxuICAgKiBSZXR1cm4gYSBTZXQgb2YgdGhlIEh1bmtzIHRoYXQgaW5jbHVkZSBhdCBsZWFzdCBvbmUgZWRpdG9yIHNlbGVjdGlvbi4gVGhlIHNlbGVjdGlvbiBuZWVkIG5vdCBjb250YWluIGFuIGFjdHVhbFxuICAgKiBjaGFuZ2Ugcm93LlxuICAgKi9cbiAgZ2V0U2VsZWN0ZWRIdW5rcygpIHtcbiAgICByZXR1cm4gdGhpcy53aXRoU2VsZWN0ZWRIdW5rcyhlYWNoID0+IGVhY2gpO1xuICB9XG5cbiAgd2l0aFNlbGVjdGVkSHVua3MoY2FsbGJhY2spIHtcbiAgICByZXR1cm4gdGhpcy5yZWZFZGl0b3IubWFwKGVkaXRvciA9PiB7XG4gICAgICBjb25zdCBzZWVuID0gbmV3IFNldCgpO1xuICAgICAgcmV0dXJuIGVkaXRvci5nZXRTZWxlY3RlZEJ1ZmZlclJhbmdlcygpLnJlZHVjZSgoYWNjLCByYW5nZSkgPT4ge1xuICAgICAgICBmb3IgKGNvbnN0IHJvdyBvZiByYW5nZS5nZXRSb3dzKCkpIHtcbiAgICAgICAgICBjb25zdCBodW5rID0gdGhpcy5wcm9wcy5tdWx0aUZpbGVQYXRjaC5nZXRIdW5rQXQocm93KTtcbiAgICAgICAgICBpZiAoIWh1bmsgfHwgc2Vlbi5oYXMoaHVuaykpIHtcbiAgICAgICAgICAgIGNvbnRpbnVlO1xuICAgICAgICAgIH1cblxuICAgICAgICAgIHNlZW4uYWRkKGh1bmspO1xuICAgICAgICAgIGFjYy5wdXNoKGNhbGxiYWNrKGh1bmspKTtcbiAgICAgICAgfVxuICAgICAgICByZXR1cm4gYWNjO1xuICAgICAgfSwgW10pO1xuICAgIH0pLmdldE9yKFtdKTtcbiAgfVxuXG4gIC8qXG4gICAqIFJldHVybiBhIFNldCBvZiBGaWxlUGF0Y2hlcyB0aGF0IGluY2x1ZGUgYXQgbGVhc3Qgb25lIGVkaXRvciBzZWxlY3Rpb24uIFRoZSBzZWxlY3Rpb24gbmVlZCBub3QgY29udGFpbiBhbiBhY3R1YWxcbiAgICogY2hhbmdlIHJvdy5cbiAgICovXG4gIGdldFNlbGVjdGVkRmlsZVBhdGNoZXMoKSB7XG4gICAgcmV0dXJuIHRoaXMucmVmRWRpdG9yLm1hcChlZGl0b3IgPT4ge1xuICAgICAgY29uc3QgcGF0Y2hlcyA9IG5ldyBTZXQoKTtcbiAgICAgIGZvciAoY29uc3QgcmFuZ2Ugb2YgZWRpdG9yLmdldFNlbGVjdGVkQnVmZmVyUmFuZ2VzKCkpIHtcbiAgICAgICAgZm9yIChjb25zdCByb3cgb2YgcmFuZ2UuZ2V0Um93cygpKSB7XG4gICAgICAgICAgY29uc3QgcGF0Y2ggPSB0aGlzLnByb3BzLm11bHRpRmlsZVBhdGNoLmdldEZpbGVQYXRjaEF0KHJvdyk7XG4gICAgICAgICAgcGF0Y2hlcy5hZGQocGF0Y2gpO1xuICAgICAgICB9XG4gICAgICB9XG4gICAgICByZXR1cm4gcGF0Y2hlcztcbiAgICB9KS5nZXRPcihuZXcgU2V0KCkpO1xuICB9XG5cbiAgZ2V0SHVua0JlZm9yZShodW5rKSB7XG4gICAgY29uc3QgcHJldlJvdyA9IGh1bmsuZ2V0UmFuZ2UoKS5zdGFydC5yb3cgLSAxO1xuICAgIHJldHVybiB0aGlzLnByb3BzLm11bHRpRmlsZVBhdGNoLmdldEh1bmtBdChwcmV2Um93KTtcbiAgfVxuXG4gIGdldEh1bmtBZnRlcihodW5rKSB7XG4gICAgY29uc3QgbmV4dFJvdyA9IGh1bmsuZ2V0UmFuZ2UoKS5lbmQucm93ICsgMTtcbiAgICByZXR1cm4gdGhpcy5wcm9wcy5tdWx0aUZpbGVQYXRjaC5nZXRIdW5rQXQobmV4dFJvdyk7XG4gIH1cblxuICBpc0NoYW5nZVJvdyhidWZmZXJSb3cpIHtcbiAgICBjb25zdCBjaGFuZ2VMYXllcnMgPSBbdGhpcy5wcm9wcy5tdWx0aUZpbGVQYXRjaC5nZXRBZGRpdGlvbkxheWVyKCksIHRoaXMucHJvcHMubXVsdGlGaWxlUGF0Y2guZ2V0RGVsZXRpb25MYXllcigpXTtcbiAgICByZXR1cm4gY2hhbmdlTGF5ZXJzLnNvbWUobGF5ZXIgPT4gbGF5ZXIuZmluZE1hcmtlcnMoe2ludGVyc2VjdHNSb3c6IGJ1ZmZlclJvd30pLmxlbmd0aCA+IDApO1xuICB9XG5cbiAgd2l0aFNlbGVjdGlvbk1vZGUoY2FsbGJhY2tzKSB7XG4gICAgY29uc3QgY2FsbGJhY2sgPSBjYWxsYmFja3NbdGhpcy5wcm9wcy5zZWxlY3Rpb25Nb2RlXTtcbiAgICAvKiBpc3RhbmJ1bCBpZ25vcmUgaWYgKi9cbiAgICBpZiAoIWNhbGxiYWNrKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoYFVua25vd24gc2VsZWN0aW9uIG1vZGU6ICR7dGhpcy5wcm9wcy5zZWxlY3Rpb25Nb2RlfWApO1xuICAgIH1cbiAgICByZXR1cm4gY2FsbGJhY2soKTtcbiAgfVxuXG4gIHBhZChudW0pIHtcbiAgICBjb25zdCBtYXhEaWdpdHMgPSB0aGlzLnByb3BzLm11bHRpRmlsZVBhdGNoLmdldE1heExpbmVOdW1iZXJXaWR0aCgpO1xuICAgIGlmIChudW0gPT09IG51bGwpIHtcbiAgICAgIHJldHVybiBOQlNQX0NIQVJBQ1RFUi5yZXBlYXQobWF4RGlnaXRzKTtcbiAgICB9IGVsc2Uge1xuICAgICAgcmV0dXJuIE5CU1BfQ0hBUkFDVEVSLnJlcGVhdChtYXhEaWdpdHMgLSBudW0udG9TdHJpbmcoKS5sZW5ndGgpICsgbnVtLnRvU3RyaW5nKCk7XG4gICAgfVxuICB9XG5cbn1cbiJdfQ==