'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.expectRelayQuery = expectRelayQuery;
exports.clearRelayExpectations = clearRelayExpectations;

var _util = require('util');

var _util2 = _interopRequireDefault(_util);

var _relayRuntime = require('relay-runtime');

var _moment = require('moment');

var _moment2 = _interopRequireDefault(_moment);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

const relayEnvironmentPerURL = new Map();
const tokenPerURL = new Map();
const fetchPerURL = new Map();

const responsesByQuery = new Map();

function logRatelimitApi(headers) {
  const remaining = headers.get('x-ratelimit-remaining');
  const total = headers.get('x-ratelimit-limit');
  const resets = headers.get('x-ratelimit-reset');
  const resetsIn = _moment2.default.unix(parseInt(resets, 10)).from();

  // eslint-disable-next-line no-console
  console.debug(`GitHub API Rate Limit: ${remaining}/${total} — resets ${resetsIn}`);
}

function expectRelayQuery(operationPattern, response) {
  let resolve, reject;
  const promise = new Promise((resolve0, reject0) => {
    resolve = () => resolve0({ data: response });
    reject = reject0;
  });

  const existing = responsesByQuery.get(operationPattern.name) || [];
  existing.push({
    promise,
    response,
    variables: operationPattern.variables || {},
    trace: operationPattern.trace
  });
  responsesByQuery.set(operationPattern.name, existing);

  const disable = () => responsesByQuery.delete(operationPattern.name);

  return { promise, resolve, reject, disable };
}

function clearRelayExpectations() {
  responsesByQuery.clear();
}

function createFetchQuery(url) {
  if (atom.inSpecMode()) {
    return function specFetchQuery(operation, variables, _cacheConfig, _uploadables) {
      const expectations = responsesByQuery.get(operation.name) || [];
      const match = expectations.find(expectation => {
        if (Object.keys(expectation.variables).length !== Object.keys(variables).length) {
          return false;
        }

        for (const key in expectation.variables) {
          if (expectation.variables[key] !== variables[key]) {
            return false;
          }
        }

        return true;
      });

      if (!match) {
        // eslint-disable-next-line no-console
        console.log(`GraphQL query ${operation.name} was:\n  ${operation.text.replace(/\n/g, '\n  ')}\n` + _util2.default.inspect(variables));

        const e = new Error(`Unexpected GraphQL query: ${operation.name}`);
        e.rawStack = e.stack;
        throw e;
      }

      if (match.trace) {
        match.promise.then(result => {
          // eslint-disable-next-line no-console
          console.log(`GraphQL query ${operation.name} was:\n` + _util2.default.inspect(variables) + '\n' + _util2.default.inspect(result, { depth: null }));
        });
      }

      return match.promise;
    };
  }

  return async function fetchQuery(operation, variables, _cacheConfig, _uploadables) {
    const currentToken = tokenPerURL.get(url);

    const response = await fetch(url, {
      method: 'POST',
      headers: {
        'content-type': 'application/json',
        'Authorization': `bearer ${currentToken}`,
        'Accept': 'application/vnd.github.graphql-profiling+json'
      },
      body: JSON.stringify({
        query: operation.text,
        variables
      })
    });

    try {
      atom && atom.inDevMode() && logRatelimitApi(response.headers);
    } catch (_e) {/* do nothing */}

    if (response.status !== 200) {
      const e = new Error(`GraphQL API endpoint at ${url} returned ${response.status}`);
      e.response = response;
      e.responseText = await response.text();
      e.rawStack = e.stack;
      throw e;
    }

    const payload = await response.json();

    if (payload.data && payload.data.errors && payload.data.errors.length > 0) {
      const e = new Error(`GraphQL API endpoint at ${url} returned an error for query ${operation.name}.`);
      e.response = response;
      e.errors = payload.data.errors;
      e.rawStack = e.stack;
      throw e;
    }

    return payload;
  };
}

class RelayNetworkLayerManager {
  static getEnvironmentForHost(endpoint, token) {
    const url = endpoint.getGraphQLRoot();
    let { environment, network } = relayEnvironmentPerURL.get(url) || {};
    tokenPerURL.set(url, token);
    if (!environment) {
      const source = new _relayRuntime.RecordSource();
      const store = new _relayRuntime.Store(source);
      network = _relayRuntime.Network.create(this.getFetchQuery(endpoint, token));
      environment = new _relayRuntime.Environment({ network, store });

      relayEnvironmentPerURL.set(url, { environment, network });
    }
    return environment;
  }

  static getFetchQuery(endpoint, token) {
    const url = endpoint.getGraphQLRoot();
    tokenPerURL.set(url, token);
    let fetch = fetchPerURL.get(url);
    if (!fetch) {
      fetch = createFetchQuery(url);
      fetchPerURL.set(fetch);
    }
    return fetch;
  }
}
exports.default = RelayNetworkLayerManager;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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