'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});
const DETACHED = Symbol('detached');
const REMOTE_TRACKING = Symbol('remote-tracking');

class Branch {
  constructor(name, upstream = nullBranch, push = upstream, head = false, attributes = {}) {
    this.name = name;
    this.upstream = upstream;
    this.push = push;
    this.head = head;
    this.attributes = attributes;
  }

  static createDetached(describe) {
    return new Branch(describe, nullBranch, nullBranch, true, { [DETACHED]: true });
  }

  static createRemoteTracking(refName, remoteName, remoteRef) {
    return new Branch(refName, nullBranch, nullBranch, false, { [REMOTE_TRACKING]: { remoteName, remoteRef } });
  }

  getName() {
    return this.name;
  }

  getShortRef() {
    return this.getName().replace(/^(refs\/)?((heads|remotes)\/)?/, '');
  }

  getFullRef() {
    if (this.isDetached()) {
      return '';
    }

    if (this.isRemoteTracking()) {
      if (this.name.startsWith('refs/')) {
        return this.name;
      } else if (this.name.startsWith('remotes/')) {
        return `refs/${this.name}`;
      }
      return `refs/remotes/${this.name}`;
    }

    if (this.name.startsWith('refs/')) {
      return this.name;
    } else if (this.name.startsWith('heads/')) {
      return `refs/${this.name}`;
    } else {
      return `refs/heads/${this.name}`;
    }
  }

  getRemoteName() {
    if (!this.isRemoteTracking()) {
      return '';
    }
    return this.attributes[REMOTE_TRACKING].remoteName || '';
  }

  getRemoteRef() {
    if (!this.isRemoteTracking()) {
      return '';
    }
    return this.attributes[REMOTE_TRACKING].remoteRef || '';
  }

  getShortRemoteRef() {
    return this.getRemoteRef().replace(/^(refs\/)?((heads|remotes)\/)?/, '');
  }

  getRefSpec(action) {
    if (this.isRemoteTracking()) {
      return '';
    }
    const remoteBranch = action === 'PUSH' ? this.push : this.upstream;
    const remoteBranchName = remoteBranch.getShortRemoteRef();
    const localBranchName = this.getName();
    if (remoteBranchName && remoteBranchName !== localBranchName) {
      if (action === 'PUSH') {
        return `${localBranchName}:${remoteBranchName}`;
      } else if (action === 'PULL') {
        return `${remoteBranchName}:${localBranchName}`;
      }
    }
    return localBranchName;
  }

  getSha() {
    return this.attributes.sha || '';
  }

  getUpstream() {
    return this.upstream;
  }

  getPush() {
    return this.push;
  }

  isHead() {
    return this.head;
  }

  isDetached() {
    return this.attributes[DETACHED] !== undefined;
  }

  isRemoteTracking() {
    return this.attributes[REMOTE_TRACKING] !== undefined;
  }

  isPresent() {
    return true;
  }

}

exports.default = Branch;
const nullBranch = exports.nullBranch = {
  getName() {
    return '';
  },

  getShortRef() {
    return '';
  },

  getFullRef() {
    return '';
  },

  getSha() {
    return '';
  },

  getUpstream() {
    return this;
  },

  getPush() {
    return this;
  },

  isHead() {
    return false;
  },

  getRemoteName() {
    return '';
  },

  getRemoteRef() {
    return '';
  },

  getShortRemoteRef() {
    return '';
  },

  isDetached() {
    return false;
  },

  isRemoteTracking() {
    return false;
  },

  isPresent() {
    return false;
  },

  inspect(depth, options) {
    return '{nullBranch}';
  }
};
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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