'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _extends = Object.assign || function (target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i]; for (var key in source) { if (Object.prototype.hasOwnProperty.call(source, key)) { target[key] = source[key]; } } } return target; };

var _eventKit = require('event-kit');

var _electron = require('electron');

var _atom = require('atom');

var _react = require('react');

var _react2 = _interopRequireDefault(_react);

var _propTypes = require('prop-types');

var _propTypes2 = _interopRequireDefault(_propTypes);

var _path = require('path');

var _path2 = _interopRequireDefault(_path);

var _propTypes3 = require('../prop-types');

var _filePatchListItemView = require('./file-patch-list-item-view');

var _filePatchListItemView2 = _interopRequireDefault(_filePatchListItemView);

var _observeModel = require('./observe-model');

var _observeModel2 = _interopRequireDefault(_observeModel);

var _mergeConflictListItemView = require('./merge-conflict-list-item-view');

var _mergeConflictListItemView2 = _interopRequireDefault(_mergeConflictListItemView);

var _compositeListSelection = require('../models/composite-list-selection');

var _compositeListSelection2 = _interopRequireDefault(_compositeListSelection);

var _resolutionProgress = require('../models/conflicts/resolution-progress');

var _resolutionProgress2 = _interopRequireDefault(_resolutionProgress);

var _commitView = require('./commit-view');

var _commitView2 = _interopRequireDefault(_commitView);

var _refHolder = require('../models/ref-holder');

var _refHolder2 = _interopRequireDefault(_refHolder);

var _changedFileItem = require('../items/changed-file-item');

var _changedFileItem2 = _interopRequireDefault(_changedFileItem);

var _commands = require('../atom/commands');

var _commands2 = _interopRequireDefault(_commands);

var _helpers = require('../helpers');

var _reporterProxy = require('../reporter-proxy');

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

const { Menu, MenuItem } = _electron.remote;


const debounce = (fn, wait) => {
  let timeout;
  return (...args) => {
    return new Promise(resolve => {
      clearTimeout(timeout);
      timeout = setTimeout(() => {
        resolve(fn(...args));
      }, wait);
    });
  };
};

function calculateTruncatedLists(lists) {
  return Object.keys(lists).reduce((acc, key) => {
    const list = lists[key];
    acc.source[key] = list;
    if (list.length <= MAXIMUM_LISTED_ENTRIES) {
      acc[key] = list;
    } else {
      acc[key] = list.slice(0, MAXIMUM_LISTED_ENTRIES);
    }
    return acc;
  }, { source: {} });
}

const noop = () => {};

const MAXIMUM_LISTED_ENTRIES = 1000;

class StagingView extends _react2.default.Component {

  constructor(props) {
    super(props);

    this.undoLastDiscardFromCoreUndo = () => {
      this.undoLastDiscard({ eventSource: { command: 'core:undo' } });
    };

    this.undoLastDiscardFromCommand = () => {
      this.undoLastDiscard({ eventSource: { command: 'github:undo-last-discard-in-git-tab' } });
    };

    this.undoLastDiscardFromButton = () => {
      this.undoLastDiscard({ eventSource: 'button' });
    };

    this.undoLastDiscardFromHeaderMenu = () => {
      this.undoLastDiscard({ eventSource: 'header-menu' });
    };

    this.discardChangesFromCommand = () => {
      this.discardChanges({ eventSource: { command: 'github:discard-changes-in-selected-files' } });
    };

    this.discardAllFromCommand = () => {
      this.discardAll({ eventSource: { command: 'github:discard-all-changes' } });
    };

    this.confirmSelectedItems = async () => {
      const itemPaths = this.getSelectedItemFilePaths();
      await this.props.attemptFileStageOperation(itemPaths, this.state.selection.getActiveListKey());
      await new Promise(resolve => {
        this.setState(prevState => ({ selection: prevState.selection.coalesce() }), resolve);
      });
    };

    (0, _helpers.autobind)(this, 'dblclickOnItem', 'contextMenuOnItem', 'mousedownOnItem', 'mousemoveOnItem', 'mouseup', 'registerItemElement', 'renderBody', 'openFile', 'discardChanges', 'activateNextList', 'activatePreviousList', 'activateLastList', 'stageAll', 'unstageAll', 'stageAllMergeConflicts', 'discardAll', 'confirmSelectedItems', 'selectAll', 'selectFirst', 'selectLast', 'diveIntoSelection', 'showDiffView', 'showBulkResolveMenu', 'showActionsMenu', 'resolveCurrentAsOurs', 'resolveCurrentAsTheirs', 'quietlySelectItem', 'didChangeSelectedItems');

    this.subs = new _eventKit.CompositeDisposable(atom.config.observe('github.keyboardNavigationDelay', value => {
      if (value === 0) {
        this.debouncedDidChangeSelectedItem = this.didChangeSelectedItems;
      } else {
        this.debouncedDidChangeSelectedItem = debounce(this.didChangeSelectedItems, value);
      }
    }));

    this.state = _extends({}, calculateTruncatedLists({
      unstagedChanges: this.props.unstagedChanges,
      stagedChanges: this.props.stagedChanges,
      mergeConflicts: this.props.mergeConflicts
    }), {
      selection: new _compositeListSelection2.default({
        listsByKey: [['unstaged', this.props.unstagedChanges], ['conflicts', this.props.mergeConflicts], ['staged', this.props.stagedChanges]],
        idForItem: item => item.filePath
      })
    });

    this.mouseSelectionInProgress = false;
    this.listElementsByItem = new WeakMap();
    this.refRoot = new _refHolder2.default();
  }

  static getDerivedStateFromProps(nextProps, prevState) {
    let nextState = {};

    if (['unstagedChanges', 'stagedChanges', 'mergeConflicts'].some(key => prevState.source[key] !== nextProps[key])) {
      const nextLists = calculateTruncatedLists({
        unstagedChanges: nextProps.unstagedChanges,
        stagedChanges: nextProps.stagedChanges,
        mergeConflicts: nextProps.mergeConflicts
      });

      nextState = _extends({}, nextLists, {
        selection: prevState.selection.updateLists([['unstaged', nextLists.unstagedChanges], ['conflicts', nextLists.mergeConflicts], ['staged', nextLists.stagedChanges]])
      });
    }

    return nextState;
  }

  componentDidMount() {
    window.addEventListener('mouseup', this.mouseup);
    this.subs.add(new _eventKit.Disposable(() => window.removeEventListener('mouseup', this.mouseup)), this.props.workspace.onDidChangeActivePaneItem(() => {
      this.syncWithWorkspace();
    }));

    if (this.isPopulated(this.props)) {
      this.syncWithWorkspace();
    }
  }

  componentDidUpdate(prevProps, prevState) {
    const isRepoSame = prevProps.workingDirectoryPath === this.props.workingDirectoryPath;
    const hasSelectionsPresent = prevState.selection.getSelectedItems().size > 0 && this.state.selection.getSelectedItems().size > 0;
    const selectionChanged = this.state.selection !== prevState.selection;

    if (isRepoSame && hasSelectionsPresent && selectionChanged) {
      this.debouncedDidChangeSelectedItem();
    }

    const headItem = this.state.selection.getHeadItem();
    if (headItem) {
      const element = this.listElementsByItem.get(headItem);
      if (element) {
        element.scrollIntoViewIfNeeded();
      }
    }

    if (!this.isPopulated(prevProps) && this.isPopulated(this.props)) {
      this.syncWithWorkspace();
    }
  }

  render() {
    return _react2.default.createElement(
      _observeModel2.default,
      { model: this.props.resolutionProgress, fetchData: noop },
      this.renderBody
    );
  }

  renderBody() {
    const selectedItems = this.state.selection.getSelectedItems();

    return _react2.default.createElement(
      'div',
      {
        ref: this.refRoot.setter,
        className: `github-StagingView ${this.state.selection.getActiveListKey()}-changes-focused`,
        tabIndex: '-1' },
      this.renderCommands(),
      _react2.default.createElement(
        'div',
        { className: `github-StagingView-group github-UnstagedChanges ${this.getFocusClass('unstaged')}` },
        _react2.default.createElement(
          'header',
          { className: 'github-StagingView-header' },
          _react2.default.createElement('span', { className: 'icon icon-list-unordered' }),
          _react2.default.createElement(
            'span',
            { className: 'github-StagingView-title' },
            'Unstaged Changes'
          ),
          this.renderActionsMenu(),
          _react2.default.createElement(
            'button',
            {
              className: 'github-StagingView-headerButton icon icon-move-down',
              disabled: this.props.unstagedChanges.length === 0,
              onClick: this.stageAll },
            'Stage All'
          )
        ),
        _react2.default.createElement(
          'div',
          { className: 'github-StagingView-list github-FilePatchListView github-StagingView-unstaged' },
          this.state.unstagedChanges.map(filePatch => _react2.default.createElement(_filePatchListItemView2.default, {
            key: filePatch.filePath,
            registerItemElement: this.registerItemElement,
            filePatch: filePatch,
            onDoubleClick: event => this.dblclickOnItem(event, filePatch),
            onContextMenu: event => this.contextMenuOnItem(event, filePatch),
            onMouseDown: event => this.mousedownOnItem(event, filePatch),
            onMouseMove: event => this.mousemoveOnItem(event, filePatch),
            selected: selectedItems.has(filePatch)
          }))
        ),
        this.renderTruncatedMessage(this.props.unstagedChanges)
      ),
      this.renderMergeConflicts(),
      _react2.default.createElement(
        'div',
        { className: `github-StagingView-group github-StagedChanges ${this.getFocusClass('staged')}` },
        _react2.default.createElement(
          'header',
          { className: 'github-StagingView-header' },
          _react2.default.createElement('span', { className: 'icon icon-tasklist' }),
          _react2.default.createElement(
            'span',
            { className: 'github-StagingView-title' },
            'Staged Changes'
          ),
          _react2.default.createElement(
            'button',
            { className: 'github-StagingView-headerButton icon icon-move-up',
              disabled: this.props.stagedChanges.length === 0,
              onClick: this.unstageAll },
            'Unstage All'
          )
        ),
        _react2.default.createElement(
          'div',
          { className: 'github-StagingView-list github-FilePatchListView github-StagingView-staged' },
          this.state.stagedChanges.map(filePatch => _react2.default.createElement(_filePatchListItemView2.default, {
            key: filePatch.filePath,
            filePatch: filePatch,
            registerItemElement: this.registerItemElement,
            onDoubleClick: event => this.dblclickOnItem(event, filePatch),
            onContextMenu: event => this.contextMenuOnItem(event, filePatch),
            onMouseDown: event => this.mousedownOnItem(event, filePatch),
            onMouseMove: event => this.mousemoveOnItem(event, filePatch),
            selected: selectedItems.has(filePatch)
          }))
        ),
        this.renderTruncatedMessage(this.props.stagedChanges)
      )
    );
  }

  renderCommands() {
    return _react2.default.createElement(
      _react.Fragment,
      null,
      _react2.default.createElement(
        _commands2.default,
        { registry: this.props.commandRegistry, target: '.github-StagingView' },
        _react2.default.createElement(_commands.Command, { command: 'core:move-up', callback: () => this.selectPrevious() }),
        _react2.default.createElement(_commands.Command, { command: 'core:move-down', callback: () => this.selectNext() }),
        _react2.default.createElement(_commands.Command, { command: 'core:move-left', callback: this.diveIntoSelection }),
        _react2.default.createElement(_commands.Command, { command: 'github:show-diff-view', callback: this.showDiffView }),
        _react2.default.createElement(_commands.Command, { command: 'core:select-up', callback: () => this.selectPrevious(true) }),
        _react2.default.createElement(_commands.Command, { command: 'core:select-down', callback: () => this.selectNext(true) }),
        _react2.default.createElement(_commands.Command, { command: 'core:select-all', callback: this.selectAll }),
        _react2.default.createElement(_commands.Command, { command: 'core:move-to-top', callback: this.selectFirst }),
        _react2.default.createElement(_commands.Command, { command: 'core:move-to-bottom', callback: this.selectLast }),
        _react2.default.createElement(_commands.Command, { command: 'core:select-to-top', callback: () => this.selectFirst(true) }),
        _react2.default.createElement(_commands.Command, { command: 'core:select-to-bottom', callback: () => this.selectLast(true) }),
        _react2.default.createElement(_commands.Command, { command: 'core:confirm', callback: this.confirmSelectedItems }),
        _react2.default.createElement(_commands.Command, { command: 'github:activate-next-list', callback: this.activateNextList }),
        _react2.default.createElement(_commands.Command, { command: 'github:activate-previous-list', callback: this.activatePreviousList }),
        _react2.default.createElement(_commands.Command, { command: 'github:jump-to-file', callback: this.openFile }),
        _react2.default.createElement(_commands.Command, { command: 'github:resolve-file-as-ours', callback: this.resolveCurrentAsOurs }),
        _react2.default.createElement(_commands.Command, { command: 'github:resolve-file-as-theirs', callback: this.resolveCurrentAsTheirs }),
        _react2.default.createElement(_commands.Command, { command: 'github:discard-changes-in-selected-files', callback: this.discardChangesFromCommand }),
        _react2.default.createElement(_commands.Command, { command: 'core:undo', callback: this.undoLastDiscardFromCoreUndo })
      ),
      _react2.default.createElement(
        _commands2.default,
        { registry: this.props.commandRegistry, target: 'atom-workspace' },
        _react2.default.createElement(_commands.Command, { command: 'github:stage-all-changes', callback: this.stageAll }),
        _react2.default.createElement(_commands.Command, { command: 'github:unstage-all-changes', callback: this.unstageAll }),
        _react2.default.createElement(_commands.Command, { command: 'github:discard-all-changes', callback: this.discardAllFromCommand }),
        _react2.default.createElement(_commands.Command, { command: 'github:undo-last-discard-in-git-tab',
          callback: this.undoLastDiscardFromCommand
        })
      )
    );
  }

  renderActionsMenu() {
    if (this.props.unstagedChanges.length || this.props.hasUndoHistory) {
      return _react2.default.createElement('button', {
        className: 'github-StagingView-headerButton github-StagingView-headerButton--iconOnly icon icon-ellipses',
        onClick: this.showActionsMenu
      });
    } else {
      return null;
    }
  }

  renderUndoButton() {
    return _react2.default.createElement(
      'button',
      { className: 'github-StagingView-headerButton github-StagingView-headerButton--fullWidth icon icon-history',
        onClick: this.undoLastDiscardFromButton },
      'Undo Discard'
    );
  }

  renderTruncatedMessage(list) {
    if (list.length > MAXIMUM_LISTED_ENTRIES) {
      return _react2.default.createElement(
        'div',
        { className: 'github-StagingView-group-truncatedMsg' },
        'List truncated to the first ',
        MAXIMUM_LISTED_ENTRIES,
        ' items'
      );
    } else {
      return null;
    }
  }

  renderMergeConflicts() {
    const mergeConflicts = this.state.mergeConflicts;

    if (mergeConflicts && mergeConflicts.length > 0) {
      const selectedItems = this.state.selection.getSelectedItems();
      const resolutionProgress = this.props.resolutionProgress;
      const anyUnresolved = mergeConflicts.map(conflict => _path2.default.join(this.props.workingDirectoryPath, conflict.filePath)).some(conflictPath => resolutionProgress.getRemaining(conflictPath) !== 0);

      const bulkResolveDropdown = anyUnresolved ? _react2.default.createElement('span', {
        className: 'inline-block icon icon-ellipses',
        onClick: this.showBulkResolveMenu
      }) : null;

      return _react2.default.createElement(
        'div',
        { className: `github-StagingView-group github-MergeConflictPaths ${this.getFocusClass('conflicts')}` },
        _react2.default.createElement(
          'header',
          { className: 'github-StagingView-header' },
          _react2.default.createElement('span', { className: 'github-FilePatchListView-icon icon icon-alert status-modified' }),
          _react2.default.createElement(
            'span',
            { className: 'github-StagingView-title' },
            'Merge Conflicts'
          ),
          bulkResolveDropdown,
          _react2.default.createElement(
            'button',
            {
              className: 'github-StagingView-headerButton icon icon-move-down',
              disabled: anyUnresolved,
              onClick: this.stageAllMergeConflicts },
            'Stage All'
          )
        ),
        _react2.default.createElement(
          'div',
          { className: 'github-StagingView-list github-FilePatchListView github-StagingView-merge' },
          mergeConflicts.map(mergeConflict => {
            const fullPath = _path2.default.join(this.props.workingDirectoryPath, mergeConflict.filePath);

            return _react2.default.createElement(_mergeConflictListItemView2.default, {
              key: fullPath,
              mergeConflict: mergeConflict,
              remainingConflicts: resolutionProgress.getRemaining(fullPath),
              registerItemElement: this.registerItemElement,
              onDoubleClick: event => this.dblclickOnItem(event, mergeConflict),
              onContextMenu: event => this.contextMenuOnItem(event, mergeConflict),
              onMouseDown: event => this.mousedownOnItem(event, mergeConflict),
              onMouseMove: event => this.mousemoveOnItem(event, mergeConflict),
              selected: selectedItems.has(mergeConflict)
            });
          })
        ),
        this.renderTruncatedMessage(mergeConflicts)
      );
    } else {
      return _react2.default.createElement('noscript', null);
    }
  }

  componentWillUnmount() {
    this.subs.dispose();
  }

  getSelectedItemFilePaths() {
    return Array.from(this.state.selection.getSelectedItems(), item => item.filePath);
  }

  getSelectedConflictPaths() {
    if (this.state.selection.getActiveListKey() !== 'conflicts') {
      return [];
    }
    return this.getSelectedItemFilePaths();
  }

  openFile() {
    const filePaths = this.getSelectedItemFilePaths();
    return this.props.openFiles(filePaths);
  }

  discardChanges({ eventSource } = {}) {
    const filePaths = this.getSelectedItemFilePaths();
    (0, _reporterProxy.addEvent)('discard-unstaged-changes', {
      package: 'github',
      component: 'StagingView',
      fileCount: filePaths.length,
      type: 'selected',
      eventSource
    });
    return this.props.discardWorkDirChangesForPaths(filePaths);
  }

  activateNextList() {
    return new Promise(resolve => {
      let advanced = false;

      this.setState(prevState => {
        const next = prevState.selection.activateNextSelection();
        if (prevState.selection === next) {
          return {};
        }

        advanced = true;
        return { selection: next.coalesce() };
      }, () => resolve(advanced));
    });
  }

  activatePreviousList() {
    return new Promise(resolve => {
      let retreated = false;
      this.setState(prevState => {
        const next = prevState.selection.activatePreviousSelection();
        if (prevState.selection === next) {
          return {};
        }

        retreated = true;
        return { selection: next.coalesce() };
      }, () => resolve(retreated));
    });
  }

  activateLastList() {
    return new Promise(resolve => {
      let emptySelection = false;
      this.setState(prevState => {
        const next = prevState.selection.activateLastSelection();
        emptySelection = next.getSelectedItems().size > 0;

        if (prevState.selection === next) {
          return {};
        }

        return { selection: next.coalesce() };
      }, () => resolve(emptySelection));
    });
  }

  stageAll() {
    if (this.props.unstagedChanges.length === 0) {
      return null;
    }
    return this.props.attemptStageAllOperation('unstaged');
  }

  unstageAll() {
    if (this.props.stagedChanges.length === 0) {
      return null;
    }
    return this.props.attemptStageAllOperation('staged');
  }

  stageAllMergeConflicts() {
    if (this.props.mergeConflicts.length === 0) {
      return null;
    }
    const filePaths = this.props.mergeConflicts.map(conflict => conflict.filePath);
    return this.props.attemptFileStageOperation(filePaths, 'unstaged');
  }

  discardAll({ eventSource } = {}) {
    if (this.props.unstagedChanges.length === 0) {
      return null;
    }
    const filePaths = this.props.unstagedChanges.map(filePatch => filePatch.filePath);
    (0, _reporterProxy.addEvent)('discard-unstaged-changes', {
      package: 'github',
      component: 'StagingView',
      fileCount: filePaths.length,
      type: 'all',
      eventSource
    });
    return this.props.discardWorkDirChangesForPaths(filePaths);
  }

  getNextListUpdatePromise() {
    return this.state.selection.getNextUpdatePromise();
  }

  selectPrevious(preserveTail = false) {
    return new Promise(resolve => {
      this.setState(prevState => ({
        selection: prevState.selection.selectPreviousItem(preserveTail).coalesce()
      }), resolve);
    });
  }

  selectNext(preserveTail = false) {
    return new Promise(resolve => {
      this.setState(prevState => ({
        selection: prevState.selection.selectNextItem(preserveTail).coalesce()
      }), resolve);
    });
  }

  selectAll() {
    return new Promise(resolve => {
      this.setState(prevState => ({
        selection: prevState.selection.selectAllItems().coalesce()
      }), resolve);
    });
  }

  selectFirst(preserveTail = false) {
    return new Promise(resolve => {
      this.setState(prevState => ({
        selection: prevState.selection.selectFirstItem(preserveTail).coalesce()
      }), resolve);
    });
  }

  selectLast(preserveTail = false) {
    return new Promise(resolve => {
      this.setState(prevState => ({
        selection: prevState.selection.selectLastItem(preserveTail).coalesce()
      }), resolve);
    });
  }

  async diveIntoSelection() {
    const selectedItems = this.state.selection.getSelectedItems();
    if (selectedItems.size !== 1) {
      return;
    }

    const selectedItem = selectedItems.values().next().value;
    const stagingStatus = this.state.selection.getActiveListKey();

    if (stagingStatus === 'conflicts') {
      this.showMergeConflictFileForPath(selectedItem.filePath, { activate: true });
    } else {
      await this.showFilePatchItem(selectedItem.filePath, this.state.selection.getActiveListKey(), { activate: true });
    }
  }

  async syncWithWorkspace() {
    const item = this.props.workspace.getActivePaneItem();
    if (!item) {
      return;
    }

    const realItemPromise = item.getRealItemPromise && item.getRealItemPromise();
    const realItem = await realItemPromise;
    if (!realItem) {
      return;
    }

    const isFilePatchItem = realItem.isFilePatchItem && realItem.isFilePatchItem();
    const isMatch = realItem.getWorkingDirectory && realItem.getWorkingDirectory() === this.props.workingDirectoryPath;

    if (isFilePatchItem && isMatch) {
      this.quietlySelectItem(realItem.getFilePath(), realItem.getStagingStatus());
    }
  }

  async showDiffView() {
    const selectedItems = this.state.selection.getSelectedItems();
    if (selectedItems.size !== 1) {
      return;
    }

    const selectedItem = selectedItems.values().next().value;
    const stagingStatus = this.state.selection.getActiveListKey();

    if (stagingStatus === 'conflicts') {
      this.showMergeConflictFileForPath(selectedItem.filePath);
    } else {
      await this.showFilePatchItem(selectedItem.filePath, this.state.selection.getActiveListKey());
    }
  }

  showBulkResolveMenu(event) {
    const conflictPaths = this.props.mergeConflicts.map(c => c.filePath);

    event.preventDefault();

    const menu = new Menu();

    menu.append(new MenuItem({
      label: 'Resolve All as Ours',
      click: () => this.props.resolveAsOurs(conflictPaths)
    }));

    menu.append(new MenuItem({
      label: 'Resolve All as Theirs',
      click: () => this.props.resolveAsTheirs(conflictPaths)
    }));

    menu.popup(_electron.remote.getCurrentWindow());
  }

  showActionsMenu(event) {
    event.preventDefault();

    const menu = new Menu();

    const selectedItemCount = this.state.selection.getSelectedItems().size;
    const pluralization = selectedItemCount > 1 ? 's' : '';

    menu.append(new MenuItem({
      label: 'Discard All Changes',
      click: () => this.discardAll({ eventSource: 'header-menu' }),
      enabled: this.props.unstagedChanges.length > 0
    }));

    menu.append(new MenuItem({
      label: 'Discard Changes in Selected File' + pluralization,
      click: () => this.discardChanges({ eventSource: 'header-menu' }),
      enabled: !!(this.props.unstagedChanges.length && selectedItemCount)
    }));

    menu.append(new MenuItem({
      label: 'Undo Last Discard',
      click: () => this.undoLastDiscard({ eventSource: 'header-menu' }),
      enabled: this.props.hasUndoHistory
    }));

    menu.popup(_electron.remote.getCurrentWindow());
  }

  resolveCurrentAsOurs() {
    this.props.resolveAsOurs(this.getSelectedConflictPaths());
  }

  resolveCurrentAsTheirs() {
    this.props.resolveAsTheirs(this.getSelectedConflictPaths());
  }

  // Directly modify the selection to include only the item identified by the file path and stagingStatus tuple.
  // Re-render the component, but don't notify didSelectSingleItem() or other callback functions. This is useful to
  // avoid circular callback loops for actions originating in FilePatchView or TextEditors with merge conflicts.
  quietlySelectItem(filePath, stagingStatus) {
    return new Promise(resolve => {
      this.setState(prevState => {
        const item = prevState.selection.findItem((each, key) => each.filePath === filePath && key === stagingStatus);
        if (!item) {
          // FIXME: make staging view display no selected item
          // eslint-disable-next-line no-console
          console.log(`Unable to find item at path ${filePath} with staging status ${stagingStatus}`);
          return null;
        }

        return { selection: prevState.selection.selectItem(item) };
      }, resolve);
    });
  }

  getSelectedItems() {
    const stagingStatus = this.state.selection.getActiveListKey();
    return Array.from(this.state.selection.getSelectedItems(), item => {
      return {
        filePath: item.filePath,
        stagingStatus
      };
    });
  }

  didChangeSelectedItems(openNew) {
    const selectedItems = Array.from(this.state.selection.getSelectedItems());
    if (selectedItems.length === 1) {
      this.didSelectSingleItem(selectedItems[0], openNew);
    }
  }

  async didSelectSingleItem(selectedItem, openNew = false) {
    if (!this.hasFocus()) {
      return;
    }

    if (this.state.selection.getActiveListKey() === 'conflicts') {
      if (openNew) {
        await this.showMergeConflictFileForPath(selectedItem.filePath, { activate: true });
      }
    } else {
      if (openNew) {
        // User explicitly asked to view diff, such as via click
        await this.showFilePatchItem(selectedItem.filePath, this.state.selection.getActiveListKey(), { activate: false });
      } else {
        const panesWithStaleItemsToUpdate = this.getPanesWithStalePendingFilePatchItem();
        if (panesWithStaleItemsToUpdate.length > 0) {
          // Update stale items to reflect new selection
          await Promise.all(panesWithStaleItemsToUpdate.map(async pane => {
            await this.showFilePatchItem(selectedItem.filePath, this.state.selection.getActiveListKey(), {
              activate: false,
              pane
            });
          }));
        } else {
          // Selection was changed via keyboard navigation, update pending item in active pane
          const activePane = this.props.workspace.getCenter().getActivePane();
          const activePendingItem = activePane.getPendingItem();
          const activePaneHasPendingFilePatchItem = activePendingItem && activePendingItem.getRealItem && activePendingItem.getRealItem() instanceof _changedFileItem2.default;
          if (activePaneHasPendingFilePatchItem) {
            await this.showFilePatchItem(selectedItem.filePath, this.state.selection.getActiveListKey(), {
              activate: false,
              pane: activePane
            });
          }
        }
      }
    }
  }

  getPanesWithStalePendingFilePatchItem() {
    // "stale" meaning there is no longer a changed file associated with item
    // due to changes being fully staged/unstaged/stashed/deleted/etc
    return this.props.workspace.getPanes().filter(pane => {
      const pendingItem = pane.getPendingItem();
      if (!pendingItem || !pendingItem.getRealItem) {
        return false;
      }
      const realItem = pendingItem.getRealItem();
      if (!(realItem instanceof _changedFileItem2.default)) {
        return false;
      }
      // We only want to update pending diff views for currently active repo
      const isInActiveRepo = realItem.getWorkingDirectory() === this.props.workingDirectoryPath;
      const isStale = !this.changedFileExists(realItem.getFilePath(), realItem.getStagingStatus());
      return isInActiveRepo && isStale;
    });
  }

  changedFileExists(filePath, stagingStatus) {
    return this.state.selection.findItem((item, key) => {
      return key === stagingStatus && item.filePath === filePath;
    });
  }

  async showFilePatchItem(filePath, stagingStatus, { activate, pane } = { activate: false }) {
    const uri = _changedFileItem2.default.buildURI(filePath, this.props.workingDirectoryPath, stagingStatus);
    const changedFileItem = await this.props.workspace.open(uri, { pending: true, activatePane: activate, activateItem: activate, pane });
    if (activate) {
      const itemRoot = changedFileItem.getElement();
      const focusRoot = itemRoot.querySelector('[tabIndex]');
      if (focusRoot) {
        focusRoot.focus();
      }
    } else {
      // simply make item visible
      this.props.workspace.paneForItem(changedFileItem).activateItem(changedFileItem);
    }
  }

  async showMergeConflictFileForPath(relativeFilePath, { activate } = { activate: false }) {
    const absolutePath = _path2.default.join(this.props.workingDirectoryPath, relativeFilePath);
    if (await this.fileExists(absolutePath)) {
      return this.props.workspace.open(absolutePath, { activatePane: activate, activateItem: activate, pending: true });
    } else {
      this.props.notificationManager.addInfo('File has been deleted.');
      return null;
    }
  }

  fileExists(absolutePath) {
    return new _atom.File(absolutePath).exists();
  }

  dblclickOnItem(event, item) {
    return this.props.attemptFileStageOperation([item.filePath], this.state.selection.listKeyForItem(item));
  }

  async contextMenuOnItem(event, item) {
    if (!this.state.selection.getSelectedItems().has(item)) {
      event.stopPropagation();

      event.persist();
      await new Promise(resolve => {
        this.setState(prevState => ({
          selection: prevState.selection.selectItem(item, event.shiftKey)
        }), resolve);
      });

      const newEvent = new MouseEvent(event.type, event);
      requestAnimationFrame(() => {
        if (!event.target.parentNode) {
          return;
        }
        event.target.parentNode.dispatchEvent(newEvent);
      });
    }
  }

  async mousedownOnItem(event, item) {
    const windows = process.platform === 'win32';
    if (event.ctrlKey && !windows) {
      return;
    } // simply open context menu
    if (event.button === 0) {
      this.mouseSelectionInProgress = true;

      event.persist();
      await new Promise(resolve => {
        if (event.metaKey || event.ctrlKey && windows) {
          this.setState(prevState => ({
            selection: prevState.selection.addOrSubtractSelection(item)
          }), resolve);
        } else {
          this.setState(prevState => ({
            selection: prevState.selection.selectItem(item, event.shiftKey)
          }), resolve);
        }
      });
    }
  }

  async mousemoveOnItem(event, item) {
    if (this.mouseSelectionInProgress) {
      await new Promise(resolve => {
        this.setState(prevState => ({
          selection: prevState.selection.selectItem(item, true)
        }), resolve);
      });
    }
  }

  async mouseup() {
    const hadSelectionInProgress = this.mouseSelectionInProgress;
    this.mouseSelectionInProgress = false;

    await new Promise(resolve => {
      this.setState(prevState => ({
        selection: prevState.selection.coalesce()
      }), resolve);
    });
    if (hadSelectionInProgress) {
      this.didChangeSelectedItems(true);
    }
  }

  undoLastDiscard({ eventSource } = {}) {
    if (!this.props.hasUndoHistory) {
      return;
    }

    (0, _reporterProxy.addEvent)('undo-last-discard', {
      package: 'github',
      component: 'StagingView',
      eventSource
    });

    this.props.undoLastDiscard();
  }

  getFocusClass(listKey) {
    return this.state.selection.getActiveListKey() === listKey ? 'is-focused' : '';
  }

  registerItemElement(item, element) {
    this.listElementsByItem.set(item, element);
  }

  getFocus(element) {
    return this.refRoot.map(root => root.contains(element)).getOr(false) ? StagingView.focus.STAGING : null;
  }

  setFocus(focus) {
    if (focus === this.constructor.focus.STAGING) {
      this.refRoot.map(root => root.focus());
      return true;
    }

    return false;
  }

  async advanceFocusFrom(focus) {
    if (focus === this.constructor.focus.STAGING) {
      if (await this.activateNextList()) {
        // There was a next list to activate.
        return this.constructor.focus.STAGING;
      }

      // We were already on the last list.
      return _commitView2.default.firstFocus;
    }

    return null;
  }

  async retreatFocusFrom(focus) {
    if (focus === _commitView2.default.firstFocus) {
      await this.activateLastList();
      return this.constructor.focus.STAGING;
    }

    if (focus === this.constructor.focus.STAGING) {
      await this.activatePreviousList();
      return this.constructor.focus.STAGING;
    }

    return false;
  }

  hasFocus() {
    return this.refRoot.map(root => root.contains(document.activeElement)).getOr(false);
  }

  isPopulated(props) {
    return props.workingDirectoryPath != null && (props.unstagedChanges.length > 0 || props.mergeConflicts.length > 0 || props.stagedChanges.length > 0);
  }
}
exports.default = StagingView;
StagingView.propTypes = {
  unstagedChanges: _propTypes2.default.arrayOf(_propTypes3.FilePatchItemPropType).isRequired,
  stagedChanges: _propTypes2.default.arrayOf(_propTypes3.FilePatchItemPropType).isRequired,
  mergeConflicts: _propTypes2.default.arrayOf(_propTypes3.MergeConflictItemPropType),
  workingDirectoryPath: _propTypes2.default.string,
  resolutionProgress: _propTypes2.default.object,
  hasUndoHistory: _propTypes2.default.bool.isRequired,
  commandRegistry: _propTypes2.default.object.isRequired,
  notificationManager: _propTypes2.default.object.isRequired,
  workspace: _propTypes2.default.object.isRequired,
  openFiles: _propTypes2.default.func.isRequired,
  attemptFileStageOperation: _propTypes2.default.func.isRequired,
  discardWorkDirChangesForPaths: _propTypes2.default.func.isRequired,
  undoLastDiscard: _propTypes2.default.func.isRequired,
  attemptStageAllOperation: _propTypes2.default.func.isRequired,
  resolveAsOurs: _propTypes2.default.func.isRequired,
  resolveAsTheirs: _propTypes2.default.func.isRequired
};
StagingView.defaultProps = {
  mergeConflicts: [],
  resolutionProgress: new _resolutionProgress2.default()
};
StagingView.focus = {
  STAGING: Symbol('staging')
};
StagingView.firstFocus = StagingView.focus.STAGING;
StagingView.lastFocus = StagingView.focus.STAGING;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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