'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _react = require('react');

var _react2 = _interopRequireDefault(_react);

var _propTypes = require('prop-types');

var _propTypes2 = _interopRequireDefault(_propTypes);

var _classnames = require('classnames');

var _classnames2 = _interopRequireDefault(_classnames);

var _propTypes3 = require('../prop-types');

var _tooltip = require('../atom/tooltip');

var _tooltip2 = _interopRequireDefault(_tooltip);

var _refHolder = require('../models/ref-holder');

var _refHolder2 = _interopRequireDefault(_refHolder);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function getIconClass(icon, animation) {
  return (0, _classnames2.default)('github-PushPull-icon', 'icon', `icon-${icon}`, { [`animate-${animation}`]: !!animation });
}

class PushPullView extends _react2.default.Component {

  constructor(props) {
    super(props);

    this.onClickPush = clickEvent => {
      if (this.props.isSyncing) {
        return;
      }
      this.props.push({
        force: clickEvent.metaKey || clickEvent.ctrlKey,
        setUpstream: !this.props.currentRemote.isPresent()
      });
    };

    this.onClickPull = clickEvent => {
      if (this.props.isSyncing) {
        return;
      }
      this.props.pull();
    };

    this.onClickPushPull = clickEvent => {
      if (this.props.isSyncing) {
        return;
      }
      if (clickEvent.metaKey || clickEvent.ctrlKey) {
        this.props.push({
          force: true
        });
      } else {
        this.props.pull();
      }
    };

    this.onClickPublish = clickEvent => {
      if (this.props.isSyncing) {
        return;
      }
      this.props.push({
        setUpstream: !this.props.currentRemote.isPresent()
      });
    };

    this.onClickFetch = clickEvent => {
      if (this.props.isSyncing) {
        return;
      }
      this.props.fetch();
    };

    this.refTileNode = new _refHolder2.default();
  }

  getTileStates() {
    const modKey = process.platform === 'darwin' ? 'Cmd' : 'Ctrl';
    return {
      fetching: {
        tooltip: 'Fetching from remote',
        icon: 'sync',
        text: 'Fetching',
        iconAnimation: 'rotate'
      },
      pulling: {
        tooltip: 'Pulling from remote',
        icon: 'arrow-down',
        text: 'Pulling',
        iconAnimation: 'down'
      },
      pushing: {
        tooltip: 'Pushing to remote',
        icon: 'arrow-up',
        text: 'Pushing',
        iconAnimation: 'up'
      },
      ahead: {
        onClick: this.onClickPush,
        tooltip: `Click to push<br />${modKey}-click to force push<br />Right-click for more`,
        icon: 'arrow-up',
        text: `Push ${this.props.aheadCount}`
      },
      behind: {
        onClick: this.onClickPull,
        tooltip: 'Click to pull<br />Right-click for more',
        icon: 'arrow-down',
        text: `Pull ${this.props.behindCount}`
      },
      aheadBehind: {
        onClick: this.onClickPushPull,
        tooltip: `Click to pull<br />${modKey}-click to force push<br />Right-click for more`,
        icon: 'arrow-down',
        text: `Pull ${this.props.behindCount}`,
        secondaryIcon: 'arrow-up',
        secondaryText: `${this.props.aheadCount} `
      },
      published: {
        onClick: this.onClickFetch,
        tooltip: 'Click to fetch<br />Right-click for more',
        icon: 'sync',
        text: 'Fetch'
      },
      unpublished: {
        onClick: this.onClickPublish,
        tooltip: 'Click to set up a remote tracking branch<br />Right-click for more',
        icon: 'cloud-upload',
        text: 'Publish'
      },
      noRemote: {
        tooltip: 'There is no remote named "origin"',
        icon: 'stop',
        text: 'No remote'
      },
      detached: {
        tooltip: 'Create a branch if you wish to push your work anywhere',
        icon: 'stop',
        text: 'Not on branch'
      }
    };
  }

  render() {
    const isAhead = this.props.aheadCount > 0;
    const isBehind = this.props.behindCount > 0;
    const isUnpublished = !this.props.currentRemote.isPresent();
    const isDetached = this.props.currentBranch.isDetached();
    const isFetching = this.props.isFetching;
    const isPulling = this.props.isPulling;
    const isPushing = this.props.isPushing;
    const hasOrigin = !!this.props.originExists;

    const tileStates = this.getTileStates();

    let tileState;

    if (isFetching) {
      tileState = tileStates.fetching;
    } else if (isPulling) {
      tileState = tileStates.pulling;
    } else if (isPushing) {
      tileState = tileStates.pushing;
    } else if (isAhead && !isBehind && !isUnpublished) {
      tileState = tileStates.ahead;
    } else if (isBehind && !isAhead && !isUnpublished) {
      tileState = tileStates.behind;
    } else if (isBehind && isAhead && !isUnpublished) {
      tileState = tileStates.aheadBehind;
    } else if (!isBehind && !isAhead && !isUnpublished && !isDetached) {
      tileState = tileStates.published;
    } else if (isUnpublished && !isDetached && hasOrigin) {
      tileState = tileStates.unpublished;
    } else if (isUnpublished && !isDetached && !hasOrigin) {
      tileState = tileStates.noRemote;
    } else if (isDetached) {
      tileState = tileStates.detached;
    }

    return _react2.default.createElement(
      'div',
      {
        onClick: tileState.onClick,
        ref: this.refTileNode.setter,
        className: (0, _classnames2.default)('github-PushPull', 'inline-block', { 'github-branch-detached': isDetached }) },
      tileState && _react2.default.createElement(
        _react.Fragment,
        null,
        _react2.default.createElement(
          'span',
          null,
          tileState.secondaryText && _react2.default.createElement(
            'span',
            { className: 'secondary' },
            _react2.default.createElement('span', { className: getIconClass(tileState.secondaryIcon) }),
            tileState.secondaryText
          ),
          _react2.default.createElement('span', { className: getIconClass(tileState.icon, tileState.iconAnimation) }),
          tileState.text
        ),
        _react2.default.createElement(_tooltip2.default, {
          key: 'tooltip',
          manager: this.props.tooltipManager,
          target: this.refTileNode,
          title: `<div style="text-align: left; line-height: 1.2em;">${tileState.tooltip}</div>`,
          showDelay: atom.tooltips.hoverDefaults.delay.show,
          hideDelay: atom.tooltips.hoverDefaults.delay.hide
        })
      )
    );
  }
}
exports.default = PushPullView;
PushPullView.propTypes = {
  currentBranch: _propTypes3.BranchPropType.isRequired,
  currentRemote: _propTypes3.RemotePropType.isRequired,
  isSyncing: _propTypes2.default.bool,
  isFetching: _propTypes2.default.bool,
  isPulling: _propTypes2.default.bool,
  isPushing: _propTypes2.default.bool,
  behindCount: _propTypes2.default.number,
  aheadCount: _propTypes2.default.number,
  push: _propTypes2.default.func.isRequired,
  pull: _propTypes2.default.func.isRequired,
  fetch: _propTypes2.default.func.isRequired,
  originExists: _propTypes2.default.bool,
  tooltipManager: _propTypes2.default.object.isRequired
};
PushPullView.defaultProps = {
  isSyncing: false,
  isFetching: false,
  isPulling: false,
  isPushing: false,
  behindCount: 0,
  aheadCount: 0
};
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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