(function() {
  var BufferedProcess, Client, CompositeDisposable, Emitter, PackageManager, _, createJsonParseError, createProcessError, handleProcessErrors, ref, semver,
    bind = function(fn, me){ return function(){ return fn.apply(me, arguments); }; };

  _ = require('underscore-plus');

  ref = require('atom'), BufferedProcess = ref.BufferedProcess, CompositeDisposable = ref.CompositeDisposable, Emitter = ref.Emitter;

  semver = require('semver');

  Client = require('./atom-io-client');

  module.exports = PackageManager = (function() {
    PackageManager.prototype.CACHE_EXPIRY = 1000 * 60 * 10;

    function PackageManager() {
      this.setProxyServersAsync = bind(this.setProxyServersAsync, this);
      this.setProxyServers = bind(this.setProxyServers, this);
      this.packagePromises = [];
      this.apmCache = {
        loadOutdated: {
          value: null,
          expiry: 0
        }
      };
      this.emitter = new Emitter;
    }

    PackageManager.prototype.getClient = function() {
      return this.client != null ? this.client : this.client = new Client(this);
    };

    PackageManager.prototype.isPackageInstalled = function(packageName) {
      if (atom.packages.isPackageLoaded(packageName)) {
        return true;
      } else {
        return atom.packages.getAvailablePackageNames().indexOf(packageName) > -1;
      }
    };

    PackageManager.prototype.packageHasSettings = function(packageName) {
      var grammar, grammars, i, len, pack, ref1, schema;
      grammars = (ref1 = atom.grammars.getGrammars()) != null ? ref1 : [];
      for (i = 0, len = grammars.length; i < len; i++) {
        grammar = grammars[i];
        if (grammar.path) {
          if (grammar.packageName === packageName) {
            return true;
          }
        }
      }
      pack = atom.packages.getLoadedPackage(packageName);
      if ((pack != null) && !atom.packages.isPackageActive(packageName)) {
        pack.activateConfig();
      }
      schema = atom.config.getSchema(packageName);
      return (schema != null) && (schema.type !== 'any');
    };

    PackageManager.prototype.setProxyServers = function(callback) {
      var session;
      session = atom.getCurrentWindow().webContents.session;
      return session.resolveProxy('http://atom.io', (function(_this) {
        return function(httpProxy) {
          _this.applyProxyToEnv('http_proxy', httpProxy);
          return session.resolveProxy('https://atom.io', function(httpsProxy) {
            _this.applyProxyToEnv('https_proxy', httpsProxy);
            return callback();
          });
        };
      })(this));
    };

    PackageManager.prototype.setProxyServersAsync = function(callback) {
      var httpProxyPromise, httpsProxyPromise;
      httpProxyPromise = atom.resolveProxy('http://atom.io').then((function(_this) {
        return function(proxy) {
          return _this.applyProxyToEnv('http_proxy', proxy);
        };
      })(this));
      httpsProxyPromise = atom.resolveProxy('https://atom.io').then((function(_this) {
        return function(proxy) {
          return _this.applyProxyToEnv('https_proxy', proxy);
        };
      })(this));
      return Promise.all([httpProxyPromise, httpsProxyPromise]).then(callback);
    };

    PackageManager.prototype.applyProxyToEnv = function(envName, proxy) {
      if (proxy != null) {
        proxy = proxy.split(' ');
        switch (proxy[0].trim().toUpperCase()) {
          case 'DIRECT':
            delete process.env[envName];
            break;
          case 'PROXY':
            process.env[envName] = 'http://' + proxy[1];
        }
      }
    };

    PackageManager.prototype.runCommand = function(args, callback) {
      var bufferedProcess, command, errorLines, exit, outputLines, stderr, stdout;
      command = atom.packages.getApmPath();
      outputLines = [];
      stdout = function(lines) {
        return outputLines.push(lines);
      };
      errorLines = [];
      stderr = function(lines) {
        return errorLines.push(lines);
      };
      exit = function(code) {
        return callback(code, outputLines.join('\n'), errorLines.join('\n'));
      };
      args.push('--no-color');
      if (atom.config.get('core.useProxySettingsWhenCallingApm')) {
        bufferedProcess = new BufferedProcess({
          command: command,
          args: args,
          stdout: stdout,
          stderr: stderr,
          exit: exit,
          autoStart: false
        });
        if (atom.resolveProxy != null) {
          this.setProxyServersAsync(function() {
            return bufferedProcess.start();
          });
        } else {
          this.setProxyServers(function() {
            return bufferedProcess.start();
          });
        }
        return bufferedProcess;
      } else {
        return new BufferedProcess({
          command: command,
          args: args,
          stdout: stdout,
          stderr: stderr,
          exit: exit
        });
      }
    };

    PackageManager.prototype.loadInstalled = function(callback) {
      var apmProcess, args, errorMessage;
      args = ['ls', '--json'];
      errorMessage = 'Fetching local packages failed.';
      apmProcess = this.runCommand(args, function(code, stdout, stderr) {
        var error, packages, parseError, ref1;
        if (code === 0) {
          try {
            packages = (ref1 = JSON.parse(stdout)) != null ? ref1 : [];
          } catch (error1) {
            parseError = error1;
            error = createJsonParseError(errorMessage, parseError, stdout);
            return callback(error);
          }
          return callback(null, packages);
        } else {
          error = new Error(errorMessage);
          error.stdout = stdout;
          error.stderr = stderr;
          return callback(error);
        }
      });
      return handleProcessErrors(apmProcess, errorMessage, callback);
    };

    PackageManager.prototype.loadFeatured = function(loadThemes, callback) {
      var apmProcess, args, errorMessage, version;
      if (!callback) {
        callback = loadThemes;
        loadThemes = false;
      }
      args = ['featured', '--json'];
      version = atom.getVersion();
      if (loadThemes) {
        args.push('--themes');
      }
      if (semver.valid(version)) {
        args.push('--compatible', version);
      }
      errorMessage = 'Fetching featured packages failed.';
      apmProcess = this.runCommand(args, function(code, stdout, stderr) {
        var error, packages, parseError, ref1;
        if (code === 0) {
          try {
            packages = (ref1 = JSON.parse(stdout)) != null ? ref1 : [];
          } catch (error1) {
            parseError = error1;
            error = createJsonParseError(errorMessage, parseError, stdout);
            return callback(error);
          }
          return callback(null, packages);
        } else {
          error = new Error(errorMessage);
          error.stdout = stdout;
          error.stderr = stderr;
          return callback(error);
        }
      });
      return handleProcessErrors(apmProcess, errorMessage, callback);
    };

    PackageManager.prototype.loadOutdated = function(clearCache, callback) {
      var apmProcess, args, errorMessage, version;
      if (clearCache) {
        this.clearOutdatedCache();
      } else if (this.apmCache.loadOutdated.value && this.apmCache.loadOutdated.expiry > Date.now()) {
        return callback(null, this.apmCache.loadOutdated.value);
      }
      args = ['outdated', '--json'];
      version = atom.getVersion();
      if (semver.valid(version)) {
        args.push('--compatible', version);
      }
      errorMessage = 'Fetching outdated packages and themes failed.';
      apmProcess = this.runCommand(args, (function(_this) {
        return function(code, stdout, stderr) {
          var error, i, len, pack, packages, parseError, ref1, updatablePackages;
          if (code === 0) {
            try {
              packages = (ref1 = JSON.parse(stdout)) != null ? ref1 : [];
            } catch (error1) {
              parseError = error1;
              error = createJsonParseError(errorMessage, parseError, stdout);
              return callback(error);
            }
            updatablePackages = (function() {
              var i, len, results;
              results = [];
              for (i = 0, len = packages.length; i < len; i++) {
                pack = packages[i];
                if (!this.getVersionPinnedPackages().includes(pack != null ? pack.name : void 0)) {
                  results.push(pack);
                }
              }
              return results;
            }).call(_this);
            _this.apmCache.loadOutdated = {
              value: updatablePackages,
              expiry: Date.now() + _this.CACHE_EXPIRY
            };
            for (i = 0, len = updatablePackages.length; i < len; i++) {
              pack = updatablePackages[i];
              _this.emitPackageEvent('update-available', pack);
            }
            return callback(null, updatablePackages);
          } else {
            error = new Error(errorMessage);
            error.stdout = stdout;
            error.stderr = stderr;
            return callback(error);
          }
        };
      })(this));
      return handleProcessErrors(apmProcess, errorMessage, callback);
    };

    PackageManager.prototype.getVersionPinnedPackages = function() {
      var ref1;
      return (ref1 = atom.config.get('core.versionPinnedPackages')) != null ? ref1 : [];
    };

    PackageManager.prototype.clearOutdatedCache = function() {
      return this.apmCache.loadOutdated = {
        value: null,
        expiry: 0
      };
    };

    PackageManager.prototype.loadPackage = function(packageName, callback) {
      var apmProcess, args, errorMessage;
      args = ['view', packageName, '--json'];
      errorMessage = "Fetching package '" + packageName + "' failed.";
      apmProcess = this.runCommand(args, function(code, stdout, stderr) {
        var error, packages, parseError, ref1;
        if (code === 0) {
          try {
            packages = (ref1 = JSON.parse(stdout)) != null ? ref1 : [];
          } catch (error1) {
            parseError = error1;
            error = createJsonParseError(errorMessage, parseError, stdout);
            return callback(error);
          }
          return callback(null, packages);
        } else {
          error = new Error(errorMessage);
          error.stdout = stdout;
          error.stderr = stderr;
          return callback(error);
        }
      });
      return handleProcessErrors(apmProcess, errorMessage, callback);
    };

    PackageManager.prototype.loadCompatiblePackageVersion = function(packageName, callback) {
      var apmProcess, args, errorMessage;
      args = ['view', packageName, '--json', '--compatible', this.normalizeVersion(atom.getVersion())];
      errorMessage = "Fetching package '" + packageName + "' failed.";
      apmProcess = this.runCommand(args, function(code, stdout, stderr) {
        var error, packages, parseError, ref1;
        if (code === 0) {
          try {
            packages = (ref1 = JSON.parse(stdout)) != null ? ref1 : [];
          } catch (error1) {
            parseError = error1;
            error = createJsonParseError(errorMessage, parseError, stdout);
            return callback(error);
          }
          return callback(null, packages);
        } else {
          error = new Error(errorMessage);
          error.stdout = stdout;
          error.stderr = stderr;
          return callback(error);
        }
      });
      return handleProcessErrors(apmProcess, errorMessage, callback);
    };

    PackageManager.prototype.getInstalled = function() {
      return new Promise((function(_this) {
        return function(resolve, reject) {
          return _this.loadInstalled(function(error, result) {
            if (error) {
              return reject(error);
            } else {
              return resolve(result);
            }
          });
        };
      })(this));
    };

    PackageManager.prototype.getFeatured = function(loadThemes) {
      return new Promise((function(_this) {
        return function(resolve, reject) {
          return _this.loadFeatured(!!loadThemes, function(error, result) {
            if (error) {
              return reject(error);
            } else {
              return resolve(result);
            }
          });
        };
      })(this));
    };

    PackageManager.prototype.getOutdated = function(clearCache) {
      if (clearCache == null) {
        clearCache = false;
      }
      return new Promise((function(_this) {
        return function(resolve, reject) {
          return _this.loadOutdated(clearCache, function(error, result) {
            if (error) {
              return reject(error);
            } else {
              return resolve(result);
            }
          });
        };
      })(this));
    };

    PackageManager.prototype.getPackage = function(packageName) {
      var base;
      return (base = this.packagePromises)[packageName] != null ? base[packageName] : base[packageName] = new Promise((function(_this) {
        return function(resolve, reject) {
          return _this.loadPackage(packageName, function(error, result) {
            if (error) {
              return reject(error);
            } else {
              return resolve(result);
            }
          });
        };
      })(this));
    };

    PackageManager.prototype.satisfiesVersion = function(version, metadata) {
      var engine, ref1, ref2;
      engine = (ref1 = (ref2 = metadata.engines) != null ? ref2.atom : void 0) != null ? ref1 : '*';
      if (!semver.validRange(engine)) {
        return false;
      }
      return semver.satisfies(version, engine);
    };

    PackageManager.prototype.normalizeVersion = function(version) {
      if (typeof version === 'string') {
        version = version.split('-')[0];
      }
      return version;
    };

    PackageManager.prototype.update = function(pack, newVersion, callback) {
      var apmInstallSource, apmProcess, args, errorMessage, exit, name, onError, theme;
      name = pack.name, theme = pack.theme, apmInstallSource = pack.apmInstallSource;
      errorMessage = newVersion ? "Updating to \u201C" + name + "@" + newVersion + "\u201D failed." : "Updating to latest sha failed.";
      onError = (function(_this) {
        return function(error) {
          error.packageInstallError = !theme;
          _this.emitPackageEvent('update-failed', pack, error);
          return typeof callback === "function" ? callback(error) : void 0;
        };
      })(this);
      if ((apmInstallSource != null ? apmInstallSource.type : void 0) === 'git') {
        args = ['install', apmInstallSource.source];
      } else {
        args = ['install', name + "@" + newVersion];
      }
      exit = (function(_this) {
        return function(code, stdout, stderr) {
          var error;
          if (code === 0) {
            _this.clearOutdatedCache();
            if (typeof callback === "function") {
              callback();
            }
            return _this.emitPackageEvent('updated', pack);
          } else {
            error = new Error(errorMessage);
            error.stdout = stdout;
            error.stderr = stderr;
            return onError(error);
          }
        };
      })(this);
      this.emitPackageEvent('updating', pack);
      apmProcess = this.runCommand(args, exit);
      return handleProcessErrors(apmProcess, errorMessage, onError);
    };

    PackageManager.prototype.unload = function(name) {
      if (atom.packages.isPackageLoaded(name)) {
        if (atom.packages.isPackageActive(name)) {
          atom.packages.deactivatePackage(name);
        }
        return atom.packages.unloadPackage(name);
      }
    };

    PackageManager.prototype.install = function(pack, callback) {
      var activateOnFailure, activateOnSuccess, apmProcess, args, errorMessage, exit, name, nameWithVersion, onError, theme, version;
      name = pack.name, version = pack.version, theme = pack.theme;
      activateOnSuccess = !theme && !atom.packages.isPackageDisabled(name);
      activateOnFailure = atom.packages.isPackageActive(name);
      nameWithVersion = version != null ? name + "@" + version : name;
      this.unload(name);
      args = ['install', nameWithVersion, '--json'];
      errorMessage = "Installing \u201C" + nameWithVersion + "\u201D failed.";
      onError = (function(_this) {
        return function(error) {
          error.packageInstallError = !theme;
          _this.emitPackageEvent('install-failed', pack, error);
          return typeof callback === "function" ? callback(error) : void 0;
        };
      })(this);
      exit = (function(_this) {
        return function(code, stdout, stderr) {
          var err, error, packageInfo;
          if (code === 0) {
            try {
              packageInfo = JSON.parse(stdout)[0];
              pack = _.extend({}, pack, packageInfo.metadata);
              name = pack.name;
            } catch (error1) {
              err = error1;
            }
            _this.clearOutdatedCache();
            if (activateOnSuccess) {
              atom.packages.activatePackage(name);
            } else {
              atom.packages.loadPackage(name);
            }
            if (typeof callback === "function") {
              callback();
            }
            return _this.emitPackageEvent('installed', pack);
          } else {
            if (activateOnFailure) {
              atom.packages.activatePackage(name);
            }
            error = new Error(errorMessage);
            error.stdout = stdout;
            error.stderr = stderr;
            return onError(error);
          }
        };
      })(this);
      this.emitPackageEvent('installing', pack);
      apmProcess = this.runCommand(args, exit);
      return handleProcessErrors(apmProcess, errorMessage, onError);
    };

    PackageManager.prototype.uninstall = function(pack, callback) {
      var apmProcess, errorMessage, name, onError;
      name = pack.name;
      if (atom.packages.isPackageActive(name)) {
        atom.packages.deactivatePackage(name);
      }
      errorMessage = "Uninstalling \u201C" + name + "\u201D failed.";
      onError = (function(_this) {
        return function(error) {
          _this.emitPackageEvent('uninstall-failed', pack, error);
          return typeof callback === "function" ? callback(error) : void 0;
        };
      })(this);
      this.emitPackageEvent('uninstalling', pack);
      apmProcess = this.runCommand(['uninstall', '--hard', name], (function(_this) {
        return function(code, stdout, stderr) {
          var error;
          if (code === 0) {
            _this.clearOutdatedCache();
            _this.unload(name);
            _this.removePackageNameFromDisabledPackages(name);
            if (typeof callback === "function") {
              callback();
            }
            return _this.emitPackageEvent('uninstalled', pack);
          } else {
            error = new Error(errorMessage);
            error.stdout = stdout;
            error.stderr = stderr;
            return onError(error);
          }
        };
      })(this));
      return handleProcessErrors(apmProcess, errorMessage, onError);
    };

    PackageManager.prototype.installAlternative = function(pack, alternativePackageName, callback) {
      var eventArg, installPromise, uninstallPromise;
      eventArg = {
        pack: pack,
        alternative: alternativePackageName
      };
      this.emitter.emit('package-installing-alternative', eventArg);
      uninstallPromise = new Promise((function(_this) {
        return function(resolve, reject) {
          return _this.uninstall(pack, function(error) {
            if (error) {
              return reject(error);
            } else {
              return resolve();
            }
          });
        };
      })(this));
      installPromise = new Promise((function(_this) {
        return function(resolve, reject) {
          return _this.install({
            name: alternativePackageName
          }, function(error) {
            if (error) {
              return reject(error);
            } else {
              return resolve();
            }
          });
        };
      })(this));
      return Promise.all([uninstallPromise, installPromise]).then((function(_this) {
        return function() {
          callback(null, eventArg);
          return _this.emitter.emit('package-installed-alternative', eventArg);
        };
      })(this))["catch"]((function(_this) {
        return function(error) {
          console.error(error.message, error.stack);
          callback(error, eventArg);
          eventArg.error = error;
          return _this.emitter.emit('package-install-alternative-failed', eventArg);
        };
      })(this));
    };

    PackageManager.prototype.canUpgrade = function(installedPackage, availableVersion) {
      var installedVersion;
      if (installedPackage == null) {
        return false;
      }
      installedVersion = installedPackage.metadata.version;
      if (!semver.valid(installedVersion)) {
        return false;
      }
      if (!semver.valid(availableVersion)) {
        return false;
      }
      return semver.gt(availableVersion, installedVersion);
    };

    PackageManager.prototype.getPackageTitle = function(arg) {
      var name;
      name = arg.name;
      return _.undasherize(_.uncamelcase(name));
    };

    PackageManager.prototype.getRepositoryUrl = function(arg) {
      var metadata, ref1, ref2, repoName, repoUrl, repository;
      metadata = arg.metadata;
      repository = metadata.repository;
      repoUrl = (ref1 = (ref2 = repository != null ? repository.url : void 0) != null ? ref2 : repository) != null ? ref1 : '';
      if (repoUrl.match('git@github')) {
        repoName = repoUrl.split(':')[1];
        repoUrl = "https://github.com/" + repoName;
      }
      return repoUrl.replace(/\.git$/, '').replace(/\/+$/, '').replace(/^git\+/, '');
    };

    PackageManager.prototype.checkNativeBuildTools = function() {
      return new Promise((function(_this) {
        return function(resolve, reject) {
          var apmProcess;
          apmProcess = _this.runCommand(['install', '--check'], function(code, stdout, stderr) {
            if (code === 0) {
              return resolve();
            } else {
              return reject(new Error());
            }
          });
          return apmProcess.onWillThrowError(function(arg) {
            var error, handle;
            error = arg.error, handle = arg.handle;
            handle();
            return reject(error);
          });
        };
      })(this));
    };

    PackageManager.prototype.removePackageNameFromDisabledPackages = function(packageName) {
      return atom.config.removeAtKeyPath('core.disabledPackages', packageName);
    };

    PackageManager.prototype.emitPackageEvent = function(eventName, pack, error) {
      var ref1, ref2, theme;
      theme = (ref1 = pack.theme) != null ? ref1 : (ref2 = pack.metadata) != null ? ref2.theme : void 0;
      eventName = theme ? "theme-" + eventName : "package-" + eventName;
      return this.emitter.emit(eventName, {
        pack: pack,
        error: error
      });
    };

    PackageManager.prototype.on = function(selectors, callback) {
      var i, len, ref1, selector, subscriptions;
      subscriptions = new CompositeDisposable;
      ref1 = selectors.split(" ");
      for (i = 0, len = ref1.length; i < len; i++) {
        selector = ref1[i];
        subscriptions.add(this.emitter.on(selector, callback));
      }
      return subscriptions;
    };

    return PackageManager;

  })();

  createJsonParseError = function(message, parseError, stdout) {
    var error;
    error = new Error(message);
    error.stdout = '';
    error.stderr = parseError.message + ": " + stdout;
    return error;
  };

  createProcessError = function(message, processError) {
    var error;
    error = new Error(message);
    error.stdout = '';
    error.stderr = processError.message;
    return error;
  };

  handleProcessErrors = function(apmProcess, message, callback) {
    return apmProcess.onWillThrowError(function(arg) {
      var error, handle;
      error = arg.error, handle = arg.handle;
      handle();
      return callback(createProcessError(message, error));
    });
  };

}).call(this);

//# sourceMappingURL=data:application/json;base64,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
