'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _extends = Object.assign || function (target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i]; for (var key in source) { if (Object.prototype.hasOwnProperty.call(source, key)) { target[key] = source[key]; } } } return target; };

var _path = require('path');

var _path2 = _interopRequireDefault(_path);

var _eventKit = require('event-kit');

var _fsExtra = require('fs-extra');

var _fsExtra2 = _interopRequireDefault(_fsExtra);

var _actionPipeline = require('../action-pipeline');

var _compositeGitStrategy = require('../composite-git-strategy');

var _compositeGitStrategy2 = _interopRequireDefault(_compositeGitStrategy);

var _author = require('./author');

var _author2 = _interopRequireDefault(_author);

var _branch = require('./branch');

var _branch2 = _interopRequireDefault(_branch);

var _repositoryStates = require('./repository-states');

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

const MERGE_MARKER_REGEX = /^(>|<){7} \S+$/m;

// Internal option keys used to designate the desired initial state of a Repository.
const initialStateSym = Symbol('initialState');

class Repository {
  constructor(workingDirectoryPath, gitStrategy = null, options = {}) {
    this.workingDirectoryPath = workingDirectoryPath;
    this.git = gitStrategy || _compositeGitStrategy2.default.create(workingDirectoryPath);

    this.emitter = new _eventKit.Emitter();

    this.loadPromise = new Promise(resolve => {
      const sub = this.onDidChangeState(() => {
        if (!this.isLoading()) {
          resolve();
          sub.dispose();
        } else if (this.isDestroyed()) {
          sub.dispose();
        }
      });
    });

    this.pipelineManager = options.pipelineManager || (0, _actionPipeline.getNullActionPipelineManager)();
    this.transitionTo(options[initialStateSym] || _repositoryStates.Loading);
  }

  static absent(options) {
    return new Repository(null, null, _extends({ [initialStateSym]: _repositoryStates.Absent }, options));
  }

  static loadingGuess(options) {
    return new Repository(null, null, _extends({ [initialStateSym]: _repositoryStates.LoadingGuess }, options));
  }

  static absentGuess(options) {
    return new Repository(null, null, _extends({ [initialStateSym]: _repositoryStates.AbsentGuess }, options));
  }

  // State management //////////////////////////////////////////////////////////////////////////////////////////////////

  transition(currentState, StateConstructor, ...payload) {
    if (currentState !== this.state) {
      // Attempted transition from a non-active state, most likely from an asynchronous start() method.
      return Promise.resolve();
    }

    const nextState = new StateConstructor(this, ...payload);
    this.state = nextState;

    this.emitter.emit('did-change-state', { from: currentState, to: this.state });
    if (!this.isDestroyed()) {
      this.emitter.emit('did-update');
    }

    return this.state.start();
  }

  transitionTo(StateConstructor, ...payload) {
    return this.transition(this.state, StateConstructor, ...payload);
  }

  getLoadPromise() {
    return this.isAbsent() ? Promise.reject(new Error('An absent repository will never load')) : this.loadPromise;
  }

  /*
   * Use `callback` to request user input from all git strategies.
   */
  setPromptCallback(callback) {
    this.git.getImplementers().forEach(strategy => strategy.setPromptCallback(callback));
  }

  // Pipeline
  getPipeline(actionName) {
    const actionKey = this.pipelineManager.actionKeys[actionName];
    return this.pipelineManager.getPipeline(actionKey);
  }

  executePipelineAction(actionName, fn, ...args) {
    const pipeline = this.getPipeline(actionName);
    return pipeline.run(fn, this, ...args);
  }

  // Event subscription ////////////////////////////////////////////////////////////////////////////////////////////////

  onDidDestroy(callback) {
    return this.emitter.on('did-destroy', callback);
  }

  onDidChangeState(callback) {
    return this.emitter.on('did-change-state', callback);
  }

  onDidUpdate(callback) {
    return this.emitter.on('did-update', callback);
  }

  onPullError(callback) {
    return this.emitter.on('pull-error', callback);
  }

  didPullError() {
    return this.emitter.emit('pull-error');
  }

  // State-independent actions /////////////////////////////////////////////////////////////////////////////////////////
  // Actions that use direct filesystem access or otherwise don't need `this.git` to be available.

  async pathHasMergeMarkers(relativePath) {
    try {
      const contents = await _fsExtra2.default.readFile(_path2.default.join(this.getWorkingDirectoryPath(), relativePath), { encoding: 'utf8' });
      return MERGE_MARKER_REGEX.test(contents);
    } catch (e) {
      // EISDIR implies this is a submodule
      if (e.code === 'ENOENT' || e.code === 'EISDIR') {
        return false;
      } else {
        throw e;
      }
    }
  }

  async getMergeMessage() {
    try {
      const contents = await _fsExtra2.default.readFile(_path2.default.join(this.getGitDirectoryPath(), 'MERGE_MSG'), { encoding: 'utf8' });
      return contents.split(/\n/).filter(line => line.length > 0 && !line.startsWith('#')).join('\n');
    } catch (e) {
      return null;
    }
  }

  // State-independent accessors ///////////////////////////////////////////////////////////////////////////////////////

  getWorkingDirectoryPath() {
    return this.workingDirectoryPath;
  }

  setGitDirectoryPath(gitDirectoryPath) {
    this._gitDirectoryPath = gitDirectoryPath;
  }

  getGitDirectoryPath() {
    if (this._gitDirectoryPath) {
      return this._gitDirectoryPath;
    } else if (this.getWorkingDirectoryPath()) {
      return _path2.default.join(this.getWorkingDirectoryPath(), '.git');
    } else {
      // Absent/Loading/etc.
      return null;
    }
  }

  isInState(stateName) {
    return this.state.constructor.name === stateName;
  }

  toString() {
    return `Repository(state=${this.state.constructor.name}, workdir="${this.getWorkingDirectoryPath()}")`;
  }

  // Compound Getters //////////////////////////////////////////////////////////////////////////////////////////////////
  // Accessor methods for data derived from other, state-provided getters.

  async getCurrentBranch() {
    const branches = await this.getBranches();
    const head = branches.getHeadBranch();
    if (head.isPresent()) {
      return head;
    }

    const description = await this.getHeadDescription();
    return _branch2.default.createDetached(description);
  }

  async getUnstagedChanges() {
    const { unstagedFiles } = await this.getStatusBundle();
    return Object.keys(unstagedFiles).sort().map(filePath => {
      return { filePath, status: unstagedFiles[filePath] };
    });
  }

  async getStagedChanges() {
    const { stagedFiles } = await this.getStatusBundle();
    return Object.keys(stagedFiles).sort().map(filePath => {
      return { filePath, status: stagedFiles[filePath] };
    });
  }

  async getMergeConflicts() {
    const { mergeConflictFiles } = await this.getStatusBundle();
    return Object.keys(mergeConflictFiles).map(filePath => {
      return { filePath, status: mergeConflictFiles[filePath] };
    });
  }

  async isPartiallyStaged(fileName) {
    const { unstagedFiles, stagedFiles } = await this.getStatusBundle();
    const u = unstagedFiles[fileName];
    const s = stagedFiles[fileName];
    return u === 'modified' && s === 'modified' || u === 'modified' && s === 'added' || u === 'added' && s === 'deleted' || u === 'deleted' && s === 'modified';
  }

  async getRemoteForBranch(branchName) {
    const name = await this.getConfig(`branch.${branchName}.remote`);
    return (await this.getRemotes()).withName(name);
  }

  async saveDiscardHistory() {
    if (this.isDestroyed()) {
      return;
    }

    const historySha = await this.createDiscardHistoryBlob();
    if (this.isDestroyed()) {
      return;
    }
    await this.setConfig('atomGithub.historySha', historySha);
  }

  async getCommitter(options = {}) {
    const output = await this.getConfig(['--get-regexp', '^user.*'], options);
    const committer = { name: null, email: null };
    // todo (tt, 4/2018): do we need null byte terminated output here for Windows?
    if (output) {
      output.trim().split('\n').forEach(line => {
        if (line.includes('user.email')) {
          committer.email = line.slice(11);
        } else if (line.includes('user.name')) {
          committer.name = line.slice(10);
        }
      });
    }

    return committer.name !== null && committer.email !== null ? new _author2.default(committer.email, committer.name) : _author.nullAuthor;
  }

  async hasGitHubRemote(host, owner, name) {
    const remotes = await this.getRemotes();
    return remotes.matchingGitHubRepository(owner, name).length > 0;
  }
}

exports.default = Repository; // The methods named here will be delegated to the current State.
//
// Duplicated here rather than just using `expectedDelegates` directly so that this file is grep-friendly for answering
// the question of "what all can a Repository do exactly".

const delegates = ['isLoadingGuess', 'isAbsentGuess', 'isAbsent', 'isLoading', 'isEmpty', 'isPresent', 'isTooLarge', 'isDestroyed', 'isUndetermined', 'showGitTabInit', 'showGitTabInitInProgress', 'showGitTabLoading', 'showStatusBarTiles', 'hasDirectory', 'init', 'clone', 'destroy', 'refresh', 'observeFilesystemChange', 'updateCommitMessageAfterFileSystemChange', 'stageFiles', 'unstageFiles', 'stageFilesFromParentCommit', 'stageFileModeChange', 'stageFileSymlinkChange', 'applyPatchToIndex', 'applyPatchToWorkdir', 'commit', 'merge', 'abortMerge', 'checkoutSide', 'mergeFile', 'writeMergeConflictToIndex', 'checkout', 'checkoutPathsAtRevision', 'undoLastCommit', 'fetch', 'pull', 'push', 'setConfig', 'createBlob', 'expandBlobToFile', 'createDiscardHistoryBlob', 'updateDiscardHistory', 'storeBeforeAndAfterBlobs', 'restoreLastDiscardInTempFiles', 'popDiscardHistory', 'clearDiscardHistory', 'discardWorkDirChangesForPaths', 'getStatusBundle', 'getStatusesForChangedFiles', 'getFilePatchForPath', 'getStagedChangesPatch', 'readFileFromIndex', 'getLastCommit', 'getCommit', 'getRecentCommits', 'isCommitPushed', 'getAuthors', 'getBranches', 'getHeadDescription', 'isMerging', 'isRebasing', 'getRemotes', 'addRemote', 'getAheadCount', 'getBehindCount', 'getConfig', 'unsetConfig', 'getBlobContents', 'hasDiscardHistory', 'getDiscardHistory', 'getLastHistorySnapshots', 'getOperationStates', 'setCommitMessage', 'getCommitMessage', 'fetchCommitMessageTemplate', 'getCache'];

for (let i = 0; i < delegates.length; i++) {
  const delegate = delegates[i];

  Repository.prototype[delegate] = function (...args) {
    return this.state[delegate](...args);
  };
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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