'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ConflictParser = undefined;

var _noopVisitor = require('./noop-visitor');

var _position = require('../position');

class Result {
  constructor(remainingSteps) {
    this.steps = remainingSteps;
  }

  wasSuccessful() {
    return this.steps.length === 0;
  }
}

class ConflictParser {
  constructor(adapter, visitor, isRebase) {
    this.adapter = adapter;
    this.visitor = visitor;
    this.isRebase = isRebase;

    this.lastBoundary = null;
    this.steps = [];

    if (this.isRebase) {
      this.steps.push(parser => parser.visitHeaderSide(_position.TOP, 'visitTheirSide'));
      this.steps.push(parser => parser.visitBaseAndSeparator());
      this.steps.push(parser => parser.visitFooterSide(_position.BOTTOM, 'visitOurSide'));
    } else {
      this.steps.push(parser => parser.visitHeaderSide(_position.TOP, 'visitOurSide'));
      this.steps.push(parser => parser.visitBaseAndSeparator());
      this.steps.push(parser => parser.visitFooterSide(_position.BOTTOM, 'visitTheirSide'));
    }
  }

  continueFrom(result) {
    this.steps = result.steps;
    return this.parse();
  }

  parse() {
    for (let i = 0; i < this.steps.length; i++) {
      if (!this.steps[i](this)) {
        return new Result(this.steps.slice(i));
      }
    }
    return new Result([]);
  }

  // Visit a side that begins with a banner and description as its first line.
  visitHeaderSide(position, visitMethod) {
    const sideRowStart = this.adapter.getCurrentRow();
    this.adapter.advanceRow();

    if (this.advanceToBoundary('|=') === null) {
      return false;
    }

    const sideRowEnd = this.adapter.getCurrentRow();

    this.visitor[visitMethod](position, sideRowStart, sideRowStart + 1, sideRowEnd);
    return true;
  }

  // Visit the base side from diff3 output, if one is present, then visit the separator.
  visitBaseAndSeparator() {
    if (this.lastBoundary === '|') {
      if (!this.visitBaseSide()) {
        return false;
      }
    }

    return this.visitSeparator();
  }

  // Visit a base side from diff3 output.
  visitBaseSide() {
    const sideRowStart = this.adapter.getCurrentRow();
    this.adapter.advanceRow();

    let b = this.advanceToBoundary('<=');
    if (b === null) {
      return false;
    }

    while (b === '<') {
      // Embedded recursive conflict within a base side, caused by a criss-cross merge.
      // Advance the input adapter beyond it without marking anything.
      const subParser = new ConflictParser(this.adapter, new _noopVisitor.NoopVisitor(), this.isRebase);
      if (!subParser.parse().wasSuccessful()) {
        return false;
      }

      b = this.advanceToBoundary('<=');
      if (b === null) {
        return false;
      }
    }

    const sideRowEnd = this.adapter.getCurrentRow();
    this.visitor.visitBaseSide(sideRowStart, sideRowStart + 1, sideRowEnd);
    return true;
  }

  // Visit a "========" separator.
  visitSeparator() {
    const sepRowStart = this.adapter.getCurrentRow();
    this.adapter.advanceRow();
    const sepRowEnd = this.adapter.getCurrentRow();

    this.visitor.visitSeparator(sepRowStart, sepRowEnd);
    return true;
  }

  // Visit a side with a banner and description as its last line.
  visitFooterSide(position, visitMethod) {
    const sideRowStart = this.adapter.getCurrentRow();
    if (this.advanceToBoundary('>') === null) {
      return false;
    }

    this.adapter.advanceRow();
    const sideRowEnd = this.adapter.getCurrentRow();

    this.visitor[visitMethod](position, sideRowEnd - 1, sideRowStart, sideRowEnd - 1);
    return true;
  }

  // Determine if the current row is a side boundary.
  //
  // boundaryKinds - [String] any combination of <, |, =, or > to limit the kinds of boundary detected.
  //
  // Returns the matching boundaryKinds character, or `null` if none match.
  isAtBoundary(boundaryKinds = '<|=>') {
    const line = this.adapter.getCurrentLine();
    for (let i = 0; i < boundaryKinds.length; i++) {
      const b = boundaryKinds[i];
      if (line.startsWith(b.repeat(7))) {
        return b;
      }
    }
    return null;
  }

  // Increment the current row until the current line matches one of the provided boundary kinds, or until there are no
  // more lines in the editor.
  //
  // boundaryKinds - [String] any combination of <, |, =, or > to limit the kinds of boundaries that halt the
  //   progression.
  //
  // Returns the matching boundaryKinds character, or 'null' if there are no matches to the end of the editor.
  advanceToBoundary(boundaryKinds = '<|=>') {
    let b = this.isAtBoundary(boundaryKinds);
    while (b === null) {
      this.adapter.advanceRow();
      if (this.adapter.isAtEnd()) {
        return null;
      }
      b = this.isAtBoundary(boundaryKinds);
    }

    this.lastBoundary = b;
    return b;
  }
}
exports.ConflictParser = ConflictParser;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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