'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});
/*
 * Input adapter to facilitate parsing conflicts from text loaded into an Editor.
 */
class EditorAdapter {
  constructor(editor, startRow) {
    this.editor = editor;
    this.currentRow = startRow;
  }

  getCurrentRow() {
    return this.currentRow;
  }

  getCurrentLine() {
    return this.editor.lineTextForBufferRow(this.currentRow);
  }

  advanceRow() {
    this.currentRow++;
  }

  isAtEnd() {
    return this.currentRow > this.editor.getLastBufferRow();
  }
}

exports.EditorAdapter = EditorAdapter; /*
                                        * Input adapter for parsing conflicts from a chunk of text arriving from a ReadStream.
                                        */

class ChunkAdapter {
  constructor(chunk) {
    this.chunk = chunk;

    this.lineEndRx = /\r?\n/g;

    this.lineStartPosition = 0;
    this.eof = false;
    this.advanceRow();
  }

  advanceTo(pattern) {
    if (this.eof) {
      return false;
    }

    const rx = new RegExp(pattern.source, 'gm');
    rx.lastIndex = this.lineStartPosition;

    const match = rx.exec(this.chunk);
    if (match) {
      this.lineEndRx.lastIndex = match.index;
      return true;
    } else {
      return false;
    }
  }

  getCurrentRow() {
    return undefined;
  }

  getCurrentLine() {
    return this.currentLine;
  }

  advanceRow() {
    this.lineStartPosition = this.lineEndRx.lastIndex;
    if (this.lineEndRx.test(this.chunk)) {
      this.currentLine = this.chunk.slice(this.lineStartPosition, this.lineEndRx.lastIndex);
    } else {
      this.currentLine = this.chunk.slice(this.lineStartPosition);
      this.eof = true;
    }
  }

  isAtEnd() {
    return this.eof;
  }

  getLastPartialMarker() {
    const match = /[<|>=]{1,7} ?[^\r\n]*\r?$/.exec(this.chunk);
    return match ? match[0] : '';
  }
}
exports.ChunkAdapter = ChunkAdapter;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbImFkYXB0ZXIuanMiXSwibmFtZXMiOlsiRWRpdG9yQWRhcHRlciIsImNvbnN0cnVjdG9yIiwiZWRpdG9yIiwic3RhcnRSb3ciLCJjdXJyZW50Um93IiwiZ2V0Q3VycmVudFJvdyIsImdldEN1cnJlbnRMaW5lIiwibGluZVRleHRGb3JCdWZmZXJSb3ciLCJhZHZhbmNlUm93IiwiaXNBdEVuZCIsImdldExhc3RCdWZmZXJSb3ciLCJDaHVua0FkYXB0ZXIiLCJjaHVuayIsImxpbmVFbmRSeCIsImxpbmVTdGFydFBvc2l0aW9uIiwiZW9mIiwiYWR2YW5jZVRvIiwicGF0dGVybiIsInJ4IiwiUmVnRXhwIiwic291cmNlIiwibGFzdEluZGV4IiwibWF0Y2giLCJleGVjIiwiaW5kZXgiLCJ1bmRlZmluZWQiLCJjdXJyZW50TGluZSIsInRlc3QiLCJzbGljZSIsImdldExhc3RQYXJ0aWFsTWFya2VyIl0sIm1hcHBpbmdzIjoiOzs7OztBQUFBOzs7QUFHTyxNQUFNQSxhQUFOLENBQW9CO0FBQ3pCQyxjQUFZQyxNQUFaLEVBQW9CQyxRQUFwQixFQUE4QjtBQUM1QixTQUFLRCxNQUFMLEdBQWNBLE1BQWQ7QUFDQSxTQUFLRSxVQUFMLEdBQWtCRCxRQUFsQjtBQUNEOztBQUVERSxrQkFBZ0I7QUFDZCxXQUFPLEtBQUtELFVBQVo7QUFDRDs7QUFFREUsbUJBQWlCO0FBQ2YsV0FBTyxLQUFLSixNQUFMLENBQVlLLG9CQUFaLENBQWlDLEtBQUtILFVBQXRDLENBQVA7QUFDRDs7QUFFREksZUFBYTtBQUNYLFNBQUtKLFVBQUw7QUFDRDs7QUFFREssWUFBVTtBQUNSLFdBQU8sS0FBS0wsVUFBTCxHQUFrQixLQUFLRixNQUFMLENBQVlRLGdCQUFaLEVBQXpCO0FBQ0Q7QUFwQndCOztRQUFkVixhLEdBQUFBLGEsRUF1QmI7Ozs7QUFHTyxNQUFNVyxZQUFOLENBQW1CO0FBQ3hCVixjQUFZVyxLQUFaLEVBQW1CO0FBQ2pCLFNBQUtBLEtBQUwsR0FBYUEsS0FBYjs7QUFFQSxTQUFLQyxTQUFMLEdBQWlCLFFBQWpCOztBQUVBLFNBQUtDLGlCQUFMLEdBQXlCLENBQXpCO0FBQ0EsU0FBS0MsR0FBTCxHQUFXLEtBQVg7QUFDQSxTQUFLUCxVQUFMO0FBQ0Q7O0FBRURRLFlBQVVDLE9BQVYsRUFBbUI7QUFDakIsUUFBSSxLQUFLRixHQUFULEVBQWM7QUFDWixhQUFPLEtBQVA7QUFDRDs7QUFFRCxVQUFNRyxLQUFLLElBQUlDLE1BQUosQ0FBV0YsUUFBUUcsTUFBbkIsRUFBMkIsSUFBM0IsQ0FBWDtBQUNBRixPQUFHRyxTQUFILEdBQWUsS0FBS1AsaUJBQXBCOztBQUVBLFVBQU1RLFFBQVFKLEdBQUdLLElBQUgsQ0FBUSxLQUFLWCxLQUFiLENBQWQ7QUFDQSxRQUFJVSxLQUFKLEVBQVc7QUFDVCxXQUFLVCxTQUFMLENBQWVRLFNBQWYsR0FBMkJDLE1BQU1FLEtBQWpDO0FBQ0EsYUFBTyxJQUFQO0FBQ0QsS0FIRCxNQUdPO0FBQ0wsYUFBTyxLQUFQO0FBQ0Q7QUFDRjs7QUFFRG5CLGtCQUFnQjtBQUNkLFdBQU9vQixTQUFQO0FBQ0Q7O0FBRURuQixtQkFBaUI7QUFDZixXQUFPLEtBQUtvQixXQUFaO0FBQ0Q7O0FBRURsQixlQUFhO0FBQ1gsU0FBS00saUJBQUwsR0FBeUIsS0FBS0QsU0FBTCxDQUFlUSxTQUF4QztBQUNBLFFBQUksS0FBS1IsU0FBTCxDQUFlYyxJQUFmLENBQW9CLEtBQUtmLEtBQXpCLENBQUosRUFBcUM7QUFDbkMsV0FBS2MsV0FBTCxHQUFtQixLQUFLZCxLQUFMLENBQVdnQixLQUFYLENBQWlCLEtBQUtkLGlCQUF0QixFQUF5QyxLQUFLRCxTQUFMLENBQWVRLFNBQXhELENBQW5CO0FBQ0QsS0FGRCxNQUVPO0FBQ0wsV0FBS0ssV0FBTCxHQUFtQixLQUFLZCxLQUFMLENBQVdnQixLQUFYLENBQWlCLEtBQUtkLGlCQUF0QixDQUFuQjtBQUNBLFdBQUtDLEdBQUwsR0FBVyxJQUFYO0FBQ0Q7QUFDRjs7QUFFRE4sWUFBVTtBQUNSLFdBQU8sS0FBS00sR0FBWjtBQUNEOztBQUVEYyx5QkFBdUI7QUFDckIsVUFBTVAsUUFBUSw0QkFBNEJDLElBQTVCLENBQWlDLEtBQUtYLEtBQXRDLENBQWQ7QUFDQSxXQUFPVSxRQUFRQSxNQUFNLENBQU4sQ0FBUixHQUFtQixFQUExQjtBQUNEO0FBckR1QjtRQUFiWCxZLEdBQUFBLFkiLCJmaWxlIjoiYWRhcHRlci5qcyIsInNvdXJjZVJvb3QiOiIvYnVpbGQvYXRvbS9zcmMvYXRvbS0xLjM1LjEvb3V0L2FwcC9ub2RlX21vZHVsZXMvZ2l0aHViL2xpYi9tb2RlbHMvY29uZmxpY3RzL3BhcnNlciIsInNvdXJjZXNDb250ZW50IjpbIi8qXG4gKiBJbnB1dCBhZGFwdGVyIHRvIGZhY2lsaXRhdGUgcGFyc2luZyBjb25mbGljdHMgZnJvbSB0ZXh0IGxvYWRlZCBpbnRvIGFuIEVkaXRvci5cbiAqL1xuZXhwb3J0IGNsYXNzIEVkaXRvckFkYXB0ZXIge1xuICBjb25zdHJ1Y3RvcihlZGl0b3IsIHN0YXJ0Um93KSB7XG4gICAgdGhpcy5lZGl0b3IgPSBlZGl0b3I7XG4gICAgdGhpcy5jdXJyZW50Um93ID0gc3RhcnRSb3c7XG4gIH1cblxuICBnZXRDdXJyZW50Um93KCkge1xuICAgIHJldHVybiB0aGlzLmN1cnJlbnRSb3c7XG4gIH1cblxuICBnZXRDdXJyZW50TGluZSgpIHtcbiAgICByZXR1cm4gdGhpcy5lZGl0b3IubGluZVRleHRGb3JCdWZmZXJSb3codGhpcy5jdXJyZW50Um93KTtcbiAgfVxuXG4gIGFkdmFuY2VSb3coKSB7XG4gICAgdGhpcy5jdXJyZW50Um93Kys7XG4gIH1cblxuICBpc0F0RW5kKCkge1xuICAgIHJldHVybiB0aGlzLmN1cnJlbnRSb3cgPiB0aGlzLmVkaXRvci5nZXRMYXN0QnVmZmVyUm93KCk7XG4gIH1cbn1cblxuLypcbiAqIElucHV0IGFkYXB0ZXIgZm9yIHBhcnNpbmcgY29uZmxpY3RzIGZyb20gYSBjaHVuayBvZiB0ZXh0IGFycml2aW5nIGZyb20gYSBSZWFkU3RyZWFtLlxuICovXG5leHBvcnQgY2xhc3MgQ2h1bmtBZGFwdGVyIHtcbiAgY29uc3RydWN0b3IoY2h1bmspIHtcbiAgICB0aGlzLmNodW5rID0gY2h1bms7XG5cbiAgICB0aGlzLmxpbmVFbmRSeCA9IC9cXHI/XFxuL2c7XG5cbiAgICB0aGlzLmxpbmVTdGFydFBvc2l0aW9uID0gMDtcbiAgICB0aGlzLmVvZiA9IGZhbHNlO1xuICAgIHRoaXMuYWR2YW5jZVJvdygpO1xuICB9XG5cbiAgYWR2YW5jZVRvKHBhdHRlcm4pIHtcbiAgICBpZiAodGhpcy5lb2YpIHtcbiAgICAgIHJldHVybiBmYWxzZTtcbiAgICB9XG5cbiAgICBjb25zdCByeCA9IG5ldyBSZWdFeHAocGF0dGVybi5zb3VyY2UsICdnbScpO1xuICAgIHJ4Lmxhc3RJbmRleCA9IHRoaXMubGluZVN0YXJ0UG9zaXRpb247XG5cbiAgICBjb25zdCBtYXRjaCA9IHJ4LmV4ZWModGhpcy5jaHVuayk7XG4gICAgaWYgKG1hdGNoKSB7XG4gICAgICB0aGlzLmxpbmVFbmRSeC5sYXN0SW5kZXggPSBtYXRjaC5pbmRleDtcbiAgICAgIHJldHVybiB0cnVlO1xuICAgIH0gZWxzZSB7XG4gICAgICByZXR1cm4gZmFsc2U7XG4gICAgfVxuICB9XG5cbiAgZ2V0Q3VycmVudFJvdygpIHtcbiAgICByZXR1cm4gdW5kZWZpbmVkO1xuICB9XG5cbiAgZ2V0Q3VycmVudExpbmUoKSB7XG4gICAgcmV0dXJuIHRoaXMuY3VycmVudExpbmU7XG4gIH1cblxuICBhZHZhbmNlUm93KCkge1xuICAgIHRoaXMubGluZVN0YXJ0UG9zaXRpb24gPSB0aGlzLmxpbmVFbmRSeC5sYXN0SW5kZXg7XG4gICAgaWYgKHRoaXMubGluZUVuZFJ4LnRlc3QodGhpcy5jaHVuaykpIHtcbiAgICAgIHRoaXMuY3VycmVudExpbmUgPSB0aGlzLmNodW5rLnNsaWNlKHRoaXMubGluZVN0YXJ0UG9zaXRpb24sIHRoaXMubGluZUVuZFJ4Lmxhc3RJbmRleCk7XG4gICAgfSBlbHNlIHtcbiAgICAgIHRoaXMuY3VycmVudExpbmUgPSB0aGlzLmNodW5rLnNsaWNlKHRoaXMubGluZVN0YXJ0UG9zaXRpb24pO1xuICAgICAgdGhpcy5lb2YgPSB0cnVlO1xuICAgIH1cbiAgfVxuXG4gIGlzQXRFbmQoKSB7XG4gICAgcmV0dXJuIHRoaXMuZW9mO1xuICB9XG5cbiAgZ2V0TGFzdFBhcnRpYWxNYXJrZXIoKSB7XG4gICAgY29uc3QgbWF0Y2ggPSAvWzx8Pj1dezEsN30gP1teXFxyXFxuXSpcXHI/JC8uZXhlYyh0aGlzLmNodW5rKTtcbiAgICByZXR1cm4gbWF0Y2ggPyBtYXRjaFswXSA6ICcnO1xuICB9XG59XG4iXX0=