/* SPDX-License-Identifier: GPL-2.0 */
/*
 * VirtualBox Guest Shared Folders support: module header.
 *
 * Copyright (C) 2006-2016 Oracle Corporation
 */

#ifndef VFSMOD_H
#define VFSMOD_H

#include <linux/backing-dev.h>
#include <linux/version.h>
#include "shfl_hostintf.h"

#define DIR_BUFFER_SIZE SZ_16K

/* The cast is to prevent assignment of void * to pointers of arbitrary type */
#define GET_GLOB_INFO(sb)       ((struct sf_glob_info *)(sb)->s_fs_info)
#define SET_GLOB_INFO(sb, sf_g) ((sb)->s_fs_info = (sf_g))
#define GET_INODE_INFO(i)       container_of(i, struct sf_inode_info, vfs_inode)

/* per-shared folder information */
struct sf_glob_info {
	struct shfl_fsobjinfo root_info;
	struct nls_table *nls;
	u32 root;
	/* mount options */
	struct shfl_string *name;
	char *nls_name;
	int ttl;
	int uid;
	int gid;
	int dmode;
	int fmode;
	int dmask;
	int fmask;
};

/* per-inode information */
struct sf_inode_info {
	/* some information was changed, update data on next revalidate */
	int force_restat;
	/* file structure, only valid between open() and release() */
	struct file *file;
	/*
	 * handle valid if a file was created with sf_create_aux until it
	 * will be opened with sf_reg_open()
	 */
	u64 handle;
	/* The VFS inode struct */
	struct inode vfs_inode;
};

struct sf_dir_info {
	struct list_head info_list;
};

struct sf_dir_buf {
	size_t entries;
	size_t free;
	size_t used;
	void *buf;
	struct list_head head;
};

struct sf_reg_info {
	u64 handle;
};

/* globals */
extern const struct inode_operations vboxsf_dir_iops;
extern const struct inode_operations vboxsf_lnk_iops;
extern const struct inode_operations vboxsf_reg_iops;
extern const struct file_operations vboxsf_dir_fops;
extern const struct file_operations vboxsf_reg_fops;
extern const struct address_space_operations vboxsf_reg_aops;

/* from utils.c */
void vboxsf_init_inode(struct sf_glob_info *sf_g, struct inode *inode,
		       const struct shfl_fsobjinfo *info);
int vboxsf_create_at_dentry(struct dentry *dentry,
			    struct shfl_createparms *params);
int vboxsf_stat(struct sf_glob_info *sf_g, struct shfl_string *path,
		struct shfl_fsobjinfo *info);
int vboxsf_stat_dentry(struct dentry *dentry, struct shfl_fsobjinfo *info);
int vboxsf_inode_revalidate(struct dentry *dentry);
int vboxsf_getattr(const struct path *path, struct kstat *kstat,
		   u32 request_mask, unsigned int query_flags);
int vboxsf_setattr(struct dentry *dentry, struct iattr *iattr);
struct shfl_string *vboxsf_path_from_dentry(struct sf_glob_info *sf_g,
					    struct dentry *dentry);
int vboxsf_nlscpy(struct sf_glob_info *sf_g, char *name, size_t name_bound_len,
		  const unsigned char *utf8_name, size_t utf8_len);
struct sf_dir_info *vboxsf_dir_info_alloc(void);
void vboxsf_dir_info_free(struct sf_dir_info *p);
int vboxsf_dir_read_all(struct sf_glob_info *sf_g, struct sf_dir_info *sf_d,
			u64 handle);

/* from vboxsf_wrappers.c */
int vboxsf_connect(void);
void vboxsf_disconnect(void);

int vboxsf_create(u32 root, struct shfl_string *parsed_path,
		  struct shfl_createparms *create_parms);

int vboxsf_close(u32 root, u64 handle);
int vboxsf_remove(u32 root, struct shfl_string *parsed_path, u32 flags);
int vboxsf_rename(u32 root, struct shfl_string *src_path,
		  struct shfl_string *dest_path, u32 flags);

int vboxsf_read(u32 root, u64 handle, u64 offset,
		u32 *buf_len, u8 *buf, bool user);
int vboxsf_write(u32 root, u64 handle, u64 offset,
		 u32 *buf_len, const u8 *buf, bool user);

int vboxsf_dirinfo(u32 root, u64 handle,
		   struct shfl_string *parsed_path, u32 flags, u32 index,
		   u32 *buf_len, struct shfl_dirinfo *buf, u32 *file_count);
int vboxsf_fsinfo(u32 root, u64 handle, u32 flags,
		  u32 *buf_len, void *buf);

int vboxsf_map_folder(struct shfl_string *folder_name, u32 *root);
int vboxsf_unmap_folder(u32 root);

int vboxsf_readlink(u32 root, struct shfl_string *parsed_path,
		    u32 buf_len, u8 *buf);
int vboxsf_symlink(u32 root, struct shfl_string *new_path,
		   struct shfl_string *old_path, struct shfl_fsobjinfo *buf);

int vboxsf_set_utf8(void);
int vboxsf_set_symlinks(void);

#endif
