'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _crypto = require('crypto');

var _crypto2 = _interopRequireDefault(_crypto);

var _eventKit = require('event-kit');

var _keytarStrategy = require('../shared/keytar-strategy');

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

let instance = null;

class GithubLoginModel {

  static get() {
    if (!instance) {
      instance = new GithubLoginModel();
    }
    return instance;
  }
  // Be sure that we're requesting at least this many scopes on the token we grant through github.atom.io or we'll
  // give everyone a really frustrating experience ;-)


  constructor(Strategy) {
    this._Strategy = Strategy;
    this._strategy = null;
    this.emitter = new _eventKit.Emitter();
    this.checked = new Map();
  }

  async getStrategy() {
    if (this._strategy) {
      return this._strategy;
    }

    if (this._Strategy) {
      this._strategy = new this._Strategy();
      return this._strategy;
    }

    this._strategy = await (0, _keytarStrategy.createStrategy)();
    return this._strategy;
  }

  async getToken(account) {
    const strategy = await this.getStrategy();
    const password = await strategy.getPassword('atom-github', account);
    if (!password || password === _keytarStrategy.UNAUTHENTICATED) {
      // User is not logged in
      return _keytarStrategy.UNAUTHENTICATED;
    }

    if (/^https?:\/\//.test(account)) {
      // Avoid storing tokens in memory longer than necessary. Let's cache token scope checks by storing a set of
      // checksums instead.
      const hash = _crypto2.default.createHash('md5');
      hash.update(password);
      const fingerprint = hash.digest('base64');

      const outcome = this.checked.get(fingerprint);
      if (outcome === _keytarStrategy.UNAUTHENTICATED || outcome === _keytarStrategy.INSUFFICIENT) {
        // Cached failure
        return outcome;
      } else if (!outcome) {
        // No cached outcome. Query for scopes.
        try {
          const scopes = await this.getScopes(account, password);
          if (scopes === _keytarStrategy.UNAUTHORIZED) {
            // Password is incorrect. Treat it as though you aren't authenticated at all.
            this.checked.set(fingerprint, _keytarStrategy.UNAUTHENTICATED);
            return _keytarStrategy.UNAUTHENTICATED;
          }
          const scopeSet = new Set(scopes);

          for (const scope of this.constructor.REQUIRED_SCOPES) {
            if (!scopeSet.has(scope)) {
              // Token doesn't have enough OAuth scopes, need to reauthenticate
              this.checked.set(fingerprint, _keytarStrategy.INSUFFICIENT);
              return _keytarStrategy.INSUFFICIENT;
            }
          }

          // Successfully authenticated and had all required scopes.
          this.checked.set(fingerprint, true);
        } catch (e) {
          // Most likely a network error. Do not cache the failure.
          // eslint-disable-next-line no-console
          console.error(`Unable to validate token scopes against ${account}`, e);
          return _keytarStrategy.UNAUTHENTICATED;
        }
      }
    }

    return password;
  }

  async setToken(account, token) {
    const strategy = await this.getStrategy();
    await strategy.replacePassword('atom-github', account, token);
    this.didUpdate();
  }

  async removeToken(account) {
    const strategy = await this.getStrategy();
    await strategy.deletePassword('atom-github', account);
    this.didUpdate();
  }

  /* istanbul ignore next */
  async getScopes(host, token) {
    if (atom.inSpecMode()) {
      if (token === 'good-token') {
        return this.constructor.REQUIRED_SCOPES;
      }

      throw new Error('Attempt to check token scopes in specs');
    }

    const response = await fetch(host, {
      method: 'HEAD',
      headers: { Authorization: `bearer ${token}` }
    });

    if (response.status === 401) {
      return _keytarStrategy.UNAUTHORIZED;
    }

    if (response.status !== 200) {
      throw new Error(`Unable to check token for OAuth scopes against ${host}: ${await response.text()}`);
    }

    return response.headers.get('X-OAuth-Scopes').split(/\s*,\s*/);
  }

  didUpdate() {
    this.emitter.emit('did-update');
  }

  onDidUpdate(cb) {
    return this.emitter.on('did-update', cb);
  }

  destroy() {
    this.emitter.dispose();
  }
}
exports.default = GithubLoginModel;
GithubLoginModel.REQUIRED_SCOPES = ['repo', 'read:org', 'user:email'];
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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