'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.nullCommit = undefined;

var _patch = require('./patch');

const UNBORN = Symbol('unborn');

// Truncation elipsis styles
const WORD_ELIPSES = '...';
const NEWLINE_ELIPSES = '\n...';
const PARAGRAPH_ELIPSES = '\n\n...';

class Commit {

  static createUnborn() {
    return new Commit({ unbornRef: UNBORN });
  }

  constructor({ sha, authorEmail, coAuthors, authorDate, messageSubject, messageBody, unbornRef, patch }) {
    this.sha = sha;
    this.authorEmail = authorEmail;
    this.coAuthors = coAuthors || [];
    this.authorDate = authorDate;
    this.messageSubject = messageSubject;
    this.messageBody = messageBody;
    this.unbornRef = unbornRef === UNBORN;

    this.multiFileDiff = patch ? (0, _patch.buildMultiFilePatch)(patch) : (0, _patch.buildMultiFilePatch)([]);
  }

  getSha() {
    return this.sha;
  }

  getAuthorEmail() {
    return this.authorEmail;
  }

  getAuthorDate() {
    return this.authorDate;
  }

  getCoAuthors() {
    return this.coAuthors;
  }

  getMessageSubject() {
    return this.messageSubject;
  }

  getMessageBody() {
    return this.messageBody;
  }

  isBodyLong() {
    if (this.getMessageBody().length > this.constructor.LONG_MESSAGE_THRESHOLD) {
      return true;
    }

    if ((this.getMessageBody().match(/\r?\n/g) || []).length > this.constructor.NEWLINE_THRESHOLD) {
      return true;
    }

    return false;
  }

  getFullMessage() {
    return `${this.getMessageSubject()}\n\n${this.getMessageBody()}`.trim();
  }

  /*
   * Return the messageBody truncated to at most LONG_MESSAGE_THRESHOLD characters or NEWLINE_THRESHOLD newlines,
   * whichever comes first.
   *
   * If NEWLINE_THRESHOLD newlines are encountered before LONG_MESSAGE_THRESHOLD characters, the body will be truncated
   * at the last counted newline and elipses added.
   *
   * If a paragraph boundary is found before LONG_MESSAGE_THRESHOLD characters, the message will be truncated at the end
   * of the previous paragraph and an elipses added. If no paragraph boundary is found, but a word boundary is, the text
   * is truncated at the last word boundary and an elipsis added. If neither are found, the text is truncated hard at
   * LONG_MESSAGE_THRESHOLD - 3 characters and an elipsis is added.
   */
  abbreviatedBody() {
    if (!this.isBodyLong()) {
      return this.getMessageBody();
    }

    const { LONG_MESSAGE_THRESHOLD, NEWLINE_THRESHOLD } = this.constructor;

    let lastNewlineCutoff = null;
    let lastParagraphCutoff = null;
    let lastWordCutoff = null;

    const searchText = this.getMessageBody().substring(0, LONG_MESSAGE_THRESHOLD);
    const boundaryRx = /\s+/g;
    let result;
    let lineCount = 0;
    while ((result = boundaryRx.exec(searchText)) !== null) {
      const newlineCount = (result[0].match(/\r?\n/g) || []).length;

      lineCount += newlineCount;
      if (lineCount > NEWLINE_THRESHOLD) {
        lastNewlineCutoff = result.index;
        break;
      }

      if (newlineCount < 2 && result.index <= LONG_MESSAGE_THRESHOLD - WORD_ELIPSES.length) {
        lastWordCutoff = result.index;
      } else if (result.index < LONG_MESSAGE_THRESHOLD - PARAGRAPH_ELIPSES.length) {
        lastParagraphCutoff = result.index;
      }
    }

    let elipses = WORD_ELIPSES;
    let cutoffIndex = LONG_MESSAGE_THRESHOLD - WORD_ELIPSES.length;
    if (lastNewlineCutoff !== null) {
      elipses = NEWLINE_ELIPSES;
      cutoffIndex = lastNewlineCutoff;
    } else if (lastParagraphCutoff !== null) {
      elipses = PARAGRAPH_ELIPSES;
      cutoffIndex = lastParagraphCutoff;
    } else if (lastWordCutoff !== null) {
      cutoffIndex = lastWordCutoff;
    }

    return this.getMessageBody().substring(0, cutoffIndex) + elipses;
  }

  setMultiFileDiff(multiFileDiff) {
    this.multiFileDiff = multiFileDiff;
  }

  getMultiFileDiff() {
    return this.multiFileDiff;
  }

  isUnbornRef() {
    return this.unbornRef;
  }

  isPresent() {
    return true;
  }

  isEqual(other) {
    // Directly comparable properties
    for (const property of ['sha', 'authorEmail', 'authorDate', 'messageSubject', 'messageBody', 'unbornRef']) {
      if (this[property] !== other[property]) {
        return false;
      }
    }

    // Co-author array
    if (this.coAuthors.length !== other.coAuthors.length) {
      return false;
    }
    for (let i = 0; i < this.coAuthors.length; i++) {
      const thisCoAuthor = this.coAuthors[i];
      const otherCoAuthor = other.coAuthors[i];

      if (thisCoAuthor.name !== otherCoAuthor.name || thisCoAuthor.email !== otherCoAuthor.email) {
        return false;
      }
    }

    // Multi-file patch
    if (!this.multiFileDiff.isEqual(other.multiFileDiff)) {
      return false;
    }

    return true;
  }
}

exports.default = Commit;
Commit.LONG_MESSAGE_THRESHOLD = 400;
Commit.NEWLINE_THRESHOLD = 5;
const nullCommit = exports.nullCommit = {
  getSha() {
    return '';
  },

  getMessageSubject() {
    return '';
  },

  isUnbornRef() {
    return false;
  },

  isPresent() {
    return false;
  },

  isBodyLong() {
    return false;
  }
};
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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