"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
var electron_1 = require("electron");
var path = require("path");
var config = require("../js/config");
var environment_1 = require("../js/environment");
var lifecycle = require("../js/lifecycle");
var windowManager = require("../js/window-manager");
var locale = require("../locale/locale");
var TrayHandler = (function () {
    function TrayHandler() {
        this.lastUnreadCount = 0;
    }
    TrayHandler.prototype.initTray = function (trayIcon) {
        if (trayIcon === void 0) { trayIcon = new electron_1.Tray(electron_1.nativeImage.createEmpty()); }
        var IMAGE_ROOT = path.join(electron_1.app.getAppPath(), 'img');
        var trayPng = 'tray.png';
        var trayBadgePng = 'tray.badge.png';
        if (environment_1.platform.IS_LINUX) {
            trayPng = "tray" + (environment_1.linuxDesktop.isGnome ? '.gnome' : '@3x') + ".png";
            trayBadgePng = "tray.badge" + (environment_1.linuxDesktop.isGnome ? '.gnome' : '@3x') + ".png";
        }
        var iconPaths = {
            badge: path.join(IMAGE_ROOT, 'taskbar.overlay.png'),
            tray: path.join(IMAGE_ROOT, 'tray-icon/tray', trayPng),
            trayWithBadge: path.join(IMAGE_ROOT, 'tray-icon/tray-with-badge', trayBadgePng),
        };
        this.icons = {
            badge: electron_1.nativeImage.createFromPath(iconPaths.badge),
            tray: electron_1.nativeImage.createFromPath(iconPaths.tray),
            trayWithBadge: electron_1.nativeImage.createFromPath(iconPaths.trayWithBadge),
        };
        this.trayIcon = trayIcon;
        this.trayIcon.setImage(this.icons.tray);
        this.buildTrayMenu();
    };
    TrayHandler.prototype.showUnreadCount = function (win, count) {
        this.updateIcons(win, count);
        this.flashApplicationWindow(win, count);
        this.updateBadgeCount(count);
    };
    TrayHandler.prototype.buildTrayMenu = function () {
        var contextMenu = electron_1.Menu.buildFromTemplate([
            {
                click: function () { return windowManager.showPrimaryWindow(); },
                label: locale.getText('trayOpen'),
            },
            {
                click: function () { return lifecycle.quit(); },
                label: locale.getText('trayQuit'),
            },
        ]);
        if (this.trayIcon) {
            this.trayIcon.on('click', function () { return windowManager.showPrimaryWindow(); });
            this.trayIcon.setContextMenu(contextMenu);
            this.trayIcon.setToolTip(config.NAME);
        }
    };
    TrayHandler.prototype.flashApplicationWindow = function (win, count) {
        if (win.isFocused() || !count) {
            win.flashFrame(false);
        }
        else if (count > this.lastUnreadCount) {
            win.flashFrame(true);
        }
    };
    TrayHandler.prototype.updateBadgeCount = function (count) {
        if (typeof count !== 'undefined') {
            electron_1.app.setBadgeCount(count);
            this.lastUnreadCount = count;
        }
    };
    TrayHandler.prototype.updateIcons = function (win, count) {
        if (this.icons) {
            var trayImage = count ? this.icons.trayWithBadge : this.icons.tray;
            if (this.trayIcon) {
                this.trayIcon.setImage(trayImage);
            }
            var overlayImage = count ? this.icons.badge : null;
            win.setOverlayIcon(overlayImage, locale.getText('unreadMessages'));
        }
    };
    return TrayHandler;
}());
exports.TrayHandler = TrayHandler;
//# sourceMappingURL=TrayHandler.js.map