/**
 * Copyright (c) 2013-present, Facebook, Inc.
 *
 * This source code is licensed under the MIT license found in the
 * LICENSE file in the root directory of this source tree.
 *
 * 
 * @format
 */

'use strict';

var _extends3 = _interopRequireDefault(require('babel-runtime/helpers/extends'));

var _classCallCheck3 = _interopRequireDefault(require('babel-runtime/helpers/classCallCheck'));

var _possibleConstructorReturn3 = _interopRequireDefault(require('babel-runtime/helpers/possibleConstructorReturn'));

var _inherits3 = _interopRequireDefault(require('babel-runtime/helpers/inherits'));

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { 'default': obj }; }

var _require = require('./RelayContainerUtils'),
    getComponentName = _require.getComponentName,
    getReactComponent = _require.getReactComponent;

var _require2 = require('./RelayContext'),
    assertRelayContext = _require2.assertRelayContext;

var _require3 = require('./ReactRelayContainerProfiler'),
    profileContainer = _require3.profileContainer;

var _require4 = require('relay-runtime'),
    RelayProfiler = _require4.RelayProfiler,
    isScalarAndEqual = _require4.isScalarAndEqual;

/**
 * Composes a React component class, returning a new class that intercepts
 * props, resolving them with the provided fragments and subscribing for
 * updates.
 */
function createContainerWithFragments(Component, fragments) {
  var ComponentClass = getReactComponent(Component);
  var componentName = getComponentName(Component);
  var containerName = 'Relay(' + componentName + ')';

  var Container = function (_React$Component) {
    (0, _inherits3['default'])(Container, _React$Component);

    function Container(props, context) {
      (0, _classCallCheck3['default'])(this, Container);

      var _this = (0, _possibleConstructorReturn3['default'])(this, _React$Component.call(this, props, context));

      _this._handleFragmentDataUpdate = function () {
        var profiler = RelayProfiler.profile('ReactRelayFragmentContainer.handleFragmentDataUpdate');
        var resolverFromThisUpdate = _this.state.resolver;
        _this.setState(function (updatedState) {
          // If this event belongs to the current data source, update.
          // Otherwise we should ignore it.
          if (resolverFromThisUpdate === updatedState.resolver) {
            return {
              data: updatedState.resolver.resolve(),
              relayProp: {
                isLoading: updatedState.resolver.isLoading(),
                environment: updatedState.relayProp.environment
              }
            };
          }

          return null;
        }, profiler.stop);
      };

      _this._legacyStringishRef = require('./makeLegacyStringishComponentRef')(_this, componentName);

      var relay = assertRelayContext(context.relay);
      var createFragmentSpecResolver = relay.environment.unstable_internal.createFragmentSpecResolver;
      // Do not provide a subscription/callback here.
      // It is possible for this render to be interrupted or aborted,
      // In which case the subscription would cause a leak.
      // We will add the subscription in componentDidMount().

      var resolver = createFragmentSpecResolver(relay, containerName, fragments, props);
      _this.state = {
        data: resolver.resolve(),
        relay: relay,
        relayEnvironment: context.relay.environment,
        prevProps: _this.props,
        relayVariables: context.relay.variables,
        relayProp: {
          isLoading: resolver.isLoading(),
          environment: relay.environment
        },
        resolver: resolver
      };
      return _this;
    }

    /**
     * When new props are received, read data for the new props and subscribe
     * for updates. Props may be the same in which case previous data and
     * subscriptions can be reused.
     */


    Container.getDerivedStateFromProps = function getDerivedStateFromProps(nextProps, prevState) {
      // Any props change could impact the query, so we mirror props in state.
      // This is an unusual pattern, but necessary for this container usecase.
      var prevProps = prevState.prevProps,
          relay = prevState.relay;
      var _relay$environment$un = relay.environment.unstable_internal,
          createFragmentSpecResolver = _relay$environment$un.createFragmentSpecResolver,
          getDataIDsFromObject = _relay$environment$un.getDataIDsFromObject;

      var prevIDs = getDataIDsFromObject(fragments, prevProps);
      var nextIDs = getDataIDsFromObject(fragments, nextProps);

      var resolver = prevState.resolver;

      // If the environment has changed or props point to new records then
      // previously fetched data and any pending fetches no longer apply:
      // - Existing references are on the old environment.
      // - Existing references are based on old variables.
      // - Pending fetches are for the previous records.
      if (prevState.relayEnvironment !== relay.environment || prevState.relayVariables !== relay.variables || !require('fbjs/lib/areEqual')(prevIDs, nextIDs)) {
        // Do not provide a subscription/callback here.
        // It is possible for this render to be interrupted or aborted,
        // In which case the subscription would cause a leak.
        // We will add the subscription in componentDidUpdate().
        resolver = createFragmentSpecResolver(relay, containerName, fragments, nextProps);

        return {
          data: resolver.resolve(),
          relayEnvironment: relay.environment,
          prevProps: nextProps,
          relayVariables: relay.variables,
          relayProp: {
            isLoading: resolver.isLoading(),
            environment: relay.environment
          },
          resolver: resolver
        };
      } else {
        resolver.setProps(nextProps);

        var _data = resolver.resolve();
        if (_data !== prevState.data) {
          return {
            data: _data,
            relayEnvironment: relay.environment,
            prevProps: nextProps,
            relayVariables: relay.variables,
            relayProp: {
              isLoading: resolver.isLoading(),
              environment: relay.environment
            }
          };
        }
      }

      return null;
    };

    Container.prototype.componentDidMount = function componentDidMount() {
      this._subscribeToNewResolver();
    };

    Container.prototype.componentDidUpdate = function componentDidUpdate(prevProps, prevState) {
      if (this.state.resolver !== prevState.resolver) {
        prevState.resolver.dispose();

        this._subscribeToNewResolver();
      }
    };

    Container.prototype.componentWillUnmount = function componentWillUnmount() {
      this.state.resolver.dispose();
    };

    Container.prototype.shouldComponentUpdate = function shouldComponentUpdate(nextProps, nextState) {
      // Short-circuit if any Relay-related data has changed
      if (nextState.data !== this.state.data) {
        return true;
      }
      // Otherwise, for convenience short-circuit if all non-Relay props
      // are scalar and equal
      var keys = Object.keys(nextProps);
      for (var ii = 0; ii < keys.length; ii++) {
        var _key = keys[ii];
        if (_key === 'relay') {
          if (nextState.relayEnvironment !== this.state.relayEnvironment || nextState.relayVariables !== this.state.relayVariables) {
            return true;
          }
        } else {
          if (!fragments.hasOwnProperty(_key) && !isScalarAndEqual(nextProps[_key], this.props[_key])) {
            return true;
          }
        }
      }
      return false;
    };

    /**
     * Render new data for the existing props/context.
     */


    Container.prototype._subscribeToNewResolver = function _subscribeToNewResolver() {
      var _state = this.state,
          data = _state.data,
          resolver = _state.resolver;

      // Event listeners are only safe to add during the commit phase,
      // So they won't leak if render is interrupted or errors.

      resolver.setCallback(this._handleFragmentDataUpdate);

      // External values could change between render and commit.
      // Check for this case, even though it requires an extra store read.
      var maybeNewData = resolver.resolve();
      if (data !== maybeNewData) {
        this.setState({ data: maybeNewData });
      }
    };

    Container.prototype.render = function render() {
      if (ComponentClass) {
        return require('react').createElement(ComponentClass, (0, _extends3['default'])({}, this.props, this.state.data, {
          // @TODO (T28161354) Remove the string ref fallback
          ref: this.props.componentRef || this._legacyStringishRef,
          relay: this.state.relayProp
        }));
      } else {
        // Stateless functional, doesn't support `ref`
        return require('react').createElement(Component, (0, _extends3['default'])({}, this.props, this.state.data, {
          relay: this.state.relayProp
        }));
      }
    };

    // @TODO (T28161354) Remove this once string ref usage is gone.


    return Container;
  }(require('react').Component);

  Container.displayName = containerName;
  Container.contextTypes = {
    relay: require('./RelayPropTypes').Relay
  };

  profileContainer(Container, 'ReactRelayFragmentContainer');

  return Container;
}

/**
 * Wrap the basic `createContainer()` function with logic to adapt to the
 * `context.relay.environment` in which it is rendered. Specifically, the
 * extraction of the environment-specific version of fragments in the
 * `fragmentSpec` is memoized once per environment, rather than once per
 * instance of the container constructed/rendered.
 */
function createContainer(Component, fragmentSpec) {
  return require('./buildReactRelayContainer')(Component, fragmentSpec, createContainerWithFragments);
}

module.exports = { createContainer: createContainer, createContainerWithFragments: createContainerWithFragments };