'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _path = require('path');

var _path2 = _interopRequireDefault(_path);

var _fsExtra = require('fs-extra');

var _fsExtra2 = _interopRequireDefault(_fsExtra);

var _compositeGitStrategy = require('../composite-git-strategy');

var _compositeGitStrategy2 = _interopRequireDefault(_compositeGitStrategy);

var _helpers = require('../helpers');

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/**
 * Locate the nearest git working directory above a given starting point, caching results.
 */
class WorkdirCache {
  constructor(maxSize = 1000) {
    this.maxSize = maxSize;
    this.known = new Map();
  }

  async find(startPath) {
    const cached = this.known.get(startPath);
    if (cached !== undefined) {
      return cached;
    }

    const workDir = await this.revParse(startPath);

    if (this.known.size >= this.maxSize) {
      this.known.clear();
    }
    this.known.set(startPath, workDir);

    return workDir;
  }

  invalidate() {
    this.known.clear();
  }

  async revParse(startPath) {
    try {
      const startDir = (await _fsExtra2.default.stat(startPath)).isDirectory() ? startPath : _path2.default.dirname(startPath);

      // Within a git worktree, return a non-empty string containing the path to the worktree root.
      // Within a gitdir or outside of a worktree, return an empty string.
      // Throw if startDir does not exist.
      const topLevel = await _compositeGitStrategy2.default.create(startDir).exec(['rev-parse', '--show-toplevel']);
      if (/\S/.test(topLevel)) {
        return (0, _helpers.toNativePathSep)(topLevel.trim());
      }

      // Within a gitdir, return the absolute path to the gitdir.
      // Outside of a gitdir or worktree, throw.
      const gitDir = await _compositeGitStrategy2.default.create(startDir).exec(['rev-parse', '--absolute-git-dir']);
      return this.revParse(_path2.default.resolve(gitDir, '..'));
    } catch (e) {
      /* istanbul ignore if */
      if (atom.config.get('github.reportCannotLocateWorkspaceError')) {
        // eslint-disable-next-line no-console
        console.error(`Unable to locate git workspace root for ${startPath}. Expected if ${startPath} is not in a git repository.`, e);
      }
      return null;
    }
  }
}
exports.default = WorkdirCache;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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