'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _extends = Object.assign || function (target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i]; for (var key in source) { if (Object.prototype.hasOwnProperty.call(source, key)) { target[key] = source[key]; } } } return target; };

var _atom = require('atom');

class MultiFilePatch {
  constructor({ buffer, layers, filePatches }) {
    this.buffer = buffer || null;

    this.patchLayer = layers && layers.patch;
    this.hunkLayer = layers && layers.hunk;
    this.unchangedLayer = layers && layers.unchanged;
    this.additionLayer = layers && layers.addition;
    this.deletionLayer = layers && layers.deletion;
    this.noNewlineLayer = layers && layers.noNewline;

    this.filePatches = filePatches || [];

    this.filePatchesByMarker = new Map();
    this.hunksByMarker = new Map();

    for (const filePatch of this.filePatches) {
      this.filePatchesByMarker.set(filePatch.getMarker(), filePatch);
      for (const hunk of filePatch.getHunks()) {
        this.hunksByMarker.set(hunk.getMarker(), hunk);
      }
    }
  }

  clone(opts = {}) {
    return new this.constructor({
      buffer: opts.buffer !== undefined ? opts.buffer : this.getBuffer(),
      layers: opts.layers !== undefined ? opts.layers : {
        patch: this.getPatchLayer(),
        hunk: this.getHunkLayer(),
        unchanged: this.getUnchangedLayer(),
        addition: this.getAdditionLayer(),
        deletion: this.getDeletionLayer(),
        noNewline: this.getNoNewlineLayer()
      },
      filePatches: opts.filePatches !== undefined ? opts.filePatches : this.getFilePatches()
    });
  }

  getBuffer() {
    return this.buffer;
  }

  getPatchLayer() {
    return this.patchLayer;
  }

  getHunkLayer() {
    return this.hunkLayer;
  }

  getUnchangedLayer() {
    return this.unchangedLayer;
  }

  getAdditionLayer() {
    return this.additionLayer;
  }

  getDeletionLayer() {
    return this.deletionLayer;
  }

  getNoNewlineLayer() {
    return this.noNewlineLayer;
  }

  getFilePatches() {
    return this.filePatches;
  }

  getPathSet() {
    return this.getFilePatches().reduce((pathSet, filePatch) => {
      for (const file of [filePatch.getOldFile(), filePatch.getNewFile()]) {
        if (file.isPresent()) {
          pathSet.add(file.getPath());
        }
      }
      return pathSet;
    }, new Set());
  }

  getFilePatchAt(bufferRow) {
    if (bufferRow < 0) {
      return undefined;
    }
    const [marker] = this.patchLayer.findMarkers({ intersectsRow: bufferRow });
    return this.filePatchesByMarker.get(marker);
  }

  getHunkAt(bufferRow) {
    if (bufferRow < 0) {
      return undefined;
    }
    const [marker] = this.hunkLayer.findMarkers({ intersectsRow: bufferRow });
    return this.hunksByMarker.get(marker);
  }

  getStagePatchForLines(selectedLineSet) {
    const nextLayeredBuffer = this.buildLayeredBuffer();
    const nextFilePatches = this.getFilePatchesContaining(selectedLineSet).map(fp => {
      return fp.buildStagePatchForLines(this.getBuffer(), nextLayeredBuffer, selectedLineSet);
    });
    return this.clone(_extends({}, nextLayeredBuffer, { filePatches: nextFilePatches }));
  }

  getStagePatchForHunk(hunk) {
    return this.getStagePatchForLines(new Set(hunk.getBufferRows()));
  }

  getUnstagePatchForLines(selectedLineSet) {
    const nextLayeredBuffer = this.buildLayeredBuffer();
    const nextFilePatches = this.getFilePatchesContaining(selectedLineSet).map(fp => {
      return fp.buildUnstagePatchForLines(this.getBuffer(), nextLayeredBuffer, selectedLineSet);
    });
    return this.clone(_extends({}, nextLayeredBuffer, { filePatches: nextFilePatches }));
  }

  getUnstagePatchForHunk(hunk) {
    return this.getUnstagePatchForLines(new Set(hunk.getBufferRows()));
  }

  getNextSelectionRange(lastMultiFilePatch, lastSelectedRows) {
    if (lastSelectedRows.size === 0) {
      const [firstPatch] = this.getFilePatches();
      if (!firstPatch) {
        return _atom.Range.fromObject([[0, 0], [0, 0]]);
      }

      return firstPatch.getFirstChangeRange();
    }

    const lastMax = Math.max(...lastSelectedRows);

    let lastSelectionIndex = 0;
    // counts unselected lines in changed regions from the old patch
    // until we get to the bottom-most selected line from the old patch (lastMax).
    patchLoop: for (const lastFilePatch of lastMultiFilePatch.getFilePatches()) {
      for (const hunk of lastFilePatch.getHunks()) {
        let includesMax = false;

        for (const change of hunk.getChanges()) {
          for (const _ref of change.intersectRows(lastSelectedRows, true)) {
            const { intersection, gap } = _ref;

            // Only include a partial range if this intersection includes the last selected buffer row.
            includesMax = intersection.intersectsRow(lastMax);
            const delta = includesMax ? lastMax - intersection.start.row + 1 : intersection.getRowCount();

            if (gap) {
              // Range of unselected changes.
              lastSelectionIndex += delta;
            }

            if (includesMax) {
              break patchLoop;
            }
          }
        }
      }
    }

    // Iterate over changed lines in new patch in order to find the
    // new row to be selected based on the last selection index.
    // As we walk through the changed lines, we whittle down the
    // remaining lines until we reach the row that corresponds to the
    // last selected index

    let newSelectionRow = 0;
    let remainingChangedLines = lastSelectionIndex;

    let foundRow = false;
    let lastChangedRow;

    patchLoop: for (const filePatch of this.getFilePatches()) {
      for (const hunk of filePatch.getHunks()) {
        for (const change of hunk.getChanges()) {
          if (remainingChangedLines < change.bufferRowCount()) {
            newSelectionRow = change.getStartBufferRow() + remainingChangedLines;
            foundRow = true;
            break patchLoop;
          } else {
            remainingChangedLines -= change.bufferRowCount();
            lastChangedRow = change.getEndBufferRow();
          }
        }
      }
    }

    // If we never got to the last selected index, that means it is
    // no longer present in the new patch (ie. we staged the last line of the file).
    // In this case we want the next selected line to be the last changed row in the file
    if (!foundRow) {
      newSelectionRow = lastChangedRow;
    }

    return _atom.Range.fromObject([[newSelectionRow, 0], [newSelectionRow, Infinity]]);
  }

  adoptBufferFrom(lastMultiFilePatch) {
    lastMultiFilePatch.getPatchLayer().clear();
    lastMultiFilePatch.getHunkLayer().clear();
    lastMultiFilePatch.getUnchangedLayer().clear();
    lastMultiFilePatch.getAdditionLayer().clear();
    lastMultiFilePatch.getDeletionLayer().clear();
    lastMultiFilePatch.getNoNewlineLayer().clear();

    this.filePatchesByMarker.clear();
    this.hunksByMarker.clear();

    const nextBuffer = lastMultiFilePatch.getBuffer();
    nextBuffer.setText(this.getBuffer().getText());

    for (const filePatch of this.getFilePatches()) {
      filePatch.getPatch().reMarkOn(lastMultiFilePatch.getPatchLayer());
      this.filePatchesByMarker.set(filePatch.getMarker(), filePatch);

      for (const hunk of filePatch.getHunks()) {
        hunk.reMarkOn(lastMultiFilePatch.getHunkLayer());
        this.hunksByMarker.set(hunk.getMarker(), hunk);

        for (const region of hunk.getRegions()) {
          const target = region.when({
            unchanged: () => lastMultiFilePatch.getUnchangedLayer(),
            addition: () => lastMultiFilePatch.getAdditionLayer(),
            deletion: () => lastMultiFilePatch.getDeletionLayer(),
            nonewline: () => lastMultiFilePatch.getNoNewlineLayer()
          });
          region.reMarkOn(target);
        }
      }
    }

    this.patchLayer = lastMultiFilePatch.getPatchLayer();
    this.hunkLayer = lastMultiFilePatch.getHunkLayer();
    this.unchangedLayer = lastMultiFilePatch.getUnchangedLayer();
    this.additionLayer = lastMultiFilePatch.getAdditionLayer();
    this.deletionLayer = lastMultiFilePatch.getDeletionLayer();
    this.noNewlineLayer = lastMultiFilePatch.getNoNewlineLayer();

    this.buffer = nextBuffer;
  }

  buildLayeredBuffer() {
    const buffer = new _atom.TextBuffer();
    buffer.retain();

    return {
      buffer,
      layers: {
        patch: buffer.addMarkerLayer(),
        hunk: buffer.addMarkerLayer(),
        unchanged: buffer.addMarkerLayer(),
        addition: buffer.addMarkerLayer(),
        deletion: buffer.addMarkerLayer(),
        noNewline: buffer.addMarkerLayer()
      }
    };
  }

  /*
   * Efficiently locate the FilePatch instances that contain at least one row from a Set.
   */
  getFilePatchesContaining(rowSet) {
    const sortedRowSet = Array.from(rowSet);
    sortedRowSet.sort((a, b) => a - b);

    const filePatches = [];
    let lastFilePatch = null;
    for (const row of sortedRowSet) {
      // Because the rows are sorted, consecutive rows will almost certainly belong to the same patch, so we can save
      // many avoidable marker index lookups by comparing with the last.
      if (lastFilePatch && lastFilePatch.containsRow(row)) {
        continue;
      }

      lastFilePatch = this.getFilePatchAt(row);
      filePatches.push(lastFilePatch);
    }

    return filePatches;
  }

  anyPresent() {
    return this.buffer !== null && this.filePatches.some(fp => fp.isPresent());
  }

  didAnyChangeExecutableMode() {
    for (const filePatch of this.getFilePatches()) {
      if (filePatch.didChangeExecutableMode()) {
        return true;
      }
    }
    return false;
  }

  anyHaveTypechange() {
    return this.getFilePatches().some(fp => fp.hasTypechange());
  }

  getMaxLineNumberWidth() {
    return this.getFilePatches().reduce((maxWidth, filePatch) => {
      const width = filePatch.getMaxLineNumberWidth();
      return maxWidth >= width ? maxWidth : width;
    }, 0);
  }

  spansMultipleFiles(rows) {
    let lastFilePatch = null;
    for (const row of rows) {
      if (lastFilePatch) {
        if (lastFilePatch.containsRow(row)) {
          continue;
        }

        return true;
      } else {
        lastFilePatch = this.getFilePatchAt(row);
      }
    }
    return false;
  }

  /*
   * Construct an apply-able patch String.
   */
  toString() {
    return this.filePatches.map(fp => fp.toStringIn(this.buffer)).join('');
  }

  isEqual(other) {
    return this.toString() === other.toString();
  }
}
exports.default = MultiFilePatch;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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