'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});
class Hunk {
  constructor({
    oldStartRow,
    newStartRow,
    oldRowCount,
    newRowCount,
    sectionHeading,
    marker,
    regions
  }) {
    this.oldStartRow = oldStartRow;
    this.newStartRow = newStartRow;
    this.oldRowCount = oldRowCount;
    this.newRowCount = newRowCount;
    this.sectionHeading = sectionHeading;

    this.marker = marker;
    this.regions = regions;
  }

  getOldStartRow() {
    return this.oldStartRow;
  }

  getNewStartRow() {
    return this.newStartRow;
  }

  getOldRowCount() {
    return this.oldRowCount;
  }

  getNewRowCount() {
    return this.newRowCount;
  }

  getHeader() {
    return `@@ -${this.oldStartRow},${this.oldRowCount} +${this.newStartRow},${this.newRowCount} @@`;
  }

  getSectionHeading() {
    return this.sectionHeading;
  }

  getRegions() {
    return this.regions;
  }

  getChanges() {
    return this.regions.filter(change => change.isChange());
  }

  getMarker() {
    return this.marker;
  }

  getRange() {
    return this.getMarker().getRange();
  }

  getBufferRows() {
    return this.getRange().getRows();
  }

  bufferRowCount() {
    return this.getRange().getRowCount();
  }

  includesBufferRow(row) {
    return this.getRange().intersectsRow(row);
  }

  getOldRowAt(row) {
    let current = this.oldStartRow;

    for (const region of this.getRegions()) {
      if (region.includesBufferRow(row)) {
        const offset = row - region.getStartBufferRow();

        return region.when({
          unchanged: () => current + offset,
          addition: () => null,
          deletion: () => current + offset,
          nonewline: () => null
        });
      } else {
        current += region.when({
          unchanged: () => region.bufferRowCount(),
          addition: () => 0,
          deletion: () => region.bufferRowCount(),
          nonewline: () => 0
        });
      }
    }

    return null;
  }

  getNewRowAt(row) {
    let current = this.newStartRow;

    for (const region of this.getRegions()) {
      if (region.includesBufferRow(row)) {
        const offset = row - region.getStartBufferRow();

        return region.when({
          unchanged: () => current + offset,
          addition: () => current + offset,
          deletion: () => null,
          nonewline: () => null
        });
      } else {
        current += region.when({
          unchanged: () => region.bufferRowCount(),
          addition: () => region.bufferRowCount(),
          deletion: () => 0,
          nonewline: () => 0
        });
      }
    }

    return null;
  }

  getMaxLineNumberWidth() {
    return Math.max((this.oldStartRow + this.oldRowCount).toString().length, (this.newStartRow + this.newRowCount).toString().length);
  }

  changedLineCount() {
    return this.regions.filter(region => region.isChange()).reduce((count, change) => count + change.bufferRowCount(), 0);
  }

  reMarkOn(markable) {
    this.marker = markable.markRange(this.getRange(), { invalidate: 'never', exclusive: false });
  }

  toStringIn(buffer) {
    return this.getRegions().reduce((str, region) => str + region.toStringIn(buffer), this.getHeader() + '\n');
  }
}
exports.default = Hunk;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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