'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _helpers = require('../helpers');

const COPY = Symbol('copy');

class ListSelection {
  constructor(options = {}) {
    (0, _helpers.autobind)(this, 'isItemSelectable');

    if (options._copy !== COPY) {
      this.options = {
        isItemSelectable: options.isItemSelectable || (item => !!item)
      };

      this.items = options.items || [];
      this.selections = this.items.length > 0 ? [{ head: 0, tail: 0 }] : [];
    } else {
      this.options = {
        isItemSelectable: options.isItemSelectable
      };
      this.items = options.items;
      this.selections = options.selections;
    }
  }

  copy(options = {}) {
    return new ListSelection({
      _copy: COPY,
      isItemSelectable: options.isItemSelectable || this.options.isItemSelectable,
      items: options.items || this.items,
      selections: options.selections || this.selections
    });
  }

  isItemSelectable(item) {
    return this.options.isItemSelectable(item);
  }

  setItems(items) {
    let newSelectionIndex;
    if (this.selections.length > 0) {
      const [{ head, tail }] = this.selections;
      newSelectionIndex = Math.min(head, tail, items.length - 1);
    } else {
      newSelectionIndex = 0;
    }

    const newSelections = items.length > 0 ? [{ head: newSelectionIndex, tail: newSelectionIndex }] : [];
    return this.copy({ items, selections: newSelections });
  }

  getItems() {
    return this.items;
  }

  getLastItem() {
    return this.items[this.items.length - 1];
  }

  selectFirstItem(preserveTail) {
    for (let i = 0; i < this.items.length; i++) {
      const item = this.items[i];
      if (this.isItemSelectable(item)) {
        return this.selectItem(item, preserveTail);
      }
    }
    return this;
  }

  selectLastItem(preserveTail) {
    for (let i = this.items.length - 1; i > 0; i--) {
      const item = this.items[i];
      if (this.isItemSelectable(item)) {
        return this.selectItem(item, preserveTail);
      }
    }
    return this;
  }

  selectAllItems() {
    return this.selectFirstItem().selectLastItem(true);
  }

  selectNextItem(preserveTail) {
    if (this.selections.length === 0) {
      return this.selectFirstItem();
    }

    let itemIndex = this.selections[0].head;
    let nextItemIndex = itemIndex;
    while (itemIndex < this.items.length - 1) {
      itemIndex++;
      if (this.isItemSelectable(this.items[itemIndex])) {
        nextItemIndex = itemIndex;
        break;
      }
    }

    return this.selectItem(this.items[nextItemIndex], preserveTail);
  }

  selectPreviousItem(preserveTail) {
    if (this.selections.length === 0) {
      return this.selectLastItem();
    }

    let itemIndex = this.selections[0].head;
    let previousItemIndex = itemIndex;

    while (itemIndex > 0) {
      itemIndex--;
      if (this.isItemSelectable(this.items[itemIndex])) {
        previousItemIndex = itemIndex;
        break;
      }
    }

    return this.selectItem(this.items[previousItemIndex], preserveTail);
  }

  selectItem(item, preserveTail, addOrSubtract) {
    if (addOrSubtract && preserveTail) {
      throw new Error('addOrSubtract and preserveTail cannot both be true at the same time');
    }

    const itemIndex = this.items.indexOf(item);
    if (preserveTail && this.selections[0]) {
      const newSelections = [{ head: itemIndex, tail: this.selections[0].tail, negate: this.selections[0].negate }, ...this.selections.slice(1)];
      return this.copy({ selections: newSelections });
    } else {
      const selection = { head: itemIndex, tail: itemIndex };
      if (addOrSubtract) {
        if (this.getSelectedItems().has(item)) {
          selection.negate = true;
        }
        return this.copy({ selections: [selection, ...this.selections] });
      } else {
        return this.copy({ selections: [selection] });
      }
    }
  }

  addOrSubtractSelection(item) {
    return this.selectItem(item, false, true);
  }

  coalesce() {
    if (this.selections.length === 0) {
      return this;
    }

    const mostRecent = this.selections[0];
    let mostRecentStart = Math.min(mostRecent.head, mostRecent.tail);
    let mostRecentEnd = Math.max(mostRecent.head, mostRecent.tail);
    while (mostRecentStart > 0 && !this.isItemSelectable(this.items[mostRecentStart - 1])) {
      mostRecentStart--;
    }
    while (mostRecentEnd < this.items.length - 1 && !this.isItemSelectable(this.items[mostRecentEnd + 1])) {
      mostRecentEnd++;
    }

    let changed = false;
    const newSelections = [mostRecent];
    for (let i = 1; i < this.selections.length;) {
      const current = this.selections[i];
      const currentStart = Math.min(current.head, current.tail);
      const currentEnd = Math.max(current.head, current.tail);
      if (mostRecentStart <= currentEnd + 1 && currentStart - 1 <= mostRecentEnd) {
        if (mostRecent.negate) {
          if (current.head > current.tail) {
            if (currentEnd > mostRecentEnd) {
              // suffix
              newSelections.push({ tail: mostRecentEnd + 1, head: currentEnd });
            }
            if (currentStart < mostRecentStart) {
              // prefix
              newSelections.push({ tail: currentStart, head: mostRecentStart - 1 });
            }
          } else {
            if (currentStart < mostRecentStart) {
              // prefix
              newSelections.push({ head: currentStart, tail: mostRecentStart - 1 });
            }
            if (currentEnd > mostRecentEnd) {
              // suffix
              newSelections.push({ head: mostRecentEnd + 1, tail: currentEnd });
            }
          }
          changed = true;
          i++;
        } else {
          mostRecentStart = Math.min(mostRecentStart, currentStart);
          mostRecentEnd = Math.max(mostRecentEnd, currentEnd);
          if (mostRecent.head >= mostRecent.tail) {
            mostRecent.head = mostRecentEnd;
            mostRecent.tail = mostRecentStart;
          } else {
            mostRecent.head = mostRecentStart;
            mostRecent.tail = mostRecentEnd;
          }
          changed = true;
          i++;
        }
      } else {
        newSelections.push(current);
        i++;
      }
    }

    if (mostRecent.negate) {
      changed = true;
      newSelections.shift();
    }

    return changed ? this.copy({ selections: newSelections }) : this;
  }

  getSelectedItems() {
    const selectedItems = new Set();
    for (const _ref of this.selections.slice().reverse()) {
      const { head, tail, negate } = _ref;

      const start = Math.min(head, tail);
      const end = Math.max(head, tail);
      for (let i = start; i <= end; i++) {
        const item = this.items[i];
        if (this.isItemSelectable(item)) {
          if (negate) {
            selectedItems.delete(item);
          } else {
            selectedItems.add(item);
          }
        }
      }
    }
    return selectedItems;
  }

  getHeadItem() {
    return this.selections.length > 0 ? this.items[this.selections[0].head] : null;
  }

  getMostRecentSelectionStartIndex() {
    const selection = this.selections[0];
    return Math.min(selection.head, selection.tail);
  }

  getTailIndex() {
    return this.selections[0] ? this.selections[0].tail : null;
  }
}
exports.default = ListSelection;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbImxpc3Qtc2VsZWN0aW9uLmpzIl0sIm5hbWVzIjpbIkNPUFkiLCJTeW1ib2wiLCJMaXN0U2VsZWN0aW9uIiwiY29uc3RydWN0b3IiLCJvcHRpb25zIiwiX2NvcHkiLCJpc0l0ZW1TZWxlY3RhYmxlIiwiaXRlbSIsIml0ZW1zIiwic2VsZWN0aW9ucyIsImxlbmd0aCIsImhlYWQiLCJ0YWlsIiwiY29weSIsInNldEl0ZW1zIiwibmV3U2VsZWN0aW9uSW5kZXgiLCJNYXRoIiwibWluIiwibmV3U2VsZWN0aW9ucyIsImdldEl0ZW1zIiwiZ2V0TGFzdEl0ZW0iLCJzZWxlY3RGaXJzdEl0ZW0iLCJwcmVzZXJ2ZVRhaWwiLCJpIiwic2VsZWN0SXRlbSIsInNlbGVjdExhc3RJdGVtIiwic2VsZWN0QWxsSXRlbXMiLCJzZWxlY3ROZXh0SXRlbSIsIml0ZW1JbmRleCIsIm5leHRJdGVtSW5kZXgiLCJzZWxlY3RQcmV2aW91c0l0ZW0iLCJwcmV2aW91c0l0ZW1JbmRleCIsImFkZE9yU3VidHJhY3QiLCJFcnJvciIsImluZGV4T2YiLCJuZWdhdGUiLCJzbGljZSIsInNlbGVjdGlvbiIsImdldFNlbGVjdGVkSXRlbXMiLCJoYXMiLCJhZGRPclN1YnRyYWN0U2VsZWN0aW9uIiwiY29hbGVzY2UiLCJtb3N0UmVjZW50IiwibW9zdFJlY2VudFN0YXJ0IiwibW9zdFJlY2VudEVuZCIsIm1heCIsImNoYW5nZWQiLCJjdXJyZW50IiwiY3VycmVudFN0YXJ0IiwiY3VycmVudEVuZCIsInB1c2giLCJzaGlmdCIsInNlbGVjdGVkSXRlbXMiLCJTZXQiLCJyZXZlcnNlIiwic3RhcnQiLCJlbmQiLCJkZWxldGUiLCJhZGQiLCJnZXRIZWFkSXRlbSIsImdldE1vc3RSZWNlbnRTZWxlY3Rpb25TdGFydEluZGV4IiwiZ2V0VGFpbEluZGV4Il0sIm1hcHBpbmdzIjoiOzs7Ozs7QUFBQTs7QUFFQSxNQUFNQSxPQUFPQyxPQUFPLE1BQVAsQ0FBYjs7QUFFZSxNQUFNQyxhQUFOLENBQW9CO0FBQ2pDQyxjQUFZQyxVQUFVLEVBQXRCLEVBQTBCO0FBQ3hCLDJCQUFTLElBQVQsRUFBZSxrQkFBZjs7QUFFQSxRQUFJQSxRQUFRQyxLQUFSLEtBQWtCTCxJQUF0QixFQUE0QjtBQUMxQixXQUFLSSxPQUFMLEdBQWU7QUFDYkUsMEJBQWtCRixRQUFRRSxnQkFBUixLQUE2QkMsUUFBUSxDQUFDLENBQUNBLElBQXZDO0FBREwsT0FBZjs7QUFJQSxXQUFLQyxLQUFMLEdBQWFKLFFBQVFJLEtBQVIsSUFBaUIsRUFBOUI7QUFDQSxXQUFLQyxVQUFMLEdBQWtCLEtBQUtELEtBQUwsQ0FBV0UsTUFBWCxHQUFvQixDQUFwQixHQUF3QixDQUFDLEVBQUNDLE1BQU0sQ0FBUCxFQUFVQyxNQUFNLENBQWhCLEVBQUQsQ0FBeEIsR0FBK0MsRUFBakU7QUFDRCxLQVBELE1BT087QUFDTCxXQUFLUixPQUFMLEdBQWU7QUFDYkUsMEJBQWtCRixRQUFRRTtBQURiLE9BQWY7QUFHQSxXQUFLRSxLQUFMLEdBQWFKLFFBQVFJLEtBQXJCO0FBQ0EsV0FBS0MsVUFBTCxHQUFrQkwsUUFBUUssVUFBMUI7QUFDRDtBQUNGOztBQUVESSxPQUFLVCxVQUFVLEVBQWYsRUFBbUI7QUFDakIsV0FBTyxJQUFJRixhQUFKLENBQWtCO0FBQ3ZCRyxhQUFPTCxJQURnQjtBQUV2Qk0sd0JBQWtCRixRQUFRRSxnQkFBUixJQUE0QixLQUFLRixPQUFMLENBQWFFLGdCQUZwQztBQUd2QkUsYUFBT0osUUFBUUksS0FBUixJQUFpQixLQUFLQSxLQUhOO0FBSXZCQyxrQkFBWUwsUUFBUUssVUFBUixJQUFzQixLQUFLQTtBQUpoQixLQUFsQixDQUFQO0FBTUQ7O0FBRURILG1CQUFpQkMsSUFBakIsRUFBdUI7QUFDckIsV0FBTyxLQUFLSCxPQUFMLENBQWFFLGdCQUFiLENBQThCQyxJQUE5QixDQUFQO0FBQ0Q7O0FBRURPLFdBQVNOLEtBQVQsRUFBZ0I7QUFDZCxRQUFJTyxpQkFBSjtBQUNBLFFBQUksS0FBS04sVUFBTCxDQUFnQkMsTUFBaEIsR0FBeUIsQ0FBN0IsRUFBZ0M7QUFDOUIsWUFBTSxDQUFDLEVBQUNDLElBQUQsRUFBT0MsSUFBUCxFQUFELElBQWlCLEtBQUtILFVBQTVCO0FBQ0FNLDBCQUFvQkMsS0FBS0MsR0FBTCxDQUFTTixJQUFULEVBQWVDLElBQWYsRUFBcUJKLE1BQU1FLE1BQU4sR0FBZSxDQUFwQyxDQUFwQjtBQUNELEtBSEQsTUFHTztBQUNMSywwQkFBb0IsQ0FBcEI7QUFDRDs7QUFFRCxVQUFNRyxnQkFBZ0JWLE1BQU1FLE1BQU4sR0FBZSxDQUFmLEdBQW1CLENBQUMsRUFBQ0MsTUFBTUksaUJBQVAsRUFBMEJILE1BQU1HLGlCQUFoQyxFQUFELENBQW5CLEdBQTBFLEVBQWhHO0FBQ0EsV0FBTyxLQUFLRixJQUFMLENBQVUsRUFBQ0wsS0FBRCxFQUFRQyxZQUFZUyxhQUFwQixFQUFWLENBQVA7QUFDRDs7QUFFREMsYUFBVztBQUNULFdBQU8sS0FBS1gsS0FBWjtBQUNEOztBQUVEWSxnQkFBYztBQUNaLFdBQU8sS0FBS1osS0FBTCxDQUFXLEtBQUtBLEtBQUwsQ0FBV0UsTUFBWCxHQUFvQixDQUEvQixDQUFQO0FBQ0Q7O0FBRURXLGtCQUFnQkMsWUFBaEIsRUFBOEI7QUFDNUIsU0FBSyxJQUFJQyxJQUFJLENBQWIsRUFBZ0JBLElBQUksS0FBS2YsS0FBTCxDQUFXRSxNQUEvQixFQUF1Q2EsR0FBdkMsRUFBNEM7QUFDMUMsWUFBTWhCLE9BQU8sS0FBS0MsS0FBTCxDQUFXZSxDQUFYLENBQWI7QUFDQSxVQUFJLEtBQUtqQixnQkFBTCxDQUFzQkMsSUFBdEIsQ0FBSixFQUFpQztBQUMvQixlQUFPLEtBQUtpQixVQUFMLENBQWdCakIsSUFBaEIsRUFBc0JlLFlBQXRCLENBQVA7QUFDRDtBQUNGO0FBQ0QsV0FBTyxJQUFQO0FBQ0Q7O0FBRURHLGlCQUFlSCxZQUFmLEVBQTZCO0FBQzNCLFNBQUssSUFBSUMsSUFBSSxLQUFLZixLQUFMLENBQVdFLE1BQVgsR0FBb0IsQ0FBakMsRUFBb0NhLElBQUksQ0FBeEMsRUFBMkNBLEdBQTNDLEVBQWdEO0FBQzlDLFlBQU1oQixPQUFPLEtBQUtDLEtBQUwsQ0FBV2UsQ0FBWCxDQUFiO0FBQ0EsVUFBSSxLQUFLakIsZ0JBQUwsQ0FBc0JDLElBQXRCLENBQUosRUFBaUM7QUFDL0IsZUFBTyxLQUFLaUIsVUFBTCxDQUFnQmpCLElBQWhCLEVBQXNCZSxZQUF0QixDQUFQO0FBQ0Q7QUFDRjtBQUNELFdBQU8sSUFBUDtBQUNEOztBQUVESSxtQkFBaUI7QUFDZixXQUFPLEtBQUtMLGVBQUwsR0FBdUJJLGNBQXZCLENBQXNDLElBQXRDLENBQVA7QUFDRDs7QUFFREUsaUJBQWVMLFlBQWYsRUFBNkI7QUFDM0IsUUFBSSxLQUFLYixVQUFMLENBQWdCQyxNQUFoQixLQUEyQixDQUEvQixFQUFrQztBQUNoQyxhQUFPLEtBQUtXLGVBQUwsRUFBUDtBQUNEOztBQUVELFFBQUlPLFlBQVksS0FBS25CLFVBQUwsQ0FBZ0IsQ0FBaEIsRUFBbUJFLElBQW5DO0FBQ0EsUUFBSWtCLGdCQUFnQkQsU0FBcEI7QUFDQSxXQUFPQSxZQUFZLEtBQUtwQixLQUFMLENBQVdFLE1BQVgsR0FBb0IsQ0FBdkMsRUFBMEM7QUFDeENrQjtBQUNBLFVBQUksS0FBS3RCLGdCQUFMLENBQXNCLEtBQUtFLEtBQUwsQ0FBV29CLFNBQVgsQ0FBdEIsQ0FBSixFQUFrRDtBQUNoREMsd0JBQWdCRCxTQUFoQjtBQUNBO0FBQ0Q7QUFDRjs7QUFFRCxXQUFPLEtBQUtKLFVBQUwsQ0FBZ0IsS0FBS2hCLEtBQUwsQ0FBV3FCLGFBQVgsQ0FBaEIsRUFBMkNQLFlBQTNDLENBQVA7QUFDRDs7QUFFRFEscUJBQW1CUixZQUFuQixFQUFpQztBQUMvQixRQUFJLEtBQUtiLFVBQUwsQ0FBZ0JDLE1BQWhCLEtBQTJCLENBQS9CLEVBQWtDO0FBQ2hDLGFBQU8sS0FBS2UsY0FBTCxFQUFQO0FBQ0Q7O0FBRUQsUUFBSUcsWUFBWSxLQUFLbkIsVUFBTCxDQUFnQixDQUFoQixFQUFtQkUsSUFBbkM7QUFDQSxRQUFJb0Isb0JBQW9CSCxTQUF4Qjs7QUFFQSxXQUFPQSxZQUFZLENBQW5CLEVBQXNCO0FBQ3BCQTtBQUNBLFVBQUksS0FBS3RCLGdCQUFMLENBQXNCLEtBQUtFLEtBQUwsQ0FBV29CLFNBQVgsQ0FBdEIsQ0FBSixFQUFrRDtBQUNoREcsNEJBQW9CSCxTQUFwQjtBQUNBO0FBQ0Q7QUFDRjs7QUFFRCxXQUFPLEtBQUtKLFVBQUwsQ0FBZ0IsS0FBS2hCLEtBQUwsQ0FBV3VCLGlCQUFYLENBQWhCLEVBQStDVCxZQUEvQyxDQUFQO0FBQ0Q7O0FBRURFLGFBQVdqQixJQUFYLEVBQWlCZSxZQUFqQixFQUErQlUsYUFBL0IsRUFBOEM7QUFDNUMsUUFBSUEsaUJBQWlCVixZQUFyQixFQUFtQztBQUNqQyxZQUFNLElBQUlXLEtBQUosQ0FBVSxxRUFBVixDQUFOO0FBQ0Q7O0FBRUQsVUFBTUwsWUFBWSxLQUFLcEIsS0FBTCxDQUFXMEIsT0FBWCxDQUFtQjNCLElBQW5CLENBQWxCO0FBQ0EsUUFBSWUsZ0JBQWdCLEtBQUtiLFVBQUwsQ0FBZ0IsQ0FBaEIsQ0FBcEIsRUFBd0M7QUFDdEMsWUFBTVMsZ0JBQWdCLENBQ3BCLEVBQUNQLE1BQU1pQixTQUFQLEVBQWtCaEIsTUFBTSxLQUFLSCxVQUFMLENBQWdCLENBQWhCLEVBQW1CRyxJQUEzQyxFQUFpRHVCLFFBQVEsS0FBSzFCLFVBQUwsQ0FBZ0IsQ0FBaEIsRUFBbUIwQixNQUE1RSxFQURvQixFQUVwQixHQUFHLEtBQUsxQixVQUFMLENBQWdCMkIsS0FBaEIsQ0FBc0IsQ0FBdEIsQ0FGaUIsQ0FBdEI7QUFJQSxhQUFPLEtBQUt2QixJQUFMLENBQVUsRUFBQ0osWUFBWVMsYUFBYixFQUFWLENBQVA7QUFDRCxLQU5ELE1BTU87QUFDTCxZQUFNbUIsWUFBWSxFQUFDMUIsTUFBTWlCLFNBQVAsRUFBa0JoQixNQUFNZ0IsU0FBeEIsRUFBbEI7QUFDQSxVQUFJSSxhQUFKLEVBQW1CO0FBQ2pCLFlBQUksS0FBS00sZ0JBQUwsR0FBd0JDLEdBQXhCLENBQTRCaEMsSUFBNUIsQ0FBSixFQUF1QztBQUFFOEIsb0JBQVVGLE1BQVYsR0FBbUIsSUFBbkI7QUFBMEI7QUFDbkUsZUFBTyxLQUFLdEIsSUFBTCxDQUFVLEVBQUNKLFlBQVksQ0FBQzRCLFNBQUQsRUFBWSxHQUFHLEtBQUs1QixVQUFwQixDQUFiLEVBQVYsQ0FBUDtBQUNELE9BSEQsTUFHTztBQUNMLGVBQU8sS0FBS0ksSUFBTCxDQUFVLEVBQUNKLFlBQVksQ0FBQzRCLFNBQUQsQ0FBYixFQUFWLENBQVA7QUFDRDtBQUNGO0FBQ0Y7O0FBRURHLHlCQUF1QmpDLElBQXZCLEVBQTZCO0FBQzNCLFdBQU8sS0FBS2lCLFVBQUwsQ0FBZ0JqQixJQUFoQixFQUFzQixLQUF0QixFQUE2QixJQUE3QixDQUFQO0FBQ0Q7O0FBRURrQyxhQUFXO0FBQ1QsUUFBSSxLQUFLaEMsVUFBTCxDQUFnQkMsTUFBaEIsS0FBMkIsQ0FBL0IsRUFBa0M7QUFBRSxhQUFPLElBQVA7QUFBYzs7QUFFbEQsVUFBTWdDLGFBQWEsS0FBS2pDLFVBQUwsQ0FBZ0IsQ0FBaEIsQ0FBbkI7QUFDQSxRQUFJa0Msa0JBQWtCM0IsS0FBS0MsR0FBTCxDQUFTeUIsV0FBVy9CLElBQXBCLEVBQTBCK0IsV0FBVzlCLElBQXJDLENBQXRCO0FBQ0EsUUFBSWdDLGdCQUFnQjVCLEtBQUs2QixHQUFMLENBQVNILFdBQVcvQixJQUFwQixFQUEwQitCLFdBQVc5QixJQUFyQyxDQUFwQjtBQUNBLFdBQU8rQixrQkFBa0IsQ0FBbEIsSUFBdUIsQ0FBQyxLQUFLckMsZ0JBQUwsQ0FBc0IsS0FBS0UsS0FBTCxDQUFXbUMsa0JBQWtCLENBQTdCLENBQXRCLENBQS9CLEVBQXVGO0FBQ3JGQTtBQUNEO0FBQ0QsV0FBT0MsZ0JBQWlCLEtBQUtwQyxLQUFMLENBQVdFLE1BQVgsR0FBb0IsQ0FBckMsSUFBMkMsQ0FBQyxLQUFLSixnQkFBTCxDQUFzQixLQUFLRSxLQUFMLENBQVdvQyxnQkFBZ0IsQ0FBM0IsQ0FBdEIsQ0FBbkQsRUFBeUc7QUFDdkdBO0FBQ0Q7O0FBRUQsUUFBSUUsVUFBVSxLQUFkO0FBQ0EsVUFBTTVCLGdCQUFnQixDQUFDd0IsVUFBRCxDQUF0QjtBQUNBLFNBQUssSUFBSW5CLElBQUksQ0FBYixFQUFnQkEsSUFBSSxLQUFLZCxVQUFMLENBQWdCQyxNQUFwQyxHQUE2QztBQUMzQyxZQUFNcUMsVUFBVSxLQUFLdEMsVUFBTCxDQUFnQmMsQ0FBaEIsQ0FBaEI7QUFDQSxZQUFNeUIsZUFBZWhDLEtBQUtDLEdBQUwsQ0FBUzhCLFFBQVFwQyxJQUFqQixFQUF1Qm9DLFFBQVFuQyxJQUEvQixDQUFyQjtBQUNBLFlBQU1xQyxhQUFhakMsS0FBSzZCLEdBQUwsQ0FBU0UsUUFBUXBDLElBQWpCLEVBQXVCb0MsUUFBUW5DLElBQS9CLENBQW5CO0FBQ0EsVUFBSStCLG1CQUFtQk0sYUFBYSxDQUFoQyxJQUFxQ0QsZUFBZSxDQUFmLElBQW9CSixhQUE3RCxFQUE0RTtBQUMxRSxZQUFJRixXQUFXUCxNQUFmLEVBQXVCO0FBQ3JCLGNBQUlZLFFBQVFwQyxJQUFSLEdBQWVvQyxRQUFRbkMsSUFBM0IsRUFBaUM7QUFDL0IsZ0JBQUlxQyxhQUFhTCxhQUFqQixFQUFnQztBQUFFO0FBQ2hDMUIsNEJBQWNnQyxJQUFkLENBQW1CLEVBQUN0QyxNQUFNZ0MsZ0JBQWdCLENBQXZCLEVBQTBCakMsTUFBTXNDLFVBQWhDLEVBQW5CO0FBQ0Q7QUFDRCxnQkFBSUQsZUFBZUwsZUFBbkIsRUFBb0M7QUFBRTtBQUNwQ3pCLDRCQUFjZ0MsSUFBZCxDQUFtQixFQUFDdEMsTUFBTW9DLFlBQVAsRUFBcUJyQyxNQUFNZ0Msa0JBQWtCLENBQTdDLEVBQW5CO0FBQ0Q7QUFDRixXQVBELE1BT087QUFDTCxnQkFBSUssZUFBZUwsZUFBbkIsRUFBb0M7QUFBRTtBQUNwQ3pCLDRCQUFjZ0MsSUFBZCxDQUFtQixFQUFDdkMsTUFBTXFDLFlBQVAsRUFBcUJwQyxNQUFNK0Isa0JBQWtCLENBQTdDLEVBQW5CO0FBQ0Q7QUFDRCxnQkFBSU0sYUFBYUwsYUFBakIsRUFBZ0M7QUFBRTtBQUNoQzFCLDRCQUFjZ0MsSUFBZCxDQUFtQixFQUFDdkMsTUFBTWlDLGdCQUFnQixDQUF2QixFQUEwQmhDLE1BQU1xQyxVQUFoQyxFQUFuQjtBQUNEO0FBQ0Y7QUFDREgsb0JBQVUsSUFBVjtBQUNBdkI7QUFDRCxTQWxCRCxNQWtCTztBQUNMb0IsNEJBQWtCM0IsS0FBS0MsR0FBTCxDQUFTMEIsZUFBVCxFQUEwQkssWUFBMUIsQ0FBbEI7QUFDQUosMEJBQWdCNUIsS0FBSzZCLEdBQUwsQ0FBU0QsYUFBVCxFQUF3QkssVUFBeEIsQ0FBaEI7QUFDQSxjQUFJUCxXQUFXL0IsSUFBWCxJQUFtQitCLFdBQVc5QixJQUFsQyxFQUF3QztBQUN0QzhCLHVCQUFXL0IsSUFBWCxHQUFrQmlDLGFBQWxCO0FBQ0FGLHVCQUFXOUIsSUFBWCxHQUFrQitCLGVBQWxCO0FBQ0QsV0FIRCxNQUdPO0FBQ0xELHVCQUFXL0IsSUFBWCxHQUFrQmdDLGVBQWxCO0FBQ0FELHVCQUFXOUIsSUFBWCxHQUFrQmdDLGFBQWxCO0FBQ0Q7QUFDREUsb0JBQVUsSUFBVjtBQUNBdkI7QUFDRDtBQUNGLE9BaENELE1BZ0NPO0FBQ0xMLHNCQUFjZ0MsSUFBZCxDQUFtQkgsT0FBbkI7QUFDQXhCO0FBQ0Q7QUFDRjs7QUFFRCxRQUFJbUIsV0FBV1AsTUFBZixFQUF1QjtBQUNyQlcsZ0JBQVUsSUFBVjtBQUNBNUIsb0JBQWNpQyxLQUFkO0FBQ0Q7O0FBRUQsV0FBT0wsVUFBVSxLQUFLakMsSUFBTCxDQUFVLEVBQUNKLFlBQVlTLGFBQWIsRUFBVixDQUFWLEdBQW1ELElBQTFEO0FBQ0Q7O0FBRURvQixxQkFBbUI7QUFDakIsVUFBTWMsZ0JBQWdCLElBQUlDLEdBQUosRUFBdEI7QUFDQSx1QkFBbUMsS0FBSzVDLFVBQUwsQ0FBZ0IyQixLQUFoQixHQUF3QmtCLE9BQXhCLEVBQW5DLEVBQXNFO0FBQUEsWUFBM0QsRUFBQzNDLElBQUQsRUFBT0MsSUFBUCxFQUFhdUIsTUFBYixFQUEyRDs7QUFDcEUsWUFBTW9CLFFBQVF2QyxLQUFLQyxHQUFMLENBQVNOLElBQVQsRUFBZUMsSUFBZixDQUFkO0FBQ0EsWUFBTTRDLE1BQU14QyxLQUFLNkIsR0FBTCxDQUFTbEMsSUFBVCxFQUFlQyxJQUFmLENBQVo7QUFDQSxXQUFLLElBQUlXLElBQUlnQyxLQUFiLEVBQW9CaEMsS0FBS2lDLEdBQXpCLEVBQThCakMsR0FBOUIsRUFBbUM7QUFDakMsY0FBTWhCLE9BQU8sS0FBS0MsS0FBTCxDQUFXZSxDQUFYLENBQWI7QUFDQSxZQUFJLEtBQUtqQixnQkFBTCxDQUFzQkMsSUFBdEIsQ0FBSixFQUFpQztBQUMvQixjQUFJNEIsTUFBSixFQUFZO0FBQ1ZpQiwwQkFBY0ssTUFBZCxDQUFxQmxELElBQXJCO0FBQ0QsV0FGRCxNQUVPO0FBQ0w2QywwQkFBY00sR0FBZCxDQUFrQm5ELElBQWxCO0FBQ0Q7QUFDRjtBQUNGO0FBQ0Y7QUFDRCxXQUFPNkMsYUFBUDtBQUNEOztBQUVETyxnQkFBYztBQUNaLFdBQU8sS0FBS2xELFVBQUwsQ0FBZ0JDLE1BQWhCLEdBQXlCLENBQXpCLEdBQTZCLEtBQUtGLEtBQUwsQ0FBVyxLQUFLQyxVQUFMLENBQWdCLENBQWhCLEVBQW1CRSxJQUE5QixDQUE3QixHQUFtRSxJQUExRTtBQUNEOztBQUVEaUQscUNBQW1DO0FBQ2pDLFVBQU12QixZQUFZLEtBQUs1QixVQUFMLENBQWdCLENBQWhCLENBQWxCO0FBQ0EsV0FBT08sS0FBS0MsR0FBTCxDQUFTb0IsVUFBVTFCLElBQW5CLEVBQXlCMEIsVUFBVXpCLElBQW5DLENBQVA7QUFDRDs7QUFFRGlELGlCQUFlO0FBQ2IsV0FBTyxLQUFLcEQsVUFBTCxDQUFnQixDQUFoQixJQUFxQixLQUFLQSxVQUFMLENBQWdCLENBQWhCLEVBQW1CRyxJQUF4QyxHQUErQyxJQUF0RDtBQUNEO0FBN09nQztrQkFBZFYsYSIsImZpbGUiOiJsaXN0LXNlbGVjdGlvbi5qcyIsInNvdXJjZVJvb3QiOiIvYnVpbGQvYXRvbS9zcmMvYXRvbS0xLjM1LjEvb3V0L2FwcC9ub2RlX21vZHVsZXMvZ2l0aHViL2xpYi9tb2RlbHMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQge2F1dG9iaW5kfSBmcm9tICcuLi9oZWxwZXJzJztcblxuY29uc3QgQ09QWSA9IFN5bWJvbCgnY29weScpO1xuXG5leHBvcnQgZGVmYXVsdCBjbGFzcyBMaXN0U2VsZWN0aW9uIHtcbiAgY29uc3RydWN0b3Iob3B0aW9ucyA9IHt9KSB7XG4gICAgYXV0b2JpbmQodGhpcywgJ2lzSXRlbVNlbGVjdGFibGUnKTtcblxuICAgIGlmIChvcHRpb25zLl9jb3B5ICE9PSBDT1BZKSB7XG4gICAgICB0aGlzLm9wdGlvbnMgPSB7XG4gICAgICAgIGlzSXRlbVNlbGVjdGFibGU6IG9wdGlvbnMuaXNJdGVtU2VsZWN0YWJsZSB8fCAoaXRlbSA9PiAhIWl0ZW0pLFxuICAgICAgfTtcblxuICAgICAgdGhpcy5pdGVtcyA9IG9wdGlvbnMuaXRlbXMgfHwgW107XG4gICAgICB0aGlzLnNlbGVjdGlvbnMgPSB0aGlzLml0ZW1zLmxlbmd0aCA+IDAgPyBbe2hlYWQ6IDAsIHRhaWw6IDB9XSA6IFtdO1xuICAgIH0gZWxzZSB7XG4gICAgICB0aGlzLm9wdGlvbnMgPSB7XG4gICAgICAgIGlzSXRlbVNlbGVjdGFibGU6IG9wdGlvbnMuaXNJdGVtU2VsZWN0YWJsZSxcbiAgICAgIH07XG4gICAgICB0aGlzLml0ZW1zID0gb3B0aW9ucy5pdGVtcztcbiAgICAgIHRoaXMuc2VsZWN0aW9ucyA9IG9wdGlvbnMuc2VsZWN0aW9ucztcbiAgICB9XG4gIH1cblxuICBjb3B5KG9wdGlvbnMgPSB7fSkge1xuICAgIHJldHVybiBuZXcgTGlzdFNlbGVjdGlvbih7XG4gICAgICBfY29weTogQ09QWSxcbiAgICAgIGlzSXRlbVNlbGVjdGFibGU6IG9wdGlvbnMuaXNJdGVtU2VsZWN0YWJsZSB8fCB0aGlzLm9wdGlvbnMuaXNJdGVtU2VsZWN0YWJsZSxcbiAgICAgIGl0ZW1zOiBvcHRpb25zLml0ZW1zIHx8IHRoaXMuaXRlbXMsXG4gICAgICBzZWxlY3Rpb25zOiBvcHRpb25zLnNlbGVjdGlvbnMgfHwgdGhpcy5zZWxlY3Rpb25zLFxuICAgIH0pO1xuICB9XG5cbiAgaXNJdGVtU2VsZWN0YWJsZShpdGVtKSB7XG4gICAgcmV0dXJuIHRoaXMub3B0aW9ucy5pc0l0ZW1TZWxlY3RhYmxlKGl0ZW0pO1xuICB9XG5cbiAgc2V0SXRlbXMoaXRlbXMpIHtcbiAgICBsZXQgbmV3U2VsZWN0aW9uSW5kZXg7XG4gICAgaWYgKHRoaXMuc2VsZWN0aW9ucy5sZW5ndGggPiAwKSB7XG4gICAgICBjb25zdCBbe2hlYWQsIHRhaWx9XSA9IHRoaXMuc2VsZWN0aW9ucztcbiAgICAgIG5ld1NlbGVjdGlvbkluZGV4ID0gTWF0aC5taW4oaGVhZCwgdGFpbCwgaXRlbXMubGVuZ3RoIC0gMSk7XG4gICAgfSBlbHNlIHtcbiAgICAgIG5ld1NlbGVjdGlvbkluZGV4ID0gMDtcbiAgICB9XG5cbiAgICBjb25zdCBuZXdTZWxlY3Rpb25zID0gaXRlbXMubGVuZ3RoID4gMCA/IFt7aGVhZDogbmV3U2VsZWN0aW9uSW5kZXgsIHRhaWw6IG5ld1NlbGVjdGlvbkluZGV4fV0gOiBbXTtcbiAgICByZXR1cm4gdGhpcy5jb3B5KHtpdGVtcywgc2VsZWN0aW9uczogbmV3U2VsZWN0aW9uc30pO1xuICB9XG5cbiAgZ2V0SXRlbXMoKSB7XG4gICAgcmV0dXJuIHRoaXMuaXRlbXM7XG4gIH1cblxuICBnZXRMYXN0SXRlbSgpIHtcbiAgICByZXR1cm4gdGhpcy5pdGVtc1t0aGlzLml0ZW1zLmxlbmd0aCAtIDFdO1xuICB9XG5cbiAgc2VsZWN0Rmlyc3RJdGVtKHByZXNlcnZlVGFpbCkge1xuICAgIGZvciAobGV0IGkgPSAwOyBpIDwgdGhpcy5pdGVtcy5sZW5ndGg7IGkrKykge1xuICAgICAgY29uc3QgaXRlbSA9IHRoaXMuaXRlbXNbaV07XG4gICAgICBpZiAodGhpcy5pc0l0ZW1TZWxlY3RhYmxlKGl0ZW0pKSB7XG4gICAgICAgIHJldHVybiB0aGlzLnNlbGVjdEl0ZW0oaXRlbSwgcHJlc2VydmVUYWlsKTtcbiAgICAgIH1cbiAgICB9XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICBzZWxlY3RMYXN0SXRlbShwcmVzZXJ2ZVRhaWwpIHtcbiAgICBmb3IgKGxldCBpID0gdGhpcy5pdGVtcy5sZW5ndGggLSAxOyBpID4gMDsgaS0tKSB7XG4gICAgICBjb25zdCBpdGVtID0gdGhpcy5pdGVtc1tpXTtcbiAgICAgIGlmICh0aGlzLmlzSXRlbVNlbGVjdGFibGUoaXRlbSkpIHtcbiAgICAgICAgcmV0dXJuIHRoaXMuc2VsZWN0SXRlbShpdGVtLCBwcmVzZXJ2ZVRhaWwpO1xuICAgICAgfVxuICAgIH1cbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIHNlbGVjdEFsbEl0ZW1zKCkge1xuICAgIHJldHVybiB0aGlzLnNlbGVjdEZpcnN0SXRlbSgpLnNlbGVjdExhc3RJdGVtKHRydWUpO1xuICB9XG5cbiAgc2VsZWN0TmV4dEl0ZW0ocHJlc2VydmVUYWlsKSB7XG4gICAgaWYgKHRoaXMuc2VsZWN0aW9ucy5sZW5ndGggPT09IDApIHtcbiAgICAgIHJldHVybiB0aGlzLnNlbGVjdEZpcnN0SXRlbSgpO1xuICAgIH1cblxuICAgIGxldCBpdGVtSW5kZXggPSB0aGlzLnNlbGVjdGlvbnNbMF0uaGVhZDtcbiAgICBsZXQgbmV4dEl0ZW1JbmRleCA9IGl0ZW1JbmRleDtcbiAgICB3aGlsZSAoaXRlbUluZGV4IDwgdGhpcy5pdGVtcy5sZW5ndGggLSAxKSB7XG4gICAgICBpdGVtSW5kZXgrKztcbiAgICAgIGlmICh0aGlzLmlzSXRlbVNlbGVjdGFibGUodGhpcy5pdGVtc1tpdGVtSW5kZXhdKSkge1xuICAgICAgICBuZXh0SXRlbUluZGV4ID0gaXRlbUluZGV4O1xuICAgICAgICBicmVhaztcbiAgICAgIH1cbiAgICB9XG5cbiAgICByZXR1cm4gdGhpcy5zZWxlY3RJdGVtKHRoaXMuaXRlbXNbbmV4dEl0ZW1JbmRleF0sIHByZXNlcnZlVGFpbCk7XG4gIH1cblxuICBzZWxlY3RQcmV2aW91c0l0ZW0ocHJlc2VydmVUYWlsKSB7XG4gICAgaWYgKHRoaXMuc2VsZWN0aW9ucy5sZW5ndGggPT09IDApIHtcbiAgICAgIHJldHVybiB0aGlzLnNlbGVjdExhc3RJdGVtKCk7XG4gICAgfVxuXG4gICAgbGV0IGl0ZW1JbmRleCA9IHRoaXMuc2VsZWN0aW9uc1swXS5oZWFkO1xuICAgIGxldCBwcmV2aW91c0l0ZW1JbmRleCA9IGl0ZW1JbmRleDtcblxuICAgIHdoaWxlIChpdGVtSW5kZXggPiAwKSB7XG4gICAgICBpdGVtSW5kZXgtLTtcbiAgICAgIGlmICh0aGlzLmlzSXRlbVNlbGVjdGFibGUodGhpcy5pdGVtc1tpdGVtSW5kZXhdKSkge1xuICAgICAgICBwcmV2aW91c0l0ZW1JbmRleCA9IGl0ZW1JbmRleDtcbiAgICAgICAgYnJlYWs7XG4gICAgICB9XG4gICAgfVxuXG4gICAgcmV0dXJuIHRoaXMuc2VsZWN0SXRlbSh0aGlzLml0ZW1zW3ByZXZpb3VzSXRlbUluZGV4XSwgcHJlc2VydmVUYWlsKTtcbiAgfVxuXG4gIHNlbGVjdEl0ZW0oaXRlbSwgcHJlc2VydmVUYWlsLCBhZGRPclN1YnRyYWN0KSB7XG4gICAgaWYgKGFkZE9yU3VidHJhY3QgJiYgcHJlc2VydmVUYWlsKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ2FkZE9yU3VidHJhY3QgYW5kIHByZXNlcnZlVGFpbCBjYW5ub3QgYm90aCBiZSB0cnVlIGF0IHRoZSBzYW1lIHRpbWUnKTtcbiAgICB9XG5cbiAgICBjb25zdCBpdGVtSW5kZXggPSB0aGlzLml0ZW1zLmluZGV4T2YoaXRlbSk7XG4gICAgaWYgKHByZXNlcnZlVGFpbCAmJiB0aGlzLnNlbGVjdGlvbnNbMF0pIHtcbiAgICAgIGNvbnN0IG5ld1NlbGVjdGlvbnMgPSBbXG4gICAgICAgIHtoZWFkOiBpdGVtSW5kZXgsIHRhaWw6IHRoaXMuc2VsZWN0aW9uc1swXS50YWlsLCBuZWdhdGU6IHRoaXMuc2VsZWN0aW9uc1swXS5uZWdhdGV9LFxuICAgICAgICAuLi50aGlzLnNlbGVjdGlvbnMuc2xpY2UoMSksXG4gICAgICBdO1xuICAgICAgcmV0dXJuIHRoaXMuY29weSh7c2VsZWN0aW9uczogbmV3U2VsZWN0aW9uc30pO1xuICAgIH0gZWxzZSB7XG4gICAgICBjb25zdCBzZWxlY3Rpb24gPSB7aGVhZDogaXRlbUluZGV4LCB0YWlsOiBpdGVtSW5kZXh9O1xuICAgICAgaWYgKGFkZE9yU3VidHJhY3QpIHtcbiAgICAgICAgaWYgKHRoaXMuZ2V0U2VsZWN0ZWRJdGVtcygpLmhhcyhpdGVtKSkgeyBzZWxlY3Rpb24ubmVnYXRlID0gdHJ1ZTsgfVxuICAgICAgICByZXR1cm4gdGhpcy5jb3B5KHtzZWxlY3Rpb25zOiBbc2VsZWN0aW9uLCAuLi50aGlzLnNlbGVjdGlvbnNdfSk7XG4gICAgICB9IGVsc2Uge1xuICAgICAgICByZXR1cm4gdGhpcy5jb3B5KHtzZWxlY3Rpb25zOiBbc2VsZWN0aW9uXX0pO1xuICAgICAgfVxuICAgIH1cbiAgfVxuXG4gIGFkZE9yU3VidHJhY3RTZWxlY3Rpb24oaXRlbSkge1xuICAgIHJldHVybiB0aGlzLnNlbGVjdEl0ZW0oaXRlbSwgZmFsc2UsIHRydWUpO1xuICB9XG5cbiAgY29hbGVzY2UoKSB7XG4gICAgaWYgKHRoaXMuc2VsZWN0aW9ucy5sZW5ndGggPT09IDApIHsgcmV0dXJuIHRoaXM7IH1cblxuICAgIGNvbnN0IG1vc3RSZWNlbnQgPSB0aGlzLnNlbGVjdGlvbnNbMF07XG4gICAgbGV0IG1vc3RSZWNlbnRTdGFydCA9IE1hdGgubWluKG1vc3RSZWNlbnQuaGVhZCwgbW9zdFJlY2VudC50YWlsKTtcbiAgICBsZXQgbW9zdFJlY2VudEVuZCA9IE1hdGgubWF4KG1vc3RSZWNlbnQuaGVhZCwgbW9zdFJlY2VudC50YWlsKTtcbiAgICB3aGlsZSAobW9zdFJlY2VudFN0YXJ0ID4gMCAmJiAhdGhpcy5pc0l0ZW1TZWxlY3RhYmxlKHRoaXMuaXRlbXNbbW9zdFJlY2VudFN0YXJ0IC0gMV0pKSB7XG4gICAgICBtb3N0UmVjZW50U3RhcnQtLTtcbiAgICB9XG4gICAgd2hpbGUgKG1vc3RSZWNlbnRFbmQgPCAodGhpcy5pdGVtcy5sZW5ndGggLSAxKSAmJiAhdGhpcy5pc0l0ZW1TZWxlY3RhYmxlKHRoaXMuaXRlbXNbbW9zdFJlY2VudEVuZCArIDFdKSkge1xuICAgICAgbW9zdFJlY2VudEVuZCsrO1xuICAgIH1cblxuICAgIGxldCBjaGFuZ2VkID0gZmFsc2U7XG4gICAgY29uc3QgbmV3U2VsZWN0aW9ucyA9IFttb3N0UmVjZW50XTtcbiAgICBmb3IgKGxldCBpID0gMTsgaSA8IHRoaXMuc2VsZWN0aW9ucy5sZW5ndGg7KSB7XG4gICAgICBjb25zdCBjdXJyZW50ID0gdGhpcy5zZWxlY3Rpb25zW2ldO1xuICAgICAgY29uc3QgY3VycmVudFN0YXJ0ID0gTWF0aC5taW4oY3VycmVudC5oZWFkLCBjdXJyZW50LnRhaWwpO1xuICAgICAgY29uc3QgY3VycmVudEVuZCA9IE1hdGgubWF4KGN1cnJlbnQuaGVhZCwgY3VycmVudC50YWlsKTtcbiAgICAgIGlmIChtb3N0UmVjZW50U3RhcnQgPD0gY3VycmVudEVuZCArIDEgJiYgY3VycmVudFN0YXJ0IC0gMSA8PSBtb3N0UmVjZW50RW5kKSB7XG4gICAgICAgIGlmIChtb3N0UmVjZW50Lm5lZ2F0ZSkge1xuICAgICAgICAgIGlmIChjdXJyZW50LmhlYWQgPiBjdXJyZW50LnRhaWwpIHtcbiAgICAgICAgICAgIGlmIChjdXJyZW50RW5kID4gbW9zdFJlY2VudEVuZCkgeyAvLyBzdWZmaXhcbiAgICAgICAgICAgICAgbmV3U2VsZWN0aW9ucy5wdXNoKHt0YWlsOiBtb3N0UmVjZW50RW5kICsgMSwgaGVhZDogY3VycmVudEVuZH0pO1xuICAgICAgICAgICAgfVxuICAgICAgICAgICAgaWYgKGN1cnJlbnRTdGFydCA8IG1vc3RSZWNlbnRTdGFydCkgeyAvLyBwcmVmaXhcbiAgICAgICAgICAgICAgbmV3U2VsZWN0aW9ucy5wdXNoKHt0YWlsOiBjdXJyZW50U3RhcnQsIGhlYWQ6IG1vc3RSZWNlbnRTdGFydCAtIDF9KTtcbiAgICAgICAgICAgIH1cbiAgICAgICAgICB9IGVsc2Uge1xuICAgICAgICAgICAgaWYgKGN1cnJlbnRTdGFydCA8IG1vc3RSZWNlbnRTdGFydCkgeyAvLyBwcmVmaXhcbiAgICAgICAgICAgICAgbmV3U2VsZWN0aW9ucy5wdXNoKHtoZWFkOiBjdXJyZW50U3RhcnQsIHRhaWw6IG1vc3RSZWNlbnRTdGFydCAtIDF9KTtcbiAgICAgICAgICAgIH1cbiAgICAgICAgICAgIGlmIChjdXJyZW50RW5kID4gbW9zdFJlY2VudEVuZCkgeyAvLyBzdWZmaXhcbiAgICAgICAgICAgICAgbmV3U2VsZWN0aW9ucy5wdXNoKHtoZWFkOiBtb3N0UmVjZW50RW5kICsgMSwgdGFpbDogY3VycmVudEVuZH0pO1xuICAgICAgICAgICAgfVxuICAgICAgICAgIH1cbiAgICAgICAgICBjaGFuZ2VkID0gdHJ1ZTtcbiAgICAgICAgICBpKys7XG4gICAgICAgIH0gZWxzZSB7XG4gICAgICAgICAgbW9zdFJlY2VudFN0YXJ0ID0gTWF0aC5taW4obW9zdFJlY2VudFN0YXJ0LCBjdXJyZW50U3RhcnQpO1xuICAgICAgICAgIG1vc3RSZWNlbnRFbmQgPSBNYXRoLm1heChtb3N0UmVjZW50RW5kLCBjdXJyZW50RW5kKTtcbiAgICAgICAgICBpZiAobW9zdFJlY2VudC5oZWFkID49IG1vc3RSZWNlbnQudGFpbCkge1xuICAgICAgICAgICAgbW9zdFJlY2VudC5oZWFkID0gbW9zdFJlY2VudEVuZDtcbiAgICAgICAgICAgIG1vc3RSZWNlbnQudGFpbCA9IG1vc3RSZWNlbnRTdGFydDtcbiAgICAgICAgICB9IGVsc2Uge1xuICAgICAgICAgICAgbW9zdFJlY2VudC5oZWFkID0gbW9zdFJlY2VudFN0YXJ0O1xuICAgICAgICAgICAgbW9zdFJlY2VudC50YWlsID0gbW9zdFJlY2VudEVuZDtcbiAgICAgICAgICB9XG4gICAgICAgICAgY2hhbmdlZCA9IHRydWU7XG4gICAgICAgICAgaSsrO1xuICAgICAgICB9XG4gICAgICB9IGVsc2Uge1xuICAgICAgICBuZXdTZWxlY3Rpb25zLnB1c2goY3VycmVudCk7XG4gICAgICAgIGkrKztcbiAgICAgIH1cbiAgICB9XG5cbiAgICBpZiAobW9zdFJlY2VudC5uZWdhdGUpIHtcbiAgICAgIGNoYW5nZWQgPSB0cnVlO1xuICAgICAgbmV3U2VsZWN0aW9ucy5zaGlmdCgpO1xuICAgIH1cblxuICAgIHJldHVybiBjaGFuZ2VkID8gdGhpcy5jb3B5KHtzZWxlY3Rpb25zOiBuZXdTZWxlY3Rpb25zfSkgOiB0aGlzO1xuICB9XG5cbiAgZ2V0U2VsZWN0ZWRJdGVtcygpIHtcbiAgICBjb25zdCBzZWxlY3RlZEl0ZW1zID0gbmV3IFNldCgpO1xuICAgIGZvciAoY29uc3Qge2hlYWQsIHRhaWwsIG5lZ2F0ZX0gb2YgdGhpcy5zZWxlY3Rpb25zLnNsaWNlKCkucmV2ZXJzZSgpKSB7XG4gICAgICBjb25zdCBzdGFydCA9IE1hdGgubWluKGhlYWQsIHRhaWwpO1xuICAgICAgY29uc3QgZW5kID0gTWF0aC5tYXgoaGVhZCwgdGFpbCk7XG4gICAgICBmb3IgKGxldCBpID0gc3RhcnQ7IGkgPD0gZW5kOyBpKyspIHtcbiAgICAgICAgY29uc3QgaXRlbSA9IHRoaXMuaXRlbXNbaV07XG4gICAgICAgIGlmICh0aGlzLmlzSXRlbVNlbGVjdGFibGUoaXRlbSkpIHtcbiAgICAgICAgICBpZiAobmVnYXRlKSB7XG4gICAgICAgICAgICBzZWxlY3RlZEl0ZW1zLmRlbGV0ZShpdGVtKTtcbiAgICAgICAgICB9IGVsc2Uge1xuICAgICAgICAgICAgc2VsZWN0ZWRJdGVtcy5hZGQoaXRlbSk7XG4gICAgICAgICAgfVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfVxuICAgIHJldHVybiBzZWxlY3RlZEl0ZW1zO1xuICB9XG5cbiAgZ2V0SGVhZEl0ZW0oKSB7XG4gICAgcmV0dXJuIHRoaXMuc2VsZWN0aW9ucy5sZW5ndGggPiAwID8gdGhpcy5pdGVtc1t0aGlzLnNlbGVjdGlvbnNbMF0uaGVhZF0gOiBudWxsO1xuICB9XG5cbiAgZ2V0TW9zdFJlY2VudFNlbGVjdGlvblN0YXJ0SW5kZXgoKSB7XG4gICAgY29uc3Qgc2VsZWN0aW9uID0gdGhpcy5zZWxlY3Rpb25zWzBdO1xuICAgIHJldHVybiBNYXRoLm1pbihzZWxlY3Rpb24uaGVhZCwgc2VsZWN0aW9uLnRhaWwpO1xuICB9XG5cbiAgZ2V0VGFpbEluZGV4KCkge1xuICAgIHJldHVybiB0aGlzLnNlbGVjdGlvbnNbMF0gPyB0aGlzLnNlbGVjdGlvbnNbMF0udGFpbCA6IG51bGw7XG4gIH1cbn1cbiJdfQ==