'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

exports.default = function ({ confirm, notificationManager, workspace }) {
  const pipelineManager = new _actionPipeline2.default({
    actionNames: ['PUSH', 'PULL', 'FETCH', 'COMMIT', 'CHECKOUT', 'ADDREMOTE']
  });

  const pushPipeline = pipelineManager.getPipeline(pipelineManager.actionKeys.PUSH);
  pushPipeline.addMiddleware('confirm-force-push', async (next, repository, branchName, options) => {
    if (options.force) {
      const choice = confirm({
        message: 'Are you sure you want to force push?',
        detailedMessage: 'This operation could result in losing data on the remote.',
        buttons: ['Force Push', 'Cancel']
      });
      if (choice !== 0) {/* do nothing */} else {
        await next();
      }
    } else {
      await next();
    }
  });
  pushPipeline.addMiddleware('set-push-in-progress', async (next, repository, branchName, options) => {
    repository.getOperationStates().setPushInProgress(true);
    try {
      await next();
    } finally {
      repository.getOperationStates().setPushInProgress(false);
    }
  });
  pushPipeline.addMiddleware('failed-to-push-error', async (next, repository, branchName, options) => {
    try {
      const result = await next();
      return result;
    } catch (error) {
      if (error instanceof _gitShellOutStrategy.GitError) {
        if (/rejected[\s\S]*failed to push/.test(error.stdErr)) {
          notificationManager.addError('Push rejected', {
            description: 'The tip of your current branch is behind its remote counterpart.' + ' Try pulling before pushing.<br />To force push, hold `cmd` or `ctrl` while clicking.',
            dismissable: true
          });
        } else {
          notificationManager.addError('Unable to push', {
            detail: error.stdErr,
            dismissable: true
          });
        }
      }
      throw error;
    }
  });

  const pullPipeline = pipelineManager.getPipeline(pipelineManager.actionKeys.PULL);
  pullPipeline.addMiddleware('set-pull-in-progress', async (next, repository, branchName) => {
    repository.getOperationStates().setPullInProgress(true);
    try {
      await next();
    } finally {
      repository.getOperationStates().setPullInProgress(false);
    }
  });
  pullPipeline.addMiddleware('failed-to-pull-error', async (next, repository, branchName) => {
    try {
      const result = await next();
      return result;
    } catch (error) {
      if (error instanceof _gitShellOutStrategy.GitError) {
        repository.didPullError();
        if (/error: Your local changes to the following files would be overwritten by merge/.test(error.stdErr)) {
          const lines = error.stdErr.split('\n');
          const files = lines.slice(3, lines.length - 3).map(l => `\`${l.trim()}\``).join('\n');
          notificationManager.addError('Pull aborted', {
            description: 'Local changes to the following would be overwritten by merge:<br/>' + files + '<br/>Please commit your changes or stash them before you merge.',
            dismissable: true
          });
        } else if (/Automatic merge failed; fix conflicts and then commit the result./.test(error.stdOut)) {
          notificationManager.addWarning('Merge conflicts', {
            description: `Your local changes conflicted with changes made on the remote branch. Resolve the conflicts
              with the Git panel and commit to continue.`,
            dismissable: true
          });
        } else if (/fatal: Not possible to fast-forward, aborting./.test(error.stdErr)) {
          notificationManager.addWarning('Unmerged changes', {
            description: 'Your local branch has diverged from its remote counterpart.<br/>' + 'Merge or rebase your local work to continue.',
            dismissable: true
          });
        } else {
          notificationManager.addError('Unable to pull', {
            detail: error.stdErr,
            dismissable: true
          });
        }
      }
      throw error;
    }
  });

  const fetchPipeline = pipelineManager.getPipeline(pipelineManager.actionKeys.FETCH);
  fetchPipeline.addMiddleware('set-fetch-in-progress', async (next, repository) => {
    repository.getOperationStates().setFetchInProgress(true);
    try {
      await next();
    } finally {
      repository.getOperationStates().setFetchInProgress(false);
    }
  });
  fetchPipeline.addMiddleware('failed-to-fetch-error', async (next, repository) => {
    try {
      const result = await next();
      return result;
    } catch (error) {
      if (error instanceof _gitShellOutStrategy.GitError) {
        notificationManager.addError('Unable to fetch', {
          detail: error.stdErr,
          dismissable: true
        });
      }
      throw error;
    }
  });

  const checkoutPipeline = pipelineManager.getPipeline(pipelineManager.actionKeys.CHECKOUT);
  checkoutPipeline.addMiddleware('set-checkout-in-progress', async (next, repository, branchName) => {
    repository.getOperationStates().setCheckoutInProgress(true);
    try {
      await next();
    } finally {
      repository.getOperationStates().setCheckoutInProgress(false);
    }
  });
  checkoutPipeline.addMiddleware('failed-to-checkout-error', async (next, repository, branchName, options) => {
    try {
      const result = await next();
      return result;
    } catch (error) {
      if (error instanceof _gitShellOutStrategy.GitError) {
        const message = options.createNew ? 'Cannot create branch' : 'Checkout aborted';
        let detail = undefined;
        let description = undefined;

        if (error.stdErr.match(/local changes.*would be overwritten/)) {
          const files = error.stdErr.split(/\r?\n/).filter(l => l.startsWith('\t')).map(l => `\`${l.trim()}\``).join('<br/>');
          description = 'Local changes to the following would be overwritten:<br/>' + files + '<br/>Please commit your changes or stash them.';
        } else if (error.stdErr.match(/branch.*already exists/)) {
          description = `\`${branchName}\` already exists. Choose another branch name.`;
        } else if (error.stdErr.match(/error: you need to resolve your current index first/)) {
          description = 'You must first resolve merge conflicts.';
        }

        if (description === undefined && detail === undefined) {
          detail = error.stdErr;
        }
        notificationManager.addError(message, { description, detail, dismissable: true });
      }
      throw error;
    }
  });

  const commitPipeline = pipelineManager.getPipeline(pipelineManager.actionKeys.COMMIT);
  commitPipeline.addMiddleware('confirm-commit', async (next, repository) => {
    function confirmCommit() {
      const choice = confirm({
        message: 'One or more text editors for the commit message are unsaved.',
        detailedMessage: 'Do you want to commit and close all open commit message editors?',
        buttons: ['Commit', 'Cancel']
      });
      return choice === 0;
    }

    const commitMessageEditors = (0, _helpers.getCommitMessageEditors)(repository, workspace);
    if (commitMessageEditors.length > 0) {
      if (!commitMessageEditors.some(e => e.isModified()) || confirmCommit()) {
        await next();
        commitMessageEditors.forEach(editor => editor.destroy());
      }
    } else {
      await next();
    }
  });
  commitPipeline.addMiddleware('clean-up-disk-commit-msg', async (next, repository) => {
    await next();
    try {
      await _fsExtra2.default.remove((0, _helpers.getCommitMessagePath)(repository));
    } catch (error) {
      // do nothing
    }
  });
  commitPipeline.addMiddleware('set-commit-in-progress', async (next, repository) => {
    repository.getOperationStates().setCommitInProgress(true);
    try {
      await next();
    } finally {
      repository.getOperationStates().setCommitInProgress(false);
    }
  });
  commitPipeline.addMiddleware('failed-to-commit-error', async (next, repository) => {
    try {
      const result = await next();
      const template = await repository.fetchCommitMessageTemplate();
      repository.setCommitMessage(template || '');
      (0, _helpers.destroyFilePatchPaneItems)({ onlyStaged: true }, workspace);
      return result;
    } catch (error) {
      if (error instanceof _gitShellOutStrategy.GitError) {
        notificationManager.addError('Unable to commit', {
          detail: error.stdErr,
          dismissable: true
        });
      }
      throw error;
    }
  });

  const addRemotePipeline = pipelineManager.getPipeline(pipelineManager.actionKeys.ADDREMOTE);
  addRemotePipeline.addMiddleware('failed-to-add-remote', async (next, repository, remoteName) => {
    try {
      return await next();
    } catch (error) {
      if (error instanceof _gitShellOutStrategy.GitError) {
        let detail = error.stdErr;
        if (error.stdErr.match(/^fatal: remote .* already exists\./)) {
          detail = `The repository already contains a remote named ${remoteName}.`;
        }
        notificationManager.addError('Cannot create remote', {
          detail,
          dismissable: true
        });
      }

      throw error;
    }
  });

  return pipelineManager;
};

var _fsExtra = require('fs-extra');

var _fsExtra2 = _interopRequireDefault(_fsExtra);

var _actionPipeline = require('./action-pipeline');

var _actionPipeline2 = _interopRequireDefault(_actionPipeline);

var _gitShellOutStrategy = require('./git-shell-out-strategy');

var _helpers = require('./helpers');

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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