"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const uuid_1 = require("./uuid");
const database_1 = require("./database");
// if you're running a local instance of central, use
// "http://localhost:4000/api/usage/" instead.
const baseUsageApi = "https://central.github.com/api/usage/";
exports.LastDailyStatsReportKey = "last-daily-stats-report";
/** The localStorage key for whether the user has opted out. */
exports.StatsOptOutKey = "stats-opt-out";
/** Have we successfully sent the stats opt-in? */
exports.HasSentOptInPingKey = "has-sent-stats-opt-in-ping";
/** milliseconds in an hour (for readability, dawg) */
const hours = 60 * 60 * 1000;
/** How often daily stats should be submitted (i.e., 24 hours). */
exports.DailyStatsReportIntervalInMs = hours * 24;
/** How often (in milliseconds) we check to see if it's time to report stats. */
exports.ReportingLoopIntervalInMs = hours * 4;
/** The goal is for this package to be app-agnostic so we can add
 * other editors in the future.
 */
var AppName;
(function (AppName) {
    AppName["Atom"] = "atom";
})(AppName = exports.AppName || (exports.AppName = {}));
/** helper for getting the date, which we pass in so that we can mock
 * in unit tests.
 */
const getISODate = () => new Date(Date.now()).toISOString();
class StatsStore {
    constructor(appName, version, isDevMode, getAccessToken = () => "") {
        /** Instance of a class thats stores metrics so they can be stored across sessions */
        this.database = new database_1.default(getISODate);
        this.version = version;
        this.appUrl = baseUsageApi + appName;
        const optOutValue = localStorage.getItem(exports.StatsOptOutKey);
        this.isDevMode = isDevMode;
        this.getAccessToken = getAccessToken;
        this.gitHubUser = null;
        this.timer = this.getTimer(exports.ReportingLoopIntervalInMs);
        if (optOutValue) {
            this.optOut = !!parseInt(optOutValue, 10);
            // If the user has set an opt out value but we haven't sent the ping yet,
            // give it a shot now.
            if (!localStorage.getItem(exports.HasSentOptInPingKey)) {
                this.sendOptInStatusPing(!this.optOut);
            }
        }
        else {
            this.optOut = false;
        }
    }
    setGitHubUser(gitHubUser) {
        this.gitHubUser = gitHubUser;
    }
    /** Set whether the user has opted out of stats reporting. */
    async setOptOut(optOut) {
        const changed = this.optOut !== optOut;
        this.optOut = optOut;
        localStorage.setItem(exports.StatsOptOutKey, optOut ? "1" : "0");
        if (changed) {
            await this.sendOptInStatusPing(!optOut);
        }
    }
    async reportStats(getDate) {
        if (this.optOut || this.isDevMode) {
            return;
        }
        const stats = await this.getDailyStats(getDate);
        try {
            const response = await this.post(stats);
            if (response.status !== 200) {
                throw new Error(`Stats reporting failure: ${response.status})`);
            }
            else {
                await localStorage.setItem(exports.LastDailyStatsReportKey, Date.now().toString());
                await this.database.clearData();
                console.log("stats successfully reported");
            }
        }
        catch (err) {
            // todo (tt, 5/2018): would be good to log these errors to Haystack/Datadog
            // so we have some kind of visibility into how often things are failing.
            console.log(err);
        }
    }
    /* send a ping to indicate that the user has changed their opt-in preferences.
    * public for testing purposes only.
    */
    async sendOptInStatusPing(optIn) {
        if (this.isDevMode) {
            return;
        }
        const direction = optIn ? "in" : "out";
        try {
            const response = await this.post({
                eventType: "ping",
                dimensions: {
                    optIn,
                },
            });
            if (response.status !== 200) {
                throw new Error(`Error sending opt in ping: ${response.status}`);
            }
            localStorage.setItem(exports.HasSentOptInPingKey, "1");
            console.log(`Opt ${direction} reported.`);
        }
        catch (err) {
            // todo (tt, 5/2018): would be good to log these errors to Haystack/Datadog
            // so we have some kind of visibility into how often things are failing.
            console.log(`Error reporting opt ${direction}`, err);
        }
    }
    // public for testing purposes only
    async getDailyStats(getDate) {
        return {
            measures: await this.database.getCounters(),
            customEvents: await this.database.getCustomEvents(),
            timings: await this.database.getTimings(),
            dimensions: {
                appVersion: this.version,
                platform: process.platform,
                guid: uuid_1.getGUID(),
                eventType: "usage",
                date: getDate(),
                language: process.env.LANG || "",
                gitHubUser: this.gitHubUser,
            },
        };
    }
    async addCustomEvent(eventType, event) {
        await this.database.addCustomEvent(eventType, event);
    }
    /**
     * Add timing data to the stats store, to be sent with the daily metrics requests.
     */
    async addTiming(eventType, durationInMilliseconds, metadata = {}) {
        // don't increment in dev mode because localStorage
        // is shared across dev and non dev windows and there's
        // no way to keep dev and non-dev metrics separate.
        // don't increment if the user has opted out, because
        // we want to respect user privacy.
        if (this.isDevMode || this.optOut) {
            return;
        }
        await this.database.addTiming(eventType, durationInMilliseconds, metadata);
    }
    /**
     * Increment a counter.  This is used to track usage statistics.
     */
    async incrementCounter(counterName) {
        // don't increment in dev mode because localStorage
        // is shared across dev and non dev windows and there's
        // no way to keep dev and non-dev metrics separate.
        // don't increment if the user has opted out, because
        // we want to respect user privacy.
        if (this.isDevMode || this.optOut) {
            return;
        }
        await this.database.incrementCounter(counterName);
    }
    /** Post some data to our stats endpoint.
     * This is public for testing purposes only.
     */
    async post(body) {
        const requestHeaders = { "Content-Type": "application/json" };
        const token = this.getAccessToken();
        if (token) {
            requestHeaders.Authorization = `token ${token}`;
        }
        const options = {
            method: "POST",
            headers: requestHeaders,
            body: JSON.stringify(body),
        };
        return this.fetch(this.appUrl, options);
    }
    /** Exists to enable us to mock fetch in tests
     * This is public for testing purposes only.
     */
    async fetch(url, options) {
        return fetch(url, options);
    }
    /** Should the app report its daily stats?
     * Public for testing purposes only.
     */
    hasReportingIntervalElapsed() {
        const lastDateString = localStorage.getItem(exports.LastDailyStatsReportKey);
        let lastDate = 0;
        if (lastDateString && lastDateString.length > 0) {
            lastDate = parseInt(lastDateString, 10);
        }
        if (isNaN(lastDate)) {
            lastDate = 0;
        }
        const now = Date.now();
        return (now - lastDate) > exports.DailyStatsReportIntervalInMs;
    }
    /** Set a timer so we can report the stats when the time comes. */
    getTimer(loopInterval) {
        // todo (tt, 5/2018): maybe we shouldn't even set up the timer
        // in dev mode or if the user has opted out.
        const timer = setInterval(() => {
            if (this.hasReportingIntervalElapsed()) {
                this.reportStats(getISODate);
            }
        }, loopInterval);
        if (timer.unref !== undefined) {
            // make sure we don't block node from exiting in tests
            // https://stackoverflow.com/questions/48172363/mocha-test-suite-never-ends-when-setinterval-running
            timer.unref();
        }
        return timer;
    }
}
exports.StatsStore = StatsStore;
//# sourceMappingURL=index.js.map