'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.source = undefined;

var _yubikiri = require('yubikiri');

var _yubikiri2 = _interopRequireDefault(_yubikiri);

var _eventKit = require('event-kit');

var _relayNetworkLayerManager = require('../relay-network-layer-manager');

var _relayNetworkLayerManager2 = _interopRequireDefault(_relayNetworkLayerManager);

var _author = require('./author');

var _author2 = _interopRequireDefault(_author);

var _keytarStrategy = require('../shared/keytar-strategy');

var _modelObserver = require('./model-observer');

var _modelObserver2 = _interopRequireDefault(_modelObserver);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

// This is a guess about what a reasonable value is. Can adjust if performance is poor.
const MAX_COMMITS = 5000;

const source = exports.source = {
  PENDING: Symbol('pending'),
  GITLOG: Symbol('git log'),
  GITHUBAPI: Symbol('github API')
};

class GraphQLCache {

  constructor() {
    this.bySlug = new Map();
  }
  // One hour


  get(remote) {
    const slug = remote.getSlug();
    const { ts, data } = this.bySlug.get(slug) || {
      ts: -Infinity,
      data: {}
    };

    if (Date.now() - ts > this.constructor.MAX_AGE_MS) {
      this.bySlug.delete(slug);
      return null;
    }
    return data;
  }

  set(remote, data) {
    this.bySlug.set(remote.getSlug(), { ts: Date.now(), data });
  }
}

GraphQLCache.MAX_AGE_MS = 3.6e6;
class UserStore {
  constructor({ repository, login, config }) {
    this.emitter = new _eventKit.Emitter();
    this.subs = new _eventKit.CompositeDisposable();

    // TODO: [ku 3/2018] Consider using Dexie (indexDB wrapper) like Desktop and persist users across sessions
    this.allUsers = new Map();
    this.excludedUsers = new Set();
    this.users = [];
    this.committer = _author.nullAuthor;

    this.last = {
      source: source.PENDING,
      repository: null,
      excludedUsers: this.excludedUsers
    };
    this.cache = new GraphQLCache();

    this.repositoryObserver = new _modelObserver2.default({
      fetchData: r => (0, _yubikiri2.default)({
        committer: r.getCommitter(),
        authors: r.getAuthors({ max: MAX_COMMITS }),
        remotes: r.getRemotes()
      }),
      didUpdate: () => this.loadUsers()
    });
    this.repositoryObserver.setActiveModel(repository);

    this.loginObserver = new _modelObserver2.default({
      didUpdate: () => this.loadUsers()
    });
    this.loginObserver.setActiveModel(login);

    this.subs.add(config.observe('github.excludedUsers', value => {
      this.excludedUsers = new Set((value || '').split(/\s*,\s*/).filter(each => each.length > 0));
      return this.loadUsers();
    }));
  }

  dispose() {
    this.subs.dispose();
    this.emitter.dispose();
  }

  async loadUsers() {
    const data = this.repositoryObserver.getActiveModelData();

    if (!data) {
      return;
    }

    this.setCommitter(data.committer);
    const githubRemotes = Array.from(data.remotes).filter(remote => remote.isGithubRepo());

    if (githubRemotes.length === 0) {
      this.addUsers(data.authors, source.GITLOG);
    } else {
      await this.loadUsersFromGraphQL(githubRemotes);
    }
  }

  loadUsersFromGraphQL(remotes) {
    return Promise.all(Array.from(remotes, remote => this.loadMentionableUsers(remote)));
  }

  async getToken(loginModel, loginAccount) {
    if (!loginModel) {
      return null;
    }
    const token = await loginModel.getToken(loginAccount);
    if (token === _keytarStrategy.UNAUTHENTICATED || token === _keytarStrategy.INSUFFICIENT) {
      return null;
    }
    return token;
  }

  async loadMentionableUsers(remote) {
    const cached = this.cache.get(remote);
    if (cached !== null) {
      this.addUsers(cached, source.GITHUBAPI);
      return;
    }

    const endpoint = remote.getEndpoint();
    const token = await this.getToken(this.loginObserver.getActiveModel(), endpoint.getLoginAccount());
    if (!token) {
      return;
    }

    const fetchQuery = _relayNetworkLayerManager2.default.getFetchQuery(endpoint, token);

    let hasMore = true;
    let cursor = null;
    const remoteUsers = [];

    while (hasMore) {
      const response = await fetchQuery({
        name: 'GetMentionableUsers',
        text: `
          query GetMentionableUsers($owner: String!, $name: String!, $first: Int!, $after: String) {
            repository(owner: $owner, name: $name) {
              mentionableUsers(first: $first, after: $after) {
                nodes {
                  login
                  email
                  name
                }
                pageInfo {
                  hasNextPage
                  endCursor
                }
              }
            }
          }
        `
      }, {
        owner: remote.getOwner(),
        name: remote.getRepo(),
        first: 100,
        after: cursor
      });

      if (response.errors && response.errors.length > 1) {
        // eslint-disable-next-line no-console
        console.error(`Error fetching mentionable users:\n${response.errors.map(e => e.message).join('\n')}`);
      }

      if (!response.data || !response.data.repository) {
        break;
      }

      const connection = response.data.repository.mentionableUsers;
      const authors = connection.nodes.map(node => {
        if (node.email === '') {
          node.email = `${node.login}@users.noreply.github.com`;
        }

        return new _author2.default(node.email, node.name, node.login);
      });
      this.addUsers(authors, source.GITHUBAPI);
      remoteUsers.push(...authors);

      cursor = connection.pageInfo.endCursor;
      hasMore = connection.pageInfo.hasNextPage;
    }

    this.cache.set(remote, remoteUsers);
  }

  addUsers(users, nextSource) {
    let changed = false;

    if (nextSource !== this.last.source || this.repositoryObserver.getActiveModel() !== this.last.repository || this.excludedUsers !== this.last.excludedUsers) {
      changed = true;
      this.allUsers.clear();
    }

    for (const author of users) {
      if (!this.allUsers.has(author.getEmail())) {
        changed = true;
      }
      this.allUsers.set(author.getEmail(), author);
    }

    if (changed) {
      this.finalize();
    }
    this.last.source = nextSource;
    this.last.repository = this.repositoryObserver.getActiveModel();
    this.last.excludedUsers = this.excludedUsers;
  }

  finalize() {
    // TODO: [ku 3/2018] consider sorting based on most recent authors or commit frequency
    const users = [];
    for (const author of this.allUsers.values()) {
      if (author.matches(this.committer)) {
        continue;
      }
      if (author.isNoReply()) {
        continue;
      }
      if (this.excludedUsers.has(author.getEmail())) {
        continue;
      }

      users.push(author);
    }
    users.sort(_author2.default.compare);
    this.users = users;
    this.didUpdate();
  }

  setRepository(repository) {
    this.repositoryObserver.setActiveModel(repository);
  }

  setLoginModel(login) {
    this.loginObserver.setActiveModel(login);
  }

  setCommitter(committer) {
    const changed = !this.committer.matches(committer);
    this.committer = committer;
    if (changed) {
      this.finalize();
    }
  }

  didUpdate() {
    this.emitter.emit('did-update', this.getUsers());
  }

  onDidUpdate(callback) {
    return this.emitter.on('did-update', callback);
  }

  getUsers() {
    return this.users;
  }
}
exports.default = UserStore;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbInVzZXItc3RvcmUuanMiXSwibmFtZXMiOlsiTUFYX0NPTU1JVFMiLCJzb3VyY2UiLCJQRU5ESU5HIiwiU3ltYm9sIiwiR0lUTE9HIiwiR0lUSFVCQVBJIiwiR3JhcGhRTENhY2hlIiwiY29uc3RydWN0b3IiLCJieVNsdWciLCJNYXAiLCJnZXQiLCJyZW1vdGUiLCJzbHVnIiwiZ2V0U2x1ZyIsInRzIiwiZGF0YSIsIkluZmluaXR5IiwiRGF0ZSIsIm5vdyIsIk1BWF9BR0VfTVMiLCJkZWxldGUiLCJzZXQiLCJVc2VyU3RvcmUiLCJyZXBvc2l0b3J5IiwibG9naW4iLCJjb25maWciLCJlbWl0dGVyIiwiRW1pdHRlciIsInN1YnMiLCJDb21wb3NpdGVEaXNwb3NhYmxlIiwiYWxsVXNlcnMiLCJleGNsdWRlZFVzZXJzIiwiU2V0IiwidXNlcnMiLCJjb21taXR0ZXIiLCJudWxsQXV0aG9yIiwibGFzdCIsImNhY2hlIiwicmVwb3NpdG9yeU9ic2VydmVyIiwiTW9kZWxPYnNlcnZlciIsImZldGNoRGF0YSIsInIiLCJnZXRDb21taXR0ZXIiLCJhdXRob3JzIiwiZ2V0QXV0aG9ycyIsIm1heCIsInJlbW90ZXMiLCJnZXRSZW1vdGVzIiwiZGlkVXBkYXRlIiwibG9hZFVzZXJzIiwic2V0QWN0aXZlTW9kZWwiLCJsb2dpbk9ic2VydmVyIiwiYWRkIiwib2JzZXJ2ZSIsInZhbHVlIiwic3BsaXQiLCJmaWx0ZXIiLCJlYWNoIiwibGVuZ3RoIiwiZGlzcG9zZSIsImdldEFjdGl2ZU1vZGVsRGF0YSIsInNldENvbW1pdHRlciIsImdpdGh1YlJlbW90ZXMiLCJBcnJheSIsImZyb20iLCJpc0dpdGh1YlJlcG8iLCJhZGRVc2VycyIsImxvYWRVc2Vyc0Zyb21HcmFwaFFMIiwiUHJvbWlzZSIsImFsbCIsImxvYWRNZW50aW9uYWJsZVVzZXJzIiwiZ2V0VG9rZW4iLCJsb2dpbk1vZGVsIiwibG9naW5BY2NvdW50IiwidG9rZW4iLCJVTkFVVEhFTlRJQ0FURUQiLCJJTlNVRkZJQ0lFTlQiLCJjYWNoZWQiLCJlbmRwb2ludCIsImdldEVuZHBvaW50IiwiZ2V0QWN0aXZlTW9kZWwiLCJnZXRMb2dpbkFjY291bnQiLCJmZXRjaFF1ZXJ5IiwiUmVsYXlOZXR3b3JrTGF5ZXJNYW5hZ2VyIiwiZ2V0RmV0Y2hRdWVyeSIsImhhc01vcmUiLCJjdXJzb3IiLCJyZW1vdGVVc2VycyIsInJlc3BvbnNlIiwibmFtZSIsInRleHQiLCJvd25lciIsImdldE93bmVyIiwiZ2V0UmVwbyIsImZpcnN0IiwiYWZ0ZXIiLCJlcnJvcnMiLCJjb25zb2xlIiwiZXJyb3IiLCJtYXAiLCJlIiwibWVzc2FnZSIsImpvaW4iLCJjb25uZWN0aW9uIiwibWVudGlvbmFibGVVc2VycyIsIm5vZGVzIiwibm9kZSIsImVtYWlsIiwiQXV0aG9yIiwicHVzaCIsInBhZ2VJbmZvIiwiZW5kQ3Vyc29yIiwiaGFzTmV4dFBhZ2UiLCJuZXh0U291cmNlIiwiY2hhbmdlZCIsImNsZWFyIiwiYXV0aG9yIiwiaGFzIiwiZ2V0RW1haWwiLCJmaW5hbGl6ZSIsInZhbHVlcyIsIm1hdGNoZXMiLCJpc05vUmVwbHkiLCJzb3J0IiwiY29tcGFyZSIsInNldFJlcG9zaXRvcnkiLCJzZXRMb2dpbk1vZGVsIiwiZW1pdCIsImdldFVzZXJzIiwib25EaWRVcGRhdGUiLCJjYWxsYmFjayIsIm9uIl0sIm1hcHBpbmdzIjoiOzs7Ozs7O0FBQUE7Ozs7QUFDQTs7QUFFQTs7OztBQUNBOzs7O0FBQ0E7O0FBQ0E7Ozs7OztBQUVBO0FBQ0EsTUFBTUEsY0FBYyxJQUFwQjs7QUFFTyxNQUFNQywwQkFBUztBQUNwQkMsV0FBU0MsT0FBTyxTQUFQLENBRFc7QUFFcEJDLFVBQVFELE9BQU8sU0FBUCxDQUZZO0FBR3BCRSxhQUFXRixPQUFPLFlBQVA7QUFIUyxDQUFmOztBQU1QLE1BQU1HLFlBQU4sQ0FBbUI7O0FBSWpCQyxnQkFBYztBQUNaLFNBQUtDLE1BQUwsR0FBYyxJQUFJQyxHQUFKLEVBQWQ7QUFDRDtBQUxEOzs7QUFPQUMsTUFBSUMsTUFBSixFQUFZO0FBQ1YsVUFBTUMsT0FBT0QsT0FBT0UsT0FBUCxFQUFiO0FBQ0EsVUFBTSxFQUFDQyxFQUFELEVBQUtDLElBQUwsS0FBYSxLQUFLUCxNQUFMLENBQVlFLEdBQVosQ0FBZ0JFLElBQWhCLEtBQXlCO0FBQzFDRSxVQUFJLENBQUNFLFFBRHFDO0FBRTFDRCxZQUFNO0FBRm9DLEtBQTVDOztBQUtBLFFBQUlFLEtBQUtDLEdBQUwsS0FBYUosRUFBYixHQUFrQixLQUFLUCxXQUFMLENBQWlCWSxVQUF2QyxFQUFtRDtBQUNqRCxXQUFLWCxNQUFMLENBQVlZLE1BQVosQ0FBbUJSLElBQW5CO0FBQ0EsYUFBTyxJQUFQO0FBQ0Q7QUFDRCxXQUFPRyxJQUFQO0FBQ0Q7O0FBRURNLE1BQUlWLE1BQUosRUFBWUksSUFBWixFQUFrQjtBQUNoQixTQUFLUCxNQUFMLENBQVlhLEdBQVosQ0FBZ0JWLE9BQU9FLE9BQVAsRUFBaEIsRUFBa0MsRUFBQ0MsSUFBSUcsS0FBS0MsR0FBTCxFQUFMLEVBQWlCSCxJQUFqQixFQUFsQztBQUNEO0FBeEJnQjs7QUFBYlQsWSxDQUVHYSxVLEdBQWEsSztBQXlCUCxNQUFNRyxTQUFOLENBQWdCO0FBQzdCZixjQUFZLEVBQUNnQixVQUFELEVBQWFDLEtBQWIsRUFBb0JDLE1BQXBCLEVBQVosRUFBeUM7QUFDdkMsU0FBS0MsT0FBTCxHQUFlLElBQUlDLGlCQUFKLEVBQWY7QUFDQSxTQUFLQyxJQUFMLEdBQVksSUFBSUMsNkJBQUosRUFBWjs7QUFFQTtBQUNBLFNBQUtDLFFBQUwsR0FBZ0IsSUFBSXJCLEdBQUosRUFBaEI7QUFDQSxTQUFLc0IsYUFBTCxHQUFxQixJQUFJQyxHQUFKLEVBQXJCO0FBQ0EsU0FBS0MsS0FBTCxHQUFhLEVBQWI7QUFDQSxTQUFLQyxTQUFMLEdBQWlCQyxrQkFBakI7O0FBRUEsU0FBS0MsSUFBTCxHQUFZO0FBQ1ZuQyxjQUFRQSxPQUFPQyxPQURMO0FBRVZxQixrQkFBWSxJQUZGO0FBR1ZRLHFCQUFlLEtBQUtBO0FBSFYsS0FBWjtBQUtBLFNBQUtNLEtBQUwsR0FBYSxJQUFJL0IsWUFBSixFQUFiOztBQUVBLFNBQUtnQyxrQkFBTCxHQUEwQixJQUFJQyx1QkFBSixDQUFrQjtBQUMxQ0MsaUJBQVdDLEtBQUssd0JBQVM7QUFDdkJQLG1CQUFXTyxFQUFFQyxZQUFGLEVBRFk7QUFFdkJDLGlCQUFTRixFQUFFRyxVQUFGLENBQWEsRUFBQ0MsS0FBSzdDLFdBQU4sRUFBYixDQUZjO0FBR3ZCOEMsaUJBQVNMLEVBQUVNLFVBQUY7QUFIYyxPQUFULENBRDBCO0FBTTFDQyxpQkFBVyxNQUFNLEtBQUtDLFNBQUw7QUFOeUIsS0FBbEIsQ0FBMUI7QUFRQSxTQUFLWCxrQkFBTCxDQUF3QlksY0FBeEIsQ0FBdUMzQixVQUF2Qzs7QUFFQSxTQUFLNEIsYUFBTCxHQUFxQixJQUFJWix1QkFBSixDQUFrQjtBQUNyQ1MsaUJBQVcsTUFBTSxLQUFLQyxTQUFMO0FBRG9CLEtBQWxCLENBQXJCO0FBR0EsU0FBS0UsYUFBTCxDQUFtQkQsY0FBbkIsQ0FBa0MxQixLQUFsQzs7QUFFQSxTQUFLSSxJQUFMLENBQVV3QixHQUFWLENBQ0UzQixPQUFPNEIsT0FBUCxDQUFlLHNCQUFmLEVBQXVDQyxTQUFTO0FBQzlDLFdBQUt2QixhQUFMLEdBQXFCLElBQUlDLEdBQUosQ0FDbkIsQ0FBQ3NCLFNBQVMsRUFBVixFQUFjQyxLQUFkLENBQW9CLFNBQXBCLEVBQStCQyxNQUEvQixDQUFzQ0MsUUFBUUEsS0FBS0MsTUFBTCxHQUFjLENBQTVELENBRG1CLENBQXJCO0FBR0EsYUFBTyxLQUFLVCxTQUFMLEVBQVA7QUFDRCxLQUxELENBREY7QUFRRDs7QUFFRFUsWUFBVTtBQUNSLFNBQUsvQixJQUFMLENBQVUrQixPQUFWO0FBQ0EsU0FBS2pDLE9BQUwsQ0FBYWlDLE9BQWI7QUFDRDs7QUFFRCxRQUFNVixTQUFOLEdBQWtCO0FBQ2hCLFVBQU1sQyxPQUFPLEtBQUt1QixrQkFBTCxDQUF3QnNCLGtCQUF4QixFQUFiOztBQUVBLFFBQUksQ0FBQzdDLElBQUwsRUFBVztBQUNUO0FBQ0Q7O0FBRUQsU0FBSzhDLFlBQUwsQ0FBa0I5QyxLQUFLbUIsU0FBdkI7QUFDQSxVQUFNNEIsZ0JBQWdCQyxNQUFNQyxJQUFOLENBQVdqRCxLQUFLK0IsT0FBaEIsRUFBeUJVLE1BQXpCLENBQWdDN0MsVUFBVUEsT0FBT3NELFlBQVAsRUFBMUMsQ0FBdEI7O0FBRUEsUUFBSUgsY0FBY0osTUFBZCxLQUF5QixDQUE3QixFQUFnQztBQUM5QixXQUFLUSxRQUFMLENBQWNuRCxLQUFLNEIsT0FBbkIsRUFBNEIxQyxPQUFPRyxNQUFuQztBQUNELEtBRkQsTUFFTztBQUNMLFlBQU0sS0FBSytELG9CQUFMLENBQTBCTCxhQUExQixDQUFOO0FBQ0Q7QUFDRjs7QUFFREssdUJBQXFCckIsT0FBckIsRUFBOEI7QUFDNUIsV0FBT3NCLFFBQVFDLEdBQVIsQ0FDTE4sTUFBTUMsSUFBTixDQUFXbEIsT0FBWCxFQUFvQm5DLFVBQVUsS0FBSzJELG9CQUFMLENBQTBCM0QsTUFBMUIsQ0FBOUIsQ0FESyxDQUFQO0FBR0Q7O0FBRUQsUUFBTTRELFFBQU4sQ0FBZUMsVUFBZixFQUEyQkMsWUFBM0IsRUFBeUM7QUFDdkMsUUFBSSxDQUFDRCxVQUFMLEVBQWlCO0FBQ2YsYUFBTyxJQUFQO0FBQ0Q7QUFDRCxVQUFNRSxRQUFRLE1BQU1GLFdBQVdELFFBQVgsQ0FBb0JFLFlBQXBCLENBQXBCO0FBQ0EsUUFBSUMsVUFBVUMsK0JBQVYsSUFBNkJELFVBQVVFLDRCQUEzQyxFQUF5RDtBQUN2RCxhQUFPLElBQVA7QUFDRDtBQUNELFdBQU9GLEtBQVA7QUFDRDs7QUFFRCxRQUFNSixvQkFBTixDQUEyQjNELE1BQTNCLEVBQW1DO0FBQ2pDLFVBQU1rRSxTQUFTLEtBQUt4QyxLQUFMLENBQVczQixHQUFYLENBQWVDLE1BQWYsQ0FBZjtBQUNBLFFBQUlrRSxXQUFXLElBQWYsRUFBcUI7QUFDbkIsV0FBS1gsUUFBTCxDQUFjVyxNQUFkLEVBQXNCNUUsT0FBT0ksU0FBN0I7QUFDQTtBQUNEOztBQUVELFVBQU15RSxXQUFXbkUsT0FBT29FLFdBQVAsRUFBakI7QUFDQSxVQUFNTCxRQUFRLE1BQU0sS0FBS0gsUUFBTCxDQUFjLEtBQUtwQixhQUFMLENBQW1CNkIsY0FBbkIsRUFBZCxFQUFtREYsU0FBU0csZUFBVCxFQUFuRCxDQUFwQjtBQUNBLFFBQUksQ0FBQ1AsS0FBTCxFQUFZO0FBQ1Y7QUFDRDs7QUFFRCxVQUFNUSxhQUFhQyxtQ0FBeUJDLGFBQXpCLENBQXVDTixRQUF2QyxFQUFpREosS0FBakQsQ0FBbkI7O0FBRUEsUUFBSVcsVUFBVSxJQUFkO0FBQ0EsUUFBSUMsU0FBUyxJQUFiO0FBQ0EsVUFBTUMsY0FBYyxFQUFwQjs7QUFFQSxXQUFPRixPQUFQLEVBQWdCO0FBQ2QsWUFBTUcsV0FBVyxNQUFNTixXQUFXO0FBQ2hDTyxjQUFNLHFCQUQwQjtBQUVoQ0MsY0FBTzs7Ozs7Ozs7Ozs7Ozs7Ozs7QUFGeUIsT0FBWCxFQW1CcEI7QUFDREMsZUFBT2hGLE9BQU9pRixRQUFQLEVBRE47QUFFREgsY0FBTTlFLE9BQU9rRixPQUFQLEVBRkw7QUFHREMsZUFBTyxHQUhOO0FBSURDLGVBQU9UO0FBSk4sT0FuQm9CLENBQXZCOztBQTBCQSxVQUFJRSxTQUFTUSxNQUFULElBQW1CUixTQUFTUSxNQUFULENBQWdCdEMsTUFBaEIsR0FBeUIsQ0FBaEQsRUFBbUQ7QUFDakQ7QUFDQXVDLGdCQUFRQyxLQUFSLENBQWUsc0NBQXFDVixTQUFTUSxNQUFULENBQWdCRyxHQUFoQixDQUFvQkMsS0FBS0EsRUFBRUMsT0FBM0IsRUFBb0NDLElBQXBDLENBQXlDLElBQXpDLENBQStDLEVBQW5HO0FBQ0Q7O0FBRUQsVUFBSSxDQUFDZCxTQUFTekUsSUFBVixJQUFrQixDQUFDeUUsU0FBU3pFLElBQVQsQ0FBY1EsVUFBckMsRUFBaUQ7QUFDL0M7QUFDRDs7QUFFRCxZQUFNZ0YsYUFBYWYsU0FBU3pFLElBQVQsQ0FBY1EsVUFBZCxDQUF5QmlGLGdCQUE1QztBQUNBLFlBQU03RCxVQUFVNEQsV0FBV0UsS0FBWCxDQUFpQk4sR0FBakIsQ0FBcUJPLFFBQVE7QUFDM0MsWUFBSUEsS0FBS0MsS0FBTCxLQUFlLEVBQW5CLEVBQXVCO0FBQ3JCRCxlQUFLQyxLQUFMLEdBQWMsR0FBRUQsS0FBS2xGLEtBQU0sMkJBQTNCO0FBQ0Q7O0FBRUQsZUFBTyxJQUFJb0YsZ0JBQUosQ0FBV0YsS0FBS0MsS0FBaEIsRUFBdUJELEtBQUtqQixJQUE1QixFQUFrQ2lCLEtBQUtsRixLQUF2QyxDQUFQO0FBQ0QsT0FOZSxDQUFoQjtBQU9BLFdBQUswQyxRQUFMLENBQWN2QixPQUFkLEVBQXVCMUMsT0FBT0ksU0FBOUI7QUFDQWtGLGtCQUFZc0IsSUFBWixDQUFpQixHQUFHbEUsT0FBcEI7O0FBRUEyQyxlQUFTaUIsV0FBV08sUUFBWCxDQUFvQkMsU0FBN0I7QUFDQTFCLGdCQUFVa0IsV0FBV08sUUFBWCxDQUFvQkUsV0FBOUI7QUFDRDs7QUFFRCxTQUFLM0UsS0FBTCxDQUFXaEIsR0FBWCxDQUFlVixNQUFmLEVBQXVCNEUsV0FBdkI7QUFDRDs7QUFFRHJCLFdBQVNqQyxLQUFULEVBQWdCZ0YsVUFBaEIsRUFBNEI7QUFDMUIsUUFBSUMsVUFBVSxLQUFkOztBQUVBLFFBQ0VELGVBQWUsS0FBSzdFLElBQUwsQ0FBVW5DLE1BQXpCLElBQ0EsS0FBS3FDLGtCQUFMLENBQXdCMEMsY0FBeEIsT0FBNkMsS0FBSzVDLElBQUwsQ0FBVWIsVUFEdkQsSUFFQSxLQUFLUSxhQUFMLEtBQXVCLEtBQUtLLElBQUwsQ0FBVUwsYUFIbkMsRUFJRTtBQUNBbUYsZ0JBQVUsSUFBVjtBQUNBLFdBQUtwRixRQUFMLENBQWNxRixLQUFkO0FBQ0Q7O0FBRUQsU0FBSyxNQUFNQyxNQUFYLElBQXFCbkYsS0FBckIsRUFBNEI7QUFDMUIsVUFBSSxDQUFDLEtBQUtILFFBQUwsQ0FBY3VGLEdBQWQsQ0FBa0JELE9BQU9FLFFBQVAsRUFBbEIsQ0FBTCxFQUEyQztBQUN6Q0osa0JBQVUsSUFBVjtBQUNEO0FBQ0QsV0FBS3BGLFFBQUwsQ0FBY1QsR0FBZCxDQUFrQitGLE9BQU9FLFFBQVAsRUFBbEIsRUFBcUNGLE1BQXJDO0FBQ0Q7O0FBRUQsUUFBSUYsT0FBSixFQUFhO0FBQ1gsV0FBS0ssUUFBTDtBQUNEO0FBQ0QsU0FBS25GLElBQUwsQ0FBVW5DLE1BQVYsR0FBbUJnSCxVQUFuQjtBQUNBLFNBQUs3RSxJQUFMLENBQVViLFVBQVYsR0FBdUIsS0FBS2Usa0JBQUwsQ0FBd0IwQyxjQUF4QixFQUF2QjtBQUNBLFNBQUs1QyxJQUFMLENBQVVMLGFBQVYsR0FBMEIsS0FBS0EsYUFBL0I7QUFDRDs7QUFFRHdGLGFBQVc7QUFDVDtBQUNBLFVBQU10RixRQUFRLEVBQWQ7QUFDQSxTQUFLLE1BQU1tRixNQUFYLElBQXFCLEtBQUt0RixRQUFMLENBQWMwRixNQUFkLEVBQXJCLEVBQTZDO0FBQzNDLFVBQUlKLE9BQU9LLE9BQVAsQ0FBZSxLQUFLdkYsU0FBcEIsQ0FBSixFQUFvQztBQUFFO0FBQVc7QUFDakQsVUFBSWtGLE9BQU9NLFNBQVAsRUFBSixFQUF3QjtBQUFFO0FBQVc7QUFDckMsVUFBSSxLQUFLM0YsYUFBTCxDQUFtQnNGLEdBQW5CLENBQXVCRCxPQUFPRSxRQUFQLEVBQXZCLENBQUosRUFBK0M7QUFBRTtBQUFXOztBQUU1RHJGLFlBQU00RSxJQUFOLENBQVdPLE1BQVg7QUFDRDtBQUNEbkYsVUFBTTBGLElBQU4sQ0FBV2YsaUJBQU9nQixPQUFsQjtBQUNBLFNBQUszRixLQUFMLEdBQWFBLEtBQWI7QUFDQSxTQUFLZSxTQUFMO0FBQ0Q7O0FBRUQ2RSxnQkFBY3RHLFVBQWQsRUFBMEI7QUFDeEIsU0FBS2Usa0JBQUwsQ0FBd0JZLGNBQXhCLENBQXVDM0IsVUFBdkM7QUFDRDs7QUFFRHVHLGdCQUFjdEcsS0FBZCxFQUFxQjtBQUNuQixTQUFLMkIsYUFBTCxDQUFtQkQsY0FBbkIsQ0FBa0MxQixLQUFsQztBQUNEOztBQUVEcUMsZUFBYTNCLFNBQWIsRUFBd0I7QUFDdEIsVUFBTWdGLFVBQVUsQ0FBQyxLQUFLaEYsU0FBTCxDQUFldUYsT0FBZixDQUF1QnZGLFNBQXZCLENBQWpCO0FBQ0EsU0FBS0EsU0FBTCxHQUFpQkEsU0FBakI7QUFDQSxRQUFJZ0YsT0FBSixFQUFhO0FBQ1gsV0FBS0ssUUFBTDtBQUNEO0FBQ0Y7O0FBRUR2RSxjQUFZO0FBQ1YsU0FBS3RCLE9BQUwsQ0FBYXFHLElBQWIsQ0FBa0IsWUFBbEIsRUFBZ0MsS0FBS0MsUUFBTCxFQUFoQztBQUNEOztBQUVEQyxjQUFZQyxRQUFaLEVBQXNCO0FBQ3BCLFdBQU8sS0FBS3hHLE9BQUwsQ0FBYXlHLEVBQWIsQ0FBZ0IsWUFBaEIsRUFBOEJELFFBQTlCLENBQVA7QUFDRDs7QUFFREYsYUFBVztBQUNULFdBQU8sS0FBSy9GLEtBQVo7QUFDRDtBQS9ONEI7a0JBQVZYLFMiLCJmaWxlIjoidXNlci1zdG9yZS5qcyIsInNvdXJjZVJvb3QiOiIvYnVpbGQvYXRvbS9zcmMvYXRvbS0xLjM1LjEvb3V0L2FwcC9ub2RlX21vZHVsZXMvZ2l0aHViL2xpYi9tb2RlbHMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgeXViaWtpcmkgZnJvbSAneXViaWtpcmknO1xuaW1wb3J0IHtFbWl0dGVyLCBDb21wb3NpdGVEaXNwb3NhYmxlfSBmcm9tICdldmVudC1raXQnO1xuXG5pbXBvcnQgUmVsYXlOZXR3b3JrTGF5ZXJNYW5hZ2VyIGZyb20gJy4uL3JlbGF5LW5ldHdvcmstbGF5ZXItbWFuYWdlcic7XG5pbXBvcnQgQXV0aG9yLCB7bnVsbEF1dGhvcn0gZnJvbSAnLi9hdXRob3InO1xuaW1wb3J0IHtVTkFVVEhFTlRJQ0FURUQsIElOU1VGRklDSUVOVH0gZnJvbSAnLi4vc2hhcmVkL2tleXRhci1zdHJhdGVneSc7XG5pbXBvcnQgTW9kZWxPYnNlcnZlciBmcm9tICcuL21vZGVsLW9ic2VydmVyJztcblxuLy8gVGhpcyBpcyBhIGd1ZXNzIGFib3V0IHdoYXQgYSByZWFzb25hYmxlIHZhbHVlIGlzLiBDYW4gYWRqdXN0IGlmIHBlcmZvcm1hbmNlIGlzIHBvb3IuXG5jb25zdCBNQVhfQ09NTUlUUyA9IDUwMDA7XG5cbmV4cG9ydCBjb25zdCBzb3VyY2UgPSB7XG4gIFBFTkRJTkc6IFN5bWJvbCgncGVuZGluZycpLFxuICBHSVRMT0c6IFN5bWJvbCgnZ2l0IGxvZycpLFxuICBHSVRIVUJBUEk6IFN5bWJvbCgnZ2l0aHViIEFQSScpLFxufTtcblxuY2xhc3MgR3JhcGhRTENhY2hlIHtcbiAgLy8gT25lIGhvdXJcbiAgc3RhdGljIE1BWF9BR0VfTVMgPSAzLjZlNlxuXG4gIGNvbnN0cnVjdG9yKCkge1xuICAgIHRoaXMuYnlTbHVnID0gbmV3IE1hcCgpO1xuICB9XG5cbiAgZ2V0KHJlbW90ZSkge1xuICAgIGNvbnN0IHNsdWcgPSByZW1vdGUuZ2V0U2x1ZygpO1xuICAgIGNvbnN0IHt0cywgZGF0YX0gPSB0aGlzLmJ5U2x1Zy5nZXQoc2x1ZykgfHwge1xuICAgICAgdHM6IC1JbmZpbml0eSxcbiAgICAgIGRhdGE6IHt9LFxuICAgIH07XG5cbiAgICBpZiAoRGF0ZS5ub3coKSAtIHRzID4gdGhpcy5jb25zdHJ1Y3Rvci5NQVhfQUdFX01TKSB7XG4gICAgICB0aGlzLmJ5U2x1Zy5kZWxldGUoc2x1Zyk7XG4gICAgICByZXR1cm4gbnVsbDtcbiAgICB9XG4gICAgcmV0dXJuIGRhdGE7XG4gIH1cblxuICBzZXQocmVtb3RlLCBkYXRhKSB7XG4gICAgdGhpcy5ieVNsdWcuc2V0KHJlbW90ZS5nZXRTbHVnKCksIHt0czogRGF0ZS5ub3coKSwgZGF0YX0pO1xuICB9XG59XG5cbmV4cG9ydCBkZWZhdWx0IGNsYXNzIFVzZXJTdG9yZSB7XG4gIGNvbnN0cnVjdG9yKHtyZXBvc2l0b3J5LCBsb2dpbiwgY29uZmlnfSkge1xuICAgIHRoaXMuZW1pdHRlciA9IG5ldyBFbWl0dGVyKCk7XG4gICAgdGhpcy5zdWJzID0gbmV3IENvbXBvc2l0ZURpc3Bvc2FibGUoKTtcblxuICAgIC8vIFRPRE86IFtrdSAzLzIwMThdIENvbnNpZGVyIHVzaW5nIERleGllIChpbmRleERCIHdyYXBwZXIpIGxpa2UgRGVza3RvcCBhbmQgcGVyc2lzdCB1c2VycyBhY3Jvc3Mgc2Vzc2lvbnNcbiAgICB0aGlzLmFsbFVzZXJzID0gbmV3IE1hcCgpO1xuICAgIHRoaXMuZXhjbHVkZWRVc2VycyA9IG5ldyBTZXQoKTtcbiAgICB0aGlzLnVzZXJzID0gW107XG4gICAgdGhpcy5jb21taXR0ZXIgPSBudWxsQXV0aG9yO1xuXG4gICAgdGhpcy5sYXN0ID0ge1xuICAgICAgc291cmNlOiBzb3VyY2UuUEVORElORyxcbiAgICAgIHJlcG9zaXRvcnk6IG51bGwsXG4gICAgICBleGNsdWRlZFVzZXJzOiB0aGlzLmV4Y2x1ZGVkVXNlcnMsXG4gICAgfTtcbiAgICB0aGlzLmNhY2hlID0gbmV3IEdyYXBoUUxDYWNoZSgpO1xuXG4gICAgdGhpcy5yZXBvc2l0b3J5T2JzZXJ2ZXIgPSBuZXcgTW9kZWxPYnNlcnZlcih7XG4gICAgICBmZXRjaERhdGE6IHIgPT4geXViaWtpcmkoe1xuICAgICAgICBjb21taXR0ZXI6IHIuZ2V0Q29tbWl0dGVyKCksXG4gICAgICAgIGF1dGhvcnM6IHIuZ2V0QXV0aG9ycyh7bWF4OiBNQVhfQ09NTUlUU30pLFxuICAgICAgICByZW1vdGVzOiByLmdldFJlbW90ZXMoKSxcbiAgICAgIH0pLFxuICAgICAgZGlkVXBkYXRlOiAoKSA9PiB0aGlzLmxvYWRVc2VycygpLFxuICAgIH0pO1xuICAgIHRoaXMucmVwb3NpdG9yeU9ic2VydmVyLnNldEFjdGl2ZU1vZGVsKHJlcG9zaXRvcnkpO1xuXG4gICAgdGhpcy5sb2dpbk9ic2VydmVyID0gbmV3IE1vZGVsT2JzZXJ2ZXIoe1xuICAgICAgZGlkVXBkYXRlOiAoKSA9PiB0aGlzLmxvYWRVc2VycygpLFxuICAgIH0pO1xuICAgIHRoaXMubG9naW5PYnNlcnZlci5zZXRBY3RpdmVNb2RlbChsb2dpbik7XG5cbiAgICB0aGlzLnN1YnMuYWRkKFxuICAgICAgY29uZmlnLm9ic2VydmUoJ2dpdGh1Yi5leGNsdWRlZFVzZXJzJywgdmFsdWUgPT4ge1xuICAgICAgICB0aGlzLmV4Y2x1ZGVkVXNlcnMgPSBuZXcgU2V0KFxuICAgICAgICAgICh2YWx1ZSB8fCAnJykuc3BsaXQoL1xccyosXFxzKi8pLmZpbHRlcihlYWNoID0+IGVhY2gubGVuZ3RoID4gMCksXG4gICAgICAgICk7XG4gICAgICAgIHJldHVybiB0aGlzLmxvYWRVc2VycygpO1xuICAgICAgfSksXG4gICAgKTtcbiAgfVxuXG4gIGRpc3Bvc2UoKSB7XG4gICAgdGhpcy5zdWJzLmRpc3Bvc2UoKTtcbiAgICB0aGlzLmVtaXR0ZXIuZGlzcG9zZSgpO1xuICB9XG5cbiAgYXN5bmMgbG9hZFVzZXJzKCkge1xuICAgIGNvbnN0IGRhdGEgPSB0aGlzLnJlcG9zaXRvcnlPYnNlcnZlci5nZXRBY3RpdmVNb2RlbERhdGEoKTtcblxuICAgIGlmICghZGF0YSkge1xuICAgICAgcmV0dXJuO1xuICAgIH1cblxuICAgIHRoaXMuc2V0Q29tbWl0dGVyKGRhdGEuY29tbWl0dGVyKTtcbiAgICBjb25zdCBnaXRodWJSZW1vdGVzID0gQXJyYXkuZnJvbShkYXRhLnJlbW90ZXMpLmZpbHRlcihyZW1vdGUgPT4gcmVtb3RlLmlzR2l0aHViUmVwbygpKTtcblxuICAgIGlmIChnaXRodWJSZW1vdGVzLmxlbmd0aCA9PT0gMCkge1xuICAgICAgdGhpcy5hZGRVc2VycyhkYXRhLmF1dGhvcnMsIHNvdXJjZS5HSVRMT0cpO1xuICAgIH0gZWxzZSB7XG4gICAgICBhd2FpdCB0aGlzLmxvYWRVc2Vyc0Zyb21HcmFwaFFMKGdpdGh1YlJlbW90ZXMpO1xuICAgIH1cbiAgfVxuXG4gIGxvYWRVc2Vyc0Zyb21HcmFwaFFMKHJlbW90ZXMpIHtcbiAgICByZXR1cm4gUHJvbWlzZS5hbGwoXG4gICAgICBBcnJheS5mcm9tKHJlbW90ZXMsIHJlbW90ZSA9PiB0aGlzLmxvYWRNZW50aW9uYWJsZVVzZXJzKHJlbW90ZSkpLFxuICAgICk7XG4gIH1cblxuICBhc3luYyBnZXRUb2tlbihsb2dpbk1vZGVsLCBsb2dpbkFjY291bnQpIHtcbiAgICBpZiAoIWxvZ2luTW9kZWwpIHtcbiAgICAgIHJldHVybiBudWxsO1xuICAgIH1cbiAgICBjb25zdCB0b2tlbiA9IGF3YWl0IGxvZ2luTW9kZWwuZ2V0VG9rZW4obG9naW5BY2NvdW50KTtcbiAgICBpZiAodG9rZW4gPT09IFVOQVVUSEVOVElDQVRFRCB8fCB0b2tlbiA9PT0gSU5TVUZGSUNJRU5UKSB7XG4gICAgICByZXR1cm4gbnVsbDtcbiAgICB9XG4gICAgcmV0dXJuIHRva2VuO1xuICB9XG5cbiAgYXN5bmMgbG9hZE1lbnRpb25hYmxlVXNlcnMocmVtb3RlKSB7XG4gICAgY29uc3QgY2FjaGVkID0gdGhpcy5jYWNoZS5nZXQocmVtb3RlKTtcbiAgICBpZiAoY2FjaGVkICE9PSBudWxsKSB7XG4gICAgICB0aGlzLmFkZFVzZXJzKGNhY2hlZCwgc291cmNlLkdJVEhVQkFQSSk7XG4gICAgICByZXR1cm47XG4gICAgfVxuXG4gICAgY29uc3QgZW5kcG9pbnQgPSByZW1vdGUuZ2V0RW5kcG9pbnQoKTtcbiAgICBjb25zdCB0b2tlbiA9IGF3YWl0IHRoaXMuZ2V0VG9rZW4odGhpcy5sb2dpbk9ic2VydmVyLmdldEFjdGl2ZU1vZGVsKCksIGVuZHBvaW50LmdldExvZ2luQWNjb3VudCgpKTtcbiAgICBpZiAoIXRva2VuKSB7XG4gICAgICByZXR1cm47XG4gICAgfVxuXG4gICAgY29uc3QgZmV0Y2hRdWVyeSA9IFJlbGF5TmV0d29ya0xheWVyTWFuYWdlci5nZXRGZXRjaFF1ZXJ5KGVuZHBvaW50LCB0b2tlbik7XG5cbiAgICBsZXQgaGFzTW9yZSA9IHRydWU7XG4gICAgbGV0IGN1cnNvciA9IG51bGw7XG4gICAgY29uc3QgcmVtb3RlVXNlcnMgPSBbXTtcblxuICAgIHdoaWxlIChoYXNNb3JlKSB7XG4gICAgICBjb25zdCByZXNwb25zZSA9IGF3YWl0IGZldGNoUXVlcnkoe1xuICAgICAgICBuYW1lOiAnR2V0TWVudGlvbmFibGVVc2VycycsXG4gICAgICAgIHRleHQ6IGBcbiAgICAgICAgICBxdWVyeSBHZXRNZW50aW9uYWJsZVVzZXJzKCRvd25lcjogU3RyaW5nISwgJG5hbWU6IFN0cmluZyEsICRmaXJzdDogSW50ISwgJGFmdGVyOiBTdHJpbmcpIHtcbiAgICAgICAgICAgIHJlcG9zaXRvcnkob3duZXI6ICRvd25lciwgbmFtZTogJG5hbWUpIHtcbiAgICAgICAgICAgICAgbWVudGlvbmFibGVVc2VycyhmaXJzdDogJGZpcnN0LCBhZnRlcjogJGFmdGVyKSB7XG4gICAgICAgICAgICAgICAgbm9kZXMge1xuICAgICAgICAgICAgICAgICAgbG9naW5cbiAgICAgICAgICAgICAgICAgIGVtYWlsXG4gICAgICAgICAgICAgICAgICBuYW1lXG4gICAgICAgICAgICAgICAgfVxuICAgICAgICAgICAgICAgIHBhZ2VJbmZvIHtcbiAgICAgICAgICAgICAgICAgIGhhc05leHRQYWdlXG4gICAgICAgICAgICAgICAgICBlbmRDdXJzb3JcbiAgICAgICAgICAgICAgICB9XG4gICAgICAgICAgICAgIH1cbiAgICAgICAgICAgIH1cbiAgICAgICAgICB9XG4gICAgICAgIGAsXG4gICAgICB9LCB7XG4gICAgICAgIG93bmVyOiByZW1vdGUuZ2V0T3duZXIoKSxcbiAgICAgICAgbmFtZTogcmVtb3RlLmdldFJlcG8oKSxcbiAgICAgICAgZmlyc3Q6IDEwMCxcbiAgICAgICAgYWZ0ZXI6IGN1cnNvcixcbiAgICAgIH0pO1xuXG4gICAgICBpZiAocmVzcG9uc2UuZXJyb3JzICYmIHJlc3BvbnNlLmVycm9ycy5sZW5ndGggPiAxKSB7XG4gICAgICAgIC8vIGVzbGludC1kaXNhYmxlLW5leHQtbGluZSBuby1jb25zb2xlXG4gICAgICAgIGNvbnNvbGUuZXJyb3IoYEVycm9yIGZldGNoaW5nIG1lbnRpb25hYmxlIHVzZXJzOlxcbiR7cmVzcG9uc2UuZXJyb3JzLm1hcChlID0+IGUubWVzc2FnZSkuam9pbignXFxuJyl9YCk7XG4gICAgICB9XG5cbiAgICAgIGlmICghcmVzcG9uc2UuZGF0YSB8fCAhcmVzcG9uc2UuZGF0YS5yZXBvc2l0b3J5KSB7XG4gICAgICAgIGJyZWFrO1xuICAgICAgfVxuXG4gICAgICBjb25zdCBjb25uZWN0aW9uID0gcmVzcG9uc2UuZGF0YS5yZXBvc2l0b3J5Lm1lbnRpb25hYmxlVXNlcnM7XG4gICAgICBjb25zdCBhdXRob3JzID0gY29ubmVjdGlvbi5ub2Rlcy5tYXAobm9kZSA9PiB7XG4gICAgICAgIGlmIChub2RlLmVtYWlsID09PSAnJykge1xuICAgICAgICAgIG5vZGUuZW1haWwgPSBgJHtub2RlLmxvZ2lufUB1c2Vycy5ub3JlcGx5LmdpdGh1Yi5jb21gO1xuICAgICAgICB9XG5cbiAgICAgICAgcmV0dXJuIG5ldyBBdXRob3Iobm9kZS5lbWFpbCwgbm9kZS5uYW1lLCBub2RlLmxvZ2luKTtcbiAgICAgIH0pO1xuICAgICAgdGhpcy5hZGRVc2VycyhhdXRob3JzLCBzb3VyY2UuR0lUSFVCQVBJKTtcbiAgICAgIHJlbW90ZVVzZXJzLnB1c2goLi4uYXV0aG9ycyk7XG5cbiAgICAgIGN1cnNvciA9IGNvbm5lY3Rpb24ucGFnZUluZm8uZW5kQ3Vyc29yO1xuICAgICAgaGFzTW9yZSA9IGNvbm5lY3Rpb24ucGFnZUluZm8uaGFzTmV4dFBhZ2U7XG4gICAgfVxuXG4gICAgdGhpcy5jYWNoZS5zZXQocmVtb3RlLCByZW1vdGVVc2Vycyk7XG4gIH1cblxuICBhZGRVc2Vycyh1c2VycywgbmV4dFNvdXJjZSkge1xuICAgIGxldCBjaGFuZ2VkID0gZmFsc2U7XG5cbiAgICBpZiAoXG4gICAgICBuZXh0U291cmNlICE9PSB0aGlzLmxhc3Quc291cmNlIHx8XG4gICAgICB0aGlzLnJlcG9zaXRvcnlPYnNlcnZlci5nZXRBY3RpdmVNb2RlbCgpICE9PSB0aGlzLmxhc3QucmVwb3NpdG9yeSB8fFxuICAgICAgdGhpcy5leGNsdWRlZFVzZXJzICE9PSB0aGlzLmxhc3QuZXhjbHVkZWRVc2Vyc1xuICAgICkge1xuICAgICAgY2hhbmdlZCA9IHRydWU7XG4gICAgICB0aGlzLmFsbFVzZXJzLmNsZWFyKCk7XG4gICAgfVxuXG4gICAgZm9yIChjb25zdCBhdXRob3Igb2YgdXNlcnMpIHtcbiAgICAgIGlmICghdGhpcy5hbGxVc2Vycy5oYXMoYXV0aG9yLmdldEVtYWlsKCkpKSB7XG4gICAgICAgIGNoYW5nZWQgPSB0cnVlO1xuICAgICAgfVxuICAgICAgdGhpcy5hbGxVc2Vycy5zZXQoYXV0aG9yLmdldEVtYWlsKCksIGF1dGhvcik7XG4gICAgfVxuXG4gICAgaWYgKGNoYW5nZWQpIHtcbiAgICAgIHRoaXMuZmluYWxpemUoKTtcbiAgICB9XG4gICAgdGhpcy5sYXN0LnNvdXJjZSA9IG5leHRTb3VyY2U7XG4gICAgdGhpcy5sYXN0LnJlcG9zaXRvcnkgPSB0aGlzLnJlcG9zaXRvcnlPYnNlcnZlci5nZXRBY3RpdmVNb2RlbCgpO1xuICAgIHRoaXMubGFzdC5leGNsdWRlZFVzZXJzID0gdGhpcy5leGNsdWRlZFVzZXJzO1xuICB9XG5cbiAgZmluYWxpemUoKSB7XG4gICAgLy8gVE9ETzogW2t1IDMvMjAxOF0gY29uc2lkZXIgc29ydGluZyBiYXNlZCBvbiBtb3N0IHJlY2VudCBhdXRob3JzIG9yIGNvbW1pdCBmcmVxdWVuY3lcbiAgICBjb25zdCB1c2VycyA9IFtdO1xuICAgIGZvciAoY29uc3QgYXV0aG9yIG9mIHRoaXMuYWxsVXNlcnMudmFsdWVzKCkpIHtcbiAgICAgIGlmIChhdXRob3IubWF0Y2hlcyh0aGlzLmNvbW1pdHRlcikpIHsgY29udGludWU7IH1cbiAgICAgIGlmIChhdXRob3IuaXNOb1JlcGx5KCkpIHsgY29udGludWU7IH1cbiAgICAgIGlmICh0aGlzLmV4Y2x1ZGVkVXNlcnMuaGFzKGF1dGhvci5nZXRFbWFpbCgpKSkgeyBjb250aW51ZTsgfVxuXG4gICAgICB1c2Vycy5wdXNoKGF1dGhvcik7XG4gICAgfVxuICAgIHVzZXJzLnNvcnQoQXV0aG9yLmNvbXBhcmUpO1xuICAgIHRoaXMudXNlcnMgPSB1c2VycztcbiAgICB0aGlzLmRpZFVwZGF0ZSgpO1xuICB9XG5cbiAgc2V0UmVwb3NpdG9yeShyZXBvc2l0b3J5KSB7XG4gICAgdGhpcy5yZXBvc2l0b3J5T2JzZXJ2ZXIuc2V0QWN0aXZlTW9kZWwocmVwb3NpdG9yeSk7XG4gIH1cblxuICBzZXRMb2dpbk1vZGVsKGxvZ2luKSB7XG4gICAgdGhpcy5sb2dpbk9ic2VydmVyLnNldEFjdGl2ZU1vZGVsKGxvZ2luKTtcbiAgfVxuXG4gIHNldENvbW1pdHRlcihjb21taXR0ZXIpIHtcbiAgICBjb25zdCBjaGFuZ2VkID0gIXRoaXMuY29tbWl0dGVyLm1hdGNoZXMoY29tbWl0dGVyKTtcbiAgICB0aGlzLmNvbW1pdHRlciA9IGNvbW1pdHRlcjtcbiAgICBpZiAoY2hhbmdlZCkge1xuICAgICAgdGhpcy5maW5hbGl6ZSgpO1xuICAgIH1cbiAgfVxuXG4gIGRpZFVwZGF0ZSgpIHtcbiAgICB0aGlzLmVtaXR0ZXIuZW1pdCgnZGlkLXVwZGF0ZScsIHRoaXMuZ2V0VXNlcnMoKSk7XG4gIH1cblxuICBvbkRpZFVwZGF0ZShjYWxsYmFjaykge1xuICAgIHJldHVybiB0aGlzLmVtaXR0ZXIub24oJ2RpZC11cGRhdGUnLCBjYWxsYmFjayk7XG4gIH1cblxuICBnZXRVc2VycygpIHtcbiAgICByZXR1cm4gdGhpcy51c2VycztcbiAgfVxufVxuIl19