'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.classNameForStatus = exports.CO_AUTHOR_REGEX = exports.LINE_ENDING_REGEX = undefined;

var _extends = Object.assign || function (target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i]; for (var key in source) { if (Object.prototype.hasOwnProperty.call(source, key)) { target[key] = source[key]; } } } return target; };

exports.autobind = autobind;
exports.extractProps = extractProps;
exports.unusedProps = unusedProps;
exports.getPackageRoot = getPackageRoot;
exports.getAtomHelperPath = getAtomHelperPath;
exports.getDugitePath = getDugitePath;
exports.getSharedModulePath = getSharedModulePath;
exports.isBinary = isBinary;
exports.firstImplementer = firstImplementer;
exports.isValidWorkdir = isValidWorkdir;
exports.fileExists = fileExists;
exports.getTempDir = getTempDir;
exports.isFileExecutable = isFileExecutable;
exports.isFileSymlink = isFileSymlink;
exports.shortenSha = shortenSha;
exports.normalizeGitHelperPath = normalizeGitHelperPath;
exports.toNativePathSep = toNativePathSep;
exports.toGitPathSep = toGitPathSep;
exports.filePathEndsWith = filePathEndsWith;
exports.toSentence = toSentence;
exports.pushAtKey = pushAtKey;
exports.getCommitMessagePath = getCommitMessagePath;
exports.getCommitMessageEditors = getCommitMessageEditors;
exports.getFilePatchPaneItems = getFilePatchPaneItems;
exports.destroyFilePatchPaneItems = destroyFilePatchPaneItems;
exports.destroyEmptyFilePatchPaneItems = destroyEmptyFilePatchPaneItems;
exports.extractCoAuthorsAndRawCommitMessage = extractCoAuthorsAndRawCommitMessage;
exports.createItem = createItem;
exports.equalSets = equalSets;

var _path = require('path');

var _path2 = _interopRequireDefault(_path);

var _fsExtra = require('fs-extra');

var _fsExtra2 = _interopRequireDefault(_fsExtra);

var _os = require('os');

var _os2 = _interopRequireDefault(_os);

var _temp = require('temp');

var _temp2 = _interopRequireDefault(_temp);

var _multiFilePatchController = require('./controllers/multi-file-patch-controller');

var _multiFilePatchController2 = _interopRequireDefault(_multiFilePatchController);

var _refHolder = require('./models/ref-holder');

var _refHolder2 = _interopRequireDefault(_refHolder);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

const LINE_ENDING_REGEX = exports.LINE_ENDING_REGEX = /\r?\n/;
const CO_AUTHOR_REGEX = exports.CO_AUTHOR_REGEX = /^co-authored-by. (.+?) <(.+?)>$/i;

function autobind(self, ...methods) {
  for (const method of methods) {
    if (typeof self[method] !== 'function') {
      throw new Error(`Unable to autobind method ${method}`);
    }
    self[method] = self[method].bind(self);
  }
}

// Extract a subset of props chosen from a propTypes object from a component's props to pass to a different API.
//
// Usage:
//
// ```js
// const apiProps = {
//   zero: PropTypes.number.isRequired,
//   one: PropTypes.string,
//   two: PropTypes.object,
// };
//
// class Component extends React.Component {
//   static propTypes = {
//     ...apiProps,
//     extra: PropTypes.func,
//   }
//
//   action() {
//     const options = extractProps(this.props, apiProps);
//     // options contains zero, one, and two, but not extra
//   }
// }
// ```
function extractProps(props, propTypes, nameMap = {}) {
  return Object.keys(propTypes).reduce((opts, propName) => {
    if (props[propName] !== undefined) {
      const destPropName = nameMap[propName] || propName;
      opts[destPropName] = props[propName];
    }
    return opts;
  }, {});
}

// The opposite of extractProps. Return a subset of props that do *not* appear in a component's prop types.
function unusedProps(props, propTypes) {
  return Object.keys(props).reduce((opts, propName) => {
    if (propTypes[propName] === undefined) {
      opts[propName] = props[propName];
    }
    return opts;
  }, {});
}

function getPackageRoot() {
  const { resourcePath } = atom.getLoadSettings();
  const currentFileWasRequiredFromSnapshot = !_path2.default.isAbsolute(__dirname);
  if (currentFileWasRequiredFromSnapshot) {
    return _path2.default.join(resourcePath, 'node_modules', 'github');
  } else {
    const packageRoot = _path2.default.resolve(__dirname, '..');
    if (_path2.default.extname(resourcePath) === '.asar') {
      if (packageRoot.indexOf(resourcePath) === 0) {
        return _path2.default.join(`${resourcePath}.unpacked`, 'node_modules', 'github');
      }
    }
    return packageRoot;
  }
}

function getAtomAppName() {
  const match = atom.getVersion().match(/-([A-Za-z]+)(\d+|-)/);
  if (match) {
    const channel = match[1];
    return `Atom ${channel.charAt(0).toUpperCase() + channel.slice(1)} Helper`;
  }

  return 'Atom Helper';
}

function getAtomHelperPath() {
  if (process.platform === 'darwin') {
    const appName = getAtomAppName();
    return _path2.default.resolve(process.resourcesPath, '..', 'Frameworks', `${appName}.app`, 'Contents', 'MacOS', appName);
  } else {
    return process.execPath;
  }
}

let DUGITE_PATH;
function getDugitePath() {
  if (!DUGITE_PATH) {
    DUGITE_PATH = require.resolve('dugite');
    if (!_path2.default.isAbsolute(DUGITE_PATH)) {
      // Assume we're snapshotted
      const { resourcePath } = atom.getLoadSettings();
      if (_path2.default.extname(resourcePath) === '.asar') {
        DUGITE_PATH = _path2.default.join(`${resourcePath}.unpacked`, 'node_modules', 'dugite');
      } else {
        DUGITE_PATH = _path2.default.join(resourcePath, 'node_modules', 'dugite');
      }
    }
  }

  return DUGITE_PATH;
}

const SHARED_MODULE_PATHS = new Map();
function getSharedModulePath(relPath) {
  let modulePath = SHARED_MODULE_PATHS.get(relPath);
  if (!modulePath) {
    modulePath = require.resolve(_path2.default.join(__dirname, 'shared', relPath));
    if (!_path2.default.isAbsolute(modulePath)) {
      // Assume we're snapshotted
      const { resourcePath } = atom.getLoadSettings();
      modulePath = _path2.default.join(resourcePath, modulePath);
    }

    SHARED_MODULE_PATHS.set(relPath, modulePath);
  }

  return modulePath;
}

function isBinary(data) {
  for (let i = 0; i < 50; i++) {
    const code = data.charCodeAt(i);
    // Char code 65533 is the "replacement character";
    // 8 and below are control characters.
    if (code === 65533 || code < 9) {
      return true;
    }
  }

  return false;
}

function descriptorsFromProto(proto) {
  return Object.getOwnPropertyNames(proto).reduce((acc, name) => {
    Object.assign(acc, {
      [name]: Reflect.getOwnPropertyDescriptor(proto, name)
    });
    return acc;
  }, {});
}

/**
 * Takes an array of targets and returns a proxy. The proxy intercepts property accessor calls and
 * returns the value of that property on the first object in `targets` where the target implements that property.
 */
function firstImplementer(...targets) {
  return new Proxy({ __implementations: targets }, {
    get(target, name) {
      if (name === 'getImplementers') {
        return () => targets;
      }

      if (Reflect.has(target, name)) {
        return target[name];
      }

      const firstValidTarget = targets.find(t => Reflect.has(t, name));
      if (firstValidTarget) {
        return firstValidTarget[name];
      } else {
        return undefined;
      }
    },

    set(target, name, value) {
      const firstValidTarget = targets.find(t => Reflect.has(t, name));
      if (firstValidTarget) {
        // eslint-disable-next-line no-return-assign
        return firstValidTarget[name] = value;
      } else {
        // eslint-disable-next-line no-return-assign
        return target[name] = value;
      }
    },

    // Used by sinon
    has(target, name) {
      if (name === 'getImplementers') {
        return true;
      }

      return targets.some(t => Reflect.has(t, name));
    },

    // Used by sinon
    getOwnPropertyDescriptor(target, name) {
      const firstValidTarget = targets.find(t => Reflect.getOwnPropertyDescriptor(t, name));
      const compositeOwnPropertyDescriptor = Reflect.getOwnPropertyDescriptor(target, name);
      if (firstValidTarget) {
        return Reflect.getOwnPropertyDescriptor(firstValidTarget, name);
      } else if (compositeOwnPropertyDescriptor) {
        return compositeOwnPropertyDescriptor;
      } else {
        return undefined;
      }
    },

    // Used by sinon
    getPrototypeOf(target) {
      return targets.reduceRight((acc, t) => {
        return Object.create(acc, descriptorsFromProto(Object.getPrototypeOf(t)));
      }, Object.prototype);
    }
  });
}

function isRoot(dir) {
  return _path2.default.resolve(dir, '..') === dir;
}

function isValidWorkdir(dir) {
  return dir !== _os2.default.homedir() && !isRoot(dir);
}

async function fileExists(absoluteFilePath) {
  try {
    await _fsExtra2.default.access(absoluteFilePath);
    return true;
  } catch (e) {
    if (e.code === 'ENOENT') {
      return false;
    }

    throw e;
  }
}

function getTempDir(options = {}) {
  _temp2.default.track();

  return new Promise((resolve, reject) => {
    _temp2.default.mkdir(options, (tempError, folder) => {
      if (tempError) {
        reject(tempError);
        return;
      }

      if (options.symlinkOk) {
        resolve(folder);
      } else {
        _fsExtra2.default.realpath(folder, (realError, rpath) => realError ? reject(realError) : resolve(rpath));
      }
    });
  });
}

async function isFileExecutable(absoluteFilePath) {
  const stat = await _fsExtra2.default.stat(absoluteFilePath);
  return stat.mode & _fsExtra2.default.constants.S_IXUSR; // eslint-disable-line no-bitwise
}

async function isFileSymlink(absoluteFilePath) {
  const stat = await _fsExtra2.default.lstat(absoluteFilePath);
  return stat.isSymbolicLink();
}

function shortenSha(sha) {
  return sha.slice(0, 8);
}

const classNameForStatus = exports.classNameForStatus = {
  added: 'added',
  deleted: 'removed',
  modified: 'modified',
  typechange: 'modified',
  equivalent: 'ignored'
};

/*
 * Apply any platform-specific munging to a path before presenting it as
 * a git environment variable or option.
 *
 * Convert a Windows-style "C:\foo\bar\baz" path to a "/c/foo/bar/baz" UNIX-y
 * path that the sh.exe used to execute git's credential helpers will
 * understand.
 */
function normalizeGitHelperPath(inPath) {
  if (process.platform === 'win32') {
    return inPath.replace(/\\/g, '/').replace(/^([^:]+):/, '/$1');
  } else {
    return inPath;
  }
}

/*
 * On Windows, git commands report paths with / delimiters. Convert them to \-delimited paths
 * so that Atom unifromly treats paths with native path separators.
 */
function toNativePathSep(rawPath) {
  if (process.platform !== 'win32') {
    return rawPath;
  } else {
    return rawPath.split('/').join(_path2.default.sep);
  }
}

/*
 * Convert Windows paths back to /-delimited paths to be presented to git.
 */
function toGitPathSep(rawPath) {
  if (process.platform !== 'win32') {
    return rawPath;
  } else {
    return rawPath.split(_path2.default.sep).join('/');
  }
}

function filePathEndsWith(filePath, ...segments) {
  return filePath.endsWith(_path2.default.join(...segments));
}

/**
 * Turns an array of things @kuychaco cannot eat
 * into a sentence containing things @kuychaco cannot eat
 *
 * ['toast'] => 'toast'
 * ['toast', 'eggs'] => 'toast and eggs'
 * ['toast', 'eggs', 'cheese'] => 'toast, eggs, and cheese'
 *
 * Oxford comma included because you're wrong, shut up.
 */
function toSentence(array) {
  const len = array.length;
  if (len === 1) {
    return `${array[0]}`;
  } else if (len === 2) {
    return `${array[0]} and ${array[1]}`;
  }

  return array.reduce((acc, item, idx) => {
    if (idx === 0) {
      return `${item}`;
    } else if (idx === len - 1) {
      return `${acc}, and ${item}`;
    } else {
      return `${acc}, ${item}`;
    }
  }, '');
}

function pushAtKey(map, key, value) {
  let existing = map.get(key);
  if (!existing) {
    existing = [];
    map.set(key, existing);
  }
  existing.push(value);
}

// Repository and workspace helpers

function getCommitMessagePath(repository) {
  return _path2.default.join(repository.getGitDirectoryPath(), 'ATOM_COMMIT_EDITMSG');
}

function getCommitMessageEditors(repository, workspace) {
  if (!repository.isPresent()) {
    return [];
  }
  return workspace.getTextEditors().filter(editor => editor.getPath() === getCommitMessagePath(repository));
}

function getFilePatchPaneItems({ onlyStaged, empty } = {}, workspace) {
  return workspace.getPaneItems().filter(item => {
    const isFilePatchItem = item && item.getRealItem && item.getRealItem() instanceof _multiFilePatchController2.default;
    if (onlyStaged) {
      return isFilePatchItem && item.stagingStatus === 'staged';
    } else if (empty) {
      return isFilePatchItem ? item.isEmpty() : false;
    } else {
      return isFilePatchItem;
    }
  });
}

function destroyFilePatchPaneItems({ onlyStaged } = {}, workspace) {
  const itemsToDestroy = getFilePatchPaneItems({ onlyStaged }, workspace);
  itemsToDestroy.forEach(item => item.destroy());
}

function destroyEmptyFilePatchPaneItems(workspace) {
  const itemsToDestroy = getFilePatchPaneItems({ empty: true }, workspace);
  itemsToDestroy.forEach(item => item.destroy());
}

function extractCoAuthorsAndRawCommitMessage(commitMessage) {
  const messageLines = [];
  const coAuthors = [];

  for (const line of commitMessage.split(LINE_ENDING_REGEX)) {
    const match = line.match(CO_AUTHOR_REGEX);
    if (match) {
      // eslint-disable-next-line no-unused-vars
      const [_, name, email] = match;
      coAuthors.push({ name, email });
    } else {
      messageLines.push(line);
    }
  }

  return { message: messageLines.join('\n'), coAuthors };
}

// Atom API pane item manipulation

function createItem(node, componentHolder = null, uri = null, extra = {}) {
  const holder = componentHolder || new _refHolder2.default();

  const override = _extends({
    getElement: () => node,

    getRealItem: () => holder.getOr(null),

    getRealItemPromise: () => holder.getPromise()

  }, extra);

  if (uri) {
    override.getURI = () => uri;
  }

  if (componentHolder) {
    return new Proxy(override, {
      get(target, name) {
        if (Reflect.has(target, name)) {
          return target[name];
        }

        // The {value: ...} wrapper prevents .map() from flattening a returned RefHolder.
        // If component[name] is a RefHolder, we want to return that RefHolder as-is.
        const { value } = holder.map(component => ({ value: component[name] })).getOr({ value: undefined });
        return value;
      },

      set(target, name, value) {
        return holder.map(component => {
          component[name] = value;
          return true;
        }).getOr(true);
      },

      has(target, name) {
        return holder.map(component => Reflect.has(component, name)).getOr(false) || Reflect.has(target, name);
      }
    });
  } else {
    return override;
  }
}

// Set functions

function equalSets(left, right) {
  if (left.size !== right.size) {
    return false;
  }

  for (const each of left) {
    if (!right.has(each)) {
      return false;
    }
  }

  return true;
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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