# -*- python -*-

# @HEADER
# ***********************************************************************
#
#          PyTrilinos: Python Interfaces to Trilinos Packages
#                 Copyright (2014) Sandia Corporation
#
# Under the terms of Contract DE-AC04-94AL85000 with Sandia
# Corporation, the U.S. Government retains certain rights in this
# software.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are
# met:
#
# 1. Redistributions of source code must retain the above copyright
# notice, this list of conditions and the following disclaimer.
#
# 2. Redistributions in binary form must reproduce the above copyright
# notice, this list of conditions and the following disclaimer in the
# documentation and/or other materials provided with the distribution.
#
# 3. Neither the name of the Corporation nor the names of the
# contributors may be used to endorse or promote products derived from
# this software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY SANDIA CORPORATION "AS IS" AND ANY
# EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
# PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL SANDIA CORPORATION OR THE
# CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
# EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
# PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
# PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
# LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
# NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
# SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
# Questions? Contact William F. Spotz (wfspotz@sandia.gov)
#
# ***********************************************************************
# @HEADER

#
# Define the class factory function
def typed_tuple(elementType):
    # These functions will be returned as class methods
    def new(cls, *args):
        return tuple.__new__(cls, args)
    def init(self, *args):
        for it in self:
            if not isinstance(it, self.elementType):
                raise TypeError('elements must be of type "%s", but received %s'
                                % (self.elementType.__name__, str(type(it))))
    # Class name
    name = 'tuple_of_' + elementType.__name__
    # Class docstring
    doc = '%s([arg1 [, arg2 ...]])\n\n' \
          'a tuple whose elements are type-checked to ensure that they\n' \
          'are of type "%s"' 
    doc = doc % (name, elementType.__name__)
    # Class members
    members = dict(__doc__     = doc,
                   elementType = elementType,
                   __new__     = new,
                   __init__    = init)
    # Return the manufactured class
    return type(name, (tuple,), members)

#
# Create a series of typed tuple classes for each of the standard python types
tuple_of_bool  = typed_tuple(bool )
tuple_of_int   = typed_tuple(int  )
tuple_of_float = typed_tuple(float)
tuple_of_str   = typed_tuple(str  )
tuple_of_tuple = typed_tuple(tuple)
tuple_of_list  = typed_tuple(list )
tuple_of_dict  = typed_tuple(dict )
