# 1 "src/dune_lang/dune_lexer.mll"
 
open! Stdune
open Lexer_shared

type block_string_line_kind =
  | With_escape_sequences
  | Raw

module Template = struct
  include Template

  let dummy_loc =
    { Loc.
      start = Lexing.dummy_pos
    ; stop = Lexing.dummy_pos
    }

  let add_text parts s =
    match parts with
    | Template.Text s' :: parts -> Template.Text (s' ^ s) :: parts
    | _ -> Template.Text s :: parts

  let token parts ~quoted ~start (lexbuf : Lexing.lexbuf) =
    lexbuf.lex_start_p <- start;
    match parts with
    | [] | [Text ""] ->
      error lexbuf "Internal error in the S-expression parser, \
                    please report upstream."
    | [Text s] ->
      Token.Atom (Atom.of_string s)
    | _ ->
      Token.Template
        { quoted
        ; loc = dummy_loc
        ; parts = List.rev parts
        }

  module Buffer : sig
    val new_token : unit -> unit
    val get : unit -> Token.t
    val add_var : part -> unit
    val add_text : string -> unit
    val add_text_c : char -> unit
  end = struct
    type state =
      | String
      | Template of Template.part list

    let text_buf = Buffer.create 256

    let new_token () = Buffer.clear text_buf

    let take_buf () =
      let contents = Buffer.contents text_buf in
      Buffer.clear text_buf;
      contents

    let state = ref String

    let add_buf_to_parts parts =
      match take_buf () with
      | "" -> parts
      | t -> add_text parts t

    let get () =
      match !state with
      | String -> Token.Quoted_string (take_buf ())
      | Template parts ->
        state := String;
        begin match add_buf_to_parts parts with
        | [] -> assert false
        | [Text s] -> Quoted_string s
        | parts ->
          Token.Template
            { quoted = true
            ; loc = dummy_loc
            ; parts = List.rev parts
            }
        end

    let add_var v =
      match !state with
      | String ->
        state := Template (v :: add_buf_to_parts []);
      | Template parts ->
        let parts = add_buf_to_parts parts in
        state := Template (v::parts)

    let add_text   = Buffer.add_string text_buf
    let add_text_c = Buffer.add_char text_buf
  end
end

# 96 "src/dune_lang/dune_lexer.ml"
let __ocaml_lex_tables = {
  Lexing.lex_base =
   "\000\000\249\255\250\255\251\255\252\255\001\000\006\000\255\255\
    \006\000\007\000\010\000\011\000\013\000\028\000\032\000\123\000\
    \254\255\065\000\188\000\254\255\255\255\014\000\252\255\253\255\
    \254\255\255\255\015\000\241\000\251\255\252\255\037\000\254\255\
    \255\255\018\000\253\255\021\000\071\000\243\000\191\000\254\255\
    \255\255\026\000\253\255\254\255\255\255\019\000\249\000\250\255\
    \251\255\252\255\021\000\041\000\254\255\255\255\253\255\250\000\
    \246\255\247\255\067\001\012\001\253\255\059\000\255\255\025\000\
    \254\255\045\001\090\001\100\001\122\001\145\001\203\001\028\000\
    \017\002\253\255\254\255\111\002\255\255\214\002\052\003\255\255\
    ";
  Lexing.lex_backtrk =
   "\007\000\255\255\255\255\255\255\255\255\002\000\001\000\255\255\
    \255\255\001\000\255\255\255\255\000\000\003\000\002\000\000\000\
    \255\255\002\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\003\000\255\255\255\255\255\255\003\000\255\255\
    \255\255\003\000\255\255\002\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\001\000\255\255\255\255\
    \255\255\255\255\004\000\004\000\255\255\255\255\255\255\005\000\
    \255\255\255\255\007\000\005\000\255\255\008\000\255\255\008\000\
    \255\255\005\000\003\000\005\000\007\000\006\000\007\000\000\000\
    \255\255\255\255\255\255\002\000\255\255\255\255\255\255\255\255\
    ";
  Lexing.lex_default =
   "\255\255\000\000\000\000\000\000\000\000\005\000\255\255\000\000\
    \255\255\255\255\255\255\255\255\012\000\255\255\255\255\255\255\
    \000\000\255\255\255\255\000\000\000\000\022\000\000\000\000\000\
    \000\000\000\000\255\255\029\000\000\000\000\000\255\255\000\000\
    \000\000\255\255\000\000\255\255\255\255\255\255\255\255\000\000\
    \000\000\043\000\000\000\000\000\000\000\255\255\048\000\000\000\
    \000\000\000\000\255\255\255\255\000\000\000\000\000\000\057\000\
    \000\000\000\000\255\255\255\255\000\000\255\255\000\000\255\255\
    \000\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \073\000\000\000\000\000\255\255\000\000\255\255\255\255\000\000\
    ";
  Lexing.lex_trans =
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\006\000\007\000\255\255\006\000\008\000\255\255\006\000\
    \007\000\010\000\006\000\010\000\011\000\010\000\010\000\255\255\
    \025\000\025\000\255\255\026\000\032\000\044\000\037\000\049\000\
    \006\000\037\000\002\000\062\000\044\000\071\000\006\000\045\000\
    \004\000\003\000\010\000\000\000\000\000\000\000\024\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\037\000\000\000\036\000\
    \000\000\000\000\000\000\005\000\071\000\015\000\000\000\015\000\
    \015\000\014\000\015\000\015\000\000\000\012\000\015\000\015\000\
    \015\000\015\000\015\000\015\000\015\000\015\000\015\000\015\000\
    \015\000\015\000\015\000\015\000\015\000\015\000\015\000\000\000\
    \015\000\015\000\015\000\015\000\015\000\015\000\015\000\015\000\
    \015\000\015\000\015\000\015\000\015\000\015\000\015\000\015\000\
    \015\000\015\000\015\000\015\000\015\000\015\000\015\000\015\000\
    \015\000\015\000\015\000\015\000\015\000\015\000\015\000\015\000\
    \015\000\015\000\015\000\015\000\015\000\015\000\015\000\015\000\
    \015\000\015\000\015\000\015\000\015\000\015\000\015\000\015\000\
    \015\000\015\000\015\000\015\000\015\000\015\000\015\000\015\000\
    \015\000\015\000\015\000\015\000\015\000\015\000\015\000\015\000\
    \015\000\015\000\015\000\016\000\015\000\018\000\015\000\015\000\
    \034\000\015\000\015\000\038\000\054\000\015\000\015\000\015\000\
    \015\000\015\000\015\000\015\000\015\000\015\000\015\000\015\000\
    \015\000\015\000\015\000\015\000\015\000\015\000\064\000\015\000\
    \015\000\015\000\015\000\015\000\015\000\015\000\015\000\015\000\
    \015\000\015\000\015\000\015\000\015\000\015\000\015\000\015\000\
    \015\000\015\000\015\000\015\000\015\000\015\000\015\000\015\000\
    \015\000\015\000\015\000\015\000\015\000\015\000\015\000\015\000\
    \015\000\015\000\015\000\015\000\015\000\015\000\015\000\015\000\
    \015\000\015\000\015\000\015\000\015\000\015\000\015\000\015\000\
    \015\000\015\000\015\000\015\000\015\000\015\000\015\000\015\000\
    \015\000\015\000\015\000\015\000\015\000\015\000\015\000\015\000\
    \015\000\015\000\019\000\032\000\037\000\039\000\033\000\037\000\
    \001\000\255\255\000\000\049\000\062\000\000\000\050\000\063\000\
    \000\000\000\000\000\000\000\000\000\000\255\255\023\000\000\000\
    \000\000\000\000\000\000\037\000\000\000\036\000\030\000\000\000\
    \000\000\000\000\042\000\053\000\060\000\000\000\051\000\061\000\
    \000\000\060\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\059\000\059\000\059\000\059\000\059\000\059\000\
    \059\000\059\000\059\000\059\000\000\000\000\000\000\000\000\000\
    \020\000\000\000\000\000\040\000\065\000\065\000\065\000\065\000\
    \065\000\065\000\065\000\065\000\065\000\065\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\031\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\052\000\060\000\000\000\
    \000\000\000\000\000\000\000\000\060\000\066\000\066\000\066\000\
    \066\000\066\000\066\000\066\000\066\000\066\000\066\000\000\000\
    \060\000\000\000\000\000\000\000\060\000\000\000\060\000\000\000\
    \000\000\000\000\058\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\069\000\069\000\069\000\069\000\069\000\069\000\
    \069\000\069\000\069\000\069\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\069\000\069\000\069\000\069\000\069\000\
    \069\000\070\000\070\000\070\000\070\000\070\000\070\000\070\000\
    \070\000\070\000\070\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\070\000\070\000\070\000\070\000\070\000\070\000\
    \000\000\000\000\000\000\069\000\069\000\069\000\069\000\069\000\
    \069\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\028\000\070\000\070\000\070\000\070\000\070\000\070\000\
    \000\000\047\000\056\000\070\000\070\000\070\000\070\000\070\000\
    \070\000\070\000\070\000\070\000\070\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\070\000\070\000\070\000\070\000\
    \070\000\070\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\070\000\070\000\070\000\070\000\
    \070\000\070\000\075\000\000\000\075\000\075\000\000\000\075\000\
    \075\000\000\000\000\000\075\000\075\000\075\000\075\000\075\000\
    \075\000\075\000\075\000\075\000\075\000\075\000\075\000\075\000\
    \075\000\075\000\075\000\000\000\000\000\075\000\075\000\075\000\
    \075\000\075\000\075\000\075\000\075\000\075\000\075\000\075\000\
    \075\000\075\000\075\000\075\000\075\000\075\000\075\000\075\000\
    \075\000\075\000\075\000\075\000\075\000\075\000\075\000\075\000\
    \075\000\075\000\075\000\075\000\075\000\075\000\075\000\075\000\
    \075\000\075\000\075\000\075\000\075\000\075\000\075\000\075\000\
    \075\000\075\000\075\000\075\000\075\000\075\000\075\000\075\000\
    \075\000\075\000\075\000\075\000\075\000\075\000\075\000\075\000\
    \075\000\075\000\075\000\075\000\000\000\075\000\074\000\075\000\
    \078\000\000\000\078\000\078\000\000\000\078\000\078\000\000\000\
    \000\000\078\000\078\000\078\000\078\000\078\000\078\000\078\000\
    \078\000\078\000\078\000\078\000\078\000\078\000\078\000\078\000\
    \078\000\077\000\000\000\078\000\078\000\078\000\078\000\078\000\
    \078\000\078\000\078\000\078\000\078\000\078\000\078\000\078\000\
    \078\000\078\000\078\000\078\000\078\000\078\000\078\000\078\000\
    \078\000\078\000\078\000\078\000\078\000\078\000\078\000\078\000\
    \078\000\078\000\078\000\078\000\078\000\078\000\078\000\078\000\
    \078\000\078\000\078\000\078\000\078\000\078\000\078\000\078\000\
    \078\000\078\000\078\000\078\000\078\000\078\000\078\000\078\000\
    \078\000\078\000\078\000\078\000\078\000\078\000\078\000\078\000\
    \078\000\078\000\000\000\078\000\076\000\078\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\077\000\
    \000\000\077\000\077\000\000\000\077\000\077\000\000\000\000\000\
    \077\000\077\000\077\000\077\000\077\000\077\000\077\000\077\000\
    \077\000\077\000\077\000\077\000\077\000\077\000\077\000\077\000\
    \077\000\074\000\077\000\077\000\077\000\077\000\077\000\077\000\
    \077\000\077\000\077\000\077\000\077\000\077\000\077\000\077\000\
    \077\000\077\000\077\000\077\000\077\000\077\000\077\000\077\000\
    \077\000\077\000\077\000\077\000\077\000\077\000\077\000\077\000\
    \077\000\077\000\077\000\077\000\077\000\077\000\077\000\077\000\
    \077\000\077\000\077\000\077\000\077\000\077\000\077\000\077\000\
    \077\000\077\000\077\000\077\000\077\000\077\000\077\000\077\000\
    \077\000\077\000\077\000\077\000\077\000\077\000\077\000\077\000\
    \077\000\000\000\077\000\079\000\077\000\078\000\000\000\078\000\
    \078\000\000\000\078\000\078\000\000\000\000\000\078\000\078\000\
    \078\000\078\000\078\000\078\000\078\000\078\000\078\000\078\000\
    \078\000\078\000\078\000\078\000\078\000\078\000\077\000\000\000\
    \078\000\078\000\078\000\078\000\078\000\078\000\078\000\078\000\
    \078\000\078\000\078\000\078\000\078\000\078\000\078\000\078\000\
    \078\000\078\000\078\000\078\000\078\000\078\000\078\000\078\000\
    \078\000\078\000\078\000\078\000\078\000\078\000\078\000\078\000\
    \078\000\078\000\078\000\078\000\078\000\078\000\078\000\078\000\
    \078\000\078\000\078\000\078\000\078\000\078\000\078\000\078\000\
    \078\000\078\000\078\000\078\000\078\000\078\000\078\000\078\000\
    \078\000\078\000\078\000\078\000\078\000\078\000\078\000\000\000\
    \078\000\076\000\078\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000";
  Lexing.lex_check =
   "\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\000\000\000\000\005\000\000\000\000\000\005\000\006\000\
    \008\000\009\000\006\000\010\000\009\000\011\000\010\000\012\000\
    \021\000\026\000\012\000\021\000\033\000\045\000\035\000\050\000\
    \000\000\035\000\000\000\063\000\041\000\071\000\006\000\041\000\
    \000\000\000\000\010\000\255\255\255\255\255\255\021\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\035\000\255\255\035\000\
    \255\255\255\255\255\255\000\000\071\000\013\000\255\255\013\000\
    \013\000\013\000\013\000\013\000\255\255\010\000\013\000\013\000\
    \013\000\013\000\013\000\013\000\013\000\013\000\013\000\013\000\
    \013\000\013\000\013\000\013\000\013\000\013\000\013\000\255\255\
    \013\000\013\000\013\000\013\000\013\000\013\000\013\000\013\000\
    \013\000\013\000\013\000\013\000\013\000\013\000\013\000\013\000\
    \013\000\013\000\013\000\013\000\013\000\013\000\013\000\013\000\
    \013\000\013\000\013\000\013\000\013\000\013\000\013\000\013\000\
    \013\000\013\000\013\000\013\000\013\000\013\000\013\000\013\000\
    \013\000\013\000\013\000\013\000\013\000\013\000\013\000\013\000\
    \013\000\013\000\013\000\013\000\013\000\013\000\013\000\013\000\
    \013\000\013\000\013\000\013\000\013\000\013\000\013\000\013\000\
    \013\000\013\000\013\000\014\000\015\000\017\000\015\000\015\000\
    \030\000\015\000\015\000\036\000\051\000\015\000\015\000\015\000\
    \015\000\015\000\015\000\015\000\015\000\015\000\015\000\015\000\
    \015\000\015\000\015\000\015\000\015\000\015\000\061\000\015\000\
    \015\000\015\000\015\000\015\000\015\000\015\000\015\000\015\000\
    \015\000\015\000\015\000\015\000\015\000\015\000\015\000\015\000\
    \015\000\015\000\015\000\015\000\015\000\015\000\015\000\015\000\
    \015\000\015\000\015\000\015\000\015\000\015\000\015\000\015\000\
    \015\000\015\000\015\000\015\000\015\000\015\000\015\000\015\000\
    \015\000\015\000\015\000\015\000\015\000\015\000\015\000\015\000\
    \015\000\015\000\015\000\015\000\015\000\015\000\015\000\015\000\
    \015\000\015\000\015\000\015\000\015\000\015\000\015\000\015\000\
    \015\000\015\000\018\000\027\000\037\000\038\000\027\000\037\000\
    \000\000\005\000\255\255\046\000\055\000\255\255\046\000\055\000\
    \255\255\255\255\255\255\255\255\255\255\012\000\021\000\255\255\
    \255\255\255\255\255\255\037\000\255\255\037\000\027\000\255\255\
    \255\255\255\255\041\000\046\000\055\000\255\255\046\000\055\000\
    \255\255\055\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\255\255\255\255\255\255\255\255\
    \018\000\255\255\255\255\038\000\059\000\059\000\059\000\059\000\
    \059\000\059\000\059\000\059\000\059\000\059\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\027\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\046\000\055\000\255\255\
    \255\255\255\255\255\255\255\255\055\000\065\000\065\000\065\000\
    \065\000\065\000\065\000\065\000\065\000\065\000\065\000\255\255\
    \055\000\255\255\255\255\255\255\055\000\255\255\055\000\255\255\
    \255\255\255\255\055\000\058\000\058\000\058\000\058\000\058\000\
    \058\000\058\000\058\000\058\000\058\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\058\000\058\000\058\000\058\000\
    \058\000\058\000\066\000\066\000\066\000\066\000\066\000\066\000\
    \066\000\066\000\066\000\066\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\058\000\058\000\058\000\058\000\
    \058\000\058\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\068\000\068\000\068\000\068\000\068\000\
    \068\000\069\000\069\000\069\000\069\000\069\000\069\000\069\000\
    \069\000\069\000\069\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\069\000\069\000\069\000\069\000\069\000\069\000\
    \255\255\255\255\255\255\068\000\068\000\068\000\068\000\068\000\
    \068\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\027\000\069\000\069\000\069\000\069\000\069\000\069\000\
    \255\255\046\000\055\000\070\000\070\000\070\000\070\000\070\000\
    \070\000\070\000\070\000\070\000\070\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\070\000\070\000\070\000\070\000\
    \070\000\070\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\070\000\070\000\070\000\070\000\
    \070\000\070\000\072\000\255\255\072\000\072\000\255\255\072\000\
    \072\000\255\255\255\255\072\000\072\000\072\000\072\000\072\000\
    \072\000\072\000\072\000\072\000\072\000\072\000\072\000\072\000\
    \072\000\072\000\072\000\255\255\255\255\072\000\072\000\072\000\
    \072\000\072\000\072\000\072\000\072\000\072\000\072\000\072\000\
    \072\000\072\000\072\000\072\000\072\000\072\000\072\000\072\000\
    \072\000\072\000\072\000\072\000\072\000\072\000\072\000\072\000\
    \072\000\072\000\072\000\072\000\072\000\072\000\072\000\072\000\
    \072\000\072\000\072\000\072\000\072\000\072\000\072\000\072\000\
    \072\000\072\000\072\000\072\000\072\000\072\000\072\000\072\000\
    \072\000\072\000\072\000\072\000\072\000\072\000\072\000\072\000\
    \072\000\072\000\072\000\072\000\255\255\072\000\072\000\072\000\
    \075\000\255\255\075\000\075\000\255\255\075\000\075\000\255\255\
    \255\255\075\000\075\000\075\000\075\000\075\000\075\000\075\000\
    \075\000\075\000\075\000\075\000\075\000\075\000\075\000\075\000\
    \075\000\075\000\255\255\075\000\075\000\075\000\075\000\075\000\
    \075\000\075\000\075\000\075\000\075\000\075\000\075\000\075\000\
    \075\000\075\000\075\000\075\000\075\000\075\000\075\000\075\000\
    \075\000\075\000\075\000\075\000\075\000\075\000\075\000\075\000\
    \075\000\075\000\075\000\075\000\075\000\075\000\075\000\075\000\
    \075\000\075\000\075\000\075\000\075\000\075\000\075\000\075\000\
    \075\000\075\000\075\000\075\000\075\000\075\000\075\000\075\000\
    \075\000\075\000\075\000\075\000\075\000\075\000\075\000\075\000\
    \075\000\075\000\255\255\075\000\075\000\075\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\077\000\
    \255\255\077\000\077\000\255\255\077\000\077\000\255\255\255\255\
    \077\000\077\000\077\000\077\000\077\000\077\000\077\000\077\000\
    \077\000\077\000\077\000\077\000\077\000\077\000\077\000\077\000\
    \077\000\072\000\077\000\077\000\077\000\077\000\077\000\077\000\
    \077\000\077\000\077\000\077\000\077\000\077\000\077\000\077\000\
    \077\000\077\000\077\000\077\000\077\000\077\000\077\000\077\000\
    \077\000\077\000\077\000\077\000\077\000\077\000\077\000\077\000\
    \077\000\077\000\077\000\077\000\077\000\077\000\077\000\077\000\
    \077\000\077\000\077\000\077\000\077\000\077\000\077\000\077\000\
    \077\000\077\000\077\000\077\000\077\000\077\000\077\000\077\000\
    \077\000\077\000\077\000\077\000\077\000\077\000\077\000\077\000\
    \077\000\255\255\077\000\077\000\077\000\078\000\255\255\078\000\
    \078\000\255\255\078\000\078\000\255\255\255\255\078\000\078\000\
    \078\000\078\000\078\000\078\000\078\000\078\000\078\000\078\000\
    \078\000\078\000\078\000\078\000\078\000\078\000\078\000\255\255\
    \078\000\078\000\078\000\078\000\078\000\078\000\078\000\078\000\
    \078\000\078\000\078\000\078\000\078\000\078\000\078\000\078\000\
    \078\000\078\000\078\000\078\000\078\000\078\000\078\000\078\000\
    \078\000\078\000\078\000\078\000\078\000\078\000\078\000\078\000\
    \078\000\078\000\078\000\078\000\078\000\078\000\078\000\078\000\
    \078\000\078\000\078\000\078\000\078\000\078\000\078\000\078\000\
    \078\000\078\000\078\000\078\000\078\000\078\000\078\000\078\000\
    \078\000\078\000\078\000\078\000\078\000\078\000\078\000\255\255\
    \078\000\078\000\078\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255";
  Lexing.lex_base_code =
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\092\000\015\000\184\000\020\001\028\000\
    ";
  Lexing.lex_backtrk_code =
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\004\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    ";
  Lexing.lex_default_code =
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    ";
  Lexing.lex_trans_code =
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\007\000\000\000\007\000\007\000\000\000\007\000\007\000\
    \000\000\000\000\007\000\007\000\007\000\007\000\007\000\007\000\
    \007\000\007\000\007\000\007\000\007\000\007\000\007\000\007\000\
    \007\000\007\000\000\000\001\000\007\000\007\000\007\000\007\000\
    \007\000\007\000\007\000\007\000\007\000\007\000\007\000\007\000\
    \007\000\007\000\007\000\007\000\007\000\007\000\007\000\007\000\
    \007\000\007\000\007\000\007\000\007\000\007\000\007\000\007\000\
    \007\000\007\000\007\000\007\000\007\000\007\000\007\000\007\000\
    \007\000\007\000\007\000\007\000\007\000\007\000\007\000\007\000\
    \007\000\007\000\007\000\007\000\007\000\007\000\007\000\007\000\
    \007\000\007\000\007\000\007\000\007\000\007\000\007\000\007\000\
    \007\000\007\000\007\000\000\000\007\000\007\000\007\000\007\000\
    \007\000\000\000\007\000\007\000\000\000\000\000\007\000\007\000\
    \007\000\007\000\007\000\007\000\007\000\007\000\007\000\007\000\
    \007\000\007\000\007\000\007\000\007\000\007\000\010\000\000\000\
    \007\000\007\000\007\000\007\000\007\000\007\000\007\000\007\000\
    \007\000\007\000\007\000\007\000\007\000\007\000\007\000\007\000\
    \007\000\007\000\007\000\007\000\007\000\007\000\007\000\007\000\
    \007\000\007\000\007\000\007\000\007\000\007\000\007\000\007\000\
    \007\000\007\000\007\000\007\000\007\000\007\000\007\000\007\000\
    \007\000\007\000\007\000\007\000\007\000\007\000\007\000\007\000\
    \007\000\007\000\007\000\007\000\007\000\007\000\007\000\007\000\
    \007\000\007\000\007\000\007\000\007\000\007\000\007\000\000\000\
    \007\000\020\000\007\000\020\000\020\000\000\000\020\000\020\000\
    \000\000\000\000\020\000\020\000\020\000\020\000\020\000\020\000\
    \020\000\020\000\020\000\020\000\020\000\020\000\020\000\020\000\
    \020\000\020\000\020\000\000\000\020\000\020\000\020\000\020\000\
    \020\000\020\000\020\000\020\000\020\000\020\000\020\000\020\000\
    \020\000\020\000\020\000\020\000\020\000\020\000\020\000\020\000\
    \020\000\020\000\020\000\020\000\020\000\020\000\020\000\020\000\
    \020\000\020\000\020\000\020\000\020\000\020\000\020\000\020\000\
    \020\000\020\000\020\000\020\000\020\000\020\000\020\000\020\000\
    \020\000\020\000\020\000\020\000\020\000\020\000\020\000\020\000\
    \020\000\020\000\020\000\020\000\020\000\020\000\020\000\020\000\
    \020\000\020\000\020\000\000\000\020\000\007\000\020\000\007\000\
    \007\000\000\000\007\000\007\000\000\000\000\000\007\000\007\000\
    \007\000\007\000\007\000\007\000\007\000\007\000\007\000\007\000\
    \007\000\007\000\007\000\007\000\007\000\007\000\023\000\000\000\
    \007\000\007\000\007\000\007\000\007\000\007\000\007\000\007\000\
    \007\000\007\000\007\000\007\000\007\000\007\000\007\000\007\000\
    \007\000\007\000\007\000\007\000\007\000\007\000\007\000\007\000\
    \007\000\007\000\007\000\007\000\007\000\007\000\007\000\007\000\
    \007\000\007\000\007\000\007\000\007\000\007\000\007\000\007\000\
    \007\000\007\000\007\000\007\000\007\000\007\000\007\000\007\000\
    \007\000\007\000\007\000\007\000\007\000\007\000\007\000\007\000\
    \007\000\007\000\007\000\007\000\007\000\007\000\007\000\000\000\
    \007\000\000\000\007\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000";
  Lexing.lex_check_code =
   "\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\072\000\255\255\072\000\072\000\255\255\072\000\072\000\
    \255\255\255\255\072\000\072\000\072\000\072\000\072\000\072\000\
    \072\000\072\000\072\000\072\000\072\000\072\000\072\000\072\000\
    \072\000\072\000\255\255\010\000\072\000\072\000\072\000\072\000\
    \072\000\072\000\072\000\072\000\072\000\072\000\072\000\072\000\
    \072\000\072\000\072\000\072\000\072\000\072\000\072\000\072\000\
    \072\000\072\000\072\000\072\000\072\000\072\000\072\000\072\000\
    \072\000\072\000\072\000\072\000\072\000\072\000\072\000\072\000\
    \072\000\072\000\072\000\072\000\072\000\072\000\072\000\072\000\
    \072\000\072\000\072\000\072\000\072\000\072\000\072\000\072\000\
    \072\000\072\000\072\000\072\000\072\000\072\000\072\000\072\000\
    \072\000\072\000\072\000\255\255\072\000\075\000\072\000\075\000\
    \075\000\255\255\075\000\075\000\255\255\255\255\075\000\075\000\
    \075\000\075\000\075\000\075\000\075\000\075\000\075\000\075\000\
    \075\000\075\000\075\000\075\000\075\000\075\000\075\000\255\255\
    \075\000\075\000\075\000\075\000\075\000\075\000\075\000\075\000\
    \075\000\075\000\075\000\075\000\075\000\075\000\075\000\075\000\
    \075\000\075\000\075\000\075\000\075\000\075\000\075\000\075\000\
    \075\000\075\000\075\000\075\000\075\000\075\000\075\000\075\000\
    \075\000\075\000\075\000\075\000\075\000\075\000\075\000\075\000\
    \075\000\075\000\075\000\075\000\075\000\075\000\075\000\075\000\
    \075\000\075\000\075\000\075\000\075\000\075\000\075\000\075\000\
    \075\000\075\000\075\000\075\000\075\000\075\000\075\000\255\255\
    \075\000\077\000\075\000\077\000\077\000\255\255\077\000\077\000\
    \255\255\255\255\077\000\077\000\077\000\077\000\077\000\077\000\
    \077\000\077\000\077\000\077\000\077\000\077\000\077\000\077\000\
    \077\000\077\000\077\000\255\255\077\000\077\000\077\000\077\000\
    \077\000\077\000\077\000\077\000\077\000\077\000\077\000\077\000\
    \077\000\077\000\077\000\077\000\077\000\077\000\077\000\077\000\
    \077\000\077\000\077\000\077\000\077\000\077\000\077\000\077\000\
    \077\000\077\000\077\000\077\000\077\000\077\000\077\000\077\000\
    \077\000\077\000\077\000\077\000\077\000\077\000\077\000\077\000\
    \077\000\077\000\077\000\077\000\077\000\077\000\077\000\077\000\
    \077\000\077\000\077\000\077\000\077\000\077\000\077\000\077\000\
    \077\000\077\000\077\000\255\255\077\000\078\000\077\000\078\000\
    \078\000\255\255\078\000\078\000\255\255\255\255\078\000\078\000\
    \078\000\078\000\078\000\078\000\078\000\078\000\078\000\078\000\
    \078\000\078\000\078\000\078\000\078\000\078\000\078\000\255\255\
    \078\000\078\000\078\000\078\000\078\000\078\000\078\000\078\000\
    \078\000\078\000\078\000\078\000\078\000\078\000\078\000\078\000\
    \078\000\078\000\078\000\078\000\078\000\078\000\078\000\078\000\
    \078\000\078\000\078\000\078\000\078\000\078\000\078\000\078\000\
    \078\000\078\000\078\000\078\000\078\000\078\000\078\000\078\000\
    \078\000\078\000\078\000\078\000\078\000\078\000\078\000\078\000\
    \078\000\078\000\078\000\078\000\078\000\078\000\078\000\078\000\
    \078\000\078\000\078\000\078\000\078\000\078\000\078\000\255\255\
    \078\000\255\255\078\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255";
  Lexing.lex_code =
   "\255\001\255\255\000\001\255\003\255\255\005\255\004\255\255\002\
    \255\000\003\255\005\255\255\004\255\005\255\255\002\004\001\005\
    \000\003\255";
}

let rec token with_comments lexbuf =
   __ocaml_lex_token_rec with_comments lexbuf 0
and __ocaml_lex_token_rec with_comments lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 107 "src/dune_lang/dune_lexer.mll"
    ( Lexing.new_line lexbuf; token with_comments lexbuf )
# 591 "src/dune_lang/dune_lexer.ml"

  | 1 ->
# 109 "src/dune_lang/dune_lexer.mll"
    ( token with_comments lexbuf )
# 596 "src/dune_lang/dune_lexer.ml"

  | 2 ->
# 111 "src/dune_lang/dune_lexer.mll"
    ( if with_comments then
        comment_trail [String.drop (Lexing.lexeme lexbuf) 1] lexbuf
      else
        token with_comments lexbuf
    )
# 605 "src/dune_lang/dune_lexer.ml"

  | 3 ->
# 117 "src/dune_lang/dune_lexer.mll"
    ( Token.Lparen )
# 610 "src/dune_lang/dune_lexer.ml"

  | 4 ->
# 119 "src/dune_lang/dune_lexer.mll"
    ( Rparen )
# 615 "src/dune_lang/dune_lexer.ml"

  | 5 ->
# 121 "src/dune_lang/dune_lexer.mll"
    ( let start = Lexing.lexeme_start_p lexbuf in
      Template.Buffer.new_token ();
      let token = start_quoted_string lexbuf in
      lexbuf.lex_start_p <- start;
      token
    )
# 625 "src/dune_lang/dune_lexer.ml"

  | 6 ->
# 128 "src/dune_lang/dune_lexer.mll"
    ( Eof )
# 630 "src/dune_lang/dune_lexer.ml"

  | 7 ->
# 130 "src/dune_lang/dune_lexer.mll"
    ( atom [] (Lexing.lexeme_start_p lexbuf) lexbuf )
# 635 "src/dune_lang/dune_lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_token_rec with_comments lexbuf __ocaml_lex_state

and comment_trail acc lexbuf =
  lexbuf.Lexing.lex_mem <- Array.make 2 (-1); __ocaml_lex_comment_trail_rec acc lexbuf 9
and __ocaml_lex_comment_trail_rec acc lexbuf __ocaml_lex_state =
  match Lexing.new_engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
let
# 133 "src/dune_lang/dune_lexer.mll"
                                        s
# 648 "src/dune_lang/dune_lexer.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_mem.(0) lexbuf.Lexing.lex_curr_pos in
# 134 "src/dune_lang/dune_lexer.mll"
    ( comment_trail (s :: acc) lexbuf )
# 652 "src/dune_lang/dune_lexer.ml"

  | 1 ->
# 136 "src/dune_lang/dune_lexer.mll"
    ( Token.Comment (Lines (List.rev acc)) )
# 657 "src/dune_lang/dune_lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_comment_trail_rec acc lexbuf __ocaml_lex_state

and atom acc start lexbuf =
   __ocaml_lex_atom_rec acc start lexbuf 13
and __ocaml_lex_atom_rec acc start lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
let
# 139 "src/dune_lang/dune_lexer.mll"
                          s
# 670 "src/dune_lang/dune_lexer.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos lexbuf.Lexing.lex_curr_pos in
# 140 "src/dune_lang/dune_lexer.mll"
    ( atom (Template.add_text acc s) start lexbuf )
# 674 "src/dune_lang/dune_lexer.ml"

  | 1 ->
# 142 "src/dune_lang/dune_lexer.mll"
    ( atom ((template_variable lexbuf) :: acc) start lexbuf )
# 679 "src/dune_lang/dune_lexer.ml"

  | 2 ->
# 144 "src/dune_lang/dune_lexer.mll"
    ( atom (Template.add_text acc "%") start lexbuf )
# 684 "src/dune_lang/dune_lexer.ml"

  | 3 ->
# 146 "src/dune_lang/dune_lexer.mll"
    ( Template.token acc ~quoted:false ~start lexbuf )
# 689 "src/dune_lang/dune_lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_atom_rec acc start lexbuf __ocaml_lex_state

and start_quoted_string lexbuf =
   __ocaml_lex_start_quoted_string_rec lexbuf 17
and __ocaml_lex_start_quoted_string_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 150 "src/dune_lang/dune_lexer.mll"
    ( block_string_start With_escape_sequences lexbuf )
# 701 "src/dune_lang/dune_lexer.ml"

  | 1 ->
# 152 "src/dune_lang/dune_lexer.mll"
    ( block_string_start Raw lexbuf )
# 706 "src/dune_lang/dune_lexer.ml"

  | 2 ->
# 154 "src/dune_lang/dune_lexer.mll"
    ( quoted_string lexbuf )
# 711 "src/dune_lang/dune_lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_start_quoted_string_rec lexbuf __ocaml_lex_state

and block_string_start kind lexbuf =
   __ocaml_lex_block_string_start_rec kind lexbuf 21
and __ocaml_lex_block_string_start_rec kind lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
let
# 157 "src/dune_lang/dune_lexer.mll"
               s
# 724 "src/dune_lang/dune_lexer.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos lexbuf.Lexing.lex_curr_pos in
# 158 "src/dune_lang/dune_lexer.mll"
    ( Lexing.new_line lexbuf;
      Template.Buffer.add_text s;
      block_string_after_newline lexbuf
    )
# 731 "src/dune_lang/dune_lexer.ml"

  | 1 ->
# 163 "src/dune_lang/dune_lexer.mll"
    ( match kind with
      | With_escape_sequences -> block_string lexbuf
      | Raw -> raw_block_string lexbuf
    )
# 739 "src/dune_lang/dune_lexer.ml"

  | 2 ->
# 168 "src/dune_lang/dune_lexer.mll"
    ( Template.Buffer.get () )
# 744 "src/dune_lang/dune_lexer.ml"

  | 3 ->
# 170 "src/dune_lang/dune_lexer.mll"
    ( error lexbuf "There must be at least one space after \"\\|"
    )
# 750 "src/dune_lang/dune_lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_block_string_start_rec kind lexbuf __ocaml_lex_state

and block_string lexbuf =
   __ocaml_lex_block_string_rec lexbuf 27
and __ocaml_lex_block_string_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
let
# 174 "src/dune_lang/dune_lexer.mll"
               s
# 763 "src/dune_lang/dune_lexer.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos lexbuf.Lexing.lex_curr_pos in
# 175 "src/dune_lang/dune_lexer.mll"
    ( Lexing.new_line lexbuf;
      Template.Buffer.add_text s;
      block_string_after_newline lexbuf
    )
# 770 "src/dune_lang/dune_lexer.ml"

  | 1 ->
# 180 "src/dune_lang/dune_lexer.mll"
    ( match escape_sequence lexbuf with
      | Newline -> block_string_after_newline lexbuf
      | Other   -> block_string               lexbuf
    )
# 778 "src/dune_lang/dune_lexer.ml"

  | 2 ->
# 184 "src/dune_lang/dune_lexer.mll"
         (
      let var = template_variable lexbuf in
      Template.Buffer.add_var var;
      block_string lexbuf
    )
# 787 "src/dune_lang/dune_lexer.ml"

  | 3 ->
let
# 189 "src/dune_lang/dune_lexer.mll"
         c
# 793 "src/dune_lang/dune_lexer.ml"
= Lexing.sub_lexeme_char lexbuf lexbuf.Lexing.lex_start_pos in
# 190 "src/dune_lang/dune_lexer.mll"
    ( Template.Buffer.add_text_c c;
      block_string lexbuf
    )
# 799 "src/dune_lang/dune_lexer.ml"

  | 4 ->
# 194 "src/dune_lang/dune_lexer.mll"
    ( Template.Buffer.get ()
    )
# 805 "src/dune_lang/dune_lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_block_string_rec lexbuf __ocaml_lex_state

and block_string_after_newline lexbuf =
   __ocaml_lex_block_string_after_newline_rec lexbuf 35
and __ocaml_lex_block_string_after_newline_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 199 "src/dune_lang/dune_lexer.mll"
    ( block_string_start With_escape_sequences lexbuf )
# 817 "src/dune_lang/dune_lexer.ml"

  | 1 ->
# 201 "src/dune_lang/dune_lexer.mll"
    ( block_string_start Raw lexbuf )
# 822 "src/dune_lang/dune_lexer.ml"

  | 2 ->
# 203 "src/dune_lang/dune_lexer.mll"
    ( Template.Buffer.get ()
    )
# 828 "src/dune_lang/dune_lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_block_string_after_newline_rec lexbuf __ocaml_lex_state

and raw_block_string lexbuf =
   __ocaml_lex_raw_block_string_rec lexbuf 41
and __ocaml_lex_raw_block_string_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
let
# 207 "src/dune_lang/dune_lexer.mll"
               s
# 841 "src/dune_lang/dune_lexer.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos lexbuf.Lexing.lex_curr_pos in
# 208 "src/dune_lang/dune_lexer.mll"
    ( Lexing.new_line lexbuf;
      Template.Buffer.add_text s;
      block_string_after_newline lexbuf
    )
# 848 "src/dune_lang/dune_lexer.ml"

  | 1 ->
let
# 212 "src/dune_lang/dune_lexer.mll"
         c
# 854 "src/dune_lang/dune_lexer.ml"
= Lexing.sub_lexeme_char lexbuf lexbuf.Lexing.lex_start_pos in
# 213 "src/dune_lang/dune_lexer.mll"
    ( Template.Buffer.add_text_c c;
      raw_block_string lexbuf
    )
# 860 "src/dune_lang/dune_lexer.ml"

  | 2 ->
# 217 "src/dune_lang/dune_lexer.mll"
    ( Template.Buffer.get ()
    )
# 866 "src/dune_lang/dune_lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_raw_block_string_rec lexbuf __ocaml_lex_state

and quoted_string lexbuf =
   __ocaml_lex_quoted_string_rec lexbuf 46
and __ocaml_lex_quoted_string_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 222 "src/dune_lang/dune_lexer.mll"
    ( Template.Buffer.get () )
# 878 "src/dune_lang/dune_lexer.ml"

  | 1 ->
# 224 "src/dune_lang/dune_lexer.mll"
    ( match escape_sequence lexbuf with
      | Newline -> quoted_string_after_escaped_newline lexbuf
      | Other   -> quoted_string                       lexbuf
    )
# 886 "src/dune_lang/dune_lexer.ml"

  | 2 ->
# 229 "src/dune_lang/dune_lexer.mll"
    ( Template.Buffer.add_var (template_variable lexbuf);
      quoted_string lexbuf
    )
# 893 "src/dune_lang/dune_lexer.ml"

  | 3 ->
let
# 232 "src/dune_lang/dune_lexer.mll"
               s
# 899 "src/dune_lang/dune_lexer.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos lexbuf.Lexing.lex_curr_pos in
# 233 "src/dune_lang/dune_lexer.mll"
    ( Lexing.new_line lexbuf;
      Template.Buffer.add_text s;
      quoted_string lexbuf
    )
# 906 "src/dune_lang/dune_lexer.ml"

  | 4 ->
let
# 237 "src/dune_lang/dune_lexer.mll"
         c
# 912 "src/dune_lang/dune_lexer.ml"
= Lexing.sub_lexeme_char lexbuf lexbuf.Lexing.lex_start_pos in
# 238 "src/dune_lang/dune_lexer.mll"
    ( Template.Buffer.add_text_c c;
      quoted_string lexbuf
    )
# 918 "src/dune_lang/dune_lexer.ml"

  | 5 ->
# 242 "src/dune_lang/dune_lexer.mll"
    ( error lexbuf "unterminated quoted string"
    )
# 924 "src/dune_lang/dune_lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_quoted_string_rec lexbuf __ocaml_lex_state

and escape_sequence lexbuf =
   __ocaml_lex_escape_sequence_rec lexbuf 55
and __ocaml_lex_escape_sequence_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 247 "src/dune_lang/dune_lexer.mll"
    ( Lexing.new_line lexbuf;
      Newline )
# 937 "src/dune_lang/dune_lexer.ml"

  | 1 ->
let
# 249 "src/dune_lang/dune_lexer.mll"
            s
# 943 "src/dune_lang/dune_lexer.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos (lexbuf.Lexing.lex_start_pos + 2) in
# 250 "src/dune_lang/dune_lexer.mll"
    ( Template.Buffer.add_text s;
      Other
    )
# 949 "src/dune_lang/dune_lexer.ml"

  | 2 ->
let
# 253 "src/dune_lang/dune_lexer.mll"
                                       c
# 955 "src/dune_lang/dune_lexer.ml"
= Lexing.sub_lexeme_char lexbuf lexbuf.Lexing.lex_start_pos in
# 254 "src/dune_lang/dune_lexer.mll"
    ( let c =
        match c with
        | 'n' -> '\n'
        | 'r' -> '\r'
        | 'b' -> '\b'
        | 't' -> '\t'
        | _   -> c
      in
      Template.Buffer.add_text_c c;
      Other
    )
# 969 "src/dune_lang/dune_lexer.ml"

  | 3 ->
let
# 265 "src/dune_lang/dune_lexer.mll"
              c1
# 975 "src/dune_lang/dune_lexer.ml"
= Lexing.sub_lexeme_char lexbuf lexbuf.Lexing.lex_start_pos
and
# 265 "src/dune_lang/dune_lexer.mll"
                            c2
# 980 "src/dune_lang/dune_lexer.ml"
= Lexing.sub_lexeme_char lexbuf (lexbuf.Lexing.lex_start_pos + 1)
and
# 265 "src/dune_lang/dune_lexer.mll"
                                          c3
# 985 "src/dune_lang/dune_lexer.ml"
= Lexing.sub_lexeme_char lexbuf (lexbuf.Lexing.lex_start_pos + 2) in
# 266 "src/dune_lang/dune_lexer.mll"
    ( let v = eval_decimal_escape c1 c2 c3 in
      if v > 255 then
        error lexbuf "escape sequence in quoted string out of range"
          ~delta:(-1);
      Template.Buffer.add_text_c (Char.chr v);
      Other
    )
# 995 "src/dune_lang/dune_lexer.ml"

  | 4 ->
# 274 "src/dune_lang/dune_lexer.mll"
    ( error lexbuf "escape sequence in quoted string out of range" ~delta:(-1);
    )
# 1001 "src/dune_lang/dune_lexer.ml"

  | 5 ->
# 277 "src/dune_lang/dune_lexer.mll"
    ( error lexbuf "unterminated decimal escape sequence" ~delta:(-1);
    )
# 1007 "src/dune_lang/dune_lexer.ml"

  | 6 ->
let
# 279 "src/dune_lang/dune_lexer.mll"
                     c1
# 1013 "src/dune_lang/dune_lexer.ml"
= Lexing.sub_lexeme_char lexbuf (lexbuf.Lexing.lex_start_pos + 1)
and
# 279 "src/dune_lang/dune_lexer.mll"
                                      c2
# 1018 "src/dune_lang/dune_lexer.ml"
= Lexing.sub_lexeme_char lexbuf (lexbuf.Lexing.lex_start_pos + 2) in
# 280 "src/dune_lang/dune_lexer.mll"
    ( let v = eval_hex_escape c1 c2 in
      Template.Buffer.add_text_c (Char.chr v);
      Other
    )
# 1025 "src/dune_lang/dune_lexer.ml"

  | 7 ->
# 285 "src/dune_lang/dune_lexer.mll"
    ( error lexbuf "unterminated hexadecimal escape sequence" ~delta:(-1);
    )
# 1031 "src/dune_lang/dune_lexer.ml"

  | 8 ->
# 288 "src/dune_lang/dune_lexer.mll"
    ( error lexbuf "unknown escape sequence" ~delta:(-1);
    )
# 1037 "src/dune_lang/dune_lexer.ml"

  | 9 ->
# 291 "src/dune_lang/dune_lexer.mll"
    ( error lexbuf "unterminated escape sequence" ~delta:(-1);
    )
# 1043 "src/dune_lang/dune_lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_escape_sequence_rec lexbuf __ocaml_lex_state

and quoted_string_after_escaped_newline lexbuf =
   __ocaml_lex_quoted_string_after_escaped_newline_rec lexbuf 71
and __ocaml_lex_quoted_string_after_escaped_newline_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 296 "src/dune_lang/dune_lexer.mll"
    ( quoted_string lexbuf )
# 1055 "src/dune_lang/dune_lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_quoted_string_after_escaped_newline_rec lexbuf __ocaml_lex_state

and template_variable lexbuf =
  lexbuf.Lexing.lex_mem <- Array.make 6 (-1); __ocaml_lex_template_variable_rec lexbuf 72
and __ocaml_lex_template_variable_rec lexbuf __ocaml_lex_state =
  match Lexing.new_engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
let
# 299 "src/dune_lang/dune_lexer.mll"
                      name
# 1068 "src/dune_lang/dune_lexer.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos lexbuf.Lexing.lex_mem.(0)
and
# 299 "src/dune_lang/dune_lexer.mll"
                                                           payload
# 1073 "src/dune_lang/dune_lexer.ml"
= Lexing.sub_lexeme_opt lexbuf lexbuf.Lexing.lex_mem.(2) lexbuf.Lexing.lex_mem.(1) in
# 300 "src/dune_lang/dune_lexer.mll"
    ( let payload =
        match payload with
        | Some "" -> error lexbuf "payload after : in variable cannot be empty"
        | p -> p
      in
      Template.Var
        { loc =
            { start = Lexing.lexeme_start_p lexbuf
            ; stop = Lexing.lexeme_end_p lexbuf
            }
        ; name
        ; payload
        ; syntax = Percent
        }
  )
# 1091 "src/dune_lang/dune_lexer.ml"

  | 1 ->
# 316 "src/dune_lang/dune_lexer.mll"
    ( error lexbuf "%{...} forms cannot be empty" )
# 1096 "src/dune_lang/dune_lexer.ml"

  | 2 ->
# 317 "src/dune_lang/dune_lexer.mll"
      ( error lexbuf "This character is not allowed inside %{...} forms" )
# 1101 "src/dune_lang/dune_lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_template_variable_rec lexbuf __ocaml_lex_state

;;

# 319 "src/dune_lang/dune_lexer.mll"
 
  let token ~with_comments lexbuf = token with_comments lexbuf

# 1112 "src/dune_lang/dune_lexer.ml"
