(function() {
  var BrowserWindow, CompositeDisposable, TabBarView, TabView, _, ipcRenderer, isAtomEvent, itemIsAllowed;

  BrowserWindow = null;

  ipcRenderer = require('electron').ipcRenderer;

  CompositeDisposable = require('atom').CompositeDisposable;

  _ = require('underscore-plus');

  TabView = require('./tab-view');

  module.exports = TabBarView = (function() {
    function TabBarView(pane1, location1) {
      var addElementCommands, item, j, len, ref;
      this.pane = pane1;
      this.location = location1;
      this.element = document.createElement('ul');
      this.element.classList.add("list-inline");
      this.element.classList.add("tab-bar");
      this.element.classList.add("inset-panel");
      this.element.setAttribute('is', 'atom-tabs');
      this.element.setAttribute("tabindex", -1);
      this.element.setAttribute("location", this.location);
      this.tabs = [];
      this.tabsByElement = new WeakMap;
      this.subscriptions = new CompositeDisposable;
      this.subscriptions.add(atom.commands.add(this.pane.getElement(), {
        'tabs:keep-pending-tab': (function(_this) {
          return function() {
            return _this.terminatePendingStates();
          };
        })(this),
        'tabs:close-tab': (function(_this) {
          return function() {
            return _this.closeTab(_this.getActiveTab());
          };
        })(this),
        'tabs:close-other-tabs': (function(_this) {
          return function() {
            return _this.closeOtherTabs(_this.getActiveTab());
          };
        })(this),
        'tabs:close-tabs-to-right': (function(_this) {
          return function() {
            return _this.closeTabsToRight(_this.getActiveTab());
          };
        })(this),
        'tabs:close-tabs-to-left': (function(_this) {
          return function() {
            return _this.closeTabsToLeft(_this.getActiveTab());
          };
        })(this),
        'tabs:close-saved-tabs': (function(_this) {
          return function() {
            return _this.closeSavedTabs();
          };
        })(this),
        'tabs:close-all-tabs': (function(_this) {
          return function(event) {
            event.stopPropagation();
            return _this.closeAllTabs();
          };
        })(this),
        'tabs:open-in-new-window': (function(_this) {
          return function() {
            return _this.openInNewWindow();
          };
        })(this)
      }));
      addElementCommands = (function(_this) {
        return function(commands) {
          var commandsWithPropagationStopped;
          commandsWithPropagationStopped = {};
          Object.keys(commands).forEach(function(name) {
            return commandsWithPropagationStopped[name] = function(event) {
              event.stopPropagation();
              return commands[name]();
            };
          });
          return _this.subscriptions.add(atom.commands.add(_this.element, commandsWithPropagationStopped));
        };
      })(this);
      addElementCommands({
        'tabs:close-tab': (function(_this) {
          return function() {
            return _this.closeTab();
          };
        })(this),
        'tabs:close-other-tabs': (function(_this) {
          return function() {
            return _this.closeOtherTabs();
          };
        })(this),
        'tabs:close-tabs-to-right': (function(_this) {
          return function() {
            return _this.closeTabsToRight();
          };
        })(this),
        'tabs:close-tabs-to-left': (function(_this) {
          return function() {
            return _this.closeTabsToLeft();
          };
        })(this),
        'tabs:close-saved-tabs': (function(_this) {
          return function() {
            return _this.closeSavedTabs();
          };
        })(this),
        'tabs:close-all-tabs': (function(_this) {
          return function() {
            return _this.closeAllTabs();
          };
        })(this),
        'tabs:split-up': (function(_this) {
          return function() {
            return _this.splitTab('splitUp');
          };
        })(this),
        'tabs:split-down': (function(_this) {
          return function() {
            return _this.splitTab('splitDown');
          };
        })(this),
        'tabs:split-left': (function(_this) {
          return function() {
            return _this.splitTab('splitLeft');
          };
        })(this),
        'tabs:split-right': (function(_this) {
          return function() {
            return _this.splitTab('splitRight');
          };
        })(this)
      });
      this.element.addEventListener("mouseenter", this.onMouseEnter.bind(this));
      this.element.addEventListener("mouseleave", this.onMouseLeave.bind(this));
      this.element.addEventListener("dragstart", this.onDragStart.bind(this));
      this.element.addEventListener("dragend", this.onDragEnd.bind(this));
      this.element.addEventListener("dragleave", this.onDragLeave.bind(this));
      this.element.addEventListener("dragover", this.onDragOver.bind(this));
      this.element.addEventListener("drop", this.onDrop.bind(this));
      this.paneContainer = this.pane.getContainer();
      ref = this.pane.getItems();
      for (j = 0, len = ref.length; j < len; j++) {
        item = ref[j];
        this.addTabForItem(item);
      }
      this.subscriptions.add(this.pane.onDidDestroy((function(_this) {
        return function() {
          return _this.destroy();
        };
      })(this)));
      this.subscriptions.add(this.pane.onDidAddItem((function(_this) {
        return function(arg) {
          var index, item;
          item = arg.item, index = arg.index;
          return _this.addTabForItem(item, index);
        };
      })(this)));
      this.subscriptions.add(this.pane.onDidMoveItem((function(_this) {
        return function(arg) {
          var item, newIndex;
          item = arg.item, newIndex = arg.newIndex;
          return _this.moveItemTabToIndex(item, newIndex);
        };
      })(this)));
      this.subscriptions.add(this.pane.onDidRemoveItem((function(_this) {
        return function(arg) {
          var item;
          item = arg.item;
          return _this.removeTabForItem(item);
        };
      })(this)));
      this.subscriptions.add(this.pane.onDidChangeActiveItem((function(_this) {
        return function(item) {
          return _this.updateActiveTab();
        };
      })(this)));
      this.subscriptions.add(atom.config.observe('tabs.tabScrolling', this.updateTabScrolling.bind(this)));
      this.subscriptions.add(atom.config.observe('tabs.tabScrollingThreshold', (function(_this) {
        return function() {
          return _this.updateTabScrollingThreshold();
        };
      })(this)));
      this.subscriptions.add(atom.config.observe('tabs.alwaysShowTabBar', (function(_this) {
        return function() {
          return _this.updateTabBarVisibility();
        };
      })(this)));
      this.updateActiveTab();
      this.element.addEventListener("mousedown", this.onMouseDown.bind(this));
      this.element.addEventListener("click", this.onClick.bind(this));
      this.element.addEventListener("auxclick", this.onClick.bind(this));
      this.element.addEventListener("dblclick", this.onDoubleClick.bind(this));
      this.onDropOnOtherWindow = this.onDropOnOtherWindow.bind(this);
      ipcRenderer.on('tab:dropped', this.onDropOnOtherWindow);
    }

    TabBarView.prototype.destroy = function() {
      ipcRenderer.removeListener('tab:dropped', this.onDropOnOtherWindow);
      this.subscriptions.dispose();
      return this.element.remove();
    };

    TabBarView.prototype.terminatePendingStates = function() {
      var j, len, ref, tab;
      ref = this.getTabs();
      for (j = 0, len = ref.length; j < len; j++) {
        tab = ref[j];
        if (typeof tab.terminatePendingState === "function") {
          tab.terminatePendingState();
        }
      }
    };

    TabBarView.prototype.addTabForItem = function(item, index) {
      var tabView;
      tabView = new TabView({
        item: item,
        pane: this.pane,
        tabs: this.tabs,
        didClickCloseIcon: (function(_this) {
          return function() {
            _this.closeTab(tabView);
          };
        })(this),
        location: this.location
      });
      if (this.isItemMovingBetweenPanes) {
        tabView.terminatePendingState();
      }
      this.tabsByElement.set(tabView.element, tabView);
      this.insertTabAtIndex(tabView, index);
      if (atom.config.get('tabs.addNewTabsAtEnd')) {
        if (!this.isItemMovingBetweenPanes) {
          return this.pane.moveItem(item, this.pane.getItems().length - 1);
        }
      }
    };

    TabBarView.prototype.moveItemTabToIndex = function(item, index) {
      var tab, tabIndex;
      tabIndex = this.tabs.findIndex(function(t) {
        return t.item === item;
      });
      if (tabIndex !== -1) {
        tab = this.tabs[tabIndex];
        tab.element.remove();
        this.tabs.splice(tabIndex, 1);
        return this.insertTabAtIndex(tab, index);
      }
    };

    TabBarView.prototype.insertTabAtIndex = function(tab, index) {
      var followingTab;
      if (index != null) {
        followingTab = this.tabs[index];
      }
      if (followingTab) {
        this.element.insertBefore(tab.element, followingTab.element);
        this.tabs.splice(index, 0, tab);
      } else {
        this.element.appendChild(tab.element);
        this.tabs.push(tab);
      }
      tab.updateTitle();
      return this.updateTabBarVisibility();
    };

    TabBarView.prototype.removeTabForItem = function(item) {
      var j, len, ref, tab, tabIndex;
      tabIndex = this.tabs.findIndex(function(t) {
        return t.item === item;
      });
      if (tabIndex !== -1) {
        tab = this.tabs[tabIndex];
        this.tabs.splice(tabIndex, 1);
        this.tabsByElement["delete"](tab);
        tab.destroy();
      }
      ref = this.getTabs();
      for (j = 0, len = ref.length; j < len; j++) {
        tab = ref[j];
        tab.updateTitle();
      }
      return this.updateTabBarVisibility();
    };

    TabBarView.prototype.updateTabBarVisibility = function() {
      if (!atom.config.get('tabs.alwaysShowTabBar') && !this.shouldAllowDrag()) {
        return this.element.classList.add('hidden');
      } else {
        return this.element.classList.remove('hidden');
      }
    };

    TabBarView.prototype.getTabs = function() {
      return this.tabs.slice();
    };

    TabBarView.prototype.tabAtIndex = function(index) {
      return this.tabs[index];
    };

    TabBarView.prototype.tabForItem = function(item) {
      return this.tabs.find(function(t) {
        return t.item === item;
      });
    };

    TabBarView.prototype.setActiveTab = function(tabView) {
      var ref;
      if ((tabView != null) && tabView !== this.activeTab) {
        if ((ref = this.activeTab) != null) {
          ref.element.classList.remove('active');
        }
        this.activeTab = tabView;
        this.activeTab.element.classList.add('active');
        return this.activeTab.element.scrollIntoView(false);
      }
    };

    TabBarView.prototype.getActiveTab = function() {
      return this.tabForItem(this.pane.getActiveItem());
    };

    TabBarView.prototype.updateActiveTab = function() {
      return this.setActiveTab(this.tabForItem(this.pane.getActiveItem()));
    };

    TabBarView.prototype.closeTab = function(tab) {
      if (tab == null) {
        tab = this.rightClickedTab;
      }
      if (tab != null) {
        return this.pane.destroyItem(tab.item);
      }
    };

    TabBarView.prototype.openInNewWindow = function(tab) {
      var item, itemURI, j, len, pathsToOpen, ref;
      if (tab == null) {
        tab = this.rightClickedTab;
      }
      item = tab != null ? tab.item : void 0;
      if (item == null) {
        return;
      }
      if (typeof item.getURI === 'function') {
        itemURI = item.getURI();
      } else if (typeof item.getPath === 'function') {
        itemURI = item.getPath();
      } else if (typeof item.getUri === 'function') {
        itemURI = item.getUri();
      }
      if (itemURI == null) {
        return;
      }
      this.closeTab(tab);
      ref = this.getTabs();
      for (j = 0, len = ref.length; j < len; j++) {
        tab = ref[j];
        tab.element.style.maxWidth = '';
      }
      pathsToOpen = [atom.project.getPaths(), itemURI].reduce((function(a, b) {
        return a.concat(b);
      }), []);
      return atom.open({
        pathsToOpen: pathsToOpen,
        newWindow: true,
        devMode: atom.devMode,
        safeMode: atom.safeMode
      });
    };

    TabBarView.prototype.splitTab = function(fn) {
      var copiedItem, item, ref;
      if (item = (ref = this.rightClickedTab) != null ? ref.item : void 0) {
        if (copiedItem = typeof item.copy === "function" ? item.copy() : void 0) {
          return this.pane[fn]({
            items: [copiedItem]
          });
        }
      }
    };

    TabBarView.prototype.closeOtherTabs = function(active) {
      var j, len, results, tab, tabs;
      tabs = this.getTabs();
      if (active == null) {
        active = this.rightClickedTab;
      }
      if (active == null) {
        return;
      }
      results = [];
      for (j = 0, len = tabs.length; j < len; j++) {
        tab = tabs[j];
        if (tab !== active) {
          results.push(this.closeTab(tab));
        }
      }
      return results;
    };

    TabBarView.prototype.closeTabsToRight = function(active) {
      var i, index, j, len, results, tab, tabs;
      tabs = this.getTabs();
      if (active == null) {
        active = this.rightClickedTab;
      }
      index = tabs.indexOf(active);
      if (index === -1) {
        return;
      }
      results = [];
      for (i = j = 0, len = tabs.length; j < len; i = ++j) {
        tab = tabs[i];
        if (i > index) {
          results.push(this.closeTab(tab));
        }
      }
      return results;
    };

    TabBarView.prototype.closeTabsToLeft = function(active) {
      var i, index, j, len, results, tab, tabs;
      tabs = this.getTabs();
      if (active == null) {
        active = this.rightClickedTab;
      }
      index = tabs.indexOf(active);
      if (index === -1) {
        return;
      }
      results = [];
      for (i = j = 0, len = tabs.length; j < len; i = ++j) {
        tab = tabs[i];
        if (i < index) {
          results.push(this.closeTab(tab));
        }
      }
      return results;
    };

    TabBarView.prototype.closeSavedTabs = function() {
      var base, j, len, ref, results, tab;
      ref = this.getTabs();
      results = [];
      for (j = 0, len = ref.length; j < len; j++) {
        tab = ref[j];
        if (!(typeof (base = tab.item).isModified === "function" ? base.isModified() : void 0)) {
          results.push(this.closeTab(tab));
        } else {
          results.push(void 0);
        }
      }
      return results;
    };

    TabBarView.prototype.closeAllTabs = function() {
      var j, len, ref, results, tab;
      ref = this.getTabs();
      results = [];
      for (j = 0, len = ref.length; j < len; j++) {
        tab = ref[j];
        results.push(this.closeTab(tab));
      }
      return results;
    };

    TabBarView.prototype.getWindowId = function() {
      return this.windowId != null ? this.windowId : this.windowId = atom.getCurrentWindow().id;
    };

    TabBarView.prototype.shouldAllowDrag = function() {
      return (this.paneContainer.getPanes().length > 1) || (this.pane.getItems().length > 1);
    };

    TabBarView.prototype.onDragStart = function(event) {
      var item, itemURI, j, len, location, paneIndex, ref, ref1, ref2, ref3, tabIndex;
      this.draggedTab = this.tabForElement(event.target);
      if (!this.draggedTab) {
        return;
      }
      this.lastDropTargetIndex = null;
      event.dataTransfer.setData('atom-event', 'true');
      this.draggedTab.element.classList.add('is-dragging');
      this.draggedTab.destroyTooltip();
      tabIndex = this.tabs.indexOf(this.draggedTab);
      event.dataTransfer.setData('sortable-index', tabIndex);
      paneIndex = this.paneContainer.getPanes().indexOf(this.pane);
      event.dataTransfer.setData('from-pane-index', paneIndex);
      event.dataTransfer.setData('from-pane-id', this.pane.id);
      event.dataTransfer.setData('from-window-id', this.getWindowId());
      item = this.pane.getItems()[this.tabs.indexOf(this.draggedTab)];
      if (item == null) {
        return;
      }
      if (typeof item.getURI === 'function') {
        itemURI = (ref = item.getURI()) != null ? ref : '';
      } else if (typeof item.getPath === 'function') {
        itemURI = (ref1 = item.getPath()) != null ? ref1 : '';
      } else if (typeof item.getUri === 'function') {
        itemURI = (ref2 = item.getUri()) != null ? ref2 : '';
      }
      if (typeof item.getAllowedLocations === 'function') {
        ref3 = item.getAllowedLocations();
        for (j = 0, len = ref3.length; j < len; j++) {
          location = ref3[j];
          event.dataTransfer.setData("allowed-location-" + location, 'true');
        }
      } else {
        event.dataTransfer.setData('allow-all-locations', 'true');
      }
      if (itemURI != null) {
        event.dataTransfer.setData('text/plain', itemURI);
        if (process.platform === 'darwin') {
          if (!this.uriHasProtocol(itemURI)) {
            itemURI = "file://" + itemURI;
          }
          event.dataTransfer.setData('text/uri-list', itemURI);
        }
        if ((typeof item.isModified === "function" ? item.isModified() : void 0) && (item.getText != null)) {
          event.dataTransfer.setData('has-unsaved-changes', 'true');
          return event.dataTransfer.setData('modified-text', item.getText());
        }
      }
    };

    TabBarView.prototype.uriHasProtocol = function(uri) {
      var error;
      try {
        return require('url').parse(uri).protocol != null;
      } catch (error1) {
        error = error1;
        return false;
      }
    };

    TabBarView.prototype.onDragLeave = function(event) {
      var j, len, ref, results, tab;
      if (!event.currentTarget.contains(event.relatedTarget)) {
        this.removePlaceholder();
        this.lastDropTargetIndex = null;
        ref = this.getTabs();
        results = [];
        for (j = 0, len = ref.length; j < len; j++) {
          tab = ref[j];
          results.push(tab.element.style.maxWidth = '');
        }
        return results;
      }
    };

    TabBarView.prototype.onDragEnd = function(event) {
      if (!this.tabForElement(event.target)) {
        return;
      }
      return this.clearDropTarget();
    };

    TabBarView.prototype.onDragOver = function(event) {
      var newDropTargetIndex, placeholder, sibling, tab, tabs;
      event.preventDefault();
      if (!isAtomEvent(event)) {
        event.stopPropagation();
        return;
      }
      if (!itemIsAllowed(event, this.location)) {
        return;
      }
      newDropTargetIndex = this.getDropTargetIndex(event);
      if (newDropTargetIndex == null) {
        return;
      }
      if (this.lastDropTargetIndex === newDropTargetIndex) {
        return;
      }
      this.lastDropTargetIndex = newDropTargetIndex;
      this.removeDropTargetClasses();
      tabs = this.getTabs();
      placeholder = this.getPlaceholder();
      if (placeholder == null) {
        return;
      }
      if (newDropTargetIndex < tabs.length) {
        tab = tabs[newDropTargetIndex];
        tab.element.classList.add('is-drop-target');
        return tab.element.parentElement.insertBefore(placeholder, tab.element);
      } else {
        if (tab = tabs[newDropTargetIndex - 1]) {
          tab.element.classList.add('drop-target-is-after');
          if (sibling = tab.element.nextSibling) {
            return tab.element.parentElement.insertBefore(placeholder, sibling);
          } else {
            return tab.element.parentElement.appendChild(placeholder);
          }
        }
      }
    };

    TabBarView.prototype.onDropOnOtherWindow = function(fromPaneId, fromItemIndex) {
      var itemToRemove;
      if (this.pane.id === fromPaneId) {
        if (itemToRemove = this.pane.getItems()[fromItemIndex]) {
          this.pane.destroyItem(itemToRemove);
        }
      }
      return this.clearDropTarget();
    };

    TabBarView.prototype.clearDropTarget = function() {
      var ref, ref1;
      if ((ref = this.draggedTab) != null) {
        ref.element.classList.remove('is-dragging');
      }
      if ((ref1 = this.draggedTab) != null) {
        ref1.updateTooltip();
      }
      this.draggedTab = null;
      this.removeDropTargetClasses();
      return this.removePlaceholder();
    };

    TabBarView.prototype.onDrop = function(event) {
      var droppedURI, fromIndex, fromPane, fromPaneId, fromPaneIndex, fromWindowId, hasUnsavedChanges, item, modifiedText, toIndex, toPane;
      event.preventDefault();
      if (event.dataTransfer.getData('atom-event') !== 'true') {
        return;
      }
      fromWindowId = parseInt(event.dataTransfer.getData('from-window-id'));
      fromPaneId = parseInt(event.dataTransfer.getData('from-pane-id'));
      fromIndex = parseInt(event.dataTransfer.getData('sortable-index'));
      fromPaneIndex = parseInt(event.dataTransfer.getData('from-pane-index'));
      hasUnsavedChanges = event.dataTransfer.getData('has-unsaved-changes') === 'true';
      modifiedText = event.dataTransfer.getData('modified-text');
      toIndex = this.getDropTargetIndex(event);
      toPane = this.pane;
      this.clearDropTarget();
      if (!itemIsAllowed(event, this.location)) {
        return;
      }
      if (fromWindowId === this.getWindowId()) {
        fromPane = this.paneContainer.getPanes()[fromPaneIndex];
        if ((fromPane != null ? fromPane.id : void 0) !== fromPaneId) {
          fromPane = Array.from(document.querySelectorAll('atom-pane')).map(function(paneEl) {
            return paneEl.model;
          }).find(function(pane) {
            return pane.id === fromPaneId;
          });
        }
        item = fromPane.getItems()[fromIndex];
        if (item != null) {
          return this.moveItemBetweenPanes(fromPane, fromIndex, toPane, toIndex, item);
        }
      } else {
        droppedURI = event.dataTransfer.getData('text/plain');
        atom.workspace.open(droppedURI).then((function(_this) {
          return function(item) {
            var activeItemIndex, activePane, browserWindow;
            activePane = atom.workspace.getActivePane();
            activeItemIndex = activePane.getItems().indexOf(item);
            _this.moveItemBetweenPanes(activePane, activeItemIndex, toPane, toIndex, item);
            if (hasUnsavedChanges) {
              if (typeof item.setText === "function") {
                item.setText(modifiedText);
              }
            }
            if (!isNaN(fromWindowId)) {
              browserWindow = _this.browserWindowForId(fromWindowId);
              return browserWindow != null ? browserWindow.webContents.send('tab:dropped', fromPaneId, fromIndex) : void 0;
            }
          };
        })(this));
        return atom.focus();
      }
    };

    TabBarView.prototype.onMouseWheel = function(event) {
      if (event.shiftKey) {
        return;
      }
      if (this.wheelDelta == null) {
        this.wheelDelta = 0;
      }
      this.wheelDelta += event.wheelDeltaY;
      if (this.wheelDelta <= -this.tabScrollingThreshold) {
        this.wheelDelta = 0;
        return this.pane.activateNextItem();
      } else if (this.wheelDelta >= this.tabScrollingThreshold) {
        this.wheelDelta = 0;
        return this.pane.activatePreviousItem();
      }
    };

    TabBarView.prototype.onMouseDown = function(event) {
      var ref, tab;
      tab = this.tabForElement(event.target);
      if (!tab) {
        return;
      }
      if (event.which === 3 || (event.which === 1 && event.ctrlKey === true)) {
        if ((ref = this.rightClickedTab) != null) {
          ref.element.classList.remove('right-clicked');
        }
        this.rightClickedTab = tab;
        this.rightClickedTab.element.classList.add('right-clicked');
        return event.preventDefault();
      } else if (event.which === 2) {
        return event.preventDefault();
      }
    };

    TabBarView.prototype.onClick = function(event) {
      var tab;
      tab = this.tabForElement(event.target);
      if (!tab) {
        return;
      }
      event.preventDefault();
      if (event.which === 3 || (event.which === 1 && event.ctrlKey === true)) {

      } else if (event.which === 1 && !event.target.classList.contains('close-icon')) {
        this.pane.activateItem(tab.item);
        if (!this.pane.isDestroyed()) {
          return this.pane.activate();
        }
      } else if (event.which === 2) {
        return this.pane.destroyItem(tab.item);
      }
    };

    TabBarView.prototype.onDoubleClick = function(event) {
      var base, tab;
      if (tab = this.tabForElement(event.target)) {
        return typeof (base = tab.item).terminatePendingState === "function" ? base.terminatePendingState() : void 0;
      } else if (event.target === this.element) {
        atom.commands.dispatch(this.element, 'application:new-file');
        return event.preventDefault();
      }
    };

    TabBarView.prototype.updateTabScrollingThreshold = function() {
      return this.tabScrollingThreshold = atom.config.get('tabs.tabScrollingThreshold');
    };

    TabBarView.prototype.updateTabScrolling = function(value) {
      if (value === 'platform') {
        this.tabScrolling = process.platform === 'linux';
      } else {
        this.tabScrolling = value;
      }
      this.tabScrollingThreshold = atom.config.get('tabs.tabScrollingThreshold');
      if (this.tabScrolling) {
        return this.element.addEventListener('mousewheel', this.onMouseWheel.bind(this));
      } else {
        return this.element.removeEventListener('mousewheel', this.onMouseWheel.bind(this));
      }
    };

    TabBarView.prototype.browserWindowForId = function(id) {
      if (BrowserWindow == null) {
        BrowserWindow = require('electron').remote.BrowserWindow;
      }
      return BrowserWindow.fromId(id);
    };

    TabBarView.prototype.moveItemBetweenPanes = function(fromPane, fromIndex, toPane, toIndex, item) {
      try {
        if (toPane === fromPane) {
          if (fromIndex < toIndex) {
            toIndex--;
          }
          toPane.moveItem(item, toIndex);
        } else {
          this.isItemMovingBetweenPanes = true;
          fromPane.moveItemToPane(item, toPane, toIndex--);
        }
        toPane.activateItem(item);
        return toPane.activate();
      } finally {
        this.isItemMovingBetweenPanes = false;
      }
    };

    TabBarView.prototype.removeDropTargetClasses = function() {
      var dropTarget, j, k, len, len1, ref, ref1, results, workspaceElement;
      workspaceElement = atom.workspace.getElement();
      ref = workspaceElement.querySelectorAll('.tab-bar .is-drop-target');
      for (j = 0, len = ref.length; j < len; j++) {
        dropTarget = ref[j];
        dropTarget.classList.remove('is-drop-target');
      }
      ref1 = workspaceElement.querySelectorAll('.tab-bar .drop-target-is-after');
      results = [];
      for (k = 0, len1 = ref1.length; k < len1; k++) {
        dropTarget = ref1[k];
        results.push(dropTarget.classList.remove('drop-target-is-after'));
      }
      return results;
    };

    TabBarView.prototype.getDropTargetIndex = function(event) {
      var elementCenter, elementIndex, left, ref, tab, tabs, target, width;
      target = event.target;
      if (this.isPlaceholder(target)) {
        return;
      }
      tabs = this.getTabs();
      tab = this.tabForElement(target);
      if (tab == null) {
        tab = tabs[tabs.length - 1];
      }
      if (tab == null) {
        return 0;
      }
      ref = tab.element.getBoundingClientRect(), left = ref.left, width = ref.width;
      elementCenter = left + width / 2;
      elementIndex = tabs.indexOf(tab);
      if (event.pageX < elementCenter) {
        return elementIndex;
      } else {
        return elementIndex + 1;
      }
    };

    TabBarView.prototype.getPlaceholder = function() {
      if (this.placeholderEl != null) {
        return this.placeholderEl;
      }
      this.placeholderEl = document.createElement("li");
      this.placeholderEl.classList.add("placeholder");
      return this.placeholderEl;
    };

    TabBarView.prototype.removePlaceholder = function() {
      var ref;
      if ((ref = this.placeholderEl) != null) {
        ref.remove();
      }
      return this.placeholderEl = null;
    };

    TabBarView.prototype.isPlaceholder = function(element) {
      return element.classList.contains('placeholder');
    };

    TabBarView.prototype.onMouseEnter = function() {
      var j, len, ref, tab, width;
      ref = this.getTabs();
      for (j = 0, len = ref.length; j < len; j++) {
        tab = ref[j];
        width = tab.element.getBoundingClientRect().width;
        tab.element.style.maxWidth = width.toFixed(2) + 'px';
      }
    };

    TabBarView.prototype.onMouseLeave = function() {
      var j, len, ref, tab;
      ref = this.getTabs();
      for (j = 0, len = ref.length; j < len; j++) {
        tab = ref[j];
        tab.element.style.maxWidth = '';
      }
    };

    TabBarView.prototype.tabForElement = function(element) {
      var currentElement, tab;
      currentElement = element;
      while (currentElement != null) {
        if (tab = this.tabsByElement.get(currentElement)) {
          return tab;
        } else {
          currentElement = currentElement.parentElement;
        }
      }
    };

    return TabBarView;

  })();

  isAtomEvent = function(event) {
    var item, j, len, ref;
    ref = event.dataTransfer.items;
    for (j = 0, len = ref.length; j < len; j++) {
      item = ref[j];
      if (item.type === 'atom-event') {
        return true;
      }
    }
    return false;
  };

  itemIsAllowed = function(event, location) {
    var item, j, len, ref;
    ref = event.dataTransfer.items;
    for (j = 0, len = ref.length; j < len; j++) {
      item = ref[j];
      if (item.type === 'allow-all-locations' || item.type === ("allowed-location-" + location)) {
        return true;
      }
    }
    return false;
  };

}).call(this);

//# sourceMappingURL=data:application/json;base64,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
