(function() {
  var CompositeDisposable, Point, Range, SnippetExpansion, ref,
    bind = function(fn, me){ return function(){ return fn.apply(me, arguments); }; },
    slice = [].slice;

  ref = require('atom'), CompositeDisposable = ref.CompositeDisposable, Range = ref.Range, Point = ref.Point;

  module.exports = SnippetExpansion = (function() {
    SnippetExpansion.prototype.settingTabStop = false;

    SnippetExpansion.prototype.isIgnoringBufferChanges = false;

    function SnippetExpansion(snippet, editor, cursor, snippets) {
      var body, indent, ref1, startPosition, tabStopList, tabStops;
      this.snippet = snippet;
      this.editor = editor;
      this.cursor = cursor;
      this.snippets = snippets;
      this.onUndoOrRedo = bind(this.onUndoOrRedo, this);
      this.subscriptions = new CompositeDisposable;
      this.tabStopMarkers = [];
      this.selections = [this.cursor.selection];
      startPosition = this.cursor.selection.getBufferRange().start;
      ref1 = this.snippet, body = ref1.body, tabStopList = ref1.tabStopList;
      tabStops = tabStopList.toArray();
      if (this.snippet.lineCount > 1 && (indent = this.editor.lineTextForBufferRow(startPosition.row).match(/^\s*/)[0])) {
        body = body.replace(/\n/g, '\n' + indent);
        tabStops = tabStops.map(function(tabStop) {
          return tabStop.copyWithIndent(indent);
        });
      }
      this.editor.transact((function(_this) {
        return function() {
          return _this.ignoringBufferChanges(function() {
            return _this.editor.transact(function() {
              var newRange;
              newRange = _this.cursor.selection.insertText(body, {
                autoIndent: false
              });
              if (_this.snippet.tabStopList.length > 0) {
                _this.subscriptions.add(_this.cursor.onDidChangePosition(function(event) {
                  return _this.cursorMoved(event);
                }));
                _this.subscriptions.add(_this.cursor.onDidDestroy(function() {
                  return _this.cursorDestroyed();
                }));
                _this.placeTabStopMarkers(startPosition, tabStops);
                _this.snippets.addExpansion(_this.editor, _this);
                return _this.editor.normalizeTabsInBufferRange(newRange);
              }
            });
          });
        };
      })(this));
    }

    SnippetExpansion.prototype.onUndoOrRedo = function(isUndo) {
      return this.isUndoingOrRedoing = true;
    };

    SnippetExpansion.prototype.cursorMoved = function(arg) {
      var itemWithCursor, newBufferPosition, oldBufferPosition, textChanged;
      oldBufferPosition = arg.oldBufferPosition, newBufferPosition = arg.newBufferPosition, textChanged = arg.textChanged;
      if (this.settingTabStop || textChanged) {
        return;
      }
      itemWithCursor = this.tabStopMarkers[this.tabStopIndex].find(function(item) {
        return item.marker.getBufferRange().containsPoint(newBufferPosition);
      });
      if (!(itemWithCursor && !itemWithCursor.insertion.isTransformation())) {
        return this.destroy();
      }
    };

    SnippetExpansion.prototype.cursorDestroyed = function() {
      if (!this.settingTabStop) {
        return this.destroy();
      }
    };

    SnippetExpansion.prototype.textChanged = function(event) {
      if (this.isIgnoringBufferChanges) {
        return;
      }
      if (this.isUndoingOrRedoing) {
        this.isUndoingOrRedoing = false;
        return;
      }
      return this.applyTransformations(this.tabStopIndex);
    };

    SnippetExpansion.prototype.ignoringBufferChanges = function(callback) {
      var wasIgnoringBufferChanges;
      wasIgnoringBufferChanges = this.isIgnoringBufferChanges;
      this.isIgnoringBufferChanges = true;
      callback();
      return this.isIgnoringBufferChanges = wasIgnoringBufferChanges;
    };

    SnippetExpansion.prototype.applyAllTransformations = function() {
      return this.editor.transact((function(_this) {
        return function() {
          var index, item, j, len, ref1, results;
          ref1 = _this.tabStopMarkers;
          results = [];
          for (index = j = 0, len = ref1.length; j < len; index = ++j) {
            item = ref1[index];
            results.push(_this.applyTransformations(index, true));
          }
          return results;
        };
      })(this));
    };

    SnippetExpansion.prototype.applyTransformations = function(tabStop, initial) {
      var inputText, items, primary, primaryRange;
      if (initial == null) {
        initial = false;
      }
      items = slice.call(this.tabStopMarkers[tabStop]);
      if (items.length === 0) {
        return;
      }
      primary = items.shift();
      primaryRange = primary.marker.getBufferRange();
      inputText = this.editor.getTextInBufferRange(primaryRange);
      return this.ignoringBufferChanges((function(_this) {
        return function() {
          var index, insertion, item, j, len, marker, newRange, outputText, range, results;
          results = [];
          for (index = j = 0, len = items.length; j < len; index = ++j) {
            item = items[index];
            marker = item.marker, insertion = item.insertion;
            range = marker.getBufferRange();
            if (!insertion.isTransformation()) {
              continue;
            }
            outputText = insertion.transform(inputText);
            _this.editor.transact(function() {
              return _this.editor.setTextInBufferRange(range, outputText);
            });
            newRange = new Range(range.start, range.start.traverse(new Point(0, outputText.length)));
            results.push(marker.setBufferRange(newRange));
          }
          return results;
        };
      })(this));
    };

    SnippetExpansion.prototype.placeTabStopMarkers = function(startPosition, tabStops) {
      var end, insertion, insertions, j, k, len, len1, marker, markers, range, start, tabStop;
      for (j = 0, len = tabStops.length; j < len; j++) {
        tabStop = tabStops[j];
        insertions = tabStop.insertions;
        markers = [];
        if (!tabStop.isValid()) {
          continue;
        }
        for (k = 0, len1 = insertions.length; k < len1; k++) {
          insertion = insertions[k];
          range = insertion.range;
          start = range.start, end = range.end;
          marker = this.getMarkerLayer(this.editor).markBufferRange([startPosition.traverse(start), startPosition.traverse(end)]);
          markers.push({
            index: markers.length,
            marker: marker,
            insertion: insertion
          });
        }
        this.tabStopMarkers.push(markers);
      }
      this.setTabStopIndex(0);
      return this.applyAllTransformations();
    };

    SnippetExpansion.prototype.goToNextTabStop = function() {
      var nextIndex, succeeded;
      nextIndex = this.tabStopIndex + 1;
      if (nextIndex < this.tabStopMarkers.length) {
        if (this.setTabStopIndex(nextIndex)) {
          return true;
        } else {
          return this.goToNextTabStop();
        }
      } else {
        if (this.snippet.tabStopList.hasEndStop) {
          this.destroy();
          return false;
        } else {
          succeeded = this.goToEndOfLastTabStop();
          this.destroy();
          return succeeded;
        }
      }
    };

    SnippetExpansion.prototype.goToPreviousTabStop = function() {
      if (this.tabStopIndex > 0) {
        return this.setTabStopIndex(this.tabStopIndex - 1);
      }
    };

    SnippetExpansion.prototype.setTabStopIndex = function(tabStopIndex) {
      var i, insertion, item, items, j, k, l, len, len1, len2, marker, markerSelected, newSelection, range, ranges, ref1, selection;
      this.tabStopIndex = tabStopIndex;
      this.settingTabStop = true;
      markerSelected = false;
      items = this.tabStopMarkers[this.tabStopIndex];
      if (items.length === 0) {
        return false;
      }
      ranges = [];
      this.hasTransforms = false;
      for (j = 0, len = items.length; j < len; j++) {
        item = items[j];
        marker = item.marker, insertion = item.insertion;
        if (marker.isDestroyed()) {
          continue;
        }
        if (!marker.isValid()) {
          continue;
        }
        if (insertion.isTransformation()) {
          this.hasTransforms = true;
          continue;
        }
        ranges.push(marker.getBufferRange());
      }
      if (ranges.length > 0) {
        ref1 = this.selections.slice(ranges.length);
        for (k = 0, len1 = ref1.length; k < len1; k++) {
          selection = ref1[k];
          selection.destroy();
        }
        this.selections = this.selections.slice(0, ranges.length);
        for (i = l = 0, len2 = ranges.length; l < len2; i = ++l) {
          range = ranges[i];
          if (this.selections[i]) {
            this.selections[i].setBufferRange(range);
          } else {
            newSelection = this.editor.addSelectionForBufferRange(range);
            this.subscriptions.add(newSelection.cursor.onDidChangePosition((function(_this) {
              return function(event) {
                return _this.cursorMoved(event);
              };
            })(this)));
            this.subscriptions.add(newSelection.cursor.onDidDestroy((function(_this) {
              return function() {
                return _this.cursorDestroyed();
              };
            })(this)));
            this.selections.push(newSelection);
          }
        }
        markerSelected = true;
      }
      this.settingTabStop = false;
      if (this.hasTransforms) {
        this.snippets.observeEditor(this.editor);
      }
      return markerSelected;
    };

    SnippetExpansion.prototype.goToEndOfLastTabStop = function() {
      var items, lastMarker;
      if (!(this.tabStopMarkers.length > 0)) {
        return;
      }
      items = this.tabStopMarkers[this.tabStopMarkers.length - 1];
      if (!(items.length > 0)) {
        return;
      }
      lastMarker = items[items.length - 1].marker;
      if (lastMarker.isDestroyed()) {
        return false;
      } else {
        this.editor.setCursorBufferPosition(lastMarker.getEndBufferPosition());
        return true;
      }
    };

    SnippetExpansion.prototype.destroy = function() {
      this.subscriptions.dispose();
      this.getMarkerLayer(this.editor).clear();
      this.tabStopMarkers = [];
      this.snippets.stopObservingEditor(this.editor);
      return this.snippets.clearExpansions(this.editor);
    };

    SnippetExpansion.prototype.getMarkerLayer = function() {
      return this.snippets.findOrCreateMarkerLayer(this.editor);
    };

    SnippetExpansion.prototype.restore = function(editor) {
      this.editor = editor;
      return this.snippets.addExpansion(this.editor, this);
    };

    return SnippetExpansion;

  })();

}).call(this);

//# sourceMappingURL=data:application/json;base64,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
