(function() {
  var AtomIoClient, fs, glob, path, remote, request;

  fs = require('fs-plus');

  path = require('path');

  remote = require('electron').remote;

  glob = require('glob');

  request = require('request');

  module.exports = AtomIoClient = (function() {
    function AtomIoClient(packageManager, baseURL) {
      this.packageManager = packageManager;
      this.baseURL = baseURL;
      if (this.baseURL == null) {
        this.baseURL = 'https://atom.io/api/';
      }
      this.expiry = 1000 * 60 * 60 * 12;
      this.createAvatarCache();
      this.expireAvatarCache();
    }

    AtomIoClient.prototype.avatar = function(login, callback) {
      return this.cachedAvatar(login, (function(_this) {
        return function(err, cached) {
          var stale;
          if (cached) {
            stale = Date.now() - parseInt(cached.split('-').pop()) > _this.expiry;
          }
          if (cached && (!stale || !_this.online())) {
            return callback(null, cached);
          } else {
            return _this.fetchAndCacheAvatar(login, callback);
          }
        };
      })(this));
    };

    AtomIoClient.prototype["package"] = function(name, callback) {
      var packagePath;
      packagePath = "packages/" + name;
      return this.fetchFromCache(packagePath, {}, (function(_this) {
        return function(err, data) {
          if (data) {
            return callback(null, data);
          } else {
            return _this.request(packagePath, callback);
          }
        };
      })(this));
    };

    AtomIoClient.prototype.featuredPackages = function(callback) {
      return this.fetchFromCache('packages/featured', {}, (function(_this) {
        return function(err, data) {
          if (data) {
            return callback(null, data);
          } else {
            return _this.getFeatured(false, callback);
          }
        };
      })(this));
    };

    AtomIoClient.prototype.featuredThemes = function(callback) {
      return this.fetchFromCache('themes/featured', {}, (function(_this) {
        return function(err, data) {
          if (data) {
            return callback(null, data);
          } else {
            return _this.getFeatured(true, callback);
          }
        };
      })(this));
    };

    AtomIoClient.prototype.getFeatured = function(loadThemes, callback) {
      return this.packageManager.getFeatured(loadThemes).then((function(_this) {
        return function(packages) {
          var cached, key;
          key = loadThemes ? 'themes/featured' : 'packages/featured';
          cached = {
            data: packages,
            createdOn: Date.now()
          };
          localStorage.setItem(_this.cacheKeyForPath(key), JSON.stringify(cached));
          return callback(null, packages);
        };
      })(this))["catch"](function(error) {
        return callback(error, null);
      });
    };

    AtomIoClient.prototype.request = function(path, callback) {
      var options;
      options = {
        url: "" + this.baseURL + path,
        headers: {
          'User-Agent': navigator.userAgent
        },
        gzip: true
      };
      return request(options, (function(_this) {
        return function(err, res, body) {
          var cached, error;
          if (err) {
            return callback(err);
          }
          try {
            body = JSON.parse(body);
            delete body.versions;
            cached = {
              data: body,
              createdOn: Date.now()
            };
            localStorage.setItem(_this.cacheKeyForPath(path), JSON.stringify(cached));
            return callback(err, cached.data);
          } catch (error1) {
            error = error1;
            return callback(error);
          }
        };
      })(this));
    };

    AtomIoClient.prototype.cacheKeyForPath = function(path) {
      return "settings-view:" + path;
    };

    AtomIoClient.prototype.online = function() {
      return navigator.onLine;
    };

    AtomIoClient.prototype.fetchFromCache = function(packagePath, options, callback) {
      var cached;
      if (!callback) {
        callback = options;
        options = {};
      }
      if (!options.force) {
        options.force = !this.online();
      }
      cached = localStorage.getItem(this.cacheKeyForPath(packagePath));
      cached = cached ? JSON.parse(cached) : void 0;
      if ((cached != null) && (!this.online() || options.force || (Date.now() - cached.createdOn < this.expiry))) {
        if (cached == null) {
          cached = {
            data: {}
          };
        }
        return callback(null, cached.data);
      } else if ((cached == null) && !this.online()) {
        return callback(null, {});
      } else {
        return callback(null, null);
      }
    };

    AtomIoClient.prototype.createAvatarCache = function() {
      return fs.makeTree(this.getCachePath());
    };

    AtomIoClient.prototype.avatarPath = function(login) {
      return path.join(this.getCachePath(), login + "-" + (Date.now()));
    };

    AtomIoClient.prototype.cachedAvatar = function(login, callback) {
      return glob(this.avatarGlob(login), (function(_this) {
        return function(err, files) {
          var createdOn, filename, i, imagePath, len, ref;
          if (err) {
            return callback(err);
          }
          files.sort().reverse();
          for (i = 0, len = files.length; i < len; i++) {
            imagePath = files[i];
            filename = path.basename(imagePath);
            ref = filename.split('-'), createdOn = ref[ref.length - 1];
            if (Date.now() - parseInt(createdOn) < _this.expiry) {
              return callback(null, imagePath);
            }
          }
          return callback(null, null);
        };
      })(this));
    };

    AtomIoClient.prototype.avatarGlob = function(login) {
      return path.join(this.getCachePath(), login + "-*([0-9])");
    };

    AtomIoClient.prototype.fetchAndCacheAvatar = function(login, callback) {
      var imagePath, requestObject;
      if (!this.online()) {
        return callback(null, null);
      } else {
        imagePath = this.avatarPath(login);
        requestObject = {
          url: "https://avatars.githubusercontent.com/" + login,
          headers: {
            'User-Agent': navigator.userAgent
          }
        };
        return request.head(requestObject, function(error, response, body) {
          var writeStream;
          if ((error != null) || response.statusCode !== 200 || !response.headers['content-type'].startsWith('image/')) {
            return callback(error);
          } else {
            writeStream = fs.createWriteStream(imagePath);
            writeStream.on('finish', function() {
              return callback(null, imagePath);
            });
            writeStream.on('error', function(error) {
              writeStream.close();
              try {
                if (fs.existsSync(imagePath)) {
                  fs.unlinkSync(imagePath);
                }
              } catch (error1) {}
              return callback(error);
            });
            return request(requestObject).pipe(writeStream);
          }
        });
      }
    };

    AtomIoClient.prototype.expireAvatarCache = function() {
      var deleteAvatar;
      deleteAvatar = (function(_this) {
        return function(child) {
          var avatarPath;
          avatarPath = path.join(_this.getCachePath(), child);
          return fs.unlink(avatarPath, function(error) {
            if (error && error.code !== 'ENOENT') {
              return console.warn("Error deleting avatar (" + error.code + "): " + avatarPath);
            }
          });
        };
      })(this);
      return fs.readdir(this.getCachePath(), function(error, _files) {
        var children, filename, files, i, key, len, parts, results, stamp;
        if (_files == null) {
          _files = [];
        }
        files = {};
        for (i = 0, len = _files.length; i < len; i++) {
          filename = _files[i];
          parts = filename.split('-');
          stamp = parts.pop();
          key = parts.join('-');
          if (files[key] == null) {
            files[key] = [];
          }
          files[key].push(key + "-" + stamp);
        }
        results = [];
        for (key in files) {
          children = files[key];
          children.sort();
          children.pop();
          results.push(children.forEach(deleteAvatar));
        }
        return results;
      });
    };

    AtomIoClient.prototype.getCachePath = function() {
      return this.cachePath != null ? this.cachePath : this.cachePath = path.join(remote.app.getPath('userData'), 'Cache', 'settings-view');
    };

    AtomIoClient.prototype.search = function(query, options) {
      var qs;
      qs = {
        q: query
      };
      if (options.themes) {
        qs.filter = 'theme';
      } else if (options.packages) {
        qs.filter = 'package';
      }
      options = {
        url: this.baseURL + "packages/search",
        headers: {
          'User-Agent': navigator.userAgent
        },
        qs: qs,
        gzip: true
      };
      return new Promise(function(resolve, reject) {
        return request(options, function(err, res, body) {
          var e, error;
          if (err) {
            error = new Error("Searching for \u201C" + query + "\u201D failed.");
            error.stderr = err.message;
            return reject(error);
          } else {
            try {
              body = JSON.parse(body);
              return resolve(body.filter(function(pkg) {
                var ref;
                return ((ref = pkg.releases) != null ? ref.latest : void 0) != null;
              }).map(function(arg) {
                var downloads, metadata, readme, stargazers_count;
                readme = arg.readme, metadata = arg.metadata, downloads = arg.downloads, stargazers_count = arg.stargazers_count;
                return Object.assign(metadata, {
                  readme: readme,
                  downloads: downloads,
                  stargazers_count: stargazers_count
                });
              }).sort(function(a, b) {
                return b.downloads - a.downloads;
              }));
            } catch (error1) {
              e = error1;
              error = new Error("Searching for \u201C" + query + "\u201D failed.");
              error.stderr = e.message + '\n' + body;
              return reject(error);
            }
          }
        });
      });
    };

    return AtomIoClient;

  })();

}).call(this);

//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiL2J1aWxkL2F0b20vc3JjL2F0b20tMS4zNC4wL291dC9hcHAvbm9kZV9tb2R1bGVzL3NldHRpbmdzLXZpZXcvbGliL2F0b20taW8tY2xpZW50LmNvZmZlZSJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiQUFBQTtBQUFBLE1BQUE7O0VBQUEsRUFBQSxHQUFLLE9BQUEsQ0FBUSxTQUFSOztFQUNMLElBQUEsR0FBTyxPQUFBLENBQVEsTUFBUjs7RUFDTixTQUFVLE9BQUEsQ0FBUSxVQUFSOztFQUVYLElBQUEsR0FBTyxPQUFBLENBQVEsTUFBUjs7RUFDUCxPQUFBLEdBQVUsT0FBQSxDQUFRLFNBQVI7O0VBRVYsTUFBTSxDQUFDLE9BQVAsR0FDTTtJQUNTLHNCQUFDLGNBQUQsRUFBa0IsT0FBbEI7TUFBQyxJQUFDLENBQUEsaUJBQUQ7TUFBaUIsSUFBQyxDQUFBLFVBQUQ7O1FBQzdCLElBQUMsQ0FBQSxVQUFXOztNQUVaLElBQUMsQ0FBQSxNQUFELEdBQVUsSUFBQSxHQUFPLEVBQVAsR0FBWSxFQUFaLEdBQWlCO01BQzNCLElBQUMsQ0FBQSxpQkFBRCxDQUFBO01BQ0EsSUFBQyxDQUFBLGlCQUFELENBQUE7SUFMVzs7MkJBUWIsTUFBQSxHQUFRLFNBQUMsS0FBRCxFQUFRLFFBQVI7YUFDTixJQUFDLENBQUEsWUFBRCxDQUFjLEtBQWQsRUFBcUIsQ0FBQSxTQUFBLEtBQUE7ZUFBQSxTQUFDLEdBQUQsRUFBTSxNQUFOO0FBQ25CLGNBQUE7VUFBQSxJQUFvRSxNQUFwRTtZQUFBLEtBQUEsR0FBUSxJQUFJLENBQUMsR0FBTCxDQUFBLENBQUEsR0FBYSxRQUFBLENBQVMsTUFBTSxDQUFDLEtBQVAsQ0FBYSxHQUFiLENBQWlCLENBQUMsR0FBbEIsQ0FBQSxDQUFULENBQWIsR0FBaUQsS0FBQyxDQUFBLE9BQTFEOztVQUNBLElBQUcsTUFBQSxJQUFXLENBQUMsQ0FBSSxLQUFKLElBQWEsQ0FBSSxLQUFDLENBQUEsTUFBRCxDQUFBLENBQWxCLENBQWQ7bUJBQ0UsUUFBQSxDQUFTLElBQVQsRUFBZSxNQUFmLEVBREY7V0FBQSxNQUFBO21CQUdFLEtBQUMsQ0FBQSxtQkFBRCxDQUFxQixLQUFyQixFQUE0QixRQUE1QixFQUhGOztRQUZtQjtNQUFBLENBQUEsQ0FBQSxDQUFBLElBQUEsQ0FBckI7SUFETTs7NEJBVVIsU0FBQSxHQUFTLFNBQUMsSUFBRCxFQUFPLFFBQVA7QUFDUCxVQUFBO01BQUEsV0FBQSxHQUFjLFdBQUEsR0FBWTthQUMxQixJQUFDLENBQUEsY0FBRCxDQUFnQixXQUFoQixFQUE2QixFQUE3QixFQUFpQyxDQUFBLFNBQUEsS0FBQTtlQUFBLFNBQUMsR0FBRCxFQUFNLElBQU47VUFDL0IsSUFBRyxJQUFIO21CQUNFLFFBQUEsQ0FBUyxJQUFULEVBQWUsSUFBZixFQURGO1dBQUEsTUFBQTttQkFHRSxLQUFDLENBQUEsT0FBRCxDQUFTLFdBQVQsRUFBc0IsUUFBdEIsRUFIRjs7UUFEK0I7TUFBQSxDQUFBLENBQUEsQ0FBQSxJQUFBLENBQWpDO0lBRk87OzJCQVFULGdCQUFBLEdBQWtCLFNBQUMsUUFBRDthQUVoQixJQUFDLENBQUEsY0FBRCxDQUFnQixtQkFBaEIsRUFBcUMsRUFBckMsRUFBeUMsQ0FBQSxTQUFBLEtBQUE7ZUFBQSxTQUFDLEdBQUQsRUFBTSxJQUFOO1VBQ3ZDLElBQUcsSUFBSDttQkFDRSxRQUFBLENBQVMsSUFBVCxFQUFlLElBQWYsRUFERjtXQUFBLE1BQUE7bUJBR0UsS0FBQyxDQUFBLFdBQUQsQ0FBYSxLQUFiLEVBQW9CLFFBQXBCLEVBSEY7O1FBRHVDO01BQUEsQ0FBQSxDQUFBLENBQUEsSUFBQSxDQUF6QztJQUZnQjs7MkJBUWxCLGNBQUEsR0FBZ0IsU0FBQyxRQUFEO2FBRWQsSUFBQyxDQUFBLGNBQUQsQ0FBZ0IsaUJBQWhCLEVBQW1DLEVBQW5DLEVBQXVDLENBQUEsU0FBQSxLQUFBO2VBQUEsU0FBQyxHQUFELEVBQU0sSUFBTjtVQUNyQyxJQUFHLElBQUg7bUJBQ0UsUUFBQSxDQUFTLElBQVQsRUFBZSxJQUFmLEVBREY7V0FBQSxNQUFBO21CQUdFLEtBQUMsQ0FBQSxXQUFELENBQWEsSUFBYixFQUFtQixRQUFuQixFQUhGOztRQURxQztNQUFBLENBQUEsQ0FBQSxDQUFBLElBQUEsQ0FBdkM7SUFGYzs7MkJBUWhCLFdBQUEsR0FBYSxTQUFDLFVBQUQsRUFBYSxRQUFiO2FBR1gsSUFBQyxDQUFBLGNBQWMsQ0FBQyxXQUFoQixDQUE0QixVQUE1QixDQUNFLENBQUMsSUFESCxDQUNRLENBQUEsU0FBQSxLQUFBO2VBQUEsU0FBQyxRQUFEO0FBRUosY0FBQTtVQUFBLEdBQUEsR0FBUyxVQUFILEdBQW1CLGlCQUFuQixHQUEwQztVQUNoRCxNQUFBLEdBQ0U7WUFBQSxJQUFBLEVBQU0sUUFBTjtZQUNBLFNBQUEsRUFBVyxJQUFJLENBQUMsR0FBTCxDQUFBLENBRFg7O1VBRUYsWUFBWSxDQUFDLE9BQWIsQ0FBcUIsS0FBQyxDQUFBLGVBQUQsQ0FBaUIsR0FBakIsQ0FBckIsRUFBNEMsSUFBSSxDQUFDLFNBQUwsQ0FBZSxNQUFmLENBQTVDO2lCQUVBLFFBQUEsQ0FBUyxJQUFULEVBQWUsUUFBZjtRQVJJO01BQUEsQ0FBQSxDQUFBLENBQUEsSUFBQSxDQURSLENBVUUsRUFBQyxLQUFELEVBVkYsQ0FVUyxTQUFDLEtBQUQ7ZUFDTCxRQUFBLENBQVMsS0FBVCxFQUFnQixJQUFoQjtNQURLLENBVlQ7SUFIVzs7MkJBZ0JiLE9BQUEsR0FBUyxTQUFDLElBQUQsRUFBTyxRQUFQO0FBQ1AsVUFBQTtNQUFBLE9BQUEsR0FBVTtRQUNSLEdBQUEsRUFBSyxFQUFBLEdBQUcsSUFBQyxDQUFBLE9BQUosR0FBYyxJQURYO1FBRVIsT0FBQSxFQUFTO1VBQUMsWUFBQSxFQUFjLFNBQVMsQ0FBQyxTQUF6QjtTQUZEO1FBR1IsSUFBQSxFQUFNLElBSEU7O2FBTVYsT0FBQSxDQUFRLE9BQVIsRUFBaUIsQ0FBQSxTQUFBLEtBQUE7ZUFBQSxTQUFDLEdBQUQsRUFBTSxHQUFOLEVBQVcsSUFBWDtBQUNmLGNBQUE7VUFBQSxJQUF3QixHQUF4QjtBQUFBLG1CQUFPLFFBQUEsQ0FBUyxHQUFULEVBQVA7O0FBRUE7WUFHRSxJQUFBLEdBQU8sSUFBSSxDQUFDLEtBQUwsQ0FBVyxJQUFYO1lBQ1AsT0FBTyxJQUFJLENBQUM7WUFFWixNQUFBLEdBQ0U7Y0FBQSxJQUFBLEVBQU0sSUFBTjtjQUNBLFNBQUEsRUFBVyxJQUFJLENBQUMsR0FBTCxDQUFBLENBRFg7O1lBRUYsWUFBWSxDQUFDLE9BQWIsQ0FBcUIsS0FBQyxDQUFBLGVBQUQsQ0FBaUIsSUFBakIsQ0FBckIsRUFBNkMsSUFBSSxDQUFDLFNBQUwsQ0FBZSxNQUFmLENBQTdDO21CQUNBLFFBQUEsQ0FBUyxHQUFULEVBQWMsTUFBTSxDQUFDLElBQXJCLEVBVkY7V0FBQSxjQUFBO1lBV007bUJBQ0osUUFBQSxDQUFTLEtBQVQsRUFaRjs7UUFIZTtNQUFBLENBQUEsQ0FBQSxDQUFBLElBQUEsQ0FBakI7SUFQTzs7MkJBd0JULGVBQUEsR0FBaUIsU0FBQyxJQUFEO2FBQ2YsZ0JBQUEsR0FBaUI7SUFERjs7MkJBR2pCLE1BQUEsR0FBUSxTQUFBO2FBQ04sU0FBUyxDQUFDO0lBREo7OzJCQUtSLGNBQUEsR0FBZ0IsU0FBQyxXQUFELEVBQWMsT0FBZCxFQUF1QixRQUF2QjtBQUNkLFVBQUE7TUFBQSxJQUFBLENBQU8sUUFBUDtRQUNFLFFBQUEsR0FBVztRQUNYLE9BQUEsR0FBVSxHQUZaOztNQUlBLElBQUEsQ0FBTyxPQUFPLENBQUMsS0FBZjtRQUVFLE9BQU8sQ0FBQyxLQUFSLEdBQWdCLENBQUksSUFBQyxDQUFBLE1BQUQsQ0FBQSxFQUZ0Qjs7TUFJQSxNQUFBLEdBQVMsWUFBWSxDQUFDLE9BQWIsQ0FBcUIsSUFBQyxDQUFBLGVBQUQsQ0FBaUIsV0FBakIsQ0FBckI7TUFDVCxNQUFBLEdBQVksTUFBSCxHQUFlLElBQUksQ0FBQyxLQUFMLENBQVcsTUFBWCxDQUFmLEdBQUE7TUFDVCxJQUFHLGdCQUFBLElBQVksQ0FBQyxDQUFJLElBQUMsQ0FBQSxNQUFELENBQUEsQ0FBSixJQUFpQixPQUFPLENBQUMsS0FBekIsSUFBa0MsQ0FBQyxJQUFJLENBQUMsR0FBTCxDQUFBLENBQUEsR0FBYSxNQUFNLENBQUMsU0FBcEIsR0FBZ0MsSUFBQyxDQUFBLE1BQWxDLENBQW5DLENBQWY7O1VBQ0UsU0FBVTtZQUFBLElBQUEsRUFBTSxFQUFOOzs7ZUFDVixRQUFBLENBQVMsSUFBVCxFQUFlLE1BQU0sQ0FBQyxJQUF0QixFQUZGO09BQUEsTUFHSyxJQUFPLGdCQUFKLElBQWdCLENBQUksSUFBQyxDQUFBLE1BQUQsQ0FBQSxDQUF2QjtlQUdILFFBQUEsQ0FBUyxJQUFULEVBQWUsRUFBZixFQUhHO09BQUEsTUFBQTtlQU1ILFFBQUEsQ0FBUyxJQUFULEVBQWUsSUFBZixFQU5HOztJQWRTOzsyQkFzQmhCLGlCQUFBLEdBQW1CLFNBQUE7YUFDakIsRUFBRSxDQUFDLFFBQUgsQ0FBWSxJQUFDLENBQUEsWUFBRCxDQUFBLENBQVo7SUFEaUI7OzJCQUduQixVQUFBLEdBQVksU0FBQyxLQUFEO2FBQ1YsSUFBSSxDQUFDLElBQUwsQ0FBVSxJQUFDLENBQUEsWUFBRCxDQUFBLENBQVYsRUFBOEIsS0FBRCxHQUFPLEdBQVAsR0FBUyxDQUFDLElBQUksQ0FBQyxHQUFMLENBQUEsQ0FBRCxDQUF0QztJQURVOzsyQkFHWixZQUFBLEdBQWMsU0FBQyxLQUFELEVBQVEsUUFBUjthQUNaLElBQUEsQ0FBSyxJQUFDLENBQUEsVUFBRCxDQUFZLEtBQVosQ0FBTCxFQUF5QixDQUFBLFNBQUEsS0FBQTtlQUFBLFNBQUMsR0FBRCxFQUFNLEtBQU47QUFDdkIsY0FBQTtVQUFBLElBQXdCLEdBQXhCO0FBQUEsbUJBQU8sUUFBQSxDQUFTLEdBQVQsRUFBUDs7VUFDQSxLQUFLLENBQUMsSUFBTixDQUFBLENBQVksQ0FBQyxPQUFiLENBQUE7QUFDQSxlQUFBLHVDQUFBOztZQUNFLFFBQUEsR0FBVyxJQUFJLENBQUMsUUFBTCxDQUFjLFNBQWQ7WUFDWCxNQUFtQixRQUFRLENBQUMsS0FBVCxDQUFlLEdBQWYsQ0FBbkIsRUFBTTtZQUNOLElBQUcsSUFBSSxDQUFDLEdBQUwsQ0FBQSxDQUFBLEdBQWEsUUFBQSxDQUFTLFNBQVQsQ0FBYixHQUFtQyxLQUFDLENBQUEsTUFBdkM7QUFDRSxxQkFBTyxRQUFBLENBQVMsSUFBVCxFQUFlLFNBQWYsRUFEVDs7QUFIRjtpQkFLQSxRQUFBLENBQVMsSUFBVCxFQUFlLElBQWY7UUFSdUI7TUFBQSxDQUFBLENBQUEsQ0FBQSxJQUFBLENBQXpCO0lBRFk7OzJCQVdkLFVBQUEsR0FBWSxTQUFDLEtBQUQ7YUFDVixJQUFJLENBQUMsSUFBTCxDQUFVLElBQUMsQ0FBQSxZQUFELENBQUEsQ0FBVixFQUE4QixLQUFELEdBQU8sV0FBcEM7SUFEVTs7MkJBR1osbUJBQUEsR0FBcUIsU0FBQyxLQUFELEVBQVEsUUFBUjtBQUNuQixVQUFBO01BQUEsSUFBRyxDQUFJLElBQUMsQ0FBQSxNQUFELENBQUEsQ0FBUDtlQUNFLFFBQUEsQ0FBUyxJQUFULEVBQWUsSUFBZixFQURGO09BQUEsTUFBQTtRQUdFLFNBQUEsR0FBWSxJQUFDLENBQUEsVUFBRCxDQUFZLEtBQVo7UUFDWixhQUFBLEdBQWdCO1VBQ2QsR0FBQSxFQUFLLHdDQUFBLEdBQXlDLEtBRGhDO1VBRWQsT0FBQSxFQUFTO1lBQUMsWUFBQSxFQUFjLFNBQVMsQ0FBQyxTQUF6QjtXQUZLOztlQUloQixPQUFPLENBQUMsSUFBUixDQUFhLGFBQWIsRUFBNEIsU0FBQyxLQUFELEVBQVEsUUFBUixFQUFrQixJQUFsQjtBQUMxQixjQUFBO1VBQUEsSUFBRyxlQUFBLElBQVUsUUFBUSxDQUFDLFVBQVQsS0FBeUIsR0FBbkMsSUFBMEMsQ0FBSSxRQUFRLENBQUMsT0FBUSxDQUFBLGNBQUEsQ0FBZSxDQUFDLFVBQWpDLENBQTRDLFFBQTVDLENBQWpEO21CQUNFLFFBQUEsQ0FBUyxLQUFULEVBREY7V0FBQSxNQUFBO1lBR0UsV0FBQSxHQUFjLEVBQUUsQ0FBQyxpQkFBSCxDQUFxQixTQUFyQjtZQUNkLFdBQVcsQ0FBQyxFQUFaLENBQWUsUUFBZixFQUF5QixTQUFBO3FCQUFHLFFBQUEsQ0FBUyxJQUFULEVBQWUsU0FBZjtZQUFILENBQXpCO1lBQ0EsV0FBVyxDQUFDLEVBQVosQ0FBZSxPQUFmLEVBQXdCLFNBQUMsS0FBRDtjQUN0QixXQUFXLENBQUMsS0FBWixDQUFBO0FBQ0E7Z0JBQ0UsSUFBMkIsRUFBRSxDQUFDLFVBQUgsQ0FBYyxTQUFkLENBQTNCO2tCQUFBLEVBQUUsQ0FBQyxVQUFILENBQWMsU0FBZCxFQUFBO2lCQURGO2VBQUE7cUJBRUEsUUFBQSxDQUFTLEtBQVQ7WUFKc0IsQ0FBeEI7bUJBS0EsT0FBQSxDQUFRLGFBQVIsQ0FBc0IsQ0FBQyxJQUF2QixDQUE0QixXQUE1QixFQVZGOztRQUQwQixDQUE1QixFQVJGOztJQURtQjs7MkJBMEJyQixpQkFBQSxHQUFtQixTQUFBO0FBQ2pCLFVBQUE7TUFBQSxZQUFBLEdBQWUsQ0FBQSxTQUFBLEtBQUE7ZUFBQSxTQUFDLEtBQUQ7QUFDYixjQUFBO1VBQUEsVUFBQSxHQUFhLElBQUksQ0FBQyxJQUFMLENBQVUsS0FBQyxDQUFBLFlBQUQsQ0FBQSxDQUFWLEVBQTJCLEtBQTNCO2lCQUNiLEVBQUUsQ0FBQyxNQUFILENBQVUsVUFBVixFQUFzQixTQUFDLEtBQUQ7WUFDcEIsSUFBRyxLQUFBLElBQVUsS0FBSyxDQUFDLElBQU4sS0FBZ0IsUUFBN0I7cUJBQ0UsT0FBTyxDQUFDLElBQVIsQ0FBYSx5QkFBQSxHQUEwQixLQUFLLENBQUMsSUFBaEMsR0FBcUMsS0FBckMsR0FBMEMsVUFBdkQsRUFERjs7VUFEb0IsQ0FBdEI7UUFGYTtNQUFBLENBQUEsQ0FBQSxDQUFBLElBQUE7YUFNZixFQUFFLENBQUMsT0FBSCxDQUFXLElBQUMsQ0FBQSxZQUFELENBQUEsQ0FBWCxFQUE0QixTQUFDLEtBQUQsRUFBUSxNQUFSO0FBQzFCLFlBQUE7O1VBQUEsU0FBVTs7UUFDVixLQUFBLEdBQVE7QUFDUixhQUFBLHdDQUFBOztVQUNFLEtBQUEsR0FBUSxRQUFRLENBQUMsS0FBVCxDQUFlLEdBQWY7VUFDUixLQUFBLEdBQVEsS0FBSyxDQUFDLEdBQU4sQ0FBQTtVQUNSLEdBQUEsR0FBTSxLQUFLLENBQUMsSUFBTixDQUFXLEdBQVg7O1lBQ04sS0FBTSxDQUFBLEdBQUEsSUFBUTs7VUFDZCxLQUFNLENBQUEsR0FBQSxDQUFJLENBQUMsSUFBWCxDQUFtQixHQUFELEdBQUssR0FBTCxHQUFRLEtBQTFCO0FBTEY7QUFPQTthQUFBLFlBQUE7O1VBQ0UsUUFBUSxDQUFDLElBQVQsQ0FBQTtVQUNBLFFBQVEsQ0FBQyxHQUFULENBQUE7dUJBSUEsUUFBUSxDQUFDLE9BQVQsQ0FBaUIsWUFBakI7QUFORjs7TUFWMEIsQ0FBNUI7SUFQaUI7OzJCQXlCbkIsWUFBQSxHQUFjLFNBQUE7c0NBQ1osSUFBQyxDQUFBLFlBQUQsSUFBQyxDQUFBLFlBQWEsSUFBSSxDQUFDLElBQUwsQ0FBVSxNQUFNLENBQUMsR0FBRyxDQUFDLE9BQVgsQ0FBbUIsVUFBbkIsQ0FBVixFQUEwQyxPQUExQyxFQUFtRCxlQUFuRDtJQURGOzsyQkFHZCxNQUFBLEdBQVEsU0FBQyxLQUFELEVBQVEsT0FBUjtBQUNOLFVBQUE7TUFBQSxFQUFBLEdBQUs7UUFBQyxDQUFBLEVBQUcsS0FBSjs7TUFFTCxJQUFHLE9BQU8sQ0FBQyxNQUFYO1FBQ0UsRUFBRSxDQUFDLE1BQUgsR0FBWSxRQURkO09BQUEsTUFFSyxJQUFHLE9BQU8sQ0FBQyxRQUFYO1FBQ0gsRUFBRSxDQUFDLE1BQUgsR0FBWSxVQURUOztNQUdMLE9BQUEsR0FBVTtRQUNSLEdBQUEsRUFBUSxJQUFDLENBQUEsT0FBRixHQUFVLGlCQURUO1FBRVIsT0FBQSxFQUFTO1VBQUMsWUFBQSxFQUFjLFNBQVMsQ0FBQyxTQUF6QjtTQUZEO1FBR1IsRUFBQSxFQUFJLEVBSEk7UUFJUixJQUFBLEVBQU0sSUFKRTs7YUFPVixJQUFJLE9BQUosQ0FBWSxTQUFDLE9BQUQsRUFBVSxNQUFWO2VBQ1YsT0FBQSxDQUFRLE9BQVIsRUFBaUIsU0FBQyxHQUFELEVBQU0sR0FBTixFQUFXLElBQVg7QUFDZixjQUFBO1VBQUEsSUFBRyxHQUFIO1lBQ0UsS0FBQSxHQUFRLElBQUksS0FBSixDQUFVLHNCQUFBLEdBQXVCLEtBQXZCLEdBQTZCLGdCQUF2QztZQUNSLEtBQUssQ0FBQyxNQUFOLEdBQWUsR0FBRyxDQUFDO21CQUNuQixNQUFBLENBQU8sS0FBUCxFQUhGO1dBQUEsTUFBQTtBQUtFO2NBR0UsSUFBQSxHQUFPLElBQUksQ0FBQyxLQUFMLENBQVcsSUFBWDtxQkFDUCxPQUFBLENBQ0UsSUFBSSxDQUFDLE1BQUwsQ0FBWSxTQUFDLEdBQUQ7QUFBUyxvQkFBQTt1QkFBQTtjQUFULENBQVosQ0FDSSxDQUFDLEdBREwsQ0FDUyxTQUFDLEdBQUQ7QUFDSCxvQkFBQTtnQkFESyxxQkFBUSx5QkFBVSwyQkFBVzt1QkFDbEMsTUFBTSxDQUFDLE1BQVAsQ0FBYyxRQUFkLEVBQXdCO2tCQUFDLFFBQUEsTUFBRDtrQkFBUyxXQUFBLFNBQVQ7a0JBQW9CLGtCQUFBLGdCQUFwQjtpQkFBeEI7Y0FERyxDQURULENBR0ksQ0FBQyxJQUhMLENBR1UsU0FBQyxDQUFELEVBQUksQ0FBSjt1QkFBVSxDQUFDLENBQUMsU0FBRixHQUFjLENBQUMsQ0FBQztjQUExQixDQUhWLENBREYsRUFKRjthQUFBLGNBQUE7Y0FVTTtjQUNKLEtBQUEsR0FBUSxJQUFJLEtBQUosQ0FBVSxzQkFBQSxHQUF1QixLQUF2QixHQUE2QixnQkFBdkM7Y0FDUixLQUFLLENBQUMsTUFBTixHQUFlLENBQUMsQ0FBQyxPQUFGLEdBQVksSUFBWixHQUFtQjtxQkFDbEMsTUFBQSxDQUFPLEtBQVAsRUFiRjthQUxGOztRQURlLENBQWpCO01BRFUsQ0FBWjtJQWZNOzs7OztBQW5NViIsInNvdXJjZXNDb250ZW50IjpbImZzID0gcmVxdWlyZSAnZnMtcGx1cydcbnBhdGggPSByZXF1aXJlICdwYXRoJ1xue3JlbW90ZX0gPSByZXF1aXJlICdlbGVjdHJvbidcblxuZ2xvYiA9IHJlcXVpcmUgJ2dsb2InXG5yZXF1ZXN0ID0gcmVxdWlyZSAncmVxdWVzdCdcblxubW9kdWxlLmV4cG9ydHMgPVxuY2xhc3MgQXRvbUlvQ2xpZW50XG4gIGNvbnN0cnVjdG9yOiAoQHBhY2thZ2VNYW5hZ2VyLCBAYmFzZVVSTCkgLT5cbiAgICBAYmFzZVVSTCA/PSAnaHR0cHM6Ly9hdG9tLmlvL2FwaS8nXG4gICAgIyAxMiBob3VyIGV4cGlyeVxuICAgIEBleHBpcnkgPSAxMDAwICogNjAgKiA2MCAqIDEyXG4gICAgQGNyZWF0ZUF2YXRhckNhY2hlKClcbiAgICBAZXhwaXJlQXZhdGFyQ2FjaGUoKVxuXG4gICMgUHVibGljOiBHZXQgYW4gYXZhdGFyIGltYWdlIGZyb20gdGhlIGZpbGVzeXN0ZW0sIGZldGNoaW5nIGl0IGZpcnN0IGlmIG5lY2Vzc2FyeVxuICBhdmF0YXI6IChsb2dpbiwgY2FsbGJhY2spIC0+XG4gICAgQGNhY2hlZEF2YXRhciBsb2dpbiwgKGVyciwgY2FjaGVkKSA9PlxuICAgICAgc3RhbGUgPSBEYXRlLm5vdygpIC0gcGFyc2VJbnQoY2FjaGVkLnNwbGl0KCctJykucG9wKCkpID4gQGV4cGlyeSBpZiBjYWNoZWRcbiAgICAgIGlmIGNhY2hlZCBhbmQgKG5vdCBzdGFsZSBvciBub3QgQG9ubGluZSgpKVxuICAgICAgICBjYWxsYmFjayBudWxsLCBjYWNoZWRcbiAgICAgIGVsc2VcbiAgICAgICAgQGZldGNoQW5kQ2FjaGVBdmF0YXIobG9naW4sIGNhbGxiYWNrKVxuXG4gICMgUHVibGljOiBnZXQgYSBwYWNrYWdlIGZyb20gdGhlIGF0b20uaW8gQVBJLCB3aXRoIHRoZSBhcHByb3ByaWF0ZSBsZXZlbCBvZlxuICAjIGNhY2hpbmcuXG4gIHBhY2thZ2U6IChuYW1lLCBjYWxsYmFjaykgLT5cbiAgICBwYWNrYWdlUGF0aCA9IFwicGFja2FnZXMvI3tuYW1lfVwiXG4gICAgQGZldGNoRnJvbUNhY2hlIHBhY2thZ2VQYXRoLCB7fSwgKGVyciwgZGF0YSkgPT5cbiAgICAgIGlmIGRhdGFcbiAgICAgICAgY2FsbGJhY2sobnVsbCwgZGF0YSlcbiAgICAgIGVsc2VcbiAgICAgICAgQHJlcXVlc3QocGFja2FnZVBhdGgsIGNhbGxiYWNrKVxuXG4gIGZlYXR1cmVkUGFja2FnZXM6IChjYWxsYmFjaykgLT5cbiAgICAjIFRPRE8gY2xlYW4gdXAgY2FjaGluZyBjb3B5cGFzdGFcbiAgICBAZmV0Y2hGcm9tQ2FjaGUgJ3BhY2thZ2VzL2ZlYXR1cmVkJywge30sIChlcnIsIGRhdGEpID0+XG4gICAgICBpZiBkYXRhXG4gICAgICAgIGNhbGxiYWNrKG51bGwsIGRhdGEpXG4gICAgICBlbHNlXG4gICAgICAgIEBnZXRGZWF0dXJlZChmYWxzZSwgY2FsbGJhY2spXG5cbiAgZmVhdHVyZWRUaGVtZXM6IChjYWxsYmFjaykgLT5cbiAgICAjIFRPRE8gY2xlYW4gdXAgY2FjaGluZyBjb3B5cGFzdGFcbiAgICBAZmV0Y2hGcm9tQ2FjaGUgJ3RoZW1lcy9mZWF0dXJlZCcsIHt9LCAoZXJyLCBkYXRhKSA9PlxuICAgICAgaWYgZGF0YVxuICAgICAgICBjYWxsYmFjayhudWxsLCBkYXRhKVxuICAgICAgZWxzZVxuICAgICAgICBAZ2V0RmVhdHVyZWQodHJ1ZSwgY2FsbGJhY2spXG5cbiAgZ2V0RmVhdHVyZWQ6IChsb2FkVGhlbWVzLCBjYWxsYmFjaykgLT5cbiAgICAjIGFwbSBhbHJlYWR5IGRvZXMgdGhpcywgbWlnaHQgYXMgd2VsbCB1c2UgaXQgaW5zdGVhZCBvZiByZXF1ZXN0IGkgZ3Vlc3M/IFRoZVxuICAgICMgZG93bnNpZGUgaXMgdGhhdCBJIG5lZWQgdG8gcmVwZWF0IGNhY2hpbmcgbG9naWMgaGVyZS5cbiAgICBAcGFja2FnZU1hbmFnZXIuZ2V0RmVhdHVyZWQobG9hZFRoZW1lcylcbiAgICAgIC50aGVuIChwYWNrYWdlcykgPT5cbiAgICAgICAgIyBjb3B5cGFzdGEgZnJvbSBiZWxvd1xuICAgICAgICBrZXkgPSBpZiBsb2FkVGhlbWVzIHRoZW4gJ3RoZW1lcy9mZWF0dXJlZCcgZWxzZSAncGFja2FnZXMvZmVhdHVyZWQnXG4gICAgICAgIGNhY2hlZCA9XG4gICAgICAgICAgZGF0YTogcGFja2FnZXNcbiAgICAgICAgICBjcmVhdGVkT246IERhdGUubm93KClcbiAgICAgICAgbG9jYWxTdG9yYWdlLnNldEl0ZW0oQGNhY2hlS2V5Rm9yUGF0aChrZXkpLCBKU09OLnN0cmluZ2lmeShjYWNoZWQpKVxuICAgICAgICAjIGVuZCBjb3B5cGFzdGFcbiAgICAgICAgY2FsbGJhY2sobnVsbCwgcGFja2FnZXMpXG4gICAgICAuY2F0Y2ggKGVycm9yKSAtPlxuICAgICAgICBjYWxsYmFjayhlcnJvciwgbnVsbClcblxuICByZXF1ZXN0OiAocGF0aCwgY2FsbGJhY2spIC0+XG4gICAgb3B0aW9ucyA9IHtcbiAgICAgIHVybDogXCIje0BiYXNlVVJMfSN7cGF0aH1cIlxuICAgICAgaGVhZGVyczogeydVc2VyLUFnZW50JzogbmF2aWdhdG9yLnVzZXJBZ2VudH1cbiAgICAgIGd6aXA6IHRydWVcbiAgICB9XG5cbiAgICByZXF1ZXN0IG9wdGlvbnMsIChlcnIsIHJlcywgYm9keSkgPT5cbiAgICAgIHJldHVybiBjYWxsYmFjayhlcnIpIGlmIGVyclxuXG4gICAgICB0cnlcbiAgICAgICAgIyBOT1RFOiByZXF1ZXN0J3MganNvbiBvcHRpb24gZG9lcyBub3QgcG9wdWxhdGUgZXJyIGlmIHBhcnNpbmcgZmFpbHMsXG4gICAgICAgICMgc28gd2UgZG8gaXQgbWFudWFsbHlcbiAgICAgICAgYm9keSA9IEpTT04ucGFyc2UoYm9keSlcbiAgICAgICAgZGVsZXRlIGJvZHkudmVyc2lvbnNcblxuICAgICAgICBjYWNoZWQgPVxuICAgICAgICAgIGRhdGE6IGJvZHlcbiAgICAgICAgICBjcmVhdGVkT246IERhdGUubm93KClcbiAgICAgICAgbG9jYWxTdG9yYWdlLnNldEl0ZW0oQGNhY2hlS2V5Rm9yUGF0aChwYXRoKSwgSlNPTi5zdHJpbmdpZnkoY2FjaGVkKSlcbiAgICAgICAgY2FsbGJhY2soZXJyLCBjYWNoZWQuZGF0YSlcbiAgICAgIGNhdGNoIGVycm9yXG4gICAgICAgIGNhbGxiYWNrKGVycm9yKVxuXG4gIGNhY2hlS2V5Rm9yUGF0aDogKHBhdGgpIC0+XG4gICAgXCJzZXR0aW5ncy12aWV3OiN7cGF0aH1cIlxuXG4gIG9ubGluZTogLT5cbiAgICBuYXZpZ2F0b3Iub25MaW5lXG5cbiAgIyBUaGlzIGNvdWxkIHVzZSBhIGJldHRlciBuYW1lLCBzaW5jZSBpdCBjaGVja3Mgd2hldGhlciBpdCdzIGFwcHJvcHJpYXRlIHRvIHJldHVyblxuICAjIHRoZSBjYWNoZWQgZGF0YSBhbmQgcHJldGVuZHMgaXQncyBudWxsIGlmIGl0J3Mgc3RhbGUgYW5kIHdlJ3JlIG9ubGluZVxuICBmZXRjaEZyb21DYWNoZTogKHBhY2thZ2VQYXRoLCBvcHRpb25zLCBjYWxsYmFjaykgLT5cbiAgICB1bmxlc3MgY2FsbGJhY2tcbiAgICAgIGNhbGxiYWNrID0gb3B0aW9uc1xuICAgICAgb3B0aW9ucyA9IHt9XG5cbiAgICB1bmxlc3Mgb3B0aW9ucy5mb3JjZVxuICAgICAgIyBTZXQgYGZvcmNlYCB0byB0cnVlIGlmIHdlIGNhbid0IHJlYWNoIHRoZSBuZXR3b3JrLlxuICAgICAgb3B0aW9ucy5mb3JjZSA9IG5vdCBAb25saW5lKClcblxuICAgIGNhY2hlZCA9IGxvY2FsU3RvcmFnZS5nZXRJdGVtKEBjYWNoZUtleUZvclBhdGgocGFja2FnZVBhdGgpKVxuICAgIGNhY2hlZCA9IGlmIGNhY2hlZCB0aGVuIEpTT04ucGFyc2UoY2FjaGVkKVxuICAgIGlmIGNhY2hlZD8gYW5kIChub3QgQG9ubGluZSgpIG9yIG9wdGlvbnMuZm9yY2Ugb3IgKERhdGUubm93KCkgLSBjYWNoZWQuY3JlYXRlZE9uIDwgQGV4cGlyeSkpXG4gICAgICBjYWNoZWQgPz0gZGF0YToge31cbiAgICAgIGNhbGxiYWNrKG51bGwsIGNhY2hlZC5kYXRhKVxuICAgIGVsc2UgaWYgbm90IGNhY2hlZD8gYW5kIG5vdCBAb25saW5lKClcbiAgICAgICMgVGhlIHVzZXIgaGFzbid0IHJlcXVlc3RlZCB0aGlzIHJlc291cmNlIGJlZm9yZSBhbmQgdGhlcmUncyBubyB3YXkgZm9yIHVzXG4gICAgICAjIHRvIGdldCBpdCB0byB0aGVtIHNvIGp1c3QgaGFuZCBiYWNrIGFuIGVtcHR5IG9iamVjdCBzbyBjYWxsZXJzIGRvbid0IGNyYXNoXG4gICAgICBjYWxsYmFjayhudWxsLCB7fSlcbiAgICBlbHNlXG4gICAgICAjIGZhbHN5IGRhdGEgbWVhbnMgXCJ0cnkgdG8gaGl0IHRoZSBuZXR3b3JrXCJcbiAgICAgIGNhbGxiYWNrKG51bGwsIG51bGwpXG5cbiAgY3JlYXRlQXZhdGFyQ2FjaGU6IC0+XG4gICAgZnMubWFrZVRyZWUoQGdldENhY2hlUGF0aCgpKVxuXG4gIGF2YXRhclBhdGg6IChsb2dpbikgLT5cbiAgICBwYXRoLmpvaW4gQGdldENhY2hlUGF0aCgpLCBcIiN7bG9naW59LSN7RGF0ZS5ub3coKX1cIlxuXG4gIGNhY2hlZEF2YXRhcjogKGxvZ2luLCBjYWxsYmFjaykgLT5cbiAgICBnbG9iIEBhdmF0YXJHbG9iKGxvZ2luKSwgKGVyciwgZmlsZXMpID0+XG4gICAgICByZXR1cm4gY2FsbGJhY2soZXJyKSBpZiBlcnJcbiAgICAgIGZpbGVzLnNvcnQoKS5yZXZlcnNlKClcbiAgICAgIGZvciBpbWFnZVBhdGggaW4gZmlsZXNcbiAgICAgICAgZmlsZW5hbWUgPSBwYXRoLmJhc2VuYW1lKGltYWdlUGF0aClcbiAgICAgICAgWy4uLiwgY3JlYXRlZE9uXSA9IGZpbGVuYW1lLnNwbGl0KCctJylcbiAgICAgICAgaWYgRGF0ZS5ub3coKSAtIHBhcnNlSW50KGNyZWF0ZWRPbikgPCBAZXhwaXJ5XG4gICAgICAgICAgcmV0dXJuIGNhbGxiYWNrKG51bGwsIGltYWdlUGF0aClcbiAgICAgIGNhbGxiYWNrKG51bGwsIG51bGwpXG5cbiAgYXZhdGFyR2xvYjogKGxvZ2luKSAtPlxuICAgIHBhdGguam9pbiBAZ2V0Q2FjaGVQYXRoKCksIFwiI3tsb2dpbn0tKihbMC05XSlcIlxuXG4gIGZldGNoQW5kQ2FjaGVBdmF0YXI6IChsb2dpbiwgY2FsbGJhY2spIC0+XG4gICAgaWYgbm90IEBvbmxpbmUoKVxuICAgICAgY2FsbGJhY2sobnVsbCwgbnVsbClcbiAgICBlbHNlXG4gICAgICBpbWFnZVBhdGggPSBAYXZhdGFyUGF0aCBsb2dpblxuICAgICAgcmVxdWVzdE9iamVjdCA9IHtcbiAgICAgICAgdXJsOiBcImh0dHBzOi8vYXZhdGFycy5naXRodWJ1c2VyY29udGVudC5jb20vI3tsb2dpbn1cIlxuICAgICAgICBoZWFkZXJzOiB7J1VzZXItQWdlbnQnOiBuYXZpZ2F0b3IudXNlckFnZW50fVxuICAgICAgfVxuICAgICAgcmVxdWVzdC5oZWFkIHJlcXVlc3RPYmplY3QsIChlcnJvciwgcmVzcG9uc2UsIGJvZHkpIC0+XG4gICAgICAgIGlmIGVycm9yPyBvciByZXNwb25zZS5zdGF0dXNDb2RlIGlzbnQgMjAwIG9yIG5vdCByZXNwb25zZS5oZWFkZXJzWydjb250ZW50LXR5cGUnXS5zdGFydHNXaXRoKCdpbWFnZS8nKVxuICAgICAgICAgIGNhbGxiYWNrKGVycm9yKVxuICAgICAgICBlbHNlXG4gICAgICAgICAgd3JpdGVTdHJlYW0gPSBmcy5jcmVhdGVXcml0ZVN0cmVhbSBpbWFnZVBhdGhcbiAgICAgICAgICB3cml0ZVN0cmVhbS5vbiAnZmluaXNoJywgLT4gY2FsbGJhY2sobnVsbCwgaW1hZ2VQYXRoKVxuICAgICAgICAgIHdyaXRlU3RyZWFtLm9uICdlcnJvcicsIChlcnJvcikgLT5cbiAgICAgICAgICAgIHdyaXRlU3RyZWFtLmNsb3NlKClcbiAgICAgICAgICAgIHRyeVxuICAgICAgICAgICAgICBmcy51bmxpbmtTeW5jIGltYWdlUGF0aCBpZiBmcy5leGlzdHNTeW5jIGltYWdlUGF0aFxuICAgICAgICAgICAgY2FsbGJhY2soZXJyb3IpXG4gICAgICAgICAgcmVxdWVzdChyZXF1ZXN0T2JqZWN0KS5waXBlKHdyaXRlU3RyZWFtKVxuXG4gICMgVGhlIGNhY2hlIGV4cGlyeSBkb2Vzbid0IG5lZWQgdG8gYmUgY2xldmVyLCBvciBldmVuIGNvbXBhcmUgZGF0ZXMsIGl0IGp1c3RcbiAgIyBuZWVkcyB0byBhbHdheXMga2VlcCBhcm91bmQgdGhlIG5ld2VzdCBpdGVtLCBhbmQgdGhhdCBpdGVtIG9ubHkuIFRoZSBsb2NhbFN0b3JhZ2VcbiAgIyBjYWNoZSB1cGRhdGVzIGluIHBsYWNlLCBzbyBpdCBkb2Vzbid0IG5lZWQgdG8gYmUgcHVyZ2VkLlxuXG4gIGV4cGlyZUF2YXRhckNhY2hlOiAtPlxuICAgIGRlbGV0ZUF2YXRhciA9IChjaGlsZCkgPT5cbiAgICAgIGF2YXRhclBhdGggPSBwYXRoLmpvaW4oQGdldENhY2hlUGF0aCgpLCBjaGlsZClcbiAgICAgIGZzLnVubGluayBhdmF0YXJQYXRoLCAoZXJyb3IpIC0+XG4gICAgICAgIGlmIGVycm9yIGFuZCBlcnJvci5jb2RlIGlzbnQgJ0VOT0VOVCcgIyBJZ25vcmUgY2FjaGUgcGF0aHMgdGhhdCBkb24ndCBleGlzdFxuICAgICAgICAgIGNvbnNvbGUud2FybihcIkVycm9yIGRlbGV0aW5nIGF2YXRhciAoI3tlcnJvci5jb2RlfSk6ICN7YXZhdGFyUGF0aH1cIilcblxuICAgIGZzLnJlYWRkaXIgQGdldENhY2hlUGF0aCgpLCAoZXJyb3IsIF9maWxlcykgLT5cbiAgICAgIF9maWxlcyA/PSBbXVxuICAgICAgZmlsZXMgPSB7fVxuICAgICAgZm9yIGZpbGVuYW1lIGluIF9maWxlc1xuICAgICAgICBwYXJ0cyA9IGZpbGVuYW1lLnNwbGl0KCctJylcbiAgICAgICAgc3RhbXAgPSBwYXJ0cy5wb3AoKVxuICAgICAgICBrZXkgPSBwYXJ0cy5qb2luKCctJylcbiAgICAgICAgZmlsZXNba2V5XSA/PSBbXVxuICAgICAgICBmaWxlc1trZXldLnB1c2ggXCIje2tleX0tI3tzdGFtcH1cIlxuXG4gICAgICBmb3Iga2V5LCBjaGlsZHJlbiBvZiBmaWxlc1xuICAgICAgICBjaGlsZHJlbi5zb3J0KClcbiAgICAgICAgY2hpbGRyZW4ucG9wKCkgIyBrZWVwXG4gICAgICAgICMgUmlnaHQgbm93IGEgYnVuY2ggb2YgY2xpZW50cyBtaWdodCBiZSBpbnN0YW50aWF0ZWQgYXQgb25jZSwgc29cbiAgICAgICAgIyB3ZSBjYW4ganVzdCBpZ25vcmUgYXR0ZW1wdHMgdG8gdW5saW5rIGZpbGVzIHRoYXQgaGF2ZSBhbHJlYWR5IGJlZW4gcmVtb3ZlZFxuICAgICAgICAjIC0gdGhpcyBzaG91bGQgYmUgZml4ZWQgd2l0aCBhIHNpbmdsZXRvbiBjbGllbnRcbiAgICAgICAgY2hpbGRyZW4uZm9yRWFjaChkZWxldGVBdmF0YXIpXG5cbiAgZ2V0Q2FjaGVQYXRoOiAtPlxuICAgIEBjYWNoZVBhdGggPz0gcGF0aC5qb2luKHJlbW90ZS5hcHAuZ2V0UGF0aCgndXNlckRhdGEnKSwgJ0NhY2hlJywgJ3NldHRpbmdzLXZpZXcnKVxuXG4gIHNlYXJjaDogKHF1ZXJ5LCBvcHRpb25zKSAtPlxuICAgIHFzID0ge3E6IHF1ZXJ5fVxuXG4gICAgaWYgb3B0aW9ucy50aGVtZXNcbiAgICAgIHFzLmZpbHRlciA9ICd0aGVtZSdcbiAgICBlbHNlIGlmIG9wdGlvbnMucGFja2FnZXNcbiAgICAgIHFzLmZpbHRlciA9ICdwYWNrYWdlJ1xuXG4gICAgb3B0aW9ucyA9IHtcbiAgICAgIHVybDogXCIje0BiYXNlVVJMfXBhY2thZ2VzL3NlYXJjaFwiXG4gICAgICBoZWFkZXJzOiB7J1VzZXItQWdlbnQnOiBuYXZpZ2F0b3IudXNlckFnZW50fVxuICAgICAgcXM6IHFzXG4gICAgICBnemlwOiB0cnVlXG4gICAgfVxuXG4gICAgbmV3IFByb21pc2UgKHJlc29sdmUsIHJlamVjdCkgLT5cbiAgICAgIHJlcXVlc3Qgb3B0aW9ucywgKGVyciwgcmVzLCBib2R5KSAtPlxuICAgICAgICBpZiBlcnJcbiAgICAgICAgICBlcnJvciA9IG5ldyBFcnJvcihcIlNlYXJjaGluZyBmb3IgXFx1MjAxQyN7cXVlcnl9XFx1MjAxRCBmYWlsZWQuXCIpXG4gICAgICAgICAgZXJyb3Iuc3RkZXJyID0gZXJyLm1lc3NhZ2VcbiAgICAgICAgICByZWplY3QgZXJyb3JcbiAgICAgICAgZWxzZVxuICAgICAgICAgIHRyeVxuICAgICAgICAgICAgIyBOT1RFOiByZXF1ZXN0J3MganNvbiBvcHRpb24gZG9lcyBub3QgcG9wdWxhdGUgZXJyIGlmIHBhcnNpbmcgZmFpbHMsXG4gICAgICAgICAgICAjIHNvIHdlIGRvIGl0IG1hbnVhbGx5XG4gICAgICAgICAgICBib2R5ID0gSlNPTi5wYXJzZShib2R5KVxuICAgICAgICAgICAgcmVzb2x2ZShcbiAgICAgICAgICAgICAgYm9keS5maWx0ZXIgKHBrZykgLT4gcGtnLnJlbGVhc2VzPy5sYXRlc3Q/XG4gICAgICAgICAgICAgICAgICAubWFwICh7cmVhZG1lLCBtZXRhZGF0YSwgZG93bmxvYWRzLCBzdGFyZ2F6ZXJzX2NvdW50fSkgLT5cbiAgICAgICAgICAgICAgICAgICAgT2JqZWN0LmFzc2lnbiBtZXRhZGF0YSwge3JlYWRtZSwgZG93bmxvYWRzLCBzdGFyZ2F6ZXJzX2NvdW50fVxuICAgICAgICAgICAgICAgICAgLnNvcnQgKGEsIGIpIC0+IGIuZG93bmxvYWRzIC0gYS5kb3dubG9hZHNcbiAgICAgICAgICAgIClcbiAgICAgICAgICBjYXRjaCBlXG4gICAgICAgICAgICBlcnJvciA9IG5ldyBFcnJvcihcIlNlYXJjaGluZyBmb3IgXFx1MjAxQyN7cXVlcnl9XFx1MjAxRCBmYWlsZWQuXCIpXG4gICAgICAgICAgICBlcnJvci5zdGRlcnIgPSBlLm1lc3NhZ2UgKyAnXFxuJyArIGJvZHlcbiAgICAgICAgICAgIHJlamVjdCBlcnJvclxuIl19
