/**
 * Copyright (c) 2013-present, Facebook, Inc.
 *
 * This source code is licensed under the MIT license found in the
 * LICENSE file in the root directory of this source tree.
 *
 * 
 * @format
 */

'use strict';

var _classCallCheck3 = _interopRequireDefault(require('babel-runtime/helpers/classCallCheck'));

var _toConsumableArray3 = _interopRequireDefault(require('babel-runtime/helpers/toConsumableArray'));

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { 'default': obj }; }

var queryID = 1;

/**
 * A network logger transaction is an object that you can construct, pass around
 * and add logs to, ultimately calling its commit method when you're done.
 * Different transactions can have different commit logic. One might use
 * `console.log`. Another might ping a logging endpoint. Another might add some
 * autogenerated logs before doing either of the foregoing.
 */
var RelayNetworkLoggerTransaction = function () {
  function RelayNetworkLoggerTransaction(_ref) {
    var request = _ref.request,
        variables = _ref.variables,
        cacheConfig = _ref.cacheConfig,
        uploadables = _ref.uploadables;
    (0, _classCallCheck3['default'])(this, RelayNetworkLoggerTransaction);
    this._hasCommittedLogs = false;
    this._logs = [];

    this._cacheConfig = cacheConfig;
    this._id = queryID++;
    this._request = request;
    this._uploadables = uploadables;
    this._variables = variables;
  }

  RelayNetworkLoggerTransaction.prototype.addLog = function addLog(label) {
    for (var _len = arguments.length, values = Array(_len > 1 ? _len - 1 : 0), _key = 1; _key < _len; _key++) {
      values[_key - 1] = arguments[_key];
    }

    this._logs.push({ label: label, values: values });
  };

  RelayNetworkLoggerTransaction.prototype.clearLogs = function clearLogs() {
    this._logs = [];
  };

  RelayNetworkLoggerTransaction.prototype.printLogs = function printLogs(error, payload, status) {
    /* eslint-disable no-console */
    var transactionId = this.getIdentifier();
    console.groupCollapsed && console.groupCollapsed('%c' + transactionId, error ? 'color:red' : '');
    console.timeEnd && console.timeEnd(transactionId);
    this.getLogsToPrint(error, payload, status).forEach(function (_ref2) {
      var _console;

      var label = _ref2.label,
          values = _ref2.values;

      (_console = console).log.apply(_console, [label + ':'].concat((0, _toConsumableArray3['default'])(values)));
    });
    console.groupEnd && console.groupEnd();
    /* eslint-enable no-console */
  };

  RelayNetworkLoggerTransaction.prototype.commitLogs = function commitLogs(error, payload, status) {
    require('fbjs/lib/invariant')(this._hasCommittedLogs === false, 'The logs for transaction #' + this._id + ' have already been committed.');
    this.printLogs(error, payload, status);
    this.markCommitted();
  };

  RelayNetworkLoggerTransaction.prototype.markCommitted = function markCommitted() {
    this._hasCommittedLogs = true;
  };

  RelayNetworkLoggerTransaction.prototype.flushLogs = function flushLogs(error, payload, status) {
    require('fbjs/lib/invariant')(this._hasCommittedLogs === false, 'The logs for transaction #' + this._id + ' have already been committed.');
    this.printLogs(error, payload, status);
    this.clearLogs();
  };

  RelayNetworkLoggerTransaction.prototype.getCacheConfig = function getCacheConfig() {
    return this._cacheConfig;
  };

  RelayNetworkLoggerTransaction.prototype.getIdentifier = function getIdentifier() {
    return '[' + this._id + '] Relay Modern: ' + this._request.name;
  };

  RelayNetworkLoggerTransaction.prototype.getLogsToPrint = function getLogsToPrint(error, payload, status) {
    return this._logs;
  };

  RelayNetworkLoggerTransaction.prototype.getRequest = function getRequest() {
    return this._request;
  };

  RelayNetworkLoggerTransaction.prototype.getUploadables = function getUploadables() {
    return this._uploadables;
  };

  RelayNetworkLoggerTransaction.prototype.getVariables = function getVariables() {
    return this._variables;
  };

  return RelayNetworkLoggerTransaction;
}();

module.exports = RelayNetworkLoggerTransaction;