'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _fsExtra = require('fs-extra');

var _fsExtra2 = _interopRequireDefault(_fsExtra);

var _path = require('path');

var _path2 = _interopRequireDefault(_path);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

// The maximum number of marks within a single DurationSet. A DurationSet will be automatically finished if this many
// marks are recorded.
// Measure elapsed durations from specific beginning points.

const MAXIMUM_MARKS = 100;

// Flush all non-active DurationSets to disk each time that this many marks have been accumulated.
const PERSIST_INTERVAL = 1000;

// A sequence of durations measured from a fixed beginning point.
class DurationSet {
  constructor(name) {
    this.name = name;
    this.startTimestamp = performance.now();
    this.marks = [];
    this.markCount = 0;

    if (atom.config.get('github.performanceToConsole')) {
      // eslint-disable-next-line no-console
      console.log('%cbegin %c%s:begin', 'font-weight: bold', 'font-weight: normal; font-style: italic; color: blue', this.name);
    }

    if (atom.config.get('github.performanceToProfile')) {
      // eslint-disable-next-line no-console
      console.profile(this.name);
    }
  }

  mark(eventName) {
    const duration = performance.now() - this.startTimestamp;

    if (atom.config.get('github.performanceToConsole')) {
      // eslint-disable-next-line no-console
      console.log('%cmark %c%s:%s %c%dms', 'font-weight: bold', 'font-weight: normal; font-style: italic; color: blue', this.name, eventName, 'font-style: normal; color: black', duration);
    }

    if (atom.config.get('github.performanceToDirectory') !== '') {
      this.marks.push({ eventName, duration });
    }

    this.markCount++;
    if (this.markCount >= MAXIMUM_MARKS) {
      this.finish();
    }
  }

  finish() {
    this.mark('finish');

    if (atom.config.get('github.performanceToProfile')) {
      // eslint-disable-next-line no-console
      console.profileEnd(this.name);
    }
  }

  serialize() {
    return {
      name: this.name,
      markers: this.marks
    };
  }

  getCount() {
    return this.marks.length;
  }
}

let durationSets = [];
let totalMarkCount = 0;
const activeSets = new Map();

function shouldCapture(seriesName, eventName) {
  const anyActive = ['Console', 'Directory', 'Profile'].some(kind => {
    const value = atom.config.get(`github.performanceTo${kind}`);
    return value !== '' && value !== false;
  });
  if (!anyActive) {
    return false;
  }

  const mask = new RegExp(atom.config.get('github.performanceMask'));
  if (!mask.test(`${seriesName}:${eventName}`)) {
    return false;
  }

  return true;
}

const yardstick = {
  async save() {
    const destDir = atom.config.get('github.performanceToDirectory');
    if (destDir === '' || destDir === undefined || destDir === null) {
      return;
    }
    const fileName = _path2.default.join(destDir, `performance-${Date.now()}.json`);

    await new Promise((resolve, reject) => {
      _fsExtra2.default.ensureDir(destDir, err => err ? reject(err) : resolve());
    });

    const payload = JSON.stringify(durationSets.map(set => set.serialize()));
    await _fsExtra2.default.writeFile(fileName, payload, { encoding: 'utf8' });

    if (atom.config.get('github.performanceToConsole')) {
      // eslint-disable-next-line no-console
      console.log('%csaved %c%d series to %s', 'font-weight: bold', 'font-weight: normal; color: black', durationSets.length, fileName);
    }

    durationSets = [];
  },

  begin(seriesName) {
    if (!shouldCapture(seriesName, 'begin')) {
      return;
    }

    const ds = new DurationSet(seriesName);
    activeSets.set(seriesName, ds);
  },

  mark(seriesName, eventName) {
    if (seriesName instanceof Array) {
      for (let i = 0; i < seriesName.length; i++) {
        this.mark(seriesName[i], eventName);
      }
      return;
    }

    if (!shouldCapture(seriesName, eventName)) {
      return;
    }

    const ds = activeSets.get(seriesName);
    if (ds === undefined) {
      return;
    }
    ds.mark(eventName);
  },

  finish(seriesName) {
    if (seriesName instanceof Array) {
      for (let i = 0; i < seriesName.length; i++) {
        this.finish(seriesName[i]);
      }
      return;
    }

    if (!shouldCapture(seriesName, 'finish')) {
      return;
    }

    const ds = activeSets.get(seriesName);
    if (ds === undefined) {
      return;
    }
    ds.finish();

    durationSets.push(ds);
    activeSets.delete(seriesName);

    totalMarkCount += ds.getCount();
    if (totalMarkCount >= PERSIST_INTERVAL) {
      totalMarkCount = 0;
      this.save();
    }
  },

  async flush() {
    durationSets.push(...activeSets.values());
    activeSets.clear();
    await this.save();
  }
};

exports.default = yardstick;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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