'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _extends = Object.assign || function (target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i]; for (var key in source) { if (Object.prototype.hasOwnProperty.call(source, key)) { target[key] = source[key]; } } } return target; };

var _eventKit = require('event-kit');

var _electron = require('electron');

var _atom = require('atom');

var _react = require('react');

var _react2 = _interopRequireDefault(_react);

var _propTypes = require('prop-types');

var _propTypes2 = _interopRequireDefault(_propTypes);

var _path = require('path');

var _path2 = _interopRequireDefault(_path);

var _propTypes3 = require('../prop-types');

var _filePatchListItemView = require('./file-patch-list-item-view');

var _filePatchListItemView2 = _interopRequireDefault(_filePatchListItemView);

var _observeModel = require('./observe-model');

var _observeModel2 = _interopRequireDefault(_observeModel);

var _mergeConflictListItemView = require('./merge-conflict-list-item-view');

var _mergeConflictListItemView2 = _interopRequireDefault(_mergeConflictListItemView);

var _compositeListSelection = require('../models/composite-list-selection');

var _compositeListSelection2 = _interopRequireDefault(_compositeListSelection);

var _resolutionProgress = require('../models/conflicts/resolution-progress');

var _resolutionProgress2 = _interopRequireDefault(_resolutionProgress);

var _refHolder = require('../models/ref-holder');

var _refHolder2 = _interopRequireDefault(_refHolder);

var _changedFileItem = require('../items/changed-file-item');

var _changedFileItem2 = _interopRequireDefault(_changedFileItem);

var _commands = require('../atom/commands');

var _commands2 = _interopRequireDefault(_commands);

var _helpers = require('../helpers');

var _reporterProxy = require('../reporter-proxy');

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

const { Menu, MenuItem } = _electron.remote;


const debounce = (fn, wait) => {
  let timeout;
  return (...args) => {
    return new Promise(resolve => {
      clearTimeout(timeout);
      timeout = setTimeout(() => {
        resolve(fn(...args));
      }, wait);
    });
  };
};

function calculateTruncatedLists(lists) {
  return Object.keys(lists).reduce((acc, key) => {
    const list = lists[key];
    acc.source[key] = list;
    if (list.length <= MAXIMUM_LISTED_ENTRIES) {
      acc[key] = list;
    } else {
      acc[key] = list.slice(0, MAXIMUM_LISTED_ENTRIES);
    }
    return acc;
  }, { source: {} });
}

const noop = () => {};

const MAXIMUM_LISTED_ENTRIES = 1000;

class StagingView extends _react2.default.Component {

  constructor(props) {
    super(props);

    this.undoLastDiscardFromCoreUndo = () => {
      this.undoLastDiscard({ eventSource: { command: 'core:undo' } });
    };

    this.undoLastDiscardFromCommand = () => {
      this.undoLastDiscard({ eventSource: { command: 'github:undo-last-discard-in-git-tab' } });
    };

    this.undoLastDiscardFromButton = () => {
      this.undoLastDiscard({ eventSource: 'button' });
    };

    this.undoLastDiscardFromHeaderMenu = () => {
      this.undoLastDiscard({ eventSource: 'header-menu' });
    };

    this.discardChangesFromCommand = () => {
      this.discardChanges({ eventSource: { command: 'github:discard-changes-in-selected-files' } });
    };

    this.discardAllFromCommand = () => {
      this.discardAll({ eventSource: { command: 'github:discard-all-changes' } });
    };

    this.confirmSelectedItems = async () => {
      const itemPaths = this.getSelectedItemFilePaths();
      await this.props.attemptFileStageOperation(itemPaths, this.state.selection.getActiveListKey());
      await new Promise(resolve => {
        this.setState(prevState => ({ selection: prevState.selection.coalesce() }), resolve);
      });
    };

    (0, _helpers.autobind)(this, 'dblclickOnItem', 'contextMenuOnItem', 'mousedownOnItem', 'mousemoveOnItem', 'mouseup', 'registerItemElement', 'renderBody', 'openFile', 'discardChanges', 'activateNextList', 'activatePreviousList', 'activateLastList', 'stageAll', 'unstageAll', 'stageAllMergeConflicts', 'discardAll', 'confirmSelectedItems', 'selectAll', 'selectFirst', 'selectLast', 'diveIntoSelection', 'showDiffView', 'showBulkResolveMenu', 'showActionsMenu', 'resolveCurrentAsOurs', 'resolveCurrentAsTheirs', 'quietlySelectItem', 'didChangeSelectedItems');

    this.subs = new _eventKit.CompositeDisposable(atom.config.observe('github.keyboardNavigationDelay', value => {
      if (value === 0) {
        this.debouncedDidChangeSelectedItem = this.didChangeSelectedItems;
      } else {
        this.debouncedDidChangeSelectedItem = debounce(this.didChangeSelectedItems, value);
      }
    }));

    this.state = _extends({}, calculateTruncatedLists({
      unstagedChanges: this.props.unstagedChanges,
      stagedChanges: this.props.stagedChanges,
      mergeConflicts: this.props.mergeConflicts
    }), {
      selection: new _compositeListSelection2.default({
        listsByKey: [['unstaged', this.props.unstagedChanges], ['conflicts', this.props.mergeConflicts], ['staged', this.props.stagedChanges]],
        idForItem: item => item.filePath
      })
    });

    this.mouseSelectionInProgress = false;
    this.listElementsByItem = new WeakMap();
    this.refRoot = new _refHolder2.default();
  }

  static getDerivedStateFromProps(nextProps, prevState) {
    let nextState = {};

    if (['unstagedChanges', 'stagedChanges', 'mergeConflicts'].some(key => prevState.source[key] !== nextProps[key])) {
      const nextLists = calculateTruncatedLists({
        unstagedChanges: nextProps.unstagedChanges,
        stagedChanges: nextProps.stagedChanges,
        mergeConflicts: nextProps.mergeConflicts
      });

      nextState = _extends({}, nextLists, {
        selection: prevState.selection.updateLists([['unstaged', nextLists.unstagedChanges], ['conflicts', nextLists.mergeConflicts], ['staged', nextLists.stagedChanges]])
      });
    }

    return nextState;
  }

  componentDidMount() {
    window.addEventListener('mouseup', this.mouseup);
    this.subs.add(new _eventKit.Disposable(() => window.removeEventListener('mouseup', this.mouseup)), this.props.workspace.onDidChangeActivePaneItem(() => {
      this.syncWithWorkspace();
    }));

    if (this.isPopulated(this.props)) {
      this.syncWithWorkspace();
    }
  }

  componentDidUpdate(prevProps, prevState) {
    const isRepoSame = prevProps.workingDirectoryPath === this.props.workingDirectoryPath;
    const hasSelectionsPresent = prevState.selection.getSelectedItems().size > 0 && this.state.selection.getSelectedItems().size > 0;
    const selectionChanged = this.state.selection !== prevState.selection;

    if (isRepoSame && hasSelectionsPresent && selectionChanged) {
      this.debouncedDidChangeSelectedItem();
    }

    const headItem = this.state.selection.getHeadItem();
    if (headItem) {
      const element = this.listElementsByItem.get(headItem);
      if (element) {
        element.scrollIntoViewIfNeeded();
      }
    }

    if (!this.isPopulated(prevProps) && this.isPopulated(this.props)) {
      this.syncWithWorkspace();
    }
  }

  render() {
    return _react2.default.createElement(
      _observeModel2.default,
      { model: this.props.resolutionProgress, fetchData: noop },
      this.renderBody
    );
  }

  renderBody() {
    const selectedItems = this.state.selection.getSelectedItems();

    return _react2.default.createElement(
      'div',
      {
        ref: this.refRoot.setter,
        className: `github-StagingView ${this.state.selection.getActiveListKey()}-changes-focused`,
        tabIndex: '-1' },
      this.renderCommands(),
      _react2.default.createElement(
        'div',
        { className: `github-StagingView-group github-UnstagedChanges ${this.getFocusClass('unstaged')}` },
        _react2.default.createElement(
          'header',
          { className: 'github-StagingView-header' },
          _react2.default.createElement('span', { className: 'icon icon-list-unordered' }),
          _react2.default.createElement(
            'span',
            { className: 'github-StagingView-title' },
            'Unstaged Changes'
          ),
          this.renderActionsMenu(),
          _react2.default.createElement(
            'button',
            {
              className: 'github-StagingView-headerButton icon icon-move-down',
              disabled: this.props.unstagedChanges.length === 0,
              onClick: this.stageAll },
            'Stage All'
          )
        ),
        _react2.default.createElement(
          'div',
          { className: 'github-StagingView-list github-FilePatchListView github-StagingView-unstaged' },
          this.state.unstagedChanges.map(filePatch => _react2.default.createElement(_filePatchListItemView2.default, {
            key: filePatch.filePath,
            registerItemElement: this.registerItemElement,
            filePatch: filePatch,
            onDoubleClick: event => this.dblclickOnItem(event, filePatch),
            onContextMenu: event => this.contextMenuOnItem(event, filePatch),
            onMouseDown: event => this.mousedownOnItem(event, filePatch),
            onMouseMove: event => this.mousemoveOnItem(event, filePatch),
            selected: selectedItems.has(filePatch)
          }))
        ),
        this.renderTruncatedMessage(this.props.unstagedChanges)
      ),
      this.renderMergeConflicts(),
      _react2.default.createElement(
        'div',
        { className: `github-StagingView-group github-StagedChanges ${this.getFocusClass('staged')}` },
        _react2.default.createElement(
          'header',
          { className: 'github-StagingView-header' },
          _react2.default.createElement('span', { className: 'icon icon-tasklist' }),
          _react2.default.createElement(
            'span',
            { className: 'github-StagingView-title' },
            'Staged Changes'
          ),
          _react2.default.createElement(
            'button',
            { className: 'github-StagingView-headerButton icon icon-move-up',
              disabled: this.props.stagedChanges.length === 0,
              onClick: this.unstageAll },
            'Unstage All'
          )
        ),
        _react2.default.createElement(
          'div',
          { className: 'github-StagingView-list github-FilePatchListView github-StagingView-staged' },
          this.state.stagedChanges.map(filePatch => _react2.default.createElement(_filePatchListItemView2.default, {
            key: filePatch.filePath,
            filePatch: filePatch,
            registerItemElement: this.registerItemElement,
            onDoubleClick: event => this.dblclickOnItem(event, filePatch),
            onContextMenu: event => this.contextMenuOnItem(event, filePatch),
            onMouseDown: event => this.mousedownOnItem(event, filePatch),
            onMouseMove: event => this.mousemoveOnItem(event, filePatch),
            selected: selectedItems.has(filePatch)
          }))
        ),
        this.renderTruncatedMessage(this.props.stagedChanges)
      )
    );
  }

  renderCommands() {
    return _react2.default.createElement(
      _react.Fragment,
      null,
      _react2.default.createElement(
        _commands2.default,
        { registry: this.props.commandRegistry, target: '.github-StagingView' },
        _react2.default.createElement(_commands.Command, { command: 'core:move-up', callback: () => this.selectPrevious() }),
        _react2.default.createElement(_commands.Command, { command: 'core:move-down', callback: () => this.selectNext() }),
        _react2.default.createElement(_commands.Command, { command: 'core:move-left', callback: this.diveIntoSelection }),
        _react2.default.createElement(_commands.Command, { command: 'github:show-diff-view', callback: this.showDiffView }),
        _react2.default.createElement(_commands.Command, { command: 'core:select-up', callback: () => this.selectPrevious(true) }),
        _react2.default.createElement(_commands.Command, { command: 'core:select-down', callback: () => this.selectNext(true) }),
        _react2.default.createElement(_commands.Command, { command: 'core:select-all', callback: this.selectAll }),
        _react2.default.createElement(_commands.Command, { command: 'core:move-to-top', callback: this.selectFirst }),
        _react2.default.createElement(_commands.Command, { command: 'core:move-to-bottom', callback: this.selectLast }),
        _react2.default.createElement(_commands.Command, { command: 'core:select-to-top', callback: () => this.selectFirst(true) }),
        _react2.default.createElement(_commands.Command, { command: 'core:select-to-bottom', callback: () => this.selectLast(true) }),
        _react2.default.createElement(_commands.Command, { command: 'core:confirm', callback: this.confirmSelectedItems }),
        _react2.default.createElement(_commands.Command, { command: 'github:activate-next-list', callback: this.activateNextList }),
        _react2.default.createElement(_commands.Command, { command: 'github:activate-previous-list', callback: this.activatePreviousList }),
        _react2.default.createElement(_commands.Command, { command: 'github:jump-to-file', callback: this.openFile }),
        _react2.default.createElement(_commands.Command, { command: 'github:resolve-file-as-ours', callback: this.resolveCurrentAsOurs }),
        _react2.default.createElement(_commands.Command, { command: 'github:resolve-file-as-theirs', callback: this.resolveCurrentAsTheirs }),
        _react2.default.createElement(_commands.Command, { command: 'github:discard-changes-in-selected-files', callback: this.discardChangesFromCommand }),
        _react2.default.createElement(_commands.Command, { command: 'core:undo', callback: this.undoLastDiscardFromCoreUndo })
      ),
      _react2.default.createElement(
        _commands2.default,
        { registry: this.props.commandRegistry, target: 'atom-workspace' },
        _react2.default.createElement(_commands.Command, { command: 'github:stage-all-changes', callback: this.stageAll }),
        _react2.default.createElement(_commands.Command, { command: 'github:unstage-all-changes', callback: this.unstageAll }),
        _react2.default.createElement(_commands.Command, { command: 'github:discard-all-changes', callback: this.discardAllFromCommand }),
        _react2.default.createElement(_commands.Command, { command: 'github:undo-last-discard-in-git-tab',
          callback: this.undoLastDiscardFromCommand
        })
      )
    );
  }

  renderActionsMenu() {
    if (this.props.unstagedChanges.length || this.props.hasUndoHistory) {
      return _react2.default.createElement('button', {
        className: 'github-StagingView-headerButton github-StagingView-headerButton--iconOnly icon icon-ellipses',
        onClick: this.showActionsMenu
      });
    } else {
      return null;
    }
  }

  renderUndoButton() {
    return _react2.default.createElement(
      'button',
      { className: 'github-StagingView-headerButton github-StagingView-headerButton--fullWidth icon icon-history',
        onClick: this.undoLastDiscardFromButton },
      'Undo Discard'
    );
  }

  renderTruncatedMessage(list) {
    if (list.length > MAXIMUM_LISTED_ENTRIES) {
      return _react2.default.createElement(
        'div',
        { className: 'github-StagingView-group-truncatedMsg' },
        'List truncated to the first ',
        MAXIMUM_LISTED_ENTRIES,
        ' items'
      );
    } else {
      return null;
    }
  }

  renderMergeConflicts() {
    const mergeConflicts = this.state.mergeConflicts;

    if (mergeConflicts && mergeConflicts.length > 0) {
      const selectedItems = this.state.selection.getSelectedItems();
      const resolutionProgress = this.props.resolutionProgress;
      const anyUnresolved = mergeConflicts.map(conflict => _path2.default.join(this.props.workingDirectoryPath, conflict.filePath)).some(conflictPath => resolutionProgress.getRemaining(conflictPath) !== 0);

      const bulkResolveDropdown = anyUnresolved ? _react2.default.createElement('span', {
        className: 'inline-block icon icon-ellipses',
        onClick: this.showBulkResolveMenu
      }) : null;

      return _react2.default.createElement(
        'div',
        { className: `github-StagingView-group github-MergeConflictPaths ${this.getFocusClass('conflicts')}` },
        _react2.default.createElement(
          'header',
          { className: 'github-StagingView-header' },
          _react2.default.createElement('span', { className: 'github-FilePatchListView-icon icon icon-alert status-modified' }),
          _react2.default.createElement(
            'span',
            { className: 'github-StagingView-title' },
            'Merge Conflicts'
          ),
          bulkResolveDropdown,
          _react2.default.createElement(
            'button',
            {
              className: 'github-StagingView-headerButton icon icon-move-down',
              disabled: anyUnresolved,
              onClick: this.stageAllMergeConflicts },
            'Stage All'
          )
        ),
        _react2.default.createElement(
          'div',
          { className: 'github-StagingView-list github-FilePatchListView github-StagingView-merge' },
          mergeConflicts.map(mergeConflict => {
            const fullPath = _path2.default.join(this.props.workingDirectoryPath, mergeConflict.filePath);

            return _react2.default.createElement(_mergeConflictListItemView2.default, {
              key: fullPath,
              mergeConflict: mergeConflict,
              remainingConflicts: resolutionProgress.getRemaining(fullPath),
              registerItemElement: this.registerItemElement,
              onDoubleClick: event => this.dblclickOnItem(event, mergeConflict),
              onContextMenu: event => this.contextMenuOnItem(event, mergeConflict),
              onMouseDown: event => this.mousedownOnItem(event, mergeConflict),
              onMouseMove: event => this.mousemoveOnItem(event, mergeConflict),
              selected: selectedItems.has(mergeConflict)
            });
          })
        ),
        this.renderTruncatedMessage(mergeConflicts)
      );
    } else {
      return _react2.default.createElement('noscript', null);
    }
  }

  componentWillUnmount() {
    this.subs.dispose();
  }

  getSelectedItemFilePaths() {
    return Array.from(this.state.selection.getSelectedItems(), item => item.filePath);
  }

  getSelectedConflictPaths() {
    if (this.state.selection.getActiveListKey() !== 'conflicts') {
      return [];
    }
    return this.getSelectedItemFilePaths();
  }

  openFile() {
    const filePaths = this.getSelectedItemFilePaths();
    return this.props.openFiles(filePaths);
  }

  discardChanges({ eventSource } = {}) {
    const filePaths = this.getSelectedItemFilePaths();
    (0, _reporterProxy.addEvent)('discard-unstaged-changes', {
      package: 'github',
      component: 'StagingView',
      fileCount: filePaths.length,
      type: 'selected',
      eventSource
    });
    return this.props.discardWorkDirChangesForPaths(filePaths);
  }

  activateNextList() {
    return new Promise(resolve => {
      let advanced = false;

      this.setState(prevState => {
        const next = prevState.selection.activateNextSelection();
        if (prevState.selection === next) {
          return {};
        }

        advanced = true;
        return { selection: next.coalesce() };
      }, () => resolve(advanced));
    });
  }

  activatePreviousList() {
    return new Promise(resolve => {
      let retreated = false;
      this.setState(prevState => {
        const next = prevState.selection.activatePreviousSelection();
        if (prevState.selection === next) {
          return {};
        }

        retreated = true;
        return { selection: next.coalesce() };
      }, () => resolve(retreated));
    });
  }

  activateLastList() {
    return new Promise(resolve => {
      let emptySelection = false;
      this.setState(prevState => {
        const next = prevState.selection.activateLastSelection();
        emptySelection = next.getSelectedItems().size > 0;

        if (prevState.selection === next) {
          return {};
        }

        return { selection: next.coalesce() };
      }, () => resolve(emptySelection));
    });
  }

  stageAll() {
    if (this.props.unstagedChanges.length === 0) {
      return null;
    }
    return this.props.attemptStageAllOperation('unstaged');
  }

  unstageAll() {
    if (this.props.stagedChanges.length === 0) {
      return null;
    }
    return this.props.attemptStageAllOperation('staged');
  }

  stageAllMergeConflicts() {
    if (this.props.mergeConflicts.length === 0) {
      return null;
    }
    const filePaths = this.props.mergeConflicts.map(conflict => conflict.filePath);
    return this.props.attemptFileStageOperation(filePaths, 'unstaged');
  }

  discardAll({ eventSource } = {}) {
    if (this.props.unstagedChanges.length === 0) {
      return null;
    }
    const filePaths = this.props.unstagedChanges.map(filePatch => filePatch.filePath);
    (0, _reporterProxy.addEvent)('discard-unstaged-changes', {
      package: 'github',
      component: 'StagingView',
      fileCount: filePaths.length,
      type: 'all',
      eventSource
    });
    return this.props.discardWorkDirChangesForPaths(filePaths);
  }

  getNextListUpdatePromise() {
    return this.state.selection.getNextUpdatePromise();
  }

  selectPrevious(preserveTail = false) {
    return new Promise(resolve => {
      this.setState(prevState => ({
        selection: prevState.selection.selectPreviousItem(preserveTail).coalesce()
      }), resolve);
    });
  }

  selectNext(preserveTail = false) {
    return new Promise(resolve => {
      this.setState(prevState => ({
        selection: prevState.selection.selectNextItem(preserveTail).coalesce()
      }), resolve);
    });
  }

  selectAll() {
    return new Promise(resolve => {
      this.setState(prevState => ({
        selection: prevState.selection.selectAllItems().coalesce()
      }), resolve);
    });
  }

  selectFirst(preserveTail = false) {
    return new Promise(resolve => {
      this.setState(prevState => ({
        selection: prevState.selection.selectFirstItem(preserveTail).coalesce()
      }), resolve);
    });
  }

  selectLast(preserveTail = false) {
    return new Promise(resolve => {
      this.setState(prevState => ({
        selection: prevState.selection.selectLastItem(preserveTail).coalesce()
      }), resolve);
    });
  }

  async diveIntoSelection() {
    const selectedItems = this.state.selection.getSelectedItems();
    if (selectedItems.size !== 1) {
      return;
    }

    const selectedItem = selectedItems.values().next().value;
    const stagingStatus = this.state.selection.getActiveListKey();

    if (stagingStatus === 'conflicts') {
      this.showMergeConflictFileForPath(selectedItem.filePath, { activate: true });
    } else {
      await this.showFilePatchItem(selectedItem.filePath, this.state.selection.getActiveListKey(), { activate: true });
    }
  }

  async syncWithWorkspace() {
    const item = this.props.workspace.getActivePaneItem();
    if (!item) {
      return;
    }

    const realItemPromise = item.getRealItemPromise && item.getRealItemPromise();
    const realItem = await realItemPromise;
    if (!realItem) {
      return;
    }

    const isFilePatchItem = realItem.isFilePatchItem && realItem.isFilePatchItem();
    const isMatch = realItem.getWorkingDirectory && realItem.getWorkingDirectory() === this.props.workingDirectoryPath;

    if (isFilePatchItem && isMatch) {
      this.quietlySelectItem(realItem.getFilePath(), realItem.getStagingStatus());
    }
  }

  async showDiffView() {
    const selectedItems = this.state.selection.getSelectedItems();
    if (selectedItems.size !== 1) {
      return;
    }

    const selectedItem = selectedItems.values().next().value;
    const stagingStatus = this.state.selection.getActiveListKey();

    if (stagingStatus === 'conflicts') {
      this.showMergeConflictFileForPath(selectedItem.filePath);
    } else {
      await this.showFilePatchItem(selectedItem.filePath, this.state.selection.getActiveListKey());
    }
  }

  showBulkResolveMenu(event) {
    const conflictPaths = this.props.mergeConflicts.map(c => c.filePath);

    event.preventDefault();

    const menu = new Menu();

    menu.append(new MenuItem({
      label: 'Resolve All as Ours',
      click: () => this.props.resolveAsOurs(conflictPaths)
    }));

    menu.append(new MenuItem({
      label: 'Resolve All as Theirs',
      click: () => this.props.resolveAsTheirs(conflictPaths)
    }));

    menu.popup(_electron.remote.getCurrentWindow());
  }

  showActionsMenu(event) {
    event.preventDefault();

    const menu = new Menu();

    const selectedItemCount = this.state.selection.getSelectedItems().size;
    const pluralization = selectedItemCount > 1 ? 's' : '';

    menu.append(new MenuItem({
      label: 'Discard All Changes',
      click: () => this.discardAll({ eventSource: 'header-menu' }),
      enabled: this.props.unstagedChanges.length > 0
    }));

    menu.append(new MenuItem({
      label: 'Discard Changes in Selected File' + pluralization,
      click: () => this.discardChanges({ eventSource: 'header-menu' }),
      enabled: !!(this.props.unstagedChanges.length && selectedItemCount)
    }));

    menu.append(new MenuItem({
      label: 'Undo Last Discard',
      click: () => this.undoLastDiscard({ eventSource: 'header-menu' }),
      enabled: this.props.hasUndoHistory
    }));

    menu.popup(_electron.remote.getCurrentWindow());
  }

  resolveCurrentAsOurs() {
    this.props.resolveAsOurs(this.getSelectedConflictPaths());
  }

  resolveCurrentAsTheirs() {
    this.props.resolveAsTheirs(this.getSelectedConflictPaths());
  }

  // Directly modify the selection to include only the item identified by the file path and stagingStatus tuple.
  // Re-render the component, but don't notify didSelectSingleItem() or other callback functions. This is useful to
  // avoid circular callback loops for actions originating in FilePatchView or TextEditors with merge conflicts.
  quietlySelectItem(filePath, stagingStatus) {
    return new Promise(resolve => {
      this.setState(prevState => {
        const item = prevState.selection.findItem((each, key) => each.filePath === filePath && key === stagingStatus);
        if (!item) {
          // FIXME: make staging view display no selected item
          // eslint-disable-next-line no-console
          console.log(`Unable to find item at path ${filePath} with staging status ${stagingStatus}`);
          return null;
        }

        return { selection: prevState.selection.selectItem(item) };
      }, resolve);
    });
  }

  getSelectedItems() {
    const stagingStatus = this.state.selection.getActiveListKey();
    return Array.from(this.state.selection.getSelectedItems(), item => {
      return {
        filePath: item.filePath,
        stagingStatus
      };
    });
  }

  didChangeSelectedItems(openNew) {
    const selectedItems = Array.from(this.state.selection.getSelectedItems());
    if (selectedItems.length === 1) {
      this.didSelectSingleItem(selectedItems[0], openNew);
    }
  }

  async didSelectSingleItem(selectedItem, openNew = false) {
    if (!this.hasFocus()) {
      return;
    }

    if (this.state.selection.getActiveListKey() === 'conflicts') {
      if (openNew) {
        await this.showMergeConflictFileForPath(selectedItem.filePath, { activate: true });
      }
    } else {
      if (openNew) {
        // User explicitly asked to view diff, such as via click
        await this.showFilePatchItem(selectedItem.filePath, this.state.selection.getActiveListKey(), { activate: false });
      } else {
        const panesWithStaleItemsToUpdate = this.getPanesWithStalePendingFilePatchItem();
        if (panesWithStaleItemsToUpdate.length > 0) {
          // Update stale items to reflect new selection
          await Promise.all(panesWithStaleItemsToUpdate.map(async pane => {
            await this.showFilePatchItem(selectedItem.filePath, this.state.selection.getActiveListKey(), {
              activate: false,
              pane
            });
          }));
        } else {
          // Selection was changed via keyboard navigation, update pending item in active pane
          const activePane = this.props.workspace.getCenter().getActivePane();
          const activePendingItem = activePane.getPendingItem();
          const activePaneHasPendingFilePatchItem = activePendingItem && activePendingItem.getRealItem && activePendingItem.getRealItem() instanceof _changedFileItem2.default;
          if (activePaneHasPendingFilePatchItem) {
            await this.showFilePatchItem(selectedItem.filePath, this.state.selection.getActiveListKey(), {
              activate: false,
              pane: activePane
            });
          }
        }
      }
    }
  }

  getPanesWithStalePendingFilePatchItem() {
    // "stale" meaning there is no longer a changed file associated with item
    // due to changes being fully staged/unstaged/stashed/deleted/etc
    return this.props.workspace.getPanes().filter(pane => {
      const pendingItem = pane.getPendingItem();
      if (!pendingItem || !pendingItem.getRealItem) {
        return false;
      }
      const realItem = pendingItem.getRealItem();
      if (!(realItem instanceof _changedFileItem2.default)) {
        return false;
      }
      // We only want to update pending diff views for currently active repo
      const isInActiveRepo = realItem.getWorkingDirectory() === this.props.workingDirectoryPath;
      const isStale = !this.changedFileExists(realItem.getFilePath(), realItem.getStagingStatus());
      return isInActiveRepo && isStale;
    });
  }

  changedFileExists(filePath, stagingStatus) {
    return this.state.selection.findItem((item, key) => {
      return key === stagingStatus && item.filePath === filePath;
    });
  }

  async showFilePatchItem(filePath, stagingStatus, { activate, pane } = { activate: false }) {
    const uri = _changedFileItem2.default.buildURI(filePath, this.props.workingDirectoryPath, stagingStatus);
    const changedFileItem = await this.props.workspace.open(uri, { pending: true, activatePane: activate, activateItem: activate, pane });
    if (activate) {
      const itemRoot = changedFileItem.getElement();
      const focusRoot = itemRoot.querySelector('[tabIndex]');
      if (focusRoot) {
        focusRoot.focus();
      }
    } else {
      // simply make item visible
      this.props.workspace.paneForItem(changedFileItem).activateItem(changedFileItem);
    }
  }

  async showMergeConflictFileForPath(relativeFilePath, { activate } = { activate: false }) {
    const absolutePath = _path2.default.join(this.props.workingDirectoryPath, relativeFilePath);
    if (await this.fileExists(absolutePath)) {
      return this.props.workspace.open(absolutePath, { activatePane: activate, activateItem: activate, pending: true });
    } else {
      this.props.notificationManager.addInfo('File has been deleted.');
      return null;
    }
  }

  fileExists(absolutePath) {
    return new _atom.File(absolutePath).exists();
  }

  dblclickOnItem(event, item) {
    return this.props.attemptFileStageOperation([item.filePath], this.state.selection.listKeyForItem(item));
  }

  async contextMenuOnItem(event, item) {
    if (!this.state.selection.getSelectedItems().has(item)) {
      event.stopPropagation();

      event.persist();
      await new Promise(resolve => {
        this.setState(prevState => ({
          selection: prevState.selection.selectItem(item, event.shiftKey)
        }), resolve);
      });

      const newEvent = new MouseEvent(event.type, event);
      requestAnimationFrame(() => {
        if (!event.target.parentNode) {
          return;
        }
        event.target.parentNode.dispatchEvent(newEvent);
      });
    }
  }

  async mousedownOnItem(event, item) {
    const windows = process.platform === 'win32';
    if (event.ctrlKey && !windows) {
      return;
    } // simply open context menu
    if (event.button === 0) {
      this.mouseSelectionInProgress = true;

      event.persist();
      await new Promise(resolve => {
        if (event.metaKey || event.ctrlKey && windows) {
          this.setState(prevState => ({
            selection: prevState.selection.addOrSubtractSelection(item)
          }), resolve);
        } else {
          this.setState(prevState => ({
            selection: prevState.selection.selectItem(item, event.shiftKey)
          }), resolve);
        }
      });
    }
  }

  async mousemoveOnItem(event, item) {
    if (this.mouseSelectionInProgress) {
      await new Promise(resolve => {
        this.setState(prevState => ({
          selection: prevState.selection.selectItem(item, true)
        }), resolve);
      });
    }
  }

  async mouseup() {
    const hadSelectionInProgress = this.mouseSelectionInProgress;
    this.mouseSelectionInProgress = false;

    await new Promise(resolve => {
      this.setState(prevState => ({
        selection: prevState.selection.coalesce()
      }), resolve);
    });
    if (hadSelectionInProgress) {
      this.didChangeSelectedItems(true);
    }
  }

  undoLastDiscard({ eventSource } = {}) {
    if (!this.props.hasUndoHistory) {
      return;
    }

    (0, _reporterProxy.addEvent)('undo-last-discard', {
      package: 'github',
      component: 'StagingView',
      eventSource
    });

    this.props.undoLastDiscard();
  }

  getFocusClass(listKey) {
    return this.state.selection.getActiveListKey() === listKey ? 'is-focused' : '';
  }

  registerItemElement(item, element) {
    this.listElementsByItem.set(item, element);
  }

  rememberFocus(event) {
    return this.refRoot.map(root => root.contains(event.target)).getOr(false) ? StagingView.focus.STAGING : null;
  }

  setFocus(focus) {
    if (focus === StagingView.focus.STAGING) {
      this.refRoot.map(root => root.focus());
      return true;
    }

    return false;
  }

  hasFocus() {
    return this.refRoot.map(root => root.contains(document.activeElement)).getOr(false);
  }

  isPopulated(props) {
    return props.workingDirectoryPath != null && (props.unstagedChanges.length > 0 || props.mergeConflicts.length > 0 || props.stagedChanges.length > 0);
  }
}
exports.default = StagingView;
StagingView.propTypes = {
  unstagedChanges: _propTypes2.default.arrayOf(_propTypes3.FilePatchItemPropType).isRequired,
  stagedChanges: _propTypes2.default.arrayOf(_propTypes3.FilePatchItemPropType).isRequired,
  mergeConflicts: _propTypes2.default.arrayOf(_propTypes3.MergeConflictItemPropType),
  workingDirectoryPath: _propTypes2.default.string,
  resolutionProgress: _propTypes2.default.object,
  hasUndoHistory: _propTypes2.default.bool.isRequired,
  commandRegistry: _propTypes2.default.object.isRequired,
  notificationManager: _propTypes2.default.object.isRequired,
  workspace: _propTypes2.default.object.isRequired,
  openFiles: _propTypes2.default.func.isRequired,
  attemptFileStageOperation: _propTypes2.default.func.isRequired,
  discardWorkDirChangesForPaths: _propTypes2.default.func.isRequired,
  undoLastDiscard: _propTypes2.default.func.isRequired,
  attemptStageAllOperation: _propTypes2.default.func.isRequired,
  resolveAsOurs: _propTypes2.default.func.isRequired,
  resolveAsTheirs: _propTypes2.default.func.isRequired
};
StagingView.defaultProps = {
  mergeConflicts: [],
  resolutionProgress: new _resolutionProgress2.default()
};
StagingView.focus = {
  STAGING: Symbol('staging')
};
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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