'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _extends = Object.assign || function (target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i]; for (var key in source) { if (Object.prototype.hasOwnProperty.call(source, key)) { target[key] = source[key]; } } } return target; };

var _react = require('react');

var _react2 = _interopRequireDefault(_react);

var _propTypes = require('prop-types');

var _propTypes2 = _interopRequireDefault(_propTypes);

var _classnames = require('classnames');

var _classnames2 = _interopRequireDefault(_classnames);

var _atom = require('atom');

var _eventKit = require('event-kit');

var _helpers = require('../helpers');

var _propTypes3 = require('../prop-types');

var _atomTextEditor = require('../atom/atom-text-editor');

var _atomTextEditor2 = _interopRequireDefault(_atomTextEditor);

var _marker = require('../atom/marker');

var _marker2 = _interopRequireDefault(_marker);

var _markerLayer = require('../atom/marker-layer');

var _markerLayer2 = _interopRequireDefault(_markerLayer);

var _decoration = require('../atom/decoration');

var _decoration2 = _interopRequireDefault(_decoration);

var _gutter = require('../atom/gutter');

var _gutter2 = _interopRequireDefault(_gutter);

var _commands = require('../atom/commands');

var _commands2 = _interopRequireDefault(_commands);

var _filePatchHeaderView = require('./file-patch-header-view');

var _filePatchHeaderView2 = _interopRequireDefault(_filePatchHeaderView);

var _filePatchMetaView = require('./file-patch-meta-view');

var _filePatchMetaView2 = _interopRequireDefault(_filePatchMetaView);

var _hunkHeaderView = require('./hunk-header-view');

var _hunkHeaderView2 = _interopRequireDefault(_hunkHeaderView);

var _refHolder = require('../models/ref-holder');

var _refHolder2 = _interopRequireDefault(_refHolder);

var _changedFileItem = require('../items/changed-file-item');

var _changedFileItem2 = _interopRequireDefault(_changedFileItem);

var _commitPreviewItem = require('../items/commit-preview-item');

var _commitPreviewItem2 = _interopRequireDefault(_commitPreviewItem);

var _file = require('../models/patch/file');

var _file2 = _interopRequireDefault(_file);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

const executableText = {
  [_file2.default.modes.NORMAL]: 'non executable',
  [_file2.default.modes.EXECUTABLE]: 'executable'
};

const NBSP_CHARACTER = '\u00a0';

const BLANK_LABEL = () => NBSP_CHARACTER;

class MultiFilePatchView extends _react2.default.Component {

  constructor(props) {
    super(props);

    this.renderFilePatchDecorations = filePatch => {
      return _react2.default.createElement(
        _react.Fragment,
        { key: filePatch.getPath() },
        _react2.default.createElement(
          _marker2.default,
          { invalidate: 'never', bufferRange: filePatch.getStartRange() },
          _react2.default.createElement(
            _decoration2.default,
            { type: 'block', className: 'github-FilePatchView-controlBlock' },
            _react2.default.createElement(_filePatchHeaderView2.default, {
              itemType: this.props.itemType,
              relPath: filePatch.getPath(),
              stagingStatus: this.props.stagingStatus,
              isPartiallyStaged: this.props.isPartiallyStaged,
              hasHunks: filePatch.getHunks().length > 0,
              hasUndoHistory: this.props.hasUndoHistory,
              hasMultipleFileSelections: this.props.hasMultipleFileSelections,

              tooltips: this.props.tooltips,

              undoLastDiscard: () => this.undoLastDiscardFromButton(filePatch),
              diveIntoMirrorPatch: () => this.props.diveIntoMirrorPatch(filePatch),
              openFile: () => this.didOpenFile({ selectedFilePatch: filePatch }),
              toggleFile: () => this.props.toggleFile(filePatch)
            }),
            this.renderSymlinkChangeMeta(filePatch),
            this.renderExecutableModeChangeMeta(filePatch)
          )
        ),
        this.renderHunkHeaders(filePatch)
      );
    };

    this.undoLastDiscardFromCoreUndo = () => {
      if (this.props.hasUndoHistory) {
        const selectedFilePatches = Array.from(this.getSelectedFilePatches());
        if (this.props.itemType === _changedFileItem2.default) {
          this.props.undoLastDiscard(selectedFilePatches[0], { eventSource: { command: 'core:undo' } });
        }
      }
    };

    this.undoLastDiscardFromButton = filePatch => {
      this.props.undoLastDiscard(filePatch, { eventSource: 'button' });
    };

    this.discardSelectionFromCommand = () => {
      return this.props.discardRows(this.props.selectedRows, this.props.selectionMode, { eventSource: { command: 'github:discard-selected-lines' } });
    };

    this.didToggleModeChange = () => {
      return Promise.all(Array.from(this.getSelectedFilePatches()).filter(fp => fp.didChangeExecutableMode()).map(this.props.toggleModeChange));
    };

    this.didToggleSymlinkChange = () => {
      return Promise.all(Array.from(this.getSelectedFilePatches()).filter(fp => fp.hasTypechange()).map(this.props.toggleSymlinkChange));
    };

    (0, _helpers.autobind)(this, 'didMouseDownOnHeader', 'didMouseDownOnLineNumber', 'didMouseMoveOnLineNumber', 'didMouseUp', 'didConfirm', 'didToggleSelectionMode', 'selectNextHunk', 'selectPreviousHunk', 'didOpenFile', 'didAddSelection', 'didChangeSelectionRange', 'didDestroySelection', 'oldLineNumberLabel', 'newLineNumberLabel');

    this.mouseSelectionInProgress = false;
    this.lastMouseMoveLine = null;
    this.nextSelectionMode = null;
    this.suppressChanges = false;
    this.refRoot = new _refHolder2.default();
    this.refEditor = new _refHolder2.default();
    this.refEditorElement = new _refHolder2.default();

    this.subs = new _eventKit.CompositeDisposable();

    this.subs.add(this.refEditor.observe(editor => {
      this.refEditorElement.setter(editor.getElement());
    }), this.refEditorElement.observe(element => {
      this.props.refInitialFocus && this.props.refInitialFocus.setter(element);
    }));
  }

  componentDidMount() {
    window.addEventListener('mouseup', this.didMouseUp);
    this.refEditor.map(editor => {
      // this.props.multiFilePatch is guaranteed to contain at least one FilePatch if <AtomTextEditor> is rendered.
      const [firstPatch] = this.props.multiFilePatch.getFilePatches();
      const [firstHunk] = firstPatch.getHunks();
      if (!firstHunk) {
        return null;
      }

      this.nextSelectionMode = 'hunk';
      editor.setSelectedBufferRange(firstHunk.getRange());
      return null;
    });

    this.subs.add(this.props.config.onDidChange('github.showDiffIconGutter', ({ newValue }) => this.forceUpdate()));
  }

  getSnapshotBeforeUpdate(prevProps) {
    let newSelectionRange = null;
    if (this.props.multiFilePatch !== prevProps.multiFilePatch) {
      // Heuristically adjust the editor selection based on the old file patch, the old row selection state, and
      // the incoming patch.
      newSelectionRange = this.props.multiFilePatch.getNextSelectionRange(prevProps.multiFilePatch, prevProps.selectedRows);

      this.suppressChanges = true;
      this.props.multiFilePatch.adoptBufferFrom(prevProps.multiFilePatch);
      this.suppressChanges = false;
    }
    return newSelectionRange;
  }

  componentDidUpdate(prevProps, prevState, newSelectionRange) {
    if (prevProps.refInitialFocus !== this.props.refInitialFocus) {
      prevProps.refInitialFocus && prevProps.refInitialFocus.setter(null);
      this.props.refInitialFocus && this.refEditorElement.map(this.props.refInitialFocus.setter);
    }

    if (newSelectionRange) {
      this.refEditor.map(editor => {
        if (this.props.selectionMode === 'line') {
          this.nextSelectionMode = 'line';
          editor.setSelectedBufferRange(newSelectionRange);
        } else {
          const nextHunks = new Set(_atom.Range.fromObject(newSelectionRange).getRows().map(row => this.props.multiFilePatch.getHunkAt(row)).filter(Boolean));
          const nextRanges = nextHunks.size > 0 ? Array.from(nextHunks, hunk => hunk.getRange()) : [[[0, 0], [0, 0]]];

          this.nextSelectionMode = 'hunk';
          editor.setSelectedBufferRanges(nextRanges);
        }

        return null;
      });
    } else {
      this.nextSelectionMode = null;
    }
  }

  componentWillUnmount() {
    window.removeEventListener('mouseup', this.didMouseUp);
    this.subs.dispose();
  }

  render() {
    const rootClass = (0, _classnames2.default)('github-FilePatchView', `github-FilePatchView--${this.props.stagingStatus}`, { 'github-FilePatchView--blank': !this.props.multiFilePatch.anyPresent() }, { 'github-FilePatchView--hunkMode': this.props.selectionMode === 'hunk' });

    return _react2.default.createElement(
      'div',
      { className: rootClass, ref: this.refRoot.setter },
      this.renderCommands(),
      _react2.default.createElement(
        'main',
        { className: 'github-FilePatchView-container' },
        this.props.multiFilePatch.anyPresent() ? this.renderNonEmptyPatch() : this.renderEmptyPatch()
      )
    );
  }

  renderCommands() {
    let stageModeCommand = null;
    let stageSymlinkCommand = null;

    if (this.props.multiFilePatch.didAnyChangeExecutableMode()) {
      const command = this.props.stagingStatus === 'unstaged' ? 'github:stage-file-mode-change' : 'github:unstage-file-mode-change';
      stageModeCommand = _react2.default.createElement(_commands.Command, { command: command, callback: this.didToggleModeChange });
    }

    if (this.props.multiFilePatch.anyHaveTypechange()) {
      const command = this.props.stagingStatus === 'unstaged' ? 'github:stage-symlink-change' : 'github:unstage-symlink-change';
      stageSymlinkCommand = _react2.default.createElement(_commands.Command, { command: command, callback: this.didToggleSymlinkChange });
    }

    return _react2.default.createElement(
      _commands2.default,
      { registry: this.props.commands, target: this.refRoot },
      _react2.default.createElement(_commands.Command, { command: 'core:confirm', callback: this.didConfirm }),
      _react2.default.createElement(_commands.Command, { command: 'core:undo', callback: this.undoLastDiscardFromCoreUndo }),
      _react2.default.createElement(_commands.Command, { command: 'github:discard-selected-lines', callback: this.discardSelectionFromCommand }),
      _react2.default.createElement(_commands.Command, { command: 'github:select-next-hunk', callback: this.selectNextHunk }),
      _react2.default.createElement(_commands.Command, { command: 'github:select-previous-hunk', callback: this.selectPreviousHunk }),
      _react2.default.createElement(_commands.Command, { command: 'github:jump-to-file', callback: this.didOpenFile }),
      _react2.default.createElement(_commands.Command, { command: 'github:surface', callback: this.props.surface }),
      _react2.default.createElement(_commands.Command, { command: 'github:toggle-patch-selection-mode', callback: this.didToggleSelectionMode }),
      stageModeCommand,
      stageSymlinkCommand
    );
  }

  renderEmptyPatch() {
    return _react2.default.createElement(
      'p',
      { className: 'github-FilePatchView-message icon icon-info' },
      'No changes to display'
    );
  }

  renderNonEmptyPatch() {
    return _react2.default.createElement(
      _atomTextEditor2.default,
      {
        workspace: this.props.workspace,

        buffer: this.props.multiFilePatch.getBuffer(),
        lineNumberGutterVisible: false,
        autoWidth: false,
        autoHeight: false,
        readOnly: true,
        softWrapped: true,

        didAddSelection: this.didAddSelection,
        didChangeSelectionRange: this.didChangeSelectionRange,
        didDestroySelection: this.didDestroySelection,
        refModel: this.refEditor },
      _react2.default.createElement(_gutter2.default, {
        name: 'old-line-numbers',
        priority: 1,
        className: 'old',
        type: 'line-number',
        labelFn: this.oldLineNumberLabel,
        onMouseDown: this.didMouseDownOnLineNumber,
        onMouseMove: this.didMouseMoveOnLineNumber
      }),
      _react2.default.createElement(_gutter2.default, {
        name: 'new-line-numbers',
        priority: 2,
        className: 'new',
        type: 'line-number',
        labelFn: this.newLineNumberLabel,
        onMouseDown: this.didMouseDownOnLineNumber,
        onMouseMove: this.didMouseMoveOnLineNumber
      }),
      this.props.config.get('github.showDiffIconGutter') && _react2.default.createElement(_gutter2.default, {
        name: 'diff-icons',
        priority: 3,
        type: 'line-number',
        className: 'icons',
        labelFn: BLANK_LABEL,
        onMouseDown: this.didMouseDownOnLineNumber,
        onMouseMove: this.didMouseMoveOnLineNumber
      }),
      this.props.multiFilePatch.getFilePatches().map(this.renderFilePatchDecorations),
      this.renderLineDecorations(Array.from(this.props.selectedRows, row => _atom.Range.fromObject([[row, 0], [row, Infinity]])), 'github-FilePatchView-line--selected', { gutter: true, icon: true, line: true }),
      this.renderDecorationsOnLayer(this.props.multiFilePatch.getAdditionLayer(), 'github-FilePatchView-line--added', { icon: true, line: true }),
      this.renderDecorationsOnLayer(this.props.multiFilePatch.getDeletionLayer(), 'github-FilePatchView-line--deleted', { icon: true, line: true }),
      this.renderDecorationsOnLayer(this.props.multiFilePatch.getNoNewlineLayer(), 'github-FilePatchView-line--nonewline', { icon: true, line: true })
    );
  }

  renderExecutableModeChangeMeta(filePatch) {
    if (!filePatch.didChangeExecutableMode()) {
      return null;
    }

    const oldMode = filePatch.getOldMode();
    const newMode = filePatch.getNewMode();

    const attrs = this.props.stagingStatus === 'unstaged' ? {
      actionIcon: 'icon-move-down',
      actionText: 'Stage Mode Change'
    } : {
      actionIcon: 'icon-move-up',
      actionText: 'Unstage Mode Change'
    };

    return _react2.default.createElement(
      _filePatchMetaView2.default,
      {
        title: 'Mode change',
        actionIcon: attrs.actionIcon,
        actionText: attrs.actionText,
        action: () => this.props.toggleModeChange(filePatch) },
      _react2.default.createElement(
        _react.Fragment,
        null,
        'File changed mode',
        _react2.default.createElement(
          'span',
          { className: 'github-FilePatchView-metaDiff github-FilePatchView-metaDiff--removed' },
          'from ',
          executableText[oldMode],
          ' ',
          _react2.default.createElement(
            'code',
            null,
            oldMode
          )
        ),
        _react2.default.createElement(
          'span',
          { className: 'github-FilePatchView-metaDiff github-FilePatchView-metaDiff--added' },
          'to ',
          executableText[newMode],
          ' ',
          _react2.default.createElement(
            'code',
            null,
            newMode
          )
        )
      )
    );
  }

  renderSymlinkChangeMeta(filePatch) {
    if (!filePatch.hasSymlink()) {
      return null;
    }

    let detail = _react2.default.createElement('div', null);
    let title = '';
    const oldSymlink = filePatch.getOldSymlink();
    const newSymlink = filePatch.getNewSymlink();
    if (oldSymlink && newSymlink) {
      detail = _react2.default.createElement(
        _react.Fragment,
        null,
        'Symlink changed',
        _react2.default.createElement(
          'span',
          { className: (0, _classnames2.default)('github-FilePatchView-metaDiff', 'github-FilePatchView-metaDiff--fullWidth', 'github-FilePatchView-metaDiff--removed') },
          'from ',
          _react2.default.createElement(
            'code',
            null,
            oldSymlink
          )
        ),
        _react2.default.createElement(
          'span',
          { className: (0, _classnames2.default)('github-FilePatchView-metaDiff', 'github-FilePatchView-metaDiff--fullWidth', 'github-FilePatchView-metaDiff--added') },
          'to ',
          _react2.default.createElement(
            'code',
            null,
            newSymlink
          )
        ),
        '.'
      );
      title = 'Symlink changed';
    } else if (oldSymlink && !newSymlink) {
      detail = _react2.default.createElement(
        _react.Fragment,
        null,
        'Symlink',
        _react2.default.createElement(
          'span',
          { className: 'github-FilePatchView-metaDiff github-FilePatchView-metaDiff--removed' },
          'to ',
          _react2.default.createElement(
            'code',
            null,
            oldSymlink
          )
        ),
        'deleted.'
      );
      title = 'Symlink deleted';
    } else {
      detail = _react2.default.createElement(
        _react.Fragment,
        null,
        'Symlink',
        _react2.default.createElement(
          'span',
          { className: 'github-FilePatchView-metaDiff github-FilePatchView-metaDiff--added' },
          'to ',
          _react2.default.createElement(
            'code',
            null,
            newSymlink
          )
        ),
        'created.'
      );
      title = 'Symlink created';
    }

    const attrs = this.props.stagingStatus === 'unstaged' ? {
      actionIcon: 'icon-move-down',
      actionText: 'Stage Symlink Change'
    } : {
      actionIcon: 'icon-move-up',
      actionText: 'Unstage Symlink Change'
    };

    return _react2.default.createElement(
      _filePatchMetaView2.default,
      {
        title: title,
        actionIcon: attrs.actionIcon,
        actionText: attrs.actionText,
        action: () => this.props.toggleSymlinkChange(filePatch) },
      _react2.default.createElement(
        _react.Fragment,
        null,
        detail
      )
    );
  }

  renderHunkHeaders(filePatch) {
    const toggleVerb = this.props.stagingStatus === 'unstaged' ? 'Stage' : 'Unstage';
    const selectedHunks = new Set(Array.from(this.props.selectedRows, row => this.props.multiFilePatch.getHunkAt(row)));

    return _react2.default.createElement(
      _react.Fragment,
      null,
      _react2.default.createElement(
        _markerLayer2.default,
        null,
        filePatch.getHunks().map((hunk, index) => {
          const containsSelection = this.props.selectionMode === 'line' && selectedHunks.has(hunk);
          const isSelected = this.props.selectionMode === 'hunk' && selectedHunks.has(hunk);

          let buttonSuffix = '';
          if (containsSelection) {
            buttonSuffix += 'Selected Line';
            if (this.props.selectedRows.size > 1) {
              buttonSuffix += 's';
            }
          } else {
            buttonSuffix += 'Hunk';
            if (selectedHunks.size > 1) {
              buttonSuffix += 's';
            }
          }

          const toggleSelectionLabel = `${toggleVerb} ${buttonSuffix}`;
          const discardSelectionLabel = `Discard ${buttonSuffix}`;

          const startPoint = hunk.getRange().start;
          const startRange = new _atom.Range(startPoint, startPoint);

          return _react2.default.createElement(
            _marker2.default,
            { key: `hunkHeader-${index}`, bufferRange: startRange, invalidate: 'never' },
            _react2.default.createElement(
              _decoration2.default,
              { type: 'block', className: 'github-FilePatchView-controlBlock' },
              _react2.default.createElement(_hunkHeaderView2.default, {
                refTarget: this.refEditorElement,
                hunk: hunk,
                isSelected: isSelected,
                stagingStatus: this.props.stagingStatus,
                selectionMode: 'line',
                toggleSelectionLabel: toggleSelectionLabel,
                discardSelectionLabel: discardSelectionLabel,

                tooltips: this.props.tooltips,
                keymaps: this.props.keymaps,

                toggleSelection: () => this.toggleHunkSelection(hunk, containsSelection),
                discardSelection: () => this.discardHunkSelection(hunk, containsSelection),
                mouseDown: this.didMouseDownOnHeader
              })
            )
          );
        })
      )
    );
  }

  renderLineDecorations(ranges, lineClass, { line, gutter, icon, refHolder }) {
    if (ranges.length === 0) {
      return null;
    }

    const holder = refHolder || new _refHolder2.default();
    return _react2.default.createElement(
      _markerLayer2.default,
      { handleLayer: holder.setter },
      ranges.map((range, index) => {
        return _react2.default.createElement(_marker2.default, {
          key: `line-${lineClass}-${index}`,
          bufferRange: range,
          invalidate: 'never'
        });
      }),
      this.renderDecorations(lineClass, { line, gutter, icon })
    );
  }

  renderDecorationsOnLayer(layer, lineClass, { line, gutter, icon }) {
    if (layer.getMarkerCount() === 0) {
      return null;
    }

    return _react2.default.createElement(
      _markerLayer2.default,
      { external: layer },
      this.renderDecorations(lineClass, { line, gutter, icon })
    );
  }

  renderDecorations(lineClass, { line, gutter, icon }) {
    return _react2.default.createElement(
      _react.Fragment,
      null,
      line && _react2.default.createElement(_decoration2.default, {
        type: 'line',
        className: lineClass,
        omitEmptyLastRow: false
      }),
      gutter && _react2.default.createElement(
        _react.Fragment,
        null,
        _react2.default.createElement(_decoration2.default, {
          type: 'line-number',
          gutterName: 'old-line-numbers',
          className: lineClass,
          omitEmptyLastRow: false
        }),
        _react2.default.createElement(_decoration2.default, {
          type: 'line-number',
          gutterName: 'new-line-numbers',
          className: lineClass,
          omitEmptyLastRow: false
        })
      ),
      icon && _react2.default.createElement(_decoration2.default, {
        type: 'line-number',
        gutterName: 'diff-icons',
        className: lineClass,
        omitEmptyLastRow: false
      })
    );
  }

  toggleHunkSelection(hunk, containsSelection) {
    if (containsSelection) {
      return this.props.toggleRows(this.props.selectedRows, this.props.selectionMode, { eventSource: 'button' });
    } else {
      const changeRows = new Set(hunk.getChanges().reduce((rows, change) => {
        rows.push(...change.getBufferRows());
        return rows;
      }, []));
      return this.props.toggleRows(changeRows, 'hunk', { eventSource: 'button' });
    }
  }

  discardHunkSelection(hunk, containsSelection) {
    if (containsSelection) {
      return this.props.discardRows(this.props.selectedRows, this.props.selectionMode, { eventSource: 'button' });
    } else {
      const changeRows = new Set(hunk.getChanges().reduce((rows, change) => {
        rows.push(...change.getBufferRows());
        return rows;
      }, []));
      return this.props.discardRows(changeRows, 'hunk', { eventSource: 'button' });
    }
  }

  didMouseDownOnHeader(event, hunk) {
    this.nextSelectionMode = 'hunk';
    this.handleSelectionEvent(event, hunk.getRange());
  }

  didMouseDownOnLineNumber(event) {
    const line = event.bufferRow;
    if (line === undefined || isNaN(line)) {
      return;
    }

    this.nextSelectionMode = 'line';
    if (this.handleSelectionEvent(event.domEvent, [[line, 0], [line, Infinity]])) {
      this.mouseSelectionInProgress = true;
    }
  }

  didMouseMoveOnLineNumber(event) {
    if (!this.mouseSelectionInProgress) {
      return;
    }

    const line = event.bufferRow;
    if (this.lastMouseMoveLine === line || line === undefined || isNaN(line)) {
      return;
    }
    this.lastMouseMoveLine = line;

    this.nextSelectionMode = 'line';
    this.handleSelectionEvent(event.domEvent, [[line, 0], [line, Infinity]], { add: true });
  }

  didMouseUp() {
    this.mouseSelectionInProgress = false;
  }

  handleSelectionEvent(event, rangeLike, opts) {
    if (event.button !== 0) {
      return false;
    }

    const isWindows = process.platform === 'win32';
    if (event.ctrlKey && !isWindows) {
      // Allow the context menu to open.
      return false;
    }

    const options = _extends({
      add: false
    }, opts);

    // Normalize the target selection range
    const converted = _atom.Range.fromObject(rangeLike);
    const range = this.refEditor.map(editor => editor.clipBufferRange(converted)).getOr(converted);

    if (event.metaKey || /* istanbul ignore next */event.ctrlKey && isWindows) {
      this.refEditor.map(editor => {
        let intersects = false;
        let without = null;

        for (const selection of editor.getSelections()) {
          if (selection.intersectsBufferRange(range)) {
            // Remove range from this selection by truncating it to the "near edge" of the range and creating a
            // new selection from the "far edge" to the previous end. Omit either side if it is empty.
            intersects = true;
            const selectionRange = selection.getBufferRange();

            const newRanges = [];

            if (!range.start.isEqual(selectionRange.start)) {
              // Include the bit from the selection's previous start to the range's start.
              let nudged = range.start;
              if (range.start.column === 0) {
                const lastColumn = editor.getBuffer().lineLengthForRow(range.start.row - 1);
                nudged = [range.start.row - 1, lastColumn];
              }

              newRanges.push([selectionRange.start, nudged]);
            }

            if (!range.end.isEqual(selectionRange.end)) {
              // Include the bit from the range's end to the selection's end.
              let nudged = range.end;
              const lastColumn = editor.getBuffer().lineLengthForRow(range.end.row);
              if (range.end.column === lastColumn) {
                nudged = [range.end.row + 1, 0];
              }

              newRanges.push([nudged, selectionRange.end]);
            }

            if (newRanges.length > 0) {
              selection.setBufferRange(newRanges[0]);
              for (const newRange of newRanges.slice(1)) {
                editor.addSelectionForBufferRange(newRange, { reversed: selection.isReversed() });
              }
            } else {
              without = selection;
            }
          }
        }

        if (without !== null) {
          const replacementRanges = editor.getSelections().filter(each => each !== without).map(each => each.getBufferRange());
          if (replacementRanges.length > 0) {
            editor.setSelectedBufferRanges(replacementRanges);
          }
        }

        if (!intersects) {
          // Add this range as a new, distinct selection.
          editor.addSelectionForBufferRange(range);
        }

        return null;
      });
    } else if (options.add || event.shiftKey) {
      // Extend the existing selection to encompass this range.
      this.refEditor.map(editor => {
        const lastSelection = editor.getLastSelection();
        const lastSelectionRange = lastSelection.getBufferRange();

        // You are now entering the wall of ternery operators. This is your last exit before the tollbooth
        const isBefore = range.start.isLessThan(lastSelectionRange.start);
        const farEdge = isBefore ? range.start : range.end;
        const newRange = isBefore ? [farEdge, lastSelectionRange.end] : [lastSelectionRange.start, farEdge];

        lastSelection.setBufferRange(newRange, { reversed: isBefore });
        return null;
      });
    } else {
      this.refEditor.map(editor => editor.setSelectedBufferRange(range));
    }

    return true;
  }

  didConfirm() {
    return this.props.toggleRows(this.props.selectedRows, this.props.selectionMode);
  }

  didToggleSelectionMode() {
    const selectedHunks = this.getSelectedHunks();
    this.withSelectionMode({
      line: () => {
        const hunkRanges = selectedHunks.map(hunk => hunk.getRange());
        this.nextSelectionMode = 'hunk';
        this.refEditor.map(editor => editor.setSelectedBufferRanges(hunkRanges));
      },
      hunk: () => {
        let firstChangeRow = Infinity;
        for (const hunk of selectedHunks) {
          const [firstChange] = hunk.getChanges();
          if (firstChange && (!firstChangeRow || firstChange.getStartBufferRow() < firstChangeRow)) {
            firstChangeRow = firstChange.getStartBufferRow();
          }
        }

        this.nextSelectionMode = 'line';
        this.refEditor.map(editor => {
          editor.setSelectedBufferRanges([[[firstChangeRow, 0], [firstChangeRow, Infinity]]]);
          return null;
        });
      }
    });
  }

  selectNextHunk() {
    this.refEditor.map(editor => {
      const nextHunks = new Set(this.withSelectedHunks(hunk => this.getHunkAfter(hunk) || hunk));
      const nextRanges = Array.from(nextHunks, hunk => hunk.getRange());
      this.nextSelectionMode = 'hunk';
      editor.setSelectedBufferRanges(nextRanges);
      return null;
    });
  }

  selectPreviousHunk() {
    this.refEditor.map(editor => {
      const nextHunks = new Set(this.withSelectedHunks(hunk => this.getHunkBefore(hunk) || hunk));
      const nextRanges = Array.from(nextHunks, hunk => hunk.getRange());
      this.nextSelectionMode = 'hunk';
      editor.setSelectedBufferRanges(nextRanges);
      return null;
    });
  }

  didOpenFile({ selectedFilePatch } = {}) {
    const cursorsByFilePatch = new Map();

    this.refEditor.map(editor => {
      const placedRows = new Set();

      for (const cursor of editor.getCursors()) {
        const cursorRow = cursor.getBufferPosition().row;
        const hunk = this.props.multiFilePatch.getHunkAt(cursorRow);
        const filePatch = this.props.multiFilePatch.getFilePatchAt(cursorRow);
        /* istanbul ignore next */
        if (!hunk) {
          continue;
        }

        let newRow = hunk.getNewRowAt(cursorRow);
        let newColumn = cursor.getBufferPosition().column;
        if (newRow === null) {
          let nearestRow = hunk.getNewStartRow();
          for (const region of hunk.getRegions()) {
            if (!region.includesBufferRow(cursorRow)) {
              region.when({
                unchanged: () => {
                  nearestRow += region.bufferRowCount();
                },
                addition: () => {
                  nearestRow += region.bufferRowCount();
                }
              });
            } else {
              break;
            }
          }

          if (!placedRows.has(nearestRow)) {
            newRow = nearestRow;
            newColumn = 0;
            placedRows.add(nearestRow);
          }
        }

        if (newRow !== null) {
          // Why is this needed? I _think_ everything is in terms of buffer position
          // so there shouldn't be an off-by-one issue
          newRow -= 1;
          const cursors = cursorsByFilePatch.get(filePatch);
          if (!cursors) {
            cursorsByFilePatch.set(filePatch, [[newRow, newColumn]]);
          } else {
            cursors.push([newRow, newColumn]);
          }
        }
      }

      return null;
    });

    const filePatchesWithCursors = new Set(cursorsByFilePatch.keys());
    if (selectedFilePatch && !filePatchesWithCursors.has(selectedFilePatch)) {
      const [firstHunk] = selectedFilePatch.getHunks();
      const cursorRow = firstHunk ? firstHunk.getNewStartRow() - 1 : 0;
      return this.props.openFile(selectedFilePatch, [[cursorRow, 0]], true);
    } else {
      const pending = cursorsByFilePatch.size === 1;
      return Promise.all(Array.from(cursorsByFilePatch, value => {
        const [filePatch, cursors] = value;
        return this.props.openFile(filePatch, cursors, pending);
      }));
    }
  }

  getSelectedRows() {
    return this.refEditor.map(editor => {
      return new Set(editor.getSelections().map(selection => selection.getBufferRange()).reduce((acc, range) => {
        for (const row of range.getRows()) {
          if (this.isChangeRow(row)) {
            acc.push(row);
          }
        }
        return acc;
      }, []));
    }).getOr(new Set());
  }

  didAddSelection() {
    this.didChangeSelectedRows();
  }

  didChangeSelectionRange(event) {
    if (!event || event.oldBufferRange.start.row !== event.newBufferRange.start.row || event.oldBufferRange.end.row !== event.newBufferRange.end.row) {
      this.didChangeSelectedRows();
    }
  }

  didDestroySelection() {
    this.didChangeSelectedRows();
  }

  didChangeSelectedRows() {
    if (this.suppressChanges) {
      return;
    }

    const nextCursorRows = this.refEditor.map(editor => {
      return editor.getCursorBufferPositions().map(position => position.row);
    }).getOr([]);
    const hasMultipleFileSelections = this.props.multiFilePatch.spansMultipleFiles(nextCursorRows);

    this.props.selectedRowsChanged(this.getSelectedRows(), this.nextSelectionMode || 'line', hasMultipleFileSelections);
  }

  oldLineNumberLabel({ bufferRow, softWrapped }) {
    const hunk = this.props.multiFilePatch.getHunkAt(bufferRow);
    if (hunk === undefined) {
      return this.pad('');
    }

    const oldRow = hunk.getOldRowAt(bufferRow);
    if (softWrapped) {
      return this.pad(oldRow === null ? '' : '•');
    }

    return this.pad(oldRow);
  }

  newLineNumberLabel({ bufferRow, softWrapped }) {
    const hunk = this.props.multiFilePatch.getHunkAt(bufferRow);
    if (hunk === undefined) {
      return this.pad('');
    }

    const newRow = hunk.getNewRowAt(bufferRow);
    if (softWrapped) {
      return this.pad(newRow === null ? '' : '•');
    }
    return this.pad(newRow);
  }

  /*
   * Return a Set of the Hunks that include at least one editor selection. The selection need not contain an actual
   * change row.
   */
  getSelectedHunks() {
    return this.withSelectedHunks(each => each);
  }

  withSelectedHunks(callback) {
    return this.refEditor.map(editor => {
      const seen = new Set();
      return editor.getSelectedBufferRanges().reduce((acc, range) => {
        for (const row of range.getRows()) {
          const hunk = this.props.multiFilePatch.getHunkAt(row);
          if (!hunk || seen.has(hunk)) {
            continue;
          }

          seen.add(hunk);
          acc.push(callback(hunk));
        }
        return acc;
      }, []);
    }).getOr([]);
  }

  /*
   * Return a Set of FilePatches that include at least one editor selection. The selection need not contain an actual
   * change row.
   */
  getSelectedFilePatches() {
    return this.refEditor.map(editor => {
      const patches = new Set();
      for (const range of editor.getSelectedBufferRanges()) {
        for (const row of range.getRows()) {
          const patch = this.props.multiFilePatch.getFilePatchAt(row);
          patches.add(patch);
        }
      }
      return patches;
    }).getOr(new Set());
  }

  getHunkBefore(hunk) {
    const prevRow = hunk.getRange().start.row - 1;
    return this.props.multiFilePatch.getHunkAt(prevRow);
  }

  getHunkAfter(hunk) {
    const nextRow = hunk.getRange().end.row + 1;
    return this.props.multiFilePatch.getHunkAt(nextRow);
  }

  isChangeRow(bufferRow) {
    const changeLayers = [this.props.multiFilePatch.getAdditionLayer(), this.props.multiFilePatch.getDeletionLayer()];
    return changeLayers.some(layer => layer.findMarkers({ intersectsRow: bufferRow }).length > 0);
  }

  withSelectionMode(callbacks) {
    const callback = callbacks[this.props.selectionMode];
    /* istanbul ignore if */
    if (!callback) {
      throw new Error(`Unknown selection mode: ${this.props.selectionMode}`);
    }
    return callback();
  }

  pad(num) {
    const maxDigits = this.props.multiFilePatch.getMaxLineNumberWidth();
    if (num === null) {
      return NBSP_CHARACTER.repeat(maxDigits);
    } else {
      return NBSP_CHARACTER.repeat(maxDigits - num.toString().length) + num.toString();
    }
  }

}
exports.default = MultiFilePatchView;
MultiFilePatchView.propTypes = {
  stagingStatus: _propTypes2.default.oneOf(['staged', 'unstaged']).isRequired,
  isPartiallyStaged: _propTypes2.default.bool,
  multiFilePatch: _propTypes3.MultiFilePatchPropType.isRequired,
  selectionMode: _propTypes2.default.oneOf(['hunk', 'line']).isRequired,
  selectedRows: _propTypes2.default.object.isRequired,
  hasMultipleFileSelections: _propTypes2.default.bool.isRequired,
  repository: _propTypes2.default.object.isRequired,
  hasUndoHistory: _propTypes2.default.bool,

  workspace: _propTypes2.default.object.isRequired,
  commands: _propTypes2.default.object.isRequired,
  keymaps: _propTypes2.default.object.isRequired,
  tooltips: _propTypes2.default.object.isRequired,
  config: _propTypes2.default.object.isRequired,

  selectedRowsChanged: _propTypes2.default.func.isRequired,

  diveIntoMirrorPatch: _propTypes2.default.func.isRequired,
  surface: _propTypes2.default.func.isRequired,
  openFile: _propTypes2.default.func.isRequired,
  toggleFile: _propTypes2.default.func.isRequired,
  toggleRows: _propTypes2.default.func.isRequired,
  toggleModeChange: _propTypes2.default.func.isRequired,
  toggleSymlinkChange: _propTypes2.default.func.isRequired,
  undoLastDiscard: _propTypes2.default.func.isRequired,
  discardRows: _propTypes2.default.func.isRequired,

  refInitialFocus: _propTypes3.RefHolderPropType,
  itemType: _propTypes2.default.oneOf([_changedFileItem2.default, _commitPreviewItem2.default]).isRequired
};
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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