'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _extends = Object.assign || function (target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i]; for (var key in source) { if (Object.prototype.hasOwnProperty.call(source, key)) { target[key] = source[key]; } } } return target; };

var _atom = require('atom');

var _eventKit = require('event-kit');

var _electron = require('electron');

var _react = require('react');

var _react2 = _interopRequireDefault(_react);

var _reactDom = require('react-dom');

var _reactDom2 = _interopRequireDefault(_reactDom);

var _propTypes = require('prop-types');

var _propTypes2 = _interopRequireDefault(_propTypes);

var _lodash = require('lodash.memoize');

var _lodash2 = _interopRequireDefault(_lodash);

var _fsExtra = require('fs-extra');

var _fsExtra2 = _interopRequireDefault(_fsExtra);

var _octicon = require('../atom/octicon');

var _octicon2 = _interopRequireDefault(_octicon);

var _helpers = require('../helpers');

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _objectWithoutProperties(obj, keys) { var target = {}; for (var i in obj) { if (keys.indexOf(i) >= 0) continue; if (!Object.prototype.hasOwnProperty.call(obj, i)) continue; target[i] = obj[i]; } return target; }

const { dialog } = _electron.remote;


const genArray = (0, _lodash2.default)(function genArray(interval, count) {
  const arr = [];
  for (let i = 1; i <= count; i++) {
    arr.push(interval * i);
  }
  return arr;
}, (interval, count) => `${interval}:${count}`);

class Marker {
  static deserialize(data) {
    const marker = new Marker(data.label, () => {});
    marker.end = data.end;
    marker.markers = data.markers;
    return marker;
  }

  constructor(label, didUpdate) {
    this.label = label;
    this.didUpdate = didUpdate;
    this.end = null;
    this.markers = [];
  }

  getStart() {
    return this.markers.length ? this.markers[0].start : null;
  }

  getEnd() {
    return this.end;
  }

  mark(sectionName, start) {
    this.markers.push({ name: sectionName, start: start || performance.now() });
  }

  finalize() {
    this.end = performance.now();
    this.didUpdate();
  }

  getTimings() {
    return this.markers.map((timing, idx, ary) => {
      const next = ary[idx + 1];
      const end = next ? next.start : this.getEnd();
      return _extends({}, timing, { end });
    });
  }

  serialize() {
    return {
      label: this.label,
      end: this.end,
      markers: this.markers.slice()
    };
  }
}

class MarkerTooltip extends _react2.default.Component {

  render() {
    const { marker } = this.props;
    const timings = marker.getTimings();

    return _react2.default.createElement(
      'div',
      { style: { textAlign: 'left', maxWidth: 300, whiteSpace: 'initial' } },
      _react2.default.createElement(
        'strong',
        null,
        _react2.default.createElement(
          'tt',
          null,
          marker.label
        )
      ),
      _react2.default.createElement(
        'ul',
        { style: { paddingLeft: 20, marginTop: 10 } },
        timings.map(({ name, start, end }) => {
          const duration = end - start;
          return _react2.default.createElement(
            'li',
            { key: name },
            name,
            ': ',
            Math.floor(duration * 100) / 100,
            'ms'
          );
        })
      )
    );
  }
}

MarkerTooltip.propTypes = {
  marker: _propTypes2.default.instanceOf(Marker).isRequired
};
const COLORS = {
  queued: 'red',
  prepare: 'cyan',
  nexttick: 'yellow',
  execute: 'green',
  ipc: 'pink'
};
class MarkerSpan extends _react2.default.Component {

  constructor(props) {
    super(props);
    (0, _helpers.autobind)(this, 'handleMouseOver', 'handleMouseOut');
  }

  render() {
    const _props = this.props,
          { marker } = _props,
          others = _objectWithoutProperties(_props, ['marker']);
    const timings = marker.getTimings();
    const totalTime = marker.getEnd() - marker.getStart();
    const percentages = timings.map(({ name, start, end }) => {
      const duration = end - start;
      return { color: COLORS[name], percent: duration / totalTime * 100 };
    });
    return _react2.default.createElement(
      'span',
      _extends({}, others, {
        ref: c => {
          this.element = c;
        },
        onMouseOver: this.handleMouseOver,
        onMouseOut: this.handleMouseOut }),
      percentages.map(({ color, percent }, i) => {
        const style = {
          width: `${percent}%`,
          background: color
        };
        return _react2.default.createElement('span', { className: 'waterfall-marker-section', key: i, style: style });
      })
    );
  }

  handleMouseOver(e) {
    const elem = document.createElement('div');
    _reactDom2.default.render(_react2.default.createElement(MarkerTooltip, { marker: this.props.marker }), elem);
    this.tooltipDisposable = atom.tooltips.add(this.element, {
      item: elem,
      placement: 'auto bottom',
      trigger: 'manual'
    });
  }

  closeTooltip() {
    this.tooltipDisposable && this.tooltipDisposable.dispose();
    this.tooltipDisposable = null;
  }

  handleMouseOut(e) {
    this.closeTooltip();
  }

  componentWillUnmount() {
    this.closeTooltip();
  }
}

MarkerSpan.propTypes = {
  marker: _propTypes2.default.instanceOf(Marker).isRequired
};
class Waterfall extends _react2.default.Component {

  constructor(props, context) {
    super(props, context);
    (0, _helpers.autobind)(this, 'renderMarker');
    this.state = this.getNextState(props);
  }

  componentWillReceiveProps(nextProps) {
    this.setState(this.getNextState(nextProps));
  }

  getNextState(props) {
    const { markers } = props;
    const firstMarker = markers[0];
    const lastMarker = markers[markers.length - 1];

    const startTime = firstMarker.getStart();
    const endTime = lastMarker.getEnd();
    const totalDuration = endTime - startTime;
    let timelineMarkInterval = null;
    if (props.zoomFactor <= 0.15) {
      timelineMarkInterval = 1000;
    } else if (props.zoomFactor <= 0.3) {
      timelineMarkInterval = 500;
    } else if (props.zoomFactor <= 0.6) {
      timelineMarkInterval = 250;
    } else {
      timelineMarkInterval = 100;
    }
    const timelineMarks = genArray(timelineMarkInterval, Math.ceil(totalDuration / timelineMarkInterval));

    return { firstMarker, lastMarker, startTime, endTime, totalDuration, timelineMarks };
  }

  render() {
    return _react2.default.createElement(
      'div',
      { className: 'waterfall-scroller' },
      _react2.default.createElement(
        'div',
        { className: 'waterfall-container' },
        this.renderTimeMarkers(),
        this.renderTimeline(),
        this.props.markers.map(this.renderMarker)
      )
    );
  }

  renderTimeline() {
    return _react2.default.createElement(
      'div',
      { className: 'waterfall-timeline' },
      '\xA0',
      this.state.timelineMarks.map(time => {
        const leftPos = time * this.props.zoomFactor;
        const style = {
          left: leftPos
        };
        return _react2.default.createElement(
          'span',
          { className: 'waterfall-timeline-label', style: style, key: `tl:${time}` },
          time,
          'ms'
        );
      })
    );
  }

  renderTimeMarkers() {
    return _react2.default.createElement(
      'div',
      { className: 'waterfall-time-markers' },
      this.state.timelineMarks.map(time => {
        const leftPos = time * this.props.zoomFactor;
        const style = {
          left: leftPos
        };
        return _react2.default.createElement('span', { className: 'waterfall-time-marker', style: style, key: `tm:${time}` });
      })
    );
  }

  renderMarker(marker, i) {
    if (marker.getStart() === null || marker.getEnd() === null) {
      return _react2.default.createElement('div', { key: i });
    }

    const startOffset = marker.getStart() - this.state.startTime;
    const duration = marker.getEnd() - marker.getStart();
    const markerStyle = {
      left: startOffset * this.props.zoomFactor,
      width: duration * this.props.zoomFactor
    };

    return _react2.default.createElement(
      'div',
      { className: 'waterfall-row', key: i },
      _react2.default.createElement(
        'span',
        {
          className: 'waterfall-row-label',
          style: { paddingLeft: markerStyle.left + markerStyle.width } },
        marker.label
      ),
      _react2.default.createElement(MarkerSpan, { className: 'waterfall-marker', style: markerStyle, marker: marker })
    );
  }
}

Waterfall.propTypes = {
  markers: _propTypes2.default.arrayOf(_propTypes2.default.instanceOf(Marker)).isRequired,
  zoomFactor: _propTypes2.default.number.isRequired
};
class WaterfallWidget extends _react2.default.Component {

  constructor(props, context) {
    super(props, context);
    (0, _helpers.autobind)(this, 'handleZoomFactorChange', 'handleCollapseClick', 'handleExportClick');
    this.state = {
      zoomFactor: 0.3,
      collapsed: false
    };
  }

  render() {
    const { markers } = this.props;
    const firstMarker = markers[0];
    const lastMarker = markers[markers.length - 1];

    const startTime = firstMarker.getStart();
    const endTime = lastMarker.getEnd();
    const duration = endTime - startTime;

    return _react2.default.createElement(
      'div',
      { className: 'waterfall-widget inset-pannel' },
      _react2.default.createElement(
        'div',
        { className: 'waterfall-header' },
        _react2.default.createElement(
          'div',
          { className: 'waterfall-header-text' },
          _react2.default.createElement(
            'span',
            { onClick: this.handleCollapseClick, className: 'collapse-toggle' },
            this.state.collapsed ? '\u25b6' : '\u25bc'
          ),
          this.props.markers.length,
          ' event(s) over ',
          Math.floor(duration),
          'ms'
        ),
        _react2.default.createElement(
          'div',
          { className: 'waterfall-header-controls' },
          _react2.default.createElement(
            'button',
            {
              className: 'waterfall-export-button btn btn-sm',
              onClick: this.handleExportClick },
            'Export'
          ),
          _react2.default.createElement(_octicon2.default, { icon: 'search' }),
          _react2.default.createElement('input', {
            type: 'range',
            className: 'input-range',
            min: 0.1,
            max: 1,
            step: 0.01,
            value: this.state.zoomFactor,
            onChange: this.handleZoomFactorChange
          })
        )
      ),
      this.state.collapsed ? null : _react2.default.createElement(Waterfall, { markers: this.props.markers, zoomFactor: this.state.zoomFactor })
    );
  }

  handleZoomFactorChange(e) {
    this.setState({ zoomFactor: parseFloat(e.target.value) });
  }

  handleCollapseClick(e) {
    this.setState(s => ({ collapsed: !s.collapsed }));
  }

  handleExportClick(e) {
    e.preventDefault();
    const json = JSON.stringify(this.props.markers.map(m => m.serialize()), null, '  ');
    const buffer = new _atom.TextBuffer({ text: json });
    dialog.showSaveDialog({
      defaultPath: 'git-timings.json'
    }, filename => {
      if (!filename) {
        return;
      }
      buffer.saveAs(filename);
    });
  }
}

WaterfallWidget.propTypes = {
  markers: _propTypes2.default.arrayOf(_propTypes2.default.instanceOf(Marker)).isRequired
};
let markers = null;
let groupId = 0;
const groups = [];
let lastMarkerTime = null;
let updateTimer = null;

class GitTimingsView extends _react2.default.Component {

  static buildURI() {
    return this.uriPattern;
  }

  static generateMarker(label) {
    const marker = new Marker(label, () => {
      GitTimingsView.scheduleUpdate();
    });
    const now = performance.now();
    if (!markers || lastMarkerTime && Math.abs(now - lastMarkerTime) >= 5000) {
      groupId++;
      markers = [];
      groups.unshift({ id: groupId, markers });
      if (groups.length > 100) {
        groups.pop();
      }
    }
    lastMarkerTime = now;
    markers.push(marker);
    GitTimingsView.scheduleUpdate();
    return marker;
  }

  static restoreGroup(group) {
    groupId++;
    groups.unshift({ id: groupId, markers: group });
    GitTimingsView.scheduleUpdate(true);
  }

  static scheduleUpdate(immediate = false) {
    if (updateTimer) {
      clearTimeout(updateTimer);
    }

    updateTimer = setTimeout(() => {
      GitTimingsView.emitter.emit('did-update');
    }, immediate ? 0 : 1000);
  }

  static onDidUpdate(callback) {
    return GitTimingsView.emitter.on('did-update', callback);
  }

  constructor(props) {
    super(props);
    (0, _helpers.autobind)(this, 'handleImportClick');
  }

  componentDidMount() {
    this.subscriptions = new _eventKit.CompositeDisposable(GitTimingsView.onDidUpdate(() => this.forceUpdate()));
  }

  componentWillUnmount() {
    this.subscriptions.dispose();
  }

  render() {
    return _react2.default.createElement(
      'div',
      { className: 'github-GitTimingsView' },
      _react2.default.createElement(
        'div',
        { className: 'github-GitTimingsView-header' },
        _react2.default.createElement(
          'button',
          { className: 'import-button btn', onClick: this.handleImportClick },
          'Import'
        )
      ),
      groups.map((group, idx) => _react2.default.createElement(WaterfallWidget, { key: group.id, markers: group.markers }))
    );
  }

  handleImportClick(e) {
    e.preventDefault();
    dialog.showOpenDialog({
      properties: ['openFile']
    }, async filenames => {
      if (!filenames) {
        return;
      }
      const filename = filenames[0];
      try {
        const contents = await _fsExtra2.default.readFile(filename, { encoding: 'utf8' });
        const data = JSON.parse(contents);
        const restoredMarkers = data.map(item => Marker.deserialize(item));
        GitTimingsView.restoreGroup(restoredMarkers);
      } catch (_err) {
        atom.notifications.addError(`Could not import timings from ${filename}`);
      }
    });
  }

  serialize() {
    return {
      deserializer: 'GitTimingsView'
    };
  }

  getURI() {
    return this.constructor.buildURI();
  }

  getTitle() {
    return 'GitHub Package Timings View';
  }
}
exports.default = GitTimingsView;
GitTimingsView.uriPattern = 'atom-github://debug/timings';
GitTimingsView.emitter = new _eventKit.Emitter();
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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