'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _react = require('react');

var _react2 = _interopRequireDefault(_react);

var _propTypes = require('prop-types');

var _propTypes2 = _interopRequireDefault(_propTypes);

var _util = require('util');

var _observeModel = require('./observe-model');

var _observeModel2 = _interopRequireDefault(_observeModel);

var _helpers = require('../helpers');

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

const sortOrders = {
  'by key': (a, b) => a.key.localeCompare(b.key),
  'oldest first': (a, b) => b.age - a.age,
  'newest first': (a, b) => a.age - b.age,
  'most hits': (a, b) => b.hits - a.hits,
  'fewest hits': (a, b) => a.hits - b.hits
};

class GitCacheView extends _react2.default.Component {

  static buildURI() {
    return this.uriPattern;
  }

  constructor(props, context) {
    super(props, context);
    (0, _helpers.autobind)(this, 'fetchRepositoryData', 'fetchCacheData', 'renderCache', 'didSelectItem', 'clearCache');

    this.state = {
      order: 'by key'
    };
  }

  getURI() {
    return 'atom-github://debug/cache';
  }

  getTitle() {
    return 'GitHub Package Cache View';
  }

  serialize() {
    return null;
  }

  fetchRepositoryData(repository) {
    return repository.getCache();
  }

  fetchCacheData(cache) {
    const cached = {};
    const promises = [];
    const now = performance.now();

    for (const [key, value] of cache) {
      cached[key] = {
        hits: value.hits,
        age: now - value.createdAt
      };

      promises.push(value.promise.then(payload => (0, _util.inspect)(payload, { depth: 3, breakLength: 30 }), err => `${err.message}\n${err.stack}`).then(resolved => {
        cached[key].value = resolved;
      }));
    }

    return Promise.all(promises).then(() => cached);
  }

  render() {
    return _react2.default.createElement(
      _observeModel2.default,
      { model: this.props.repository, fetchData: this.fetchRepositoryData },
      cache => _react2.default.createElement(
        _observeModel2.default,
        { model: cache, fetchData: this.fetchCacheData },
        this.renderCache
      )
    );
  }

  renderCache(contents) {
    const rows = Object.keys(contents || {}).map(key => {
      return {
        key,
        age: contents[key].age,
        hits: contents[key].hits,
        content: contents[key].value
      };
    });

    rows.sort(sortOrders[this.state.order]);

    const orders = Object.keys(sortOrders);

    return _react2.default.createElement(
      'div',
      { className: 'github-CacheView' },
      _react2.default.createElement(
        'header',
        null,
        _react2.default.createElement(
          'h1',
          null,
          'Cache contents'
        ),
        _react2.default.createElement(
          'p',
          null,
          _react2.default.createElement(
            'span',
            { className: 'badge' },
            rows.length
          ),
          ' cached items'
        )
      ),
      _react2.default.createElement(
        'main',
        null,
        _react2.default.createElement(
          'p',
          { className: 'github-CacheView-Controls' },
          _react2.default.createElement(
            'span',
            { className: 'github-CacheView-Order' },
            'order',
            _react2.default.createElement(
              'select',
              {
                className: 'input-select',
                onChange: this.didSelectItem,
                value: this.state.order },
              orders.map(order => {
                return _react2.default.createElement(
                  'option',
                  { key: order, value: order },
                  order
                );
              })
            )
          ),
          _react2.default.createElement(
            'span',
            { className: 'github-CacheView-Clear' },
            _react2.default.createElement(
              'button',
              { className: 'btn icon icon-trashcan', onClick: this.clearCache },
              'Clear'
            )
          )
        ),
        _react2.default.createElement(
          'table',
          null,
          _react2.default.createElement(
            'thead',
            null,
            _react2.default.createElement(
              'tr',
              null,
              _react2.default.createElement(
                'td',
                { className: 'github-CacheView-Key' },
                'key'
              ),
              _react2.default.createElement(
                'td',
                { className: 'github-CacheView-Age' },
                'age'
              ),
              _react2.default.createElement(
                'td',
                { className: 'github-CacheView-Hits' },
                'hits'
              ),
              _react2.default.createElement(
                'td',
                { className: 'github-CacheView-Content' },
                'content'
              )
            )
          ),
          _react2.default.createElement(
            'tbody',
            null,
            rows.map(row => _react2.default.createElement(
              'tr',
              { key: row.key, className: 'github-CacheView-Row' },
              _react2.default.createElement(
                'td',
                { className: 'github-CacheView-Key' },
                _react2.default.createElement(
                  'button',
                  { className: 'btn', onClick: () => this.didClickKey(row.key) },
                  row.key
                )
              ),
              _react2.default.createElement(
                'td',
                { className: 'github-CacheView-Age' },
                this.formatAge(row.age)
              ),
              _react2.default.createElement(
                'td',
                { className: 'github-CacheView-Hits' },
                row.hits
              ),
              _react2.default.createElement(
                'td',
                { className: 'github-CacheView-Content' },
                _react2.default.createElement(
                  'code',
                  null,
                  row.content
                )
              )
            ))
          )
        )
      )
    );
  }

  formatAge(ageMs) {
    let remaining = ageMs;
    const parts = [];

    if (remaining > 3600000) {
      const hours = Math.floor(remaining / 3600000);
      parts.push(`${hours}h`);
      remaining -= 3600000 * hours;
    }

    if (remaining > 60000) {
      const minutes = Math.floor(remaining / 60000);
      parts.push(`${minutes}m`);
      remaining -= 60000 * minutes;
    }

    if (remaining > 1000) {
      const seconds = Math.floor(remaining / 1000);
      parts.push(`${seconds}s`);
      remaining -= 1000 * seconds;
    }

    parts.push(`${Math.floor(remaining)}ms`);

    return parts.slice(parts.length - 2).join(' ');
  }

  didSelectItem(event) {
    this.setState({ order: event.target.value });
  }

  didClickKey(key) {
    const cache = this.props.repository.getCache();
    if (!cache) {
      return;
    }

    cache.removePrimary(key);
  }

  clearCache() {
    const cache = this.props.repository.getCache();
    if (!cache) {
      return;
    }

    cache.clear();
  }
}
exports.default = GitCacheView;
GitCacheView.uriPattern = 'atom-github://debug/cache';
GitCacheView.propTypes = {
  repository: _propTypes2.default.object.isRequired
};
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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