'use strict';

/* eslint comma-dangle: ["error", {
    "arrays": "never",
    "objects": "never",
    "imports": "never",
    "exports": "never",
    "functions": "never"
  }] */

const { execFile } = require('child_process');
const fs = require('fs');

if (typeof atom === 'undefined') {
  global.atom = {
    inSpecMode() {
      return !!process.env.ATOM_GITHUB_SPEC_MODE;
    },
    inDevMode() {
      return false;
    }
  };
}

// No token available in your OS keychain.
const UNAUTHENTICATED = Symbol('UNAUTHENTICATED');

// The token in your keychain isn't granted all of the required OAuth scopes.
const INSUFFICIENT = Symbol('INSUFFICIENT');

// The token in your keychain is not accepted by GitHub.
const UNAUTHORIZED = Symbol('UNAUTHORIZED');

class KeytarStrategy {
  static get keytar() {
    return require('keytar');
  }

  static async isValid() {
    // Allow for disabling Keytar on problematic CI environments
    if (process.env.ATOM_GITHUB_DISABLE_KEYTAR) {
      return false;
    }

    const keytar = this.keytar;

    try {
      const rand = Math.floor(Math.random() * 10e20).toString(16);
      await keytar.setPassword('atom-test-service', rand, rand);
      const pass = await keytar.getPassword('atom-test-service', rand);
      const success = pass === rand;
      keytar.deletePassword('atom-test-service', rand);
      return success;
    } catch (err) {
      return false;
    }
  }

  async getPassword(service, account) {
    const password = await this.constructor.keytar.getPassword(service, account);
    return password !== null ? password : UNAUTHENTICATED;
  }

  replacePassword(service, account, password) {
    return this.constructor.keytar.setPassword(service, account, password);
  }

  deletePassword(service, account) {
    return this.constructor.keytar.deletePassword(service, account);
  }
}

class SecurityBinaryStrategy {
  static isValid() {
    return process.platform === 'darwin';
  }

  async getPassword(service, account) {
    try {
      const password = await this.exec(['find-generic-password', '-s', service, '-a', account, '-w']);
      return password.trim() || UNAUTHENTICATED;
    } catch (err) {
      return UNAUTHENTICATED;
    }
  }

  replacePassword(service, account, newPassword) {
    return this.exec(['add-generic-password', '-s', service, '-a', account, '-w', newPassword, '-U']);
  }

  deletePassword(service, account) {
    return this.exec(['delete-generic-password', '-s', service, '-a', account]);
  }

  exec(securityArgs, { binary } = { binary: 'security' }) {
    return new Promise((resolve, reject) => {
      execFile(binary, securityArgs, (error, stdout) => {
        if (error) {
          return reject(error);
        }
        return resolve(stdout);
      });
    });
  }
}

class InMemoryStrategy {
  static isValid() {
    return true;
  }

  constructor() {
    if (!atom.inSpecMode()) {
      // eslint-disable-next-line no-console
      console.warn('Using an InMemoryStrategy strategy for storing tokens. ' + 'The tokens will only be stored for the current window.');
    }
    this.passwordsByService = new Map();
  }

  getPassword(service, account) {
    const passwords = this.passwordsByService.get(service) || new Map();
    const password = passwords.get(account);
    return password || UNAUTHENTICATED;
  }

  replacePassword(service, account, newPassword) {
    const passwords = this.passwordsByService.get(service) || new Map();
    passwords.set(account, newPassword);
    this.passwordsByService.set(service, passwords);
  }

  deletePassword(service, account) {
    const passwords = this.passwordsByService.get(service);
    if (passwords) {
      passwords.delete(account);
    }
  }
}

class FileStrategy {
  static isValid() {
    if (!atom.inSpecMode() && !atom.inDevMode()) {
      return false;
    }

    return Boolean(process.env.ATOM_GITHUB_KEYTAR_FILE);
  }

  constructor() {
    this.filePath = process.env.ATOM_GITHUB_KEYTAR_FILE;

    if (!atom.inSpecMode()) {
      // eslint-disable-next-line no-console
      console.warn('Using a FileStrategy strategy for storing tokens. ' + 'The tokens will be stored %cin the clear%c in a file at %s. ' + "You probably shouldn't use real credentials while this strategy is in use. " + 'Unset ATOM_GITHUB_KEYTAR_FILE_STRATEGY to disable it.', 'color: red; font-weight: bold; font-style: italic', 'color: black; font-weight: normal; font-style: normal', this.filePath);
    }
  }

  async getPassword(service, account) {
    const payload = await this.load();
    const forService = payload[service];
    if (forService === undefined) {
      return UNAUTHENTICATED;
    }
    const passwd = forService[account];
    if (passwd === undefined) {
      return UNAUTHENTICATED;
    }
    return passwd;
  }

  replacePassword(service, account, password) {
    return this.modify(payload => {
      let forService = payload[service];
      if (forService === undefined) {
        forService = {};
        payload[service] = forService;
      }
      forService[account] = password;
    });
  }

  deletePassword(service, account) {
    return this.modify(payload => {
      const forService = payload[service];
      if (forService === undefined) {
        return;
      }
      delete forService[account];
      if (Object.keys(forService).length === 0) {
        delete payload[service];
      }
    });
  }

  load() {
    return new Promise((resolve, reject) => {
      fs.readFile(this.filePath, 'utf8', (err, content) => {
        if (err && err.code === 'ENOENT') {
          return resolve({});
        }
        if (err) {
          return reject(err);
        }
        return resolve(JSON.parse(content));
      });
    });
  }

  save(payload) {
    return new Promise((resolve, reject) => {
      fs.writeFile(this.filePath, JSON.stringify(payload), 'utf8', err => {
        if (err) {
          reject(err);
        } else {
          resolve();
        }
      });
    });
  }

  async modify(callback) {
    const payload = await this.load();
    callback(payload);
    await this.save(payload);
  }
}

const strategies = [FileStrategy, KeytarStrategy, SecurityBinaryStrategy, InMemoryStrategy];
let ValidStrategy = null;

async function createStrategy() {
  if (ValidStrategy) {
    return new ValidStrategy();
  }

  for (let i = 0; i < strategies.length; i++) {
    const strat = strategies[i];
    const isValid = await strat.isValid();
    if (isValid) {
      ValidStrategy = strat;
      break;
    }
  }
  if (!ValidStrategy) {
    throw new Error('None of the listed keytar strategies returned true for `isValid`');
  }
  return new ValidStrategy();
}

module.exports = {
  UNAUTHENTICATED,
  INSUFFICIENT,
  UNAUTHORIZED,
  KeytarStrategy,
  SecurityBinaryStrategy,
  InMemoryStrategy,
  FileStrategy,
  createStrategy
};
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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