'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.expectRelayQuery = expectRelayQuery;
exports.clearRelayExpectations = clearRelayExpectations;

var _util = require('util');

var _util2 = _interopRequireDefault(_util);

var _relayRuntime = require('relay-runtime');

var _moment = require('moment');

var _moment2 = _interopRequireDefault(_moment);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

const relayEnvironmentPerGithubHost = new Map();

function logRatelimitApi(headers) {
  const remaining = headers.get('x-ratelimit-remaining');
  const total = headers.get('x-ratelimit-limit');
  const resets = headers.get('x-ratelimit-reset');
  const resetsIn = _moment2.default.unix(parseInt(resets, 10)).from();

  // eslint-disable-next-line no-console
  console.debug(`GitHub API Rate Limit: ${remaining}/${total} — resets ${resetsIn}`);
}

const responsesByQuery = new Map();

function expectRelayQuery(operationPattern, response) {
  let resolve, reject;
  const promise = new Promise((resolve0, reject0) => {
    resolve = () => resolve0({ data: response });
    reject = reject0;
  });

  const existing = responsesByQuery.get(operationPattern.name) || [];
  existing.push({
    promise,
    response,
    variables: operationPattern.variables || {},
    trace: operationPattern.trace
  });
  responsesByQuery.set(operationPattern.name, existing);

  const disable = () => responsesByQuery.delete(operationPattern.name);

  return { promise, resolve, reject, disable };
}

function clearRelayExpectations() {
  responsesByQuery.clear();
}

const tokenPerURL = new Map();
const fetchPerURL = new Map();

function createFetchQuery(url) {
  if (atom.inSpecMode()) {
    return function specFetchQuery(operation, variables, cacheConfig, uploadables) {
      const expectations = responsesByQuery.get(operation.name) || [];
      const match = expectations.find(expectation => {
        if (Object.keys(expectation.variables).length !== Object.keys(variables).length) {
          return false;
        }

        for (const key in expectation.variables) {
          if (expectation.variables[key] !== variables[key]) {
            return false;
          }
        }

        return true;
      });

      if (!match) {
        // eslint-disable-next-line no-console
        console.log(`GraphQL query ${operation.name} was:\n  ${operation.text.replace(/\n/g, '\n  ')}\n` + _util2.default.inspect(variables));

        const e = new Error(`Unexpected GraphQL query: ${operation.name}`);
        e.rawStack = e.stack;
        throw e;
      }

      if (match.trace) {
        match.promise.then(result => {
          // eslint-disable-next-line no-console
          console.log(`GraphQL query ${operation.name} was:\n` + _util2.default.inspect(variables) + '\n' + _util2.default.inspect(result, { depth: null }));
        });
      }

      return match.promise;
    };
  }

  return async function fetchQuery(operation, variables, cacheConfig, uploadables) {
    const currentToken = tokenPerURL.get(url);

    const response = await fetch(url, {
      method: 'POST',
      headers: {
        'content-type': 'application/json',
        'Authorization': `bearer ${currentToken}`,
        'Accept': 'application/vnd.github.graphql-profiling+json'
      },
      body: JSON.stringify({
        query: operation.text,
        variables
      })
    });

    try {
      atom && atom.inDevMode() && logRatelimitApi(response.headers);
    } catch (_e) {/* do nothing */}

    if (response.status !== 200) {
      const e = new Error(`GraphQL API endpoint at ${url} returned ${response.status}`);
      e.response = response;
      e.responseText = await response.text();
      e.rawStack = e.stack;
      throw e;
    }

    const payload = await response.json();

    if (payload.data && payload.data.errors && payload.data.errors.length > 0) {
      const e = new Error(`GraphQL API endpoint at ${url} returned an error for query ${operation.name}.`);
      e.response = response;
      e.errors = payload.data.errors;
      e.rawStack = e.stack;
      throw e;
    }

    return payload;
  };
}

class RelayNetworkLayerManager {
  static getEnvironmentForHost(host, token) {
    host = host === 'github.com' ? 'https://api.github.com' : host;
    const url = host === 'https://api.github.com' ? `${host}/graphql` : `${host}/api/v3/graphql`;
    let { environment, network } = relayEnvironmentPerGithubHost.get(host) || {};
    tokenPerURL.set(url, token);
    if (!environment) {
      const source = new _relayRuntime.RecordSource();
      const store = new _relayRuntime.Store(source);
      network = _relayRuntime.Network.create(this.getFetchQuery(url, token));
      environment = new _relayRuntime.Environment({ network, store });

      relayEnvironmentPerGithubHost.set(host, { environment, network });
    }
    return environment;
  }

  static getFetchQuery(url, token) {
    tokenPerURL.set(url, token);
    let fetch = fetchPerURL.get(url);
    if (!fetch) {
      fetch = createFetchQuery(url);
      fetchPerURL.set(fetch);
    }
    return fetch;
  }
}
exports.default = RelayNetworkLayerManager;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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