'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _extends = Object.assign || function (target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i]; for (var key in source) { if (Object.prototype.hasOwnProperty.call(source, key)) { target[key] = source[key]; } } } return target; };

var _compareSets = require('compare-sets');

var _compareSets2 = _interopRequireDefault(_compareSets);

var _eventKit = require('event-kit');

var _workdirContext = require('./workdir-context');

var _workdirContext2 = _interopRequireDefault(_workdirContext);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/**
 * Manage a WorkdirContext for each open directory.
 */
class WorkdirContextPool {

  /**
   * Options will be passed to each `WorkdirContext` as it is created.
   */
  constructor(options = {}) {
    this.options = options;

    this.contexts = new Map();
    this.emitter = new _eventKit.Emitter();
  }

  size() {
    return this.contexts.size;
  }

  /**
   * Access the context mapped to a known directory.
   */
  getContext(directory) {
    const { pipelineManager } = this.options;
    return this.contexts.get(directory) || _workdirContext2.default.absent({ pipelineManager });
  }

  add(directory, options = {}) {
    if (this.contexts.has(directory)) {
      return this.getContext(directory);
    }

    const context = new _workdirContext2.default(directory, _extends({}, this.options, options));
    this.contexts.set(directory, context);

    const disposable = new _eventKit.CompositeDisposable();

    const forwardEvent = (subMethod, emitEventName) => {
      const emit = () => this.emitter.emit(emitEventName, context);
      disposable.add(context[subMethod](emit));
    };

    forwardEvent('onDidStartObserver', 'did-start-observer');
    forwardEvent('onDidChangeWorkdirOrHead', 'did-change-workdir-or-head');
    forwardEvent('onDidChangeRepositoryState', 'did-change-repository-state');
    forwardEvent('onDidUpdateRepository', 'did-update-repository');
    forwardEvent('onDidDestroyRepository', 'did-destroy-repository');

    disposable.add(this.onDidRemoveContext(removed => {
      if (removed === context) {
        disposable.dispose();
      }
    }));

    return context;
  }

  replace(directory, options = {}) {
    this.remove(directory);
    this.add(directory, options);
  }

  remove(directory) {
    const existing = this.contexts.get(directory);
    this.contexts.delete(directory);

    if (existing) {
      this.emitter.emit('did-remove-context', existing);
      existing.destroy();
    }
  }

  set(directories, options = {}) {
    const previous = new Set(this.contexts.keys());
    const { added, removed } = (0, _compareSets2.default)(previous, directories);

    for (const directory of added) {
      this.add(directory, options);
    }
    for (const directory of removed) {
      this.remove(directory);
    }
  }

  withResidentContexts(callback) {
    const results = [];
    for (const [workdir, context] of this.contexts) {
      results.push(callback(workdir, context));
    }
    return results;
  }

  onDidStartObserver(callback) {
    return this.emitter.on('did-start-observer', callback);
  }

  onDidChangeWorkdirOrHead(callback) {
    return this.emitter.on('did-change-workdir-or-head', callback);
  }

  onDidRemoveContext(callback) {
    return this.emitter.on('did-remove-context', callback);
  }

  onDidChangeRepositoryState(callback) {
    return this.emitter.on('did-change-repository-state', callback);
  }

  onDidUpdateRepository(callback) {
    return this.emitter.on('did-update-repository', callback);
  }

  onDidDestroyRepository(callback) {
    return this.emitter.on('did-destroy-repository', callback);
  }

  clear() {
    this.withResidentContexts(workdir => this.remove(workdir));
    _workdirContext2.default.destroyAbsent();
  }
}
exports.default = WorkdirContextPool;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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