'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _path = require('path');

var _path2 = _interopRequireDefault(_path);

var _fsExtra = require('fs-extra');

var _fsExtra2 = _interopRequireDefault(_fsExtra);

var _compositeGitStrategy = require('../composite-git-strategy');

var _compositeGitStrategy2 = _interopRequireDefault(_compositeGitStrategy);

var _helpers = require('../helpers');

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/**
 * Locate the nearest git working directory above a given starting point, caching results.
 */
class WorkdirCache {
  constructor(maxSize = 1000) {
    this.maxSize = maxSize;
    this.known = new Map();
  }

  async find(startPath) {
    const cached = this.known.get(startPath);
    if (cached !== undefined) {
      return cached;
    }

    const workDir = await this.revParse(startPath);

    if (this.known.size >= this.maxSize) {
      this.known.clear();
    }
    this.known.set(startPath, workDir);

    return workDir;
  }

  invalidate() {
    this.known.clear();
  }

  async revParse(startPath) {
    try {
      const startDir = (await _fsExtra2.default.stat(startPath)).isDirectory() ? startPath : _path2.default.dirname(startPath);

      // Within a git worktree, return a non-empty string containing the path to the worktree root.
      // Within a gitdir or outside of a worktree, return an empty string.
      // Throw if startDir does not exist.
      const topLevel = await _compositeGitStrategy2.default.create(startDir).exec(['rev-parse', '--show-toplevel']);
      if (/\S/.test(topLevel)) {
        return (0, _helpers.toNativePathSep)(topLevel.trim());
      }

      // Within a gitdir, return the absolute path to the gitdir.
      // Outside of a gitdir or worktree, throw.
      const gitDir = await _compositeGitStrategy2.default.create(startDir).exec(['rev-parse', '--absolute-git-dir']);
      return this.revParse(_path2.default.resolve(gitDir, '..'));
    } catch (e) {
      if (atom.config.get('github.reportCannotLocateWorkspaceError')) {
        // eslint-disable-next-line no-console
        console.error(`Unable to locate git workspace root for ${startPath}. Expected if ${startPath} is not in a git repository.`, e);
      }
      return null;
    }
  }
}
exports.default = WorkdirCache;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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