'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.source = undefined;

var _yubikiri = require('yubikiri');

var _yubikiri2 = _interopRequireDefault(_yubikiri);

var _eventKit = require('event-kit');

var _relayNetworkLayerManager = require('../relay-network-layer-manager');

var _relayNetworkLayerManager2 = _interopRequireDefault(_relayNetworkLayerManager);

var _author = require('./author');

var _author2 = _interopRequireDefault(_author);

var _keytarStrategy = require('../shared/keytar-strategy');

var _modelObserver = require('./model-observer');

var _modelObserver2 = _interopRequireDefault(_modelObserver);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

// This is a guess about what a reasonable value is. Can adjust if performance is poor.
const MAX_COMMITS = 5000;

const source = exports.source = {
  PENDING: Symbol('pending'),
  GITLOG: Symbol('git log'),
  GITHUBAPI: Symbol('github API')
};

class GraphQLCache {

  constructor() {
    this.bySlug = new Map();
  }
  // One hour


  get(remote) {
    const slug = remote.getSlug();
    const { ts, data } = this.bySlug.get(slug) || {
      ts: -Infinity,
      data: {}
    };

    if (Date.now() - ts > this.constructor.MAX_AGE_MS) {
      this.bySlug.delete(slug);
      return null;
    }
    return data;
  }

  set(remote, data) {
    this.bySlug.set(remote.getSlug(), { ts: Date.now(), data });
  }
}

GraphQLCache.MAX_AGE_MS = 3.6e6;
class UserStore {
  constructor({ repository, login, config }) {
    this.emitter = new _eventKit.Emitter();
    this.subs = new _eventKit.CompositeDisposable();

    // TODO: [ku 3/2018] Consider using Dexie (indexDB wrapper) like Desktop and persist users across sessions
    this.allUsers = new Map();
    this.excludedUsers = new Set();
    this.users = [];
    this.committer = _author.nullAuthor;

    this.last = {
      source: source.PENDING,
      repository: null,
      excludedUsers: this.excludedUsers
    };
    this.cache = new GraphQLCache();

    this.repositoryObserver = new _modelObserver2.default({
      fetchData: r => (0, _yubikiri2.default)({
        committer: r.getCommitter(),
        authors: r.getAuthors({ max: MAX_COMMITS }),
        remotes: r.getRemotes()
      }),
      didUpdate: () => this.loadUsers()
    });
    this.repositoryObserver.setActiveModel(repository);

    this.loginObserver = new _modelObserver2.default({
      didUpdate: () => this.loadUsers()
    });
    this.loginObserver.setActiveModel(login);

    this.subs.add(config.observe('github.excludedUsers', value => {
      this.excludedUsers = new Set((value || '').split(/\s*,\s*/).filter(each => each.length > 0));
      return this.loadUsers();
    }));
  }

  dispose() {
    this.subs.dispose();
    this.emitter.dispose();
  }

  async loadUsers() {
    const data = this.repositoryObserver.getActiveModelData();

    if (!data) {
      return;
    }

    this.setCommitter(data.committer);
    const githubRemotes = Array.from(data.remotes).filter(remote => remote.isGithubRepo());

    if (githubRemotes.length === 0) {
      this.addUsers(data.authors, source.GITLOG);
    } else {
      await this.loadUsersFromGraphQL(githubRemotes);
    }
  }

  loadUsersFromGraphQL(remotes) {
    return Promise.all(Array.from(remotes, remote => this.loadMentionableUsers(remote)));
  }

  async loadMentionableUsers(remote) {
    const cached = this.cache.get(remote);
    if (cached !== null) {
      this.addUsers(cached, source.GITHUBAPI);
      return;
    }

    const loginModel = this.loginObserver.getActiveModel();
    if (!loginModel) {
      return;
    }

    const token = await loginModel.getToken('https://api.github.com');
    if (token === _keytarStrategy.UNAUTHENTICATED || token === _keytarStrategy.INSUFFICIENT) {
      return;
    }

    const fetchQuery = _relayNetworkLayerManager2.default.getFetchQuery('https://api.github.com/graphql', token);

    let hasMore = true;
    let cursor = null;
    const remoteUsers = [];

    while (hasMore) {
      const response = await fetchQuery({
        name: 'GetMentionableUsers',
        text: `
          query GetMentionableUsers($owner: String!, $name: String!, $first: Int!, $after: String) {
            repository(owner: $owner, name: $name) {
              mentionableUsers(first: $first, after: $after) {
                nodes {
                  login
                  email
                  name
                }
                pageInfo {
                  hasNextPage
                  endCursor
                }
              }
            }
          }
        `
      }, {
        owner: remote.getOwner(),
        name: remote.getRepo(),
        first: 100,
        after: cursor
      });

      if (response.errors && response.errors.length > 1) {
        // eslint-disable-next-line no-console
        console.error(`Error fetching mentionable users:\n${response.errors.map(e => e.message).join('\n')}`);
      }

      if (!response.data || !response.data.repository) {
        break;
      }

      const connection = response.data.repository.mentionableUsers;
      const authors = connection.nodes.map(node => {
        if (node.email === '') {
          node.email = `${node.login}@users.noreply.github.com`;
        }

        return new _author2.default(node.email, node.name, node.login);
      });
      this.addUsers(authors, source.GITHUBAPI);
      remoteUsers.push(...authors);

      cursor = connection.pageInfo.endCursor;
      hasMore = connection.pageInfo.hasNextPage;
    }

    this.cache.set(remote, remoteUsers);
  }

  addUsers(users, nextSource) {
    let changed = false;

    if (nextSource !== this.last.source || this.repositoryObserver.getActiveModel() !== this.last.repository || this.excludedUsers !== this.last.excludedUsers) {
      changed = true;
      this.allUsers.clear();
    }

    for (const author of users) {
      if (!this.allUsers.has(author.getEmail())) {
        changed = true;
      }
      this.allUsers.set(author.getEmail(), author);
    }

    if (changed) {
      this.finalize();
    }
    this.last.source = nextSource;
    this.last.repository = this.repositoryObserver.getActiveModel();
    this.last.excludedUsers = this.excludedUsers;
  }

  finalize() {
    // TODO: [ku 3/2018] consider sorting based on most recent authors or commit frequency
    const users = [];
    for (const author of this.allUsers.values()) {
      if (author.matches(this.committer)) {
        continue;
      }
      if (author.isNoReply()) {
        continue;
      }
      if (this.excludedUsers.has(author.getEmail())) {
        continue;
      }

      users.push(author);
    }
    users.sort(_author2.default.compare);
    this.users = users;
    this.didUpdate();
  }

  setRepository(repository) {
    this.repositoryObserver.setActiveModel(repository);
  }

  setLoginModel(login) {
    this.loginObserver.setActiveModel(login);
  }

  setCommitter(committer) {
    const changed = !this.committer.matches(committer);
    this.committer = committer;
    if (changed) {
      this.finalize();
    }
  }

  didUpdate() {
    this.emitter.emit('did-update', this.getUsers());
  }

  onDidUpdate(callback) {
    return this.emitter.on('did-update', callback);
  }

  getUsers() {
    return this.users;
  }
}
exports.default = UserStore;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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